'''Object classes for HapticMaster

Classes
-------
Object
Block
Sphere
FlatPlane
Cylinder
Torus

Functions
---------
get_stiffness()
set_stiffness(value)
get_dampfactor()
set_dampfactor(value)
get_no_pull()
set_no_pull(value)
get_tang_damping()
set_tang_damping(value)
get_damping_forcemax()
set_damping_forcemax(value)
get_friction()
set_friction(value)
get_ejection_velmax()
set_ejection_velmax(value)
get_ejection_damping()
set_ejection_damping(value)
get_outward_forcemax()
set_outward_forcemax(value)
get_powermax()
get_powermax(value)

create()
get_size()
set_size(value)

'''

from dataclasses import dataclass
import logging
from haptic_master.base import Base
import numpy.typing as npt


@dataclass(frozen=True, slots=True)
class Object(Base):
    '''A class for haptic objects

    Methods
    -------
    get_stiffness()
        Get the stiffness of the object's surface from the robot
    set_stiffness(value)
        Set the stiffness of the object's surface on the robot
    get_dampfactor()
        Get the damping factor coefficient of the object's surface from the robot
    set_dampfactor(value)
        Set the damping factor coefficient of the object's surface on the robot
    get_no_pull()
        Get the pulling condition of the surface during contact from the robot
    set_no_pull(value)
        Set the pulling condition of the surface during contact on the robot
    get_tang_damping()
        Get the tangential damping of the object's surface from the robot
    set_tang_damping(value)
        Set the tangential damping of the object's surface on the robot
    get_damping_forcemax()
        Get the maximum damping force of the object's surface from the robot
    set_damping_forcemax(value)
        Set the maximum damping force of the object's surface on the robot
    get_friction()
        Get the friction coefficient of the object's surface from the robot
    set_friction(value)
        Set the friction coefficient of the object's surface on the robot
    get_ejection_velmax()
        Get the maximum ejection velocity of the object's surface from the robot
    set_ejection_velmax(value)
        Set the maximum ejection velocity of the object's surface on the robot
    get_ejection_damping()
        Get the ejection damping of the object from the robot
    set_ejection_damping(value)
        Set the ejection damping of the object on the robot
    get_outward_forcemax()
        Get the maximum outward force of the object from the robot
    set_outward_forcemax(value)
        Set the maximum outward force of the object on the robot
    get_powermax()
        Get the maximum power generated by the surface from the robot
    get_powermax(value)
        Set the maximum power generated by the surface on the robot

    '''

    def get_stiffness(self) -> float:
        '''Get the spring coefficient from the robot

        Returns
        -------
        float: Stiffness of the object [N/m]

        '''

        msg = f'get {self.name} stiffness'

        return float(self.robot.send_message(msg))

    def set_stiffness(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} stiffness {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shapes\'s stiffness set' in response

    def get_dampfactor(self) -> float:
        '''Get the damping factor of the object from the robot

        Returns
        -------
        float: Damping factor [non-dimensional]

        '''

        msg = f'get {self.name} dampfactor'

        return float(self.robot.send_message(msg))

    def set_dampfactor(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} dampfactor {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s damp factor set' in response

    def get_no_pull(self) -> bool:
        '''Get the object's pulling condition during contact from the robot

        Returns
        -------
        float: Pulling condition [non-dimensional]

        '''

        msg = f'get {self.name} no_pull'

        return self.robot.string_to_bool(self.robot.send_message(msg))

    def set_no_pull(self, value: bool) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} no_pull {str(value).lower()}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s no pull set' in response

    def get_tang_damping(self) -> float:
        '''Get the tangential damping of the object during contact from the robot

        Returns
        -------
        float: Tangential damping [non-dimensional]

        '''

        msg = f'get {self.name} tang_damping'

        return float(self.robot.send_message(msg))

    def set_tang_damping(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} tang_damping {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s tangential damping set' in response

    def get_damping_forcemax(self) -> float:
        '''Get the maximum force by the object during contact from the robot

        Returns
        -------
        float: Maximum force [N]

        '''

        msg = f'get {self.name} damping_forcemax'

        return float(self.robot.send_message(msg))

    def set_damping_forcemax(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} damping_forcemax {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s damping force max set' in response

    def get_friction(self) -> float:
        '''Get the friction of the object from the robot

        Returns
        -------
        float: Friction coefficient [non-dimensional]

        '''

        msg = f'get {self.name} friction'

        return float(self.robot.send_message(msg))

    def set_friction(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} friction {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s friction mu set' in response

    def get_ejection_velmax(self) -> float:
        '''Get the maximum ejection velocity when inside the object from the robot

        Returns
        -------
        float: Maximum velocity [m/s]

        '''

        msg = f'get {self.name} ejection_velmax'

        return float(self.robot.send_message(msg))

    def set_ejection_velmax(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} ejection_velmax {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s ejection velocity max set' in response

    def get_ejection_damping(self) -> float:
        '''Get the ejection damping when inside the object from the robot

        Returns
        -------
        float: Damping coefficient [Ns/m]

        '''

        msg = f'get {self.name} ejection_damping'

        return float(self.robot.send_message(msg))

    def set_ejection_damping(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} ejection_damping {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s ejection velocity damping set' in response

    def get_outward_forcemax(self) -> float:
        '''Get the maximum force outwards from the robot

        Returns
        -------
        float: Maximum force [N]

        '''

        msg = f'get {self.name} outward_forcemax'

        return float(self.robot.send_message(msg))

    def set_outward_forcemax(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} outward_forcemax {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s outward force max set' in response

    def get_powermax(self) -> float:
        '''Get the maximum power generated by the of the surface from the robot

        Returns
        -------
        float: Maximum power [W]

        '''

        msg = f'get {self.name} powermax'

        return float(self.robot.send_message(msg))

    def set_powermax(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} powermax {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Shape\'s power max set' in response


@dataclass(frozen=True, slots=True)
class Block(Object):
    '''A class for block objects

    Methods
    -------
    create()
        Create a block
    get_size()
        Get the size of the block
    set_size(value)
        Set the size of the block

    '''

    def create(self) -> bool:
        '''Create a block object on the robot

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'create block {self.name}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return f'Effect block with name {self.name} created' in response

    def get_size(self) -> npt.NDArray:
        '''Get the dimensions of the block from the robot

        Returns
        -------
        npt.NDArray: Size vector [m]

        '''

        msg = f'get {self.name} size'

        return self.robot.string_to_array(self.robot.send_message(msg))

    def set_size(self, value: npt.NDArray) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} size {str(value).replace(' ', '')}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Block\'s size set' in response


@dataclass(frozen=True, slots=True)
class Sphere(Object):
    '''A class for sphere objects

    Methods
    -------
    create()
        Create a sphere
    get_radius()
        Get the radius of the sphere
    set_radius(value)
        Set the radius of the sphere

    '''

    def create(self) -> bool:
        '''Create a sphere object on the robot

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'create sphere {self.name}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return f'Effect sphere with name {self.name} created' in response

    def get_radius(self) -> float:
        '''Get the radius of the sphere from the robot

        Returns
        -------
        float: Radius [m]

        '''

        msg = f'get {self.name} radius'

        return float(self.robot.send_message(msg))

    def set_radius(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} radius {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Sphere\'s radius set' in response


@dataclass(frozen=True, slots=True)
class FlatPlane(Object):
    '''A class for flat plane objects

    Methods
    -------
    create()
        Create a flat plane
    get_normal()
        Get the normal of the flat plane
    set_normal(value)
        Set the normal of the flat plane

    '''

    def create(self) -> bool:
        '''Create a flat plane object on the robot

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'create flatplane {self.name}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return f'Effect flatplane with name {self.name} created' in response

    def get_normal(self) -> npt.NDArray:
        '''Get the normal of the flat surface from the robot

        Returns
        -------
        npt.NDArray: Unit plane normal vector [non-dimensional]

        '''

        msg = f'get {self.name} normal'

        return self.robot.string_to_array(self.robot.send_message(msg))

    def set_normal(self, value: npt.NDArray) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (npt.NDArray): Normal of the plane

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} normal {str(value).replace(' ', '')}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Flat plane\'s normal set' in response


@dataclass(frozen=True, slots=True)
class Cylinder(Object):
    '''A class for cylinder objects

    Methods
    -------
    create()
        Create a cylinder
    get_radius()
        Get the radius of the cylinder
    set_radius(value)
        Set the radius of the cylinder
    get_length()
        Get the length of the cylinder
    set_length(value)
        Set the length of the cylinder

    '''

    def create(self) -> bool:
        '''Create a cylinder object on the robot

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'create cylinder {self.name}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return f'Effect cylinder with name {self.name} created' in response

    def get_radius(self) -> float:
        '''Get the radius of the cylinder from the robot

        Returns
        -------
        float: Radius [m]

        '''

        msg = f'get {self.name} radius'

        return float(self.robot.send_message(msg))

    def set_radius(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} radius {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Cylinder\'s radius set' in response

    def get_length(self) -> float:
        '''Get the length of the cylinder from the robot

        Returns
        -------
        float: Length [m]

        '''

        msg = f'get {self.name} length'

        return float(self.robot.send_message(msg))

    def set_length(self, value: float) -> bool:
        '''Set the stiffness of the object on the robot

        Parameters
        ----------
        value (float): Stiffness of the object [N/m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} length {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Cylinder\'s length set' in response


@dataclass(frozen=True, slots=True)
class Torus(Object):
    '''A class for torus objects

    Methods
    -------
    create()
        Create a torus
    get_ringradius()
        Get the ring radius of the torus
    set_ringradius(value)
        Set the ring radius of the torus
    get_tuberadius()
        Get the tube radius of the torus
    set_tuberadius(value)
        Set the tube radius of the torus

    '''

    def create(self) -> bool:
        '''Create a torus object on the robot

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'create torus {self.name}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return f'Effect torus with name {self.name} created' in response

    def get_ringradius(self) -> float:
        '''Get the ring (horizontal) radius of the torus from the robot

        Returns
        -------
        float: Radius [m]

        '''

        msg = f'get {self.name} ringradius'

        return float(self.robot.send_message(msg))

    def set_ringradius(self, value: float) -> bool:
        '''Set the ring (horizontal) radius of the torus on the robot

        Parameters
        ----------
        value (float): Radius [m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} ringradius {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Torus\'s ring radius set' in response

    def get_tuberadius(self) -> float:
        '''Get the tube (vertical) radius of the torus from the robot

        Returns
        -------
        float: Radius [m]

        '''

        msg = f'get {self.name} tuberadius'

        return float(self.robot.send_message(msg))

    def set_tuberadius(self, value: float) -> bool:
        '''Set the tube (vertical) radius of the torus on the robot

        Parameters
        ----------
        value (float): Radius [m]

        Returns
        -------
        bool: True if successful, False otherwise

        '''

        msg = f'set {self.name} tuberadius {value}'

        response = self.robot.send_message(msg)

        logging.info(response)

        return 'Torus\'s tube radius set' in response

