"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancedServiceBase = void 0;
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
/**
 * The base class for ApplicationLoadBalancedEc2Service and ApplicationLoadBalancedFargateService services.
 */
class ApplicationLoadBalancedServiceBase extends cdk.Construct {
    /**
     * Constructs a new instance of the ApplicationLoadBalancedServiceBase class.
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        this.desiredCount = props.desiredCount || 1;
        const internetFacing = props.publicLoadBalancer !== undefined ? props.publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        const loadBalancer = props.loadBalancer !== undefined ? props.loadBalancer
            : new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', lbProps);
        if (props.certificate !== undefined && props.protocol !== undefined && props.protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            throw new Error('The HTTPS protocol must be used when a certificate is given');
        }
        const protocol = props.protocol !== undefined ? props.protocol :
            (props.certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
        const targetProps = {
            port: 80,
        };
        this.listener = loadBalancer.addListener('PublicListener', {
            protocol,
            port: props.listenerPort,
            open: (_a = props.openListener) !== null && _a !== void 0 ? _a : true,
        });
        this.targetGroup = this.listener.addTargets('ECS', targetProps);
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            if (typeof props.domainName === 'undefined' || typeof props.domainZone === 'undefined') {
                throw new Error('A domain name and zone is required when using the HTTPS protocol');
            }
            if (props.certificate !== undefined) {
                this.certificate = props.certificate;
            }
            else {
                this.certificate = new aws_certificatemanager_1.Certificate(this, 'Certificate', {
                    domainName: props.domainName,
                    validation: aws_certificatemanager_1.CertificateValidation.fromDns(props.domainZone),
                });
            }
        }
        if (this.certificate !== undefined) {
            this.listener.addCertificates('Arns', [aws_elasticloadbalancingv2_1.ListenerCertificate.fromCertificateManager(this.certificate)]);
        }
        let domainName = loadBalancer.loadBalancerDnsName;
        if (typeof props.domainName !== 'undefined') {
            if (typeof props.domainZone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            const record = new aws_route53_1.ARecord(this, 'DNS', {
                zone: props.domainZone,
                recordName: props.domainName,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
            domainName = record.domainName;
        }
        if (loadBalancer instanceof aws_elasticloadbalancingv2_1.ApplicationLoadBalancer) {
            this._applicationLoadBalancer = loadBalancer;
        }
        new cdk.CfnOutput(this, 'LoadBalancerDNS', { value: loadBalancer.loadBalancerDnsName });
        new cdk.CfnOutput(this, 'ServiceURL', { value: protocol.toLowerCase() + '://' + domainName });
    }
    /**
     * The Application Load Balancer for the service.
     */
    get loadBalancer() {
        if (!this._applicationLoadBalancer) {
            throw new Error('.loadBalancer can only be accessed if the class was constructed with an owned, not imported, load balancer');
        }
        return this._applicationLoadBalancer;
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = cdk.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Adds service as a target of the target group.
     */
    addServiceAsTarget(service) {
        this.targetGroup.addTarget(service);
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.ApplicationLoadBalancedServiceBase = ApplicationLoadBalancedServiceBase;
//# sourceMappingURL=data:application/json;base64,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