"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isComplete = exports.onTimeout = exports.handler = void 0;
/* eslint-disable no-console */
const assertion_1 = require("./assertion");
const sdk_1 = require("./sdk");
const types = require("./types");
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify({ ...event, ResponseURL: '...' })}`);
    const provider = createResourceHandler(event, context);
    try {
        // if we are deleting the custom resource, just respond
        // with 'SUCCESS' since there is nothing to do.
        if (event.RequestType === 'Delete') {
            await provider.respond({
                status: 'SUCCESS',
                reason: 'OK',
            });
            return;
        }
        const result = await provider.handle();
        // if there is a `stateMachineArn` then we have already started a state machine
        // execution and the rest will be handled there
        if ('stateMachineArn' in event.ResourceProperties) {
            console.info('Found "stateMachineArn", waiter statemachine started');
            return;
        }
        else if ('expected' in event.ResourceProperties) {
            console.info('Found "expected", testing assertions');
            const actualPath = event.ResourceProperties.actualPath;
            // if we are providing a path to make the assertion at, that means that we have
            // flattened the response, otherwise the path to assert against in the entire response
            const actual = actualPath ? result[`apiCallResponse.${actualPath}`] : result.apiCallResponse;
            const assertion = new assertion_1.AssertionHandler({
                ...event,
                ResourceProperties: {
                    ServiceToken: event.ServiceToken,
                    actual,
                    expected: event.ResourceProperties.expected,
                },
            }, context);
            try {
                const assertionResult = await assertion.handle();
                await provider.respond({
                    status: 'SUCCESS',
                    reason: 'OK',
                    // return both the result of the API call _and_ the assertion results
                    data: {
                        ...assertionResult,
                        ...result,
                    },
                });
                return;
            }
            catch (e) {
                await provider.respond({
                    status: 'FAILED',
                    reason: e.message ?? 'Internal Error',
                });
                return;
            }
        }
        await provider.respond({
            status: 'SUCCESS',
            reason: 'OK',
            data: result,
        });
    }
    catch (e) {
        await provider.respond({
            status: 'FAILED',
            reason: e.message ?? 'Internal Error',
        });
        return;
    }
    return;
}
exports.handler = handler;
/**
 * Invoked by the waiter statemachine when the retry
 * attempts are exhausted
 */
async function onTimeout(timeoutEvent) {
    // the event payload is passed through the `errorMessage` in the state machine
    // timeout event
    const isCompleteRequest = JSON.parse(JSON.parse(timeoutEvent.Cause).errorMessage);
    const provider = createResourceHandler(isCompleteRequest, standardContext);
    await provider.respond({
        status: 'FAILED',
        reason: 'Operation timed out: ' + JSON.stringify(isCompleteRequest),
    });
}
exports.onTimeout = onTimeout;
/**
 * Invoked by the waiter statemachine when the user is waiting for a specific
 * result.
 *
 * If the result of the assertion is not successful then it will throw an error
 * which will cause the state machine to try again
 */
async function isComplete(event, context) {
    console.log(`Event: ${JSON.stringify({ ...event, ResponseURL: '...' })}`);
    const provider = createResourceHandler(event, context);
    try {
        const result = await provider.handleIsComplete();
        const actualPath = event.ResourceProperties.actualPath;
        if (result) {
            const actual = actualPath ? result[`apiCallResponse.${actualPath}`] : result.apiCallResponse;
            if ('expected' in event.ResourceProperties) {
                const assertion = new assertion_1.AssertionHandler({
                    ...event,
                    ResourceProperties: {
                        ServiceToken: event.ServiceToken,
                        actual,
                        expected: event.ResourceProperties.expected,
                    },
                }, context);
                const assertionResult = await assertion.handleIsComplete();
                if (!(assertionResult?.failed)) {
                    await provider.respond({
                        status: 'SUCCESS',
                        reason: 'OK',
                        data: {
                            ...assertionResult,
                            ...result,
                        },
                    });
                    return;
                }
                else {
                    console.log(`Assertion Failed: ${JSON.stringify(assertionResult)}`);
                    throw new Error(JSON.stringify(event));
                }
            }
            await provider.respond({
                status: 'SUCCESS',
                reason: 'OK',
                data: result,
            });
        }
        else {
            console.log('No result');
            throw new Error(JSON.stringify(event));
        }
        return;
    }
    catch (e) {
        console.log(e);
        throw new Error(JSON.stringify(event));
    }
}
exports.isComplete = isComplete;
function createResourceHandler(event, context) {
    if (event.ResourceType.startsWith(types.SDK_RESOURCE_TYPE_PREFIX)) {
        return new sdk_1.AwsApiCallHandler(event, context);
    }
    else if (event.ResourceType.startsWith(types.ASSERT_RESOURCE_TYPE)) {
        return new assertion_1.AssertionHandler(event, context);
    }
    else {
        throw new Error(`Unsupported resource type "${event.ResourceType}`);
    }
}
const standardContext = {
    getRemainingTimeInMillis: () => 90000,
};
//# sourceMappingURL=data:application/json;base64,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