"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionHandler = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const helpers_internal_1 = require("aws-cdk-lib/assertions/lib/helpers-internal");
const base_1 = require("./base");
class AssertionHandler extends base_1.CustomResourceHandler {
    async processEvent(request) {
        let actual = decodeCall(request.actual);
        const expected = decodeCall(request.expected);
        let result;
        const matcher = new MatchCreator(expected).getMatcher();
        console.log(`Testing equality between ${JSON.stringify(request.actual)} and ${JSON.stringify(request.expected)}`);
        const matchResult = matcher.test(actual);
        matchResult.finished();
        if (matchResult.hasFailed()) {
            result = {
                failed: true,
                assertion: JSON.stringify({
                    status: 'fail',
                    message: matchResult.renderMismatch(),
                }),
            };
            if (request.failDeployment) {
                throw new Error(result.assertion);
            }
        }
        else {
            result = {
                assertion: JSON.stringify({
                    status: 'success',
                }),
            };
        }
        return result;
    }
}
exports.AssertionHandler = AssertionHandler;
class MatchCreator {
    constructor(obj) {
        this.parsedObj = {
            matcher: obj,
        };
    }
    /**
     * Return a Matcher that can be tested against the actual results.
     * This will convert the encoded matchers into their corresponding
     * assertions matcher.
     *
     * For example:
     *
     * ExpectedResult.objectLike({
     *   Messages: [{
     *     Body: Match.objectLike({
     *       Elements: Match.arrayWith([{ Asdf: 3 }]),
     *       Payload: Match.serializedJson({ key: 'value' }),
     *     }),
     *   }],
     * });
     *
     * Will be encoded as:
     * {
     *   $ObjectLike: {
     *     Messages: [{
     *       Body: {
     *         $ObjectLike: {
     *           Elements: {
     *             $ArrayWith: [{ Asdf: 3 }],
     *           },
     *           Payload: {
     *             $SerializedJson: { key: 'value' }
     *           }
     *         },
     *       },
     *     }],
     *   },
     * }
     *
     * Which can then be parsed by this function. For each key (recursively)
     * the parser will check if the value has one of the encoded matchers as a key
     * and if so, it will set the value as the Matcher. So,
     *
     * {
     *   Body: {
     *     $ObjectLike: {
     *       Elements: {
     *         $ArrayWith: [{ Asdf: 3 }],
     *       },
     *       Payload: {
     *         $SerializedJson: { key: 'value' }
     *       }
     *     },
     *   },
     * }
     *
     * Will be converted to
     * {
     *   Body: Match.objectLike({
     *     Elements: Match.arrayWith([{ Asdf: 3 }]),
     *     Payload: Match.serializedJson({ key: 'value' }),
     *   }),
     * }
     */
    getMatcher() {
        try {
            const final = JSON.parse(JSON.stringify(this.parsedObj), function (_k, v) {
                const nested = Object.keys(v)[0];
                switch (nested) {
                    case '$ArrayWith':
                        return helpers_internal_1.Match.arrayWith(v[nested]);
                    case '$ObjectLike':
                        return helpers_internal_1.Match.objectLike(v[nested]);
                    case '$StringLike':
                        return helpers_internal_1.Match.stringLikeRegexp(v[nested]);
                    case '$SerializedJson':
                        return helpers_internal_1.Match.serializedJson(v[nested]);
                    default:
                        return v;
                }
            });
            if (helpers_internal_1.Matcher.isMatcher(final.matcher)) {
                return final.matcher;
            }
            return helpers_internal_1.Match.exact(final.matcher);
        }
        catch {
            return helpers_internal_1.Match.exact(this.parsedObj.matcher);
        }
    }
}
function decodeCall(call) {
    if (!call) {
        return undefined;
    }
    try {
        const parsed = JSON.parse(call);
        return parsed;
    }
    catch (e) {
        return call;
    }
}
//# sourceMappingURL=data:application/json;base64,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