import os
import subprocess
import uuid
from datetime import datetime, timezone
from pathlib import Path

import typer
from rich.console import Console
from rich.progress import Progress, SpinnerColumn, TextColumn

from .models import (
    GeneralConfig,
    Iperf3JsonResult,
    IperfClientConfig,
    IperfResult,
    IperfServerConfig,
)


class Iperf3Runner:
    """Main class for running iperf3 commands and processing results."""

    def __init__(self, iperf3_path: Path):
        self.iperf3_path = str(iperf3_path)
        self.console = Console()
        try:
            self._validate_iperf3_availability()
        except RuntimeError as e:
            self.console.print(f"[red]Error: {e}[/red]")
            raise typer.Exit(1) from e

    def build_command(self, config: IperfClientConfig | IperfServerConfig) -> list[str]:
        cmd = [self.iperf3_path]
        if isinstance(config, IperfClientConfig):
            if not config.server_host:
                raise ValueError("Client mode requires server_host to be set")
            cmd.extend(["-c", config.server_host])
        elif isinstance(config, IperfServerConfig):
            cmd.append("-s")
        cmd.extend(config.build_cli_args())
        return cmd

    def run(
        self,
        config: IperfClientConfig | IperfServerConfig,
        general_config: GeneralConfig,
        timeout: int | None = None,
    ) -> IperfResult:
        # Determine the description based on config type
        if isinstance(config, IperfClientConfig):
            description = "Running iperf3 client test..."
        else:
            description = "Running iperf3 server..."

        with Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            console=self.console,
        ) as progress:
            task = progress.add_task(description, total=None)
            try:
                run_id = general_config.run_id or str(uuid.uuid4())
                output_dir = self._create_output_directory(general_config, run_id)
                cmd = self.build_command(config)
                self._save_command(output_dir, cmd)
                start_time = datetime.now(timezone.utc)

                process_result = self._run_subprocess(
                    cmd, cwd=output_dir, timeout=timeout
                )
                json_results = None
                if config.json_output:
                    json_results = Iperf3JsonResult.model_validate_json(
                        process_result.stdout
                    )
                result = IperfResult(
                    run_id=run_id,
                    start_time=start_time,
                    config=config,
                    output_directory=output_dir,
                    stdout=process_result.stdout,
                    stderr=process_result.stderr,
                    return_code=process_result.returncode,
                    end_time=datetime.now(timezone.utc),
                    general_config=general_config,
                    json_results=json_results,
                )

                result.save()
                progress.update(task, description="✅ Completed")
                return result
            except Exception as e:
                progress.update(task, description=f"❌ Failed: {e}")
                self.console.print(f"[red]Error: {e}[/red]")
                raise typer.Exit(1) from e

    def _validate_iperf3_availability(self) -> None:
        try:
            result = self._run_subprocess([self.iperf3_path, "--version"], timeout=10)
            if result.returncode != 0:
                raise RuntimeError(f"iperf3 is not working: {result.stderr}")
        except FileNotFoundError as err:
            raise RuntimeError(f"iperf3 not found at: {self.iperf3_path}") from err
        except subprocess.TimeoutExpired as err:
            raise RuntimeError("iperf3 version check timed out") from err

    @staticmethod
    def _run_subprocess(
        cmd: list[str], cwd: Path | None = None, timeout: int | None = None
    ) -> subprocess.CompletedProcess:
        return subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=timeout,
            cwd=cwd,
            env=os.environ,
        )

    @staticmethod
    def _save_command(output_dir: Path, cmd: list[str]) -> None:
        (output_dir / "command.txt").write_text(" ".join(cmd))

    @staticmethod
    def _create_output_directory(general_config: GeneralConfig, run_id: str) -> Path:
        base_dir = general_config.output_directory
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        dir_name = f"{timestamp}_{run_id[:8]}"
        output_dir = base_dir / dir_name
        output_dir.mkdir(parents=True, exist_ok=True)
        return output_dir
