# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/utils.ipynb.

# %% auto 0
__all__ = ['generate_daily_series', 'generate_prices_for_series', 'backtest_splits']

# %% ../nbs/utils.ipynb 2
import random
import reprlib
from itertools import chain
from math import ceil, log10
from typing import Optional, Union

import numpy as np
import pandas as pd

# %% ../nbs/utils.ipynb 4
def generate_daily_series(
    n_series: int,
    min_length: int = 50,
    max_length: int = 500,
    n_static_features: int = 0,
    equal_ends: bool = False,
    static_as_categorical: bool = True,
    with_trend: bool = False,
    seed: int = 0,
) -> pd.DataFrame:
    """Generates `n_series` of different lengths in the interval [`min_length`, `max_length`].

    If `n_static_features > 0`, then each serie gets static features with random values.
    If `equal_ends == True` then all series end at the same date."""
    rng = np.random.RandomState(seed)
    random.seed(seed)
    series_lengths = rng.randint(min_length, max_length + 1, n_series)
    total_length = series_lengths.sum()
    n_digits = ceil(log10(n_series))

    dates = pd.date_range("2000-01-01", periods=max_length, freq="D").values
    uids = [
        [f"id_{i:0{n_digits}}"] * serie_length
        for i, serie_length in enumerate(series_lengths)
    ]
    if equal_ends:
        ds = [dates[-serie_length:] for serie_length in series_lengths]
    else:
        ds = [dates[:serie_length] for serie_length in series_lengths]
    y = np.arange(total_length) % 7 + rng.rand(total_length) * 0.5
    series = pd.DataFrame(
        {
            "unique_id": list(chain.from_iterable(uids)),
            "ds": list(chain.from_iterable(ds)),
            "y": y,
        }
    )
    for i in range(n_static_features):
        static_values = np.repeat(rng.randint(0, 100, n_series), series_lengths)
        series[f"static_{i}"] = static_values
        if static_as_categorical:
            series[f"static_{i}"] = series[f"static_{i}"].astype("category")
        if i == 0:
            series["y"] = series["y"] * 0.1 * (1 + static_values)
    series["unique_id"] = series["unique_id"].astype("category")
    series["unique_id"] = series["unique_id"].cat.as_ordered()
    if with_trend:
        coefs = pd.Series(
            rng.rand(n_series), index=[f"id_{i:0{n_digits}}" for i in range(n_series)]
        )
        trends = series.groupby("unique_id").cumcount()
        trends.index = series["unique_id"]
        series["y"] += (coefs * trends).values
    return series

# %% ../nbs/utils.ipynb 15
def generate_prices_for_series(
    series: pd.DataFrame, horizon: int = 7, seed: int = 0
) -> pd.DataFrame:
    rng = np.random.RandomState(seed)
    unique_last_dates = series.groupby("unique_id")["ds"].max().nunique()
    if unique_last_dates > 1:
        raise ValueError("series must have equal ends.")
    if "product_id" not in series:
        raise ValueError("series must have a product_id column.")
    day_offset = pd.tseries.frequencies.Day()
    starts_ends = series.groupby("product_id")["ds"].agg([min, max])
    dfs = []
    for idx, (start, end) in starts_ends.iterrows():
        product_df = pd.DataFrame(
            {
                "product_id": idx,
                "price": rng.rand((end - start).days + 1 + horizon),
            },
            index=pd.date_range(start, end + horizon * day_offset, name="ds"),
        )
        dfs.append(product_df)
    prices_catalog = pd.concat(dfs).reset_index()
    return prices_catalog

# %% ../nbs/utils.ipynb 18
def single_split(
    data: pd.DataFrame,
    i_window: int,
    n_windows: int,
    window_size: int,
    id_col: str,
    time_col: str,
    freq: Union[pd.offsets.BaseOffset, int],
    max_dates: pd.Series,
    step_size: Optional[int] = None,
    input_size: Optional[int] = None,
):
    if step_size is None:
        step_size = window_size
    test_size = window_size + step_size * (n_windows - 1)
    offset = test_size - i_window * step_size
    train_ends = max_dates - offset * freq
    valid_ends = train_ends + window_size * freq
    train_mask = data[time_col].le(train_ends)
    if input_size is not None:
        train_mask &= data[time_col].gt(train_ends - input_size * freq)
    train_sizes = train_mask.groupby(data[id_col], observed=True).sum()
    if train_sizes.eq(0).any():
        ids = reprlib.repr(train_sizes[train_sizes.eq(0)].index.tolist())
        raise ValueError(f"The following series are too short for the window: {ids}")
    valid_mask = data[time_col].gt(train_ends) & data[time_col].le(valid_ends)
    cutoffs = (
        train_ends.set_axis(data[id_col])
        .groupby(id_col, observed=True)
        .head(1)
        .rename("cutoff")
    )
    return cutoffs, train_mask, valid_mask

# %% ../nbs/utils.ipynb 19
def backtest_splits(
    data: pd.DataFrame,
    n_windows: int,
    window_size: int,
    id_col: str,
    time_col: str,
    freq: Union[pd.offsets.BaseOffset, int],
    step_size: Optional[int] = None,
    input_size: Optional[int] = None,
):
    max_dates = data.groupby(id_col, observed=True)[time_col].transform("max")
    for i in range(n_windows):
        cutoffs, train_mask, valid_mask = single_split(
            data,
            i_window=i,
            n_windows=n_windows,
            window_size=window_size,
            id_col=id_col,
            time_col=time_col,
            freq=freq,
            max_dates=max_dates,
            step_size=step_size,
            input_size=input_size,
        )
        train, valid = data[train_mask], data[valid_mask]
        yield cutoffs, train, valid

# %% ../nbs/utils.ipynb 22
class PredictionIntervals:
    """Class for storing prediction intervals metadata information."""

    def __init__(
        self,
        n_windows: int = 2,
        window_size: int = 1,
        method: str = "conformal_distribution",
    ):
        if n_windows < 2:
            raise ValueError(
                "You need at least two windows to compute conformal intervals"
            )
        allowed_methods = ["conformal_error", "conformal_distribution"]
        if method not in allowed_methods:
            raise ValueError(f"method must be one of {allowed_methods}")
        self.n_windows = n_windows
        self.window_size = window_size
        self.method = method
