# -*- coding: utf-8 -*-

import numpy as np
import pandas as pd
import math
import cmath

import scipy.special


def arterial_model(qb, temp, ti):
    # Calculate ha, hv by using the 138 arterial tree model.
    b_den = 1.05  # 血液密度 [g/ml]
    b_vis = 0.04  # 血液黏性 mpa.s
    Wvi0 = 15 * (np.pi / 180)  # 血管壁黏性 mpa.s
    poisson_ratio = 0.5  # 泊松比
    reflection_coefficient = 0.8  # 名义反射系数

    L = [4.0, 2.0, 3.4, 8.9, 3.9, 3.4, 15.0, 6.8, 14.8, 8.9, 5.2, 9.9, 14.8, 6.8, 15.0, 5.0,
         6.1, 10.0, 5.0, 3.1, 5.2, 8.9, 5.0, 10.0, 6.1, 5.0, 3.0, 5.6, 5.0, 8.0, 5.9, 11.8, 4.0, 5.2, 4.0, 11.8,
         5.9, 8.0, 5.0, 5.6, 3.0, 6.3, 3.0, 5.9, 4.0, 3.0, 5.9, 3.0, 1.0, 5.3, 3.0, 5.9, 3.0, 4.0, 5.9, 3.0, 6.3, 15.0,
         6.3, 5.9, 7.1, 6.3, 6.6, 3.2, 5.3, 5.9, 3.2, 5.9, 6.3, 15.0, 6.3, 5.0, 4.0, 5.0, 5.3, 5.0, 4.0, 5.0, 6.3, 5.0,
         4.6, 5.8,
         5.0, 5.8, 4.6, 5.0, 6.7, 11.7, 8.3, 5.0, 5.0, 8.3, 11.7, 6.7, 8.5, 7.9, 11.7, 6.1, 6.1, 11.7, 7.9, 8.5, 8.5,
         12.7,
         12.6, 12.6, 12.7, 8.5, 12.7, 12.7, 9.4, 9.4, 9.4, 9.4, 2.5, 16.1, 16.1, 2.5, 15.0, 15.9, 22.1, 16.1, 15.9,
         15.0,
         15.0, 15.9, 15.9, 15.0, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6]  # 138段动脉血管的长度(cm)
    R = [1.45, 1.12, 0.42, 0.37, 1.07, 0.62, 0.1, 0.4, 0.19, 0.37, 1.0, 0.37, 0.19, 0.4, 0.1, 0.1, 0.35, 0.2, 0.1,
         0.37, 0.95, 0.37, 0.1, 0.2, 0.35, 0.1, 0.15, 0.31, 0.1, 0.15, 0.18, 0.15, 0.07, 0.95, 0.07, 0.15, 0.18, 0.15,
         0.1, 0.31, 0.15, 0.28, 0.1, 0.13, 0.1, 0.06, 0.08, 0.07, 0.39, 0.87, 0.07, 0.08, 0.06, 0.1, 0.13, 0.1, 0.28,
         0.15, 0.26, 0.08, 0.18, 0.28, 0.22, 0.26, 0.57, 0.43, 0.26, 0.08, 0.26, 0.15, 0.25, 0.07, 0.06, 0.07, 0.57,
         0.07, 0.06, 0.07, 0.25, 0.06, 0.24, 0.52, 0.16, 0.52, 0.24, 0.06, 0.21, 0.16, 0.29, 0.2, 0.2, 0.29, 0.16, 0.21,
         0.19, 0.09, 0.16, 0.27, 0.27, 0.16, 0.09, 0.19, 0.19, 0.24, 0.23, 0.23, 0.24, 0.19, 0.24, 0.24, 0.2, 0.2,
         0.2, 0.2, 0.13, 0.18, 0.18, 0.13, 0.1, 0.13, 0.18, 0.18, 0.13, 0.1, 0.1, 0.13, 0.13, 0.1, 0.05, 0.05, 0.05,
         0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.05]  # 138段动脉血管的半径(cm)

    h = [0.163, 0.132, 0.067, 0.063, 0.127, 0.086, 0.03, 0.066, 0.045, 0.063, 0.12, 0.063, 0.045, 0.066,
         0.03, 0.03, 0.062, 0.052, 0.03, 0.063, 0.116, 0.063, 0.03, 0.052, 0.062, 0.03, 0.035, 0.057, 0.03,
         0.035, 0.045, 0.042, 0.02, 0.116, 0.02, 0.042, 0.045, 0.035, 0.03, 0.057, 0.035, 0.055, 0.03, 0.039,
         0.03, 0.02, 0.026, 0.02, 0.064, 0.108, 0.02, 0.026, 0.02, 0.03, 0.039, 0.03, 0.055, 0.035, 0.053,
         0.026, 0.045, 0.054, 0.049, 0.053, 0.08, 0.069, 0.053, 0.026, 0.053, 0.035, 0.052, 0.02, 0.02,
         0.02, 0.08, 0.02, 0.02, 0.02, 0.052, 0.02, 0.05, 0.076, 0.043, 0.076, 0.05, 0.02, 0.049, 0.043, 0.055,
         0.04, 0.04, 0.055, 0.043, 0.049, 0.462, 0.028, 0.043, 0.053, 0.053, 0.043, 0.028, 0.462, 0.046, 0.05,
         0.049, 0.049, 0.05, 0.046, 0.05, 0.05, 0.047, 0.047, 0.05, 0.05, 0.039, 0.045, 0.045, 0.039, 0.02, 0.039,
         0.045, 0.045, 0.039, 0.02, 0.02, 0.019, 0.019, 0.02, 0.019, 0.019, 0.019, 0.019, 0.019, 0.019, 0.019, 0.019,
         0.019, 0.019]  # 138段动脉血管壁面厚度(cm)

    Eyan = np.array([4, 4, 4, 4, 4, 4, 8, 4, 8, 4, 4, 4, 8, 4, 8, 8, 4, 8, 8, 4, 4, 4, 8, 8, 4, 8, 16, 4, 16, 16, 8, 8, 8, 4, 8,
            8, 8, 16, 16, 4, 16.0,
            4, 8, 8, 16, 16, 16, 16, 4, 4, 16, 16, 16, 16, 8, 8, 4, 8, 4, 16, 4, 4, 4, 4, 4, 4, 4, 16, 4, 8, 4, 16, 16,
            16, 4, 16, 16.0,
            16, 4, 16, 4, 4, 4, 4, 4, 16, 8, 8, 4, 16, 16, 4, 8, 8, 8, 16, 8, 4, 4, 8, 16, 8, 8, 8, 16, 16, 8, 8, 8, 8,
            8, 8, 4, 4, 16, 16.0,
            16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8]) * (10 ** 6)  # 138段动脉血管壁杨氏模量
    Qb = qb  # 人体各区段各层基础代谢率，W

    temp_mean_skin = 0.07 * temp[3] + 0.35 * temp[35] + 0.14 * temp[19] + 0.05 * temp[27] + 0.19 * temp[59] + 0.13 * temp[67] + 0.07 * temp[75]
    Mt = 7 / 0.0476
    Mb = sum(Qb)
    CObascal = 290000  # ml / h
    if temp[40] > 41.0:
        COdil = 427500
    elif temp[40] >= 37.2:
        COdil = (((temp[40] - 37.2) / (508000 - 400000)) * (41.0 - 37.2)) + 400000
    elif temp[40] > 36.8:
        COdil = (((temp[40] - 36.8) / (427500 - 290000)) * (37.2 - 36.8)) + 290000
    else:
        COdil = 290000

    if temp_mean_skin >= 33.7:
        COcon = 290000
    elif temp_mean_skin > 10.7:
        COcon = ((temp_mean_skin - 10.7) * (290000 - 270000) / (33.7 - 10.7)) + 270000
    else:
        COcon = 270000

    CO = (COdil * COcon) * (Mt / Mb) / 290000
    A = np.empty(138)
    Rb = np.empty(138)
    Cin = np.empty(138)
    c0 = np.empty(138)
    for i in range(len(R)):
        A[i] = (R[i] ** 2) * np.pi
        Rb[i] = (8 * b_vis) / (np.pi * (R[i] ** 4))
        Cin[i] = (3 * np.pi * (R[i] ** 2)) / (2 * Eyan[i] * h[i])
        c0[i] = ((Eyan[i] * h[i]) / (2 * b_den * R[i])) ** 0.5
    Wvi = np.empty(8)
    a = np.empty([138, 8])
    F10 = np.empty([138, 8], dtype=np.complex64)
    c1 = np.empty([138, 8], dtype=np.complex64)
    Z0 = np.empty([138, 8], dtype=np.complex64)
    r = np.empty([138, 8], dtype=np.complex64)
    ZT = np.empty([138, 8], dtype=np.complex64)
    T = np.empty([138, 8], dtype=np.complex64)
    Z = np.empty([138, 8], dtype=np.complex64)
    for n in range(8):
        w = 7.854 * (n + 1)
        # 频率源程序初始值为8.353
        k = 2
        Wvi[n] = Wvi0 * (1 - math.exp(-k * w))
        # 压力相对于壁位移的相位超前，Wvi0为渐进值
        for m in range(138):
            a[m, n] = R[m] * (((w * b_den) / b_vis) ** 0.5)
            F10[m, n] = (2 * scipy.special.jv(1, (a[m, n] * ((0 + 1j) ** 1.5)))) / (
                        ((0 + 1j) ** 1.5) * a[m, n] * scipy.special.jv(0, a[m, n] * ((0 + 1j) ** 1.5)))
            c1[m, n] = c0[m] * (cmath.exp(((0 + 1j) * Wvi[n] / 2)))
            # 波速
            Z0[m, n] = ((b_den * c0[m]) / ((np.pi * (R[m] ** 2)) * ((1 - (poisson_ratio ** 2)) ** 0.5))) * (
                        (1 - F10[m, n]) ** (-0.5)) * cmath.exp(((0 + 1j) * Wvi[n]) / 2)
            r[m, n] = (((0 + 1j) * w) / c0[m]) * (((1 - F10[m, n]) / (1 - (poisson_ratio ** 2))) ** 0.5) * cmath.exp(((0 - 1j) * Wvi[n]) / 2)
        for mm in range(128, 138):
            r[mm, n] = 0.975
        # 计算输入阻抗
        # 复数
        i = 0 + 1j
        # 第138区段
        ZT[137, n] = Z0[137, n] + (1 / ((1 / Rb[137]) + (i * w * Cin[137])))
        T[137, n] = 0.8  # 波反射，对于末梢为0.8
        Z[137, n] = Z0[137, n] * ((1 + (T[137, n] * cmath.exp(-2 * r[137, 0] * L[137]))) / (
                    1 - (T[137, n] * cmath.exp(-2 * r[137, 0] * L[137]))))
        # 第137-129区段, 公式同上
        for mm in [136, 135, 134, 133, 132, 131, 130, 129, 128]:
            ZT[mm, n] = Z0[mm, n] + (1 / ((1 / Rb[mm]) + (i * w * Cin[mm])))
            T[mm, n] = 0.8  # 波反射，对于末梢为0.8
            Z[mm, n] = Z0[mm, n] * ((1 + (T[mm, n] * cmath.exp(-2 * r[mm, 0] * L[mm]))) / (
                        1 - (T[mm, n] * cmath.exp(-2 * r[mm, 0] * L[mm]))))

        # function T[mi, n] = (ZT[mi, n] - Z0[mi, n]) / (ZT[mi, n] + Z0[mi, n])
        def f2(mi):
            T[mi, n] = (ZT[mi, n] - Z0[mi, n]) / (ZT[mi, n] + Z0[mi, n])
            return T[mi, n]

        # function Z[127, n] = Z0[127, n] * ((1 + (T[127, n] * math.exp(-2 * r[127] * L[127]))) / (1 - (T[127, n] * math.exp(-2 * r[127] * L[127]))))
        def f3(mi):
            Z[mi, n] = Z0[mi, n] * ((1 + (T[mi, n] * cmath.exp(-2 * r[mi, 0] * L[mi]))) / (
                    1 - (T[mi, n] * cmath.exp(-2 * r[mi, 0] * L[mi]))))
            return Z[mi, n]

        # 第128区段
        ZT[127, n] = Z0[127, n] + (1 / ((1 / Rb[127]) + (i * w * Cin[127])))
        T[127, n] = (ZT[127, n] - Z0[127, n]) / (ZT[127, n] + Z0[127, n])
        Z[127, n] = Z0[127, n] * ((1 + (T[127, n] * cmath.exp(-2 * r[127, 0] * L[127]))) / (1 - (T[127, n] * cmath.exp(-2 * r[127, 0] * L[127]))))
        # 第124区段
        ZT[123, n] = Z[127, n]
        T[123, n] = f2(123)
        Z[123, n] = f3(123)
        # 第127区段
        ZT[126, n] = Z0[126, n] + (1 / ((1 / Rb[126]) + (i * w * Cin[126])))
        T[126, n] = f2(126)
        Z[126, n] = f3(126)
        # 第123区段
        ZT[122, n] = Z[126, n]
        T[122, n] = f2(122)
        Z[122, n] = f3(122)
        # 第118区段
        ZT[117, n] = (Z[122, n] * Z[123, n]) / (Z[122, n] + Z[123, n])
        T[117, n] = f2(117)
        Z[117, n] = f3(117)
        # 第122区段
        ZT[121, n] = Z0[121, n] + (1 / ((1 / Rb[121]) + (i * w * Cin[121])))
        T[121, n] = f2(121)
        Z[121, n] = f3(121)
        # 第117区段
        ZT[116, n] = Z[121, n]
        T[116, n] = f2(116)
        Z[116, n] = f3(116)
        # 第114区段
        ZT[113, n] = (Z[116, n] * Z[117, n]) / (Z[116, n] + Z[117, n])
        T[113, n] = f2(113)
        Z[113, n] = f3(113)
        # 第112区段
        ZT[111, n] = Z[113, n]
        T[111, n] = f2(111)
        Z[111, n] = f3(111)
        # 第110区段
        ZT[109, n] = Z[111, n]
        T[109, n] = f2(109)
        Z[109, n] = f3(109)
        # 第107区段
        ZT[106, n] = Z[109, n]
        T[106, n] = f2(106)
        Z[106, n] = f3(106)
        # 第106区段
        ZT[105, n] = Z0[105, n] + (1 / ((1 / Rb[105]) + (i * w * Cin[105])))
        T[105, n] = f2(105)
        Z[105, n] = f3(105)
        # 第99区段
        ZT[98, n] = (Z[106, n] * Z[105, n]) / (Z[106, n] + Z[105, n])
        T[98, n] = f2(98)
        Z[98, n] = f3(98)
        # 第92区段
        ZT[91, n] = Z[98, n]
        T[91, n] = f2(91)
        Z[91, n] = f3(91)
        # 第91区段
        ZT[90, n] = Z0[90, n] + (1 / ((1 / Rb[90]) + (i * w * Cin[90])))
        T[90, n] = f2(90)
        Z[90, n] = f3(90)
        # 第84区段
        ZT[83, n] = (Z[90, n] * Z[91, n]) / (Z[90, n] + Z[91, n])
        T[83, n] = f2(83)
        Z[83, n] = f3(83)
        # 第125区段
        ZT[124, n] = Z0[124, n] + (1 / ((1 / Rb[124]) + (i * w * Cin[124])))
        T[124, n] = f2(124)
        Z[124, n] = f3(124)
        # 第119区段
        ZT[118, n] = Z[124, n]
        T[118, n] = f2(118)
        Z[118, n] = f3(118)
        # 第126区段
        ZT[125, n] = Z0[125, n] + (1 / ((1 / Rb[125]) + (i * w * Cin[125])))
        T[125, n] = f2(125)
        Z[125, n] = f3(125)
        # 第120区段
        ZT[119, n] = Z[125, n]
        T[119, n] = f2(119)
        Z[119, n] = f3(119)
        # 第115区段
        ZT[114, n] = (Z[118, n] * Z[119, n]) / (Z[118, n] + Z[119, n])
        T[114, n] = f2(114)
        Z[114, n] = f3(114)
        # 第121区段
        ZT[120, n] = Z0[120, n] + (1 / ((1 / Rb[120]) + (i * w * Cin[120])))
        T[120, n] = f2(120)
        Z[120, n] = f3(120)
        # 第116区段
        ZT[115, n] = Z[120, n]
        T[115, n] = f2(115)
        Z[115, n] = f3(115)
        # 第113区段
        ZT[112, n] = (Z[114, n] * Z[115, n]) / (Z[114, n] + Z[115, n])
        T[112, n] = f2(112)
        Z[112, n] = f3(112)
        # 第111区段
        ZT[110, n] = Z[112, n]
        T[110, n] = f2(110)
        Z[110, n] = f3(110)
        # 第109区段
        ZT[108, n] = Z[110, n]
        T[108, n] = f2(108)
        Z[108, n] = f3(108)
        # 第104区段
        ZT[103, n] = Z[108, n]
        T[103, n] = f2(103)
        Z[103, n] = f3(103)
        # 第105区段
        ZT[104, n] = Z0[104, n] + (1 / ((1 / Rb[104]) + (i * w * Cin[104])))
        T[104, n] = f2(104)
        Z[104, n] = f3(104)
        # 第98区段
        ZT[97, n] = (Z[103, n] * Z[104, n]) / (Z[103, n] + Z[104, n])
        T[97, n] = f2(97)
        Z[97, n] = f3(97)
        # 第89区段
        ZT[88, n] = Z[97, n]
        T[88, n] = f2(88)
        Z[88, n] = f3(88)
        # 第90区段
        ZT[89, n] = Z0[89, n] + (1 / ((1 / Rb[89]) + (i * w * Cin[89])))
        T[89, n] = f2(89)
        Z[89, n] = f3(89)
        # 第82区段
        ZT[81, n] = (Z[88, n] * Z[89, n]) / (Z[88, n] + Z[89, n])
        T[81, n] = f2(81)
        Z[81, n] = f3(81)
        # 第83区段
        ZT[82, n] = Z0[82, n] + (1 / ((1 / Rb[82]) + (i * w * Cin[82])))
        T[82, n] = f2(82)
        Z[82, n] = f3(82)
        # 第75区段
        ZT[74, n] = (Z[81, n] * Z[82, n] * Z[83, n]) / (
                    (Z[82, n] * Z[83, n]) + (Z[81, n] * Z[83, n]) + (Z[81, n] * Z[82, n]))
        T[74, n] = f2(74)
        Z[74, n] = f3(74)
        # 第65区段
        ZT[64, n] = Z[74, n]
        T[64, n] = f2(64)
        Z[64, n] = f3(64)
        # 第66区段
        ZT[65, n] = Z0[65, n] + (1 / ((1 / Rb[65]) + (i * w * Cin[65])))
        T[65, n] = f2(65)
        Z[65, n] = f3(65)
        # 第67区段
        ZT[66, n] = Z0[66, n] + (1 / ((1 / Rb[66]) + (i * w * Cin[66])))
        T[66, n] = f2(66)
        Z[66, n] = f3(66)
        # 第64区段
        ZT[63, n] = Z0[63, n] + (1 / ((1 / Rb[63]) + (i * w * Cin[63])))
        T[63, n] = f2(63)
        Z[63, n] = f3(63)
        # 第50区段
        ZT[49, n] = (Z[63, n] * Z[64, n] * Z[65, n] * Z[66, n]) / (
                    (Z[63, n] * Z[65, n] * Z[66, n]) + (Z[64, n] * Z[65, n] * Z[66, n]) + (
                        Z[63, n] * Z[64, n] * Z[66, n]) + (Z[63, n] * Z[64, n] * Z[65, n]))
        T[49, n] = f2(49)
        Z[49, n] = f3(49)
        # 第61区段
        ZT[60, n] = Z0[60, n] + (1 / ((1 / Rb[60]) + (i * w * Cin[60])))
        T[60, n] = f2(60)
        Z[60, n] = f3(60)
        # 第62区段
        ZT[61, n] = Z0[61, n] + (1 / ((1 / Rb[61]) + (i * w * Cin[61])))
        T[61, n] = f2(61)
        Z[61, n] = f3(61)
        # 第63区段
        ZT[62, n] = Z0[62, n] + (1 / ((1 / Rb[62]) + (i * w * Cin[62])))
        T[62, n] = f2(62)
        Z[62, n] = f3(62)
        # 第49区段
        ZT[48, n] = (Z[60, n] * Z[61, n] * Z[62, n]) / (
                    (Z[61, n] * Z[62, n]) + (Z[60, n] * Z[62, n]) + (Z[60, n] * Z[61, n]))
        T[48, n] = f2(48)
        Z[48, n] = f3(48)
        # 第34区段
        ZT[33, n] = (Z[48, n] * Z[49, n]) / (Z[48, n] + Z[49, n])
        T[33, n] = f2(33)
        Z[33, n] = f3(33)
        # 第21区段
        ZT[20, n] = Z[33, n]
        T[20, n] = f2(20)
        Z[20, n] = f3(20)
        # 第11区段
        ZT[10, n] = Z[20, n]
        T[10, n] = f2(10)
        Z[10, n] = f3(10)
        # 第5区段
        ZT[4, n] = Z[10, n]
        T[4, n] = f2(4)
        Z[4, n] = f3(4)
        # 第108区段
        ZT[107, n] = (Z[128, n] * Z[129, n] * Z[130, n] * Z[131, n] * Z[132, n]) / (
                    (Z[129, n] * Z[130, n] * Z[131, n] * Z[132, n]) + (
                        Z[128, n] * Z[130, n] * Z[131, n] * Z[132, n]) + (
                                Z[128, n] * Z[129, n] * Z[131, n] * Z[132, n]) + (
                                Z[128, n] * Z[129, n] * Z[130, n] * Z[132, n]) + (
                                Z[128, n] * Z[129, n] * Z[130, n] * Z[131, n]))
        T[107, n] = f2(107)
        Z[107, n] = f3(107)
        # 第102区段
        ZT[101, n] = Z[107, n]
        T[101, n] = f2(101)
        Z[101, n] = f3(101)
        # 第101区段
        ZT[100, n] = Z0[100, n] + (1 / ((1 / Rb[100]) + (i * w * Cin[100])))
        T[100, n] = f2(100)
        Z[100, n] = f3(100)
        # 第94区段
        ZT[93, n] = (Z[100, n] * Z[101, n]) / (Z[100, n] + Z[101, n])
        T[93, n] = f2(93)
        Z[93, n] = f3(93)
        # 第100区段
        ZT[99, n] = Z0[99, n] + (1 / ((1 / Rb[99]) + (i * w * Cin[99])))
        T[99, n] = f2(99)
        Z[99, n] = f3(99)
        # 第93区段
        ZT[92, n] = Z[99, n]
        T[92, n] = f2(92)
        Z[92, n] = f3(92)
        # 第85区段
        ZT[84, n] = (Z[92, n] * Z[93, n]) / (Z[92, n] + Z[93, n])
        T[84, n] = f2(84)
        Z[84, n] = f3(84)
        # 第86区段
        ZT[85, n] = Z0[85, n] + (1 / ((1 / Rb[85]) + (i * w * Cin[85])))
        T[85, n] = f2(85)
        Z[85, n] = f3(85)
        # 第79区段
        ZT[78, n] = (Z[84, n] * Z[85, n]) / (Z[84, n] + Z[85, n])
        T[78, n] = f2(78)
        Z[78, n] = f3(78)
        # 第78区段
        ZT[77, n] = Z0[77, n] + (1 / ((1 / Rb[77]) + (i * w * Cin[77])))
        T[77, n] = f2(77)
        Z[77, n] = f3(77)
        # 第69区段
        ZT[68, n] = (Z[77, n] * Z[78, n]) / (Z[77, n] + Z[78, n])
        T[68, n] = f2(68)
        Z[68, n] = f3(68)
        # 第70区段
        ZT[69, n] = Z0[69, n] + (1 / ((1 / Rb[69]) + (i * w * Cin[69])))
        T[69, n] = f2(69)
        Z[69, n] = f3(69)
        # 第57区段
        ZT[56, n] = (Z[68, n] * Z[69, n]) / (Z[68, n] + Z[69, n])
        T[56, n] = f2(56)
        Z[56, n] = f3(56)
        # 第40区段
        ZT[39, n] = Z[56, n]
        T[39, n] = f2(39)
        Z[39, n] = f3(39)
        # 第41区段
        ZT[40, n] = Z0[40, n] + (1 / ((1 / Rb[40]) + (i * w * Cin[40])))
        T[40, n] = f2(40)
        Z[40, n] = f3(40)
        # 第38区段
        ZT[37, n] = Z0[37, n] + (1 / ((1 / Rb[37]) + (i * w * Cin[37])))
        T[37, n] = f2(37)
        Z[37, n] = f3(37)
        # 第39区段
        ZT[38, n] = Z0[38, n] + (1 / ((1 / Rb[38]) + (i * w * Cin[38])))
        T[38, n] = f2(38)
        Z[38, n] = f3(38)
        # 第25区段
        ZT[24, n] = (Z[37, n] * Z[38, n] * Z[39, n] * Z[40, n]) / (
                    (Z[37, n] * Z[39, n] * Z[40, n]) + (Z[38, n] * Z[39, n] * Z[40, n]) + (
                        Z[37, n] * Z[38, n] * Z[40, n]) + (Z[37, n] * Z[38, n] * Z[39, n]))
        T[24, n] = f2(24)
        Z[24, n] = f3(24)
        # 第26区段
        ZT[25, n] = Z0[25, n] + (1 / ((1 / Rb[25]) + (i * w * Cin[25])))
        T[25, n] = f2(25)
        Z[25, n] = f3(25)
        # 第24区段
        ZT[23, n] = Z0[23, n] + (1 / ((1 / Rb[23]) + (i * w * Cin[23])))
        T[23, n] = f2(23)
        Z[23, n] = f3(23)
        # 第23区段
        ZT[22, n] = Z0[22, n] + (1 / ((1 / Rb[22]) + (i * w * Cin[22])))
        T[22, n] = f2(22)
        Z[22, n] = f3(22)
        # 第14区段
        ZT[13, n] = (Z[22, n] * Z[23, n] * Z[24, n] * Z[25, n]) / (
                    (Z[23, n] * Z[24, n] * Z[25, n]) + (Z[22, n] * Z[24, n] * Z[25, n]) + (
                        Z[22, n] * Z[23, n] * Z[25, n]) + (Z[22, n] * Z[23, n] * Z[24, n]))
        T[13, n] = f2(13)
        Z[13, n] = f3(13)
        # 第15区段
        ZT[14, n] = Z0[14, n] + (1 / ((1 / Rb[14]) + (i * w * Cin[14])))
        T[14, n] = f2(14)
        Z[14, n] = f3(14)
        # 第13区段
        ZT[12, n] = Z0[12, n] + (1 / ((1 / Rb[12]) + (i * w * Cin[12])))
        T[12, n] = f2(12)
        Z[12, n] = f3(12)
        # 第77区段
        ZT[76, n] = Z0[76, n] + (1 / ((1 / Rb[76]) + (i * w * Cin[76])))
        T[76, n] = f2(76)
        Z[76, n] = f3(76)
        # 第76区段
        ZT[75, n] = Z0[75, n] + (1 / ((1 / Rb[75]) + (i * w * Cin[75])))
        T[75, n] = f2(75)
        Z[75, n] = f3(75)
        # 第68区段
        ZT[67, n] = (Z[75, n] * Z[76, n]) / (Z[75, n] + Z[76, n])
        T[67, n] = f2(67)
        Z[67, n] = f3(67)
        # 第55区段
        ZT[54, n] = Z[67, n]
        T[54, n] = f2(54)
        Z[54, n] = f3(54)
        # 第54区段
        ZT[53, n] = Z0[53, n] + (1 / ((1 / Rb[53]) + (i * w * Cin[53])))
        T[53, n] = f2(53)
        Z[53, n] = f3(53)
        # 第56区段
        ZT[55, n] = Z0[55, n] + (1 / ((1 / Rb[55]) + (i * w * Cin[55])))
        T[55, n] = f2(55)
        Z[55, n] = f3(55)
        # 第37区段
        ZT[36, n] = (Z[54, n] * Z[55, n] * Z[53, n]) / (
                    (Z[55, n] * Z[53, n]) + (Z[54, n] * Z[53, n]) + (Z[54, n] * Z[55, n]))
        T[36, n] = f2(36)
        Z[36, n] = f3(36)
        # 第51区段
        ZT[50, n] = Z0[50, n] + (1 / ((1 / Rb[50]) + (i * w * Cin[50])))
        T[50, n] = f2(50)
        Z[50, n] = f3(50)
        # 第52区段
        ZT[51, n] = Z0[51, n] + (1 / ((1 / Rb[51]) + (i * w * Cin[51])))
        T[51, n] = f2(51)
        Z[51, n] = f3(51)
        # 第53区段
        ZT[52, n] = Z0[52, n] + (1 / ((1 / Rb[52]) + (i * 2 * Cin[52])))
        T[52, n] = f2(52)
        Z[52, n] = f3(52)
        # 第36区段
        ZT[35, n] = (Z[50, n] * Z[51, n] * Z[52, n]) / (
                    (Z[51, n] * Z[52, n]) + (Z[50, n] * Z[52, n]) + (Z[50, n] * Z[51, n]))
        T[35, n] = f2(35)
        Z[35, n] = f3(35)
        # 第35区段
        ZT[34, n] = Z0[34, n] + (1 / ((1 / Rb[34]) + (i * w * Cin[34])))
        T[34, n] = f2(34)
        Z[34, n] = f3(34)
        # 第22区段
        ZT[21, n] = (Z[34, n] * Z[35, n] * Z[36, n]) / (
                    (Z[35, n] * Z[36, n]) + (Z[34, n] * Z[36, n]) + (Z[34, n] * Z[35, n]))
        T[21, n] = f2(21)
        Z[21, n] = f3(21)
        # 第12区段
        ZT[11, n] = Z[21, n]
        T[11, n] = f2(11)
        Z[11, n] = f3(11) / 2
        # 第6区段
        ZT[5, n] = (Z[11, n] * Z[12, n] * Z[13, n] * Z[14, n]) / (
                    (Z[12, n] * Z[13, n] * Z[14, n]) + (Z[11, n] * Z[13, n] * Z[14, n]) + (
                        Z[11, n] * Z[12, n] * Z[14, n]) + (Z[11, n] * Z[12, n] * Z[13, n]))
        T[5, n] = f2(5)
        Z[5, n] = f3(5)
        # 第103区段
        ZT[102, n] = (Z[133, n] * Z[134, n] * Z[135, n] * Z[136, n] * Z[137, n]) / (
                (Z[134, n] * Z[135, n] * Z[136, n] * Z[137, n]) + (
                Z[133, n] * Z[135, n] * Z[136, n] * Z[137, n]) + (
                        Z[133, n] * Z[134, n] * Z[136, n] * Z[137, n]) + (
                        Z[133, n] * Z[134, n] * Z[135, n] * Z[137, n]) + (
                        Z[133, n] * Z[134, n] * Z[135, n] * Z[136, n]))
        T[102, n] = f2(102)
        Z[102, n] = f3(102)
        # 第95区段
        ZT[94, n] = Z[102, n]
        T[94, n] = f2(94)
        Z[94, n] = f3(94)
        # 第96区段
        ZT[95, n] = Z0[95, n] + (1 / ((1 / Rb[95]) + (i * w * Cin[95])))
        T[95, n] = f2(95)
        Z[95, n] = f3(95)
        # 第87区段
        ZT[86, n] = (Z[94, n] * Z[95, n]) / (Z[94, n] + Z[95, n])
        T[86, n] = f2(86)
        Z[86, n] = f3(86)
        # 第97区段
        ZT[96, n] = Z0[96, n] + (1 / ((1 / Rb[96]) + (i * w * Cin[96])))
        T[96, n] = f2(96)
        Z[96, n] = f3(96)
        # 第88区段
        ZT[87, n] = Z[96, n]
        T[87, n] = f2(87)
        Z[87, n] = f3(87)
        # 第81区段
        ZT[80, n] = (Z[86, n] * Z[87, n]) / (Z[86, n] + Z[87, n])
        T[80, n] = f2(80)
        Z[80, n] = f3(80)
        # 第80区段
        ZT[79, n] = Z0[79, n] + (1 / ((1 / Rb[79]) + (i * w * Cin[79])))
        T[79, n] = f2(79)
        Z[79, n] = f3(79)
        # 第71区段
        ZT[70, n] = (Z[79, n] * Z[80, n]) / (Z[79, n] + Z[80, n])
        T[70, n] = f2(70)
        Z[70, n] = f3(70)
        # 第72区段
        ZT[71, n] = Z0[71, n] + (1 / ((1 / Rb[71]) + (i * w * Cin[71])))
        T[71, n] = f2(71)
        Z[71, n] = f3(71)
        # 第59区段
        ZT[58, n] = (Z[70, n] * Z[71, n]) / (Z[70, n] + Z[71, n])
        T[58, n] = f2(58)
        Z[58, n] = f3(58)
        # 第58区段
        ZT[57, n] = Z0[57, n] + (1 / ((1 / Rb[57]) + (i * w * Cin[57])))
        T[57, n] = f2(57)
        Z[57, n] = f3(57)
        # 第42区段
        ZT[41, n] = (Z[57, n] * Z[58, n]) / (Z[57, n] + Z[58, n])
        T[41, n] = f2(41)
        Z[41, n] = f3(41)
        # 第28区段
        ZT[27, n] = Z[41, n]
        T[27, n] = f2(27)
        Z[27, n] = f3(27)
        # 第27区段
        ZT[26, n] = Z0[26, n] + (1 / ((1 / Rb[26]) + (i * w * Cin[26])))
        T[26, n] = f2(26)
        Z[26, n] = f3(26)
        # 第29区段
        ZT[28, n] = Z0[28, n] + (1 / ((1 / Rb[28]) + (i * w * Cin[28])))
        T[28, n] = f2(28)
        Z[28, n] = f3(28)
        # 第30区段
        ZT[29, n] = Z0[29, n] + (1 / ((1 / Rb[29]) + (i * w * Cin[29])))
        T[29, n] = f2(29)
        Z[29, n] = f3(29)
        # 第17区段
        ZT[16, n] = (Z[26, n] * Z[28, n] * Z[27, n] * Z[29, n]) / (
                (Z[28, n] * Z[27, n] * Z[29, n]) + (Z[26, n] * Z[27, n] * Z[29, n]) + (
                Z[26, n] * Z[28, n] * Z[29, n]) + (Z[26, n] * Z[28, n] * Z[27, n]))
        T[16, n] = f2(16)
        Z[16, n] = f3(16)
        # 第16区段
        ZT[15, n] = Z0[15, n] + (1 / ((1 / Rb[15]) + (i * w * Cin[15])))
        T[15, n] = f2(15)
        Z[15, n] = f3(15)
        # 第18区段
        ZT[17, n] = Z0[17, n] + (1 / ((1 / Rb[17]) + (i * w * Cin[17])))
        T[17, n] = f2(17)
        Z[17, n] = f3(17)
        # 第19区段
        ZT[18, n] = Z0[18, n] + (1 / ((1 / Rb[18]) + (i * w * Cin[18])))
        T[18, n] = f2(18)
        Z[18, n] = f3(18)
        # 第8区段
        ZT[7, n] = (Z[15, n] * Z[16, n] * Z[17, n] * Z[18, n]) / (
                    (Z[16, n] * Z[17, n] * Z[18, n]) + (Z[15, n] * Z[17, n] * Z[18, n]) + (
                        Z[15, n] * Z[16, n] * Z[18, n]) + (Z[15, n] * Z[16, n] * Z[17, n]))
        T[7, n] = f2(7)
        Z[7, n] = f3(7)
        # 第9区段
        ZT[8, n] = Z0[8, n] + (1 / ((1 / Rb[8]) + (i * w * Cin[8])))
        T[8, n] = f2(8)
        Z[8, n] = f3(8)
        # 第7区段
        ZT[6, n] = Z0[6, n] + (1 / ((1 / Rb[6]) + (i * w * Cin[6])))
        T[6, n] = f2(6)
        Z[6, n] = f3(6)
        # 第3区段
        ZT[2, n] = (Z[6, n] * Z[7, n] * Z[8, n]) / ((Z[7, n] * Z[8, n]) + (Z[6, n] * Z[8, n]) + (Z[6, n] * Z[7, n]))
        T[2, n] = f2(2)
        Z[2, n] = f3(2)
        # 第73区段
        ZT[72, n] = Z0[72, n] + (1 / ((1 / Rb[72]) + (i * w * Cin[72])))
        T[72, n] = f2(72)
        Z[72, n] = f3(72)
        # 第74区段
        ZT[73, n] = Z0[73, n] + (1 / ((1 / Rb[73]) + (i * w * Cin[73])))
        T[73, n] = f2(73)
        Z[73, n] = f3(73)
        # 第60区段
        ZT[59, n] = (Z[72, n] * Z[73, n]) / (Z[72, n] + Z[73, n])
        T[59, n] = f2(59)
        Z[59, n] = f3(59)
        # 第44区段
        ZT[43, n] = Z[59, n]
        T[43, n] = f2(43)
        Z[43, n] = f3(43)
        # 第43区段
        ZT[42, n] = Z0[42, n] + (1 / ((1 / Rb[42]) + (i * w * Cin[42])))
        T[42, n] = f2(42)
        Z[42, n] = f3(42)
        # 第45区段
        ZT[44, n] = Z0[44, n] + (1 / ((1 / Rb[44]) + (i * w * Cin[44])))
        T[44, n] = f2(44)
        Z[44, n] = f3(44)
        # 第31区段
        ZT[30, n] = (Z[42, n] * Z[43, n] * Z[44, n]) / (
                    (Z[43, n] * Z[44, n]) + (Z[42, n] * Z[44, n]) + (Z[42, n] * Z[43, n]))
        T[30, n] = f2(30)
        Z[30, n] = f3(30)
        # 第48区段
        ZT[47, n] = Z0[47, n] + (1 / ((1 / Rb[47]) + (i * w * Cin[47])))
        T[47, n] = f2(47)
        Z[47, n] = f3(47)
        # 第47区段
        ZT[46, n] = Z0[46, n] + (1 / ((1 / Rb[46]) + (i * w * Cin[46])))
        T[46, n] = f2(46)
        Z[46, n] = f3(46)
        # 第46区段
        ZT[45, n] = Z0[45, n] + (1 / ((1 / Rb[45]) + (i * w * Cin[45])))
        T[45, n] = f2(45)
        Z[45, n] = f3(45)
        # 第32区段
        ZT[31, n] = (Z[45, n] * Z[46, n] * Z[47, n]) / (
                    (Z[46, n] * Z[47, n]) + (Z[45, n] * Z[47, n]) + (Z[45, n] * Z[46, n]))
        T[31, n] = f2(31)
        Z[31, n] = f3(31)
        # 第33区段
        ZT[32, n] = Z0[32, n] + (1 / ((1 / Rb[32]) + (i * w * Cin[32])))
        T[32, n] = f2(32)
        Z[32, n] = f3(32)
        # 第20区段
        ZT[19, n] = (Z[30, n] * Z[31, n] * Z[32, n]) / (
                    (Z[31, n] * Z[32, n]) + (Z[30, n] * Z[32, n]) + (Z[30, n] * Z[31, n]))
        T[19, n] = f2(19)
        Z[19, n] = f3(19)
        # 第10区段
        ZT[9, n] = Z[19, n]
        T[9, n] = f2(9)
        Z[9, n] = f3(9)
        # 第4区段
        ZT[3, n] = Z[9, n]
        T[3, n] = f2(3)
        Z[3, n] = f3(3) / 2
        # 第2区段
        ZT[1, n] = (Z[2, n] * Z[3, n] * Z[4, n] * Z[5, n]) / (
                    (Z[3, n] * Z[4, n] * Z[5, n]) + (Z[2, n] * Z[4, n] * Z[5, n]) + (Z[2, n] * Z[3, n] * Z[5, n]) + (
                        Z[2, n] * Z[3, n] * Z[4, n]))
        T[1, n] = f2(1)
        Z[1, n] = f3(1)
        # 第1区段
        ZT[0, n] = Z[1, n]
        T[0, n] = f2(0)
        Z[0, n] = f3(0)

    t = np.arange(0, 0.81, 0.01)
    if ti > 0:
        a0 = 105.9 * (CO / CObascal)
        a1 = 116.4 * (CO / CObascal)
        b1 = 155.1 * (CO / CObascal)
        a2 = -48.27 * (CO / CObascal)
        b2 = 137.1 * (CO / CObascal)
        a3 = -66.94 * (CO / CObascal)
        b3 = 16.83 * (CO / CObascal)
        a4 = -51.71 * (CO / CObascal)
        b4 = 27.41 * (CO / CObascal)
        a5 = -44.3 * (CO / CObascal)
        b5 = -27.53 * (CO / CObascal)
        a6 = -4.866 * (CO / CObascal)
        b6 = -20.53 * (CO / CObascal)
        a7 = -7.162 * (CO / CObascal)
        b7 = -12.47 * (CO / CObascal)
        a8 = 0.6105 * (CO / CObascal)
        b8 = -13.76 * (CO / CObascal)
    else:
        a0 = 105.9
        a1 = 116.4
        b1 = 155.1
        a2 = -48.27
        b2 = 137.1
        a3 = -66.94
        b3 = 16.83
        a4 = -51.71
        b4 = 27.41
        a5 = -44.3
        b5 = -27.53
        a6 = -4.866
        b6 = -20.53
        a7 = -7.162
        b7 = -12.47
        a8 = 0.6105
        b8 = -13.76


    n = 7.854

    # 八个波的叠加
    m1 = a1 + b1 * 1j
    x1 = cmath.phase(m1)
    ff1 = abs(m1)
    m2 = a2 + b2 * 1j
    x2 = cmath.phase(m2)
    ff2 = abs(m2)
    m3 = a3 + b3 * 1j
    x3 = cmath.phase(m3)
    ff3 = abs(m3)
    m4 = a4 + b4 * 1j
    x4 = cmath.phase(m4)
    ff4 = abs(m4)
    m5 = a5 + b5 * 1j
    x5 = cmath.phase(m5)
    ff5 = abs(m5)
    m6 = a6 + b6 * 1j
    x6 = cmath.phase(m6)
    ff6 = abs(m6)
    m7 = a7 + b7 * 1j
    x7 = cmath.phase(m7)
    ff7 = abs(m7)
    m8 = a8 + b8 * 1j
    x8 = cmath.phase(m8)
    ff8 = abs(m8)

    har1 = ff1 * np.exp((0 + 1j) * ((n * t) - x1))
    har2 = ff2 * np.exp((0 + 1j) * ((2 * n * t) - x2))
    har3 = ff3 * np.exp((0 + 1j) * ((3 * n * t) - x3))
    har4 = ff4 * np.exp((0 + 1j) * ((4 * n * t) - x4))
    har5 = ff5 * np.exp((0 + 1j) * ((5 * n * t) - x5))
    har6 = ff6 * np.exp((0 + 1j) * ((6 * n * t) - x6))
    har7 = ff7 * np.exp((0 + 1j) * ((7 * n * t) - x7))
    har8 = ff8 * np.exp((0 + 1j) * ((8 * n * t) - x8))
    a_chu = a0 + har1 + har2 + har3 + har4 + har5 + har6 + har7 + har8

    q1 = a_chu
    qv1 = cmax(q1) * 0.251
    q2 = q1 * abs((1 - T[1, 3]) / abs(np.exp(r[1, 0] * L[1]) - (T[1, 3] * np.exp(-r[1, 0] * L[1]))))
    qv2 = cmax(q2) * 0.251
    q3 = (((Z[1, 0] * har1) / Z[2, 0]) + ((Z[1, 1] * har2) / Z[2, 1]) + ((Z[1, 2] * har3) / Z[2, 2]) + ((Z[1, 3] * har4) / Z[2, 3]) + ((Z[1, 4] * har5) / Z[2, 4]) + ((Z[1, 5] * har6) / Z[2, 5]) + ((Z[1, 6] * har7) / Z[2, 6]) + ((Z[1, 7] * har8) / Z[2, 7]))
    q3 = q3 + (cmax(q3) * 0.251) + 15
    qv3 = cmax(q3) * 0.251
    q4 = (((Z[1, 0] * har1) / Z[3, 0]) + ((Z[1, 1] * har2) / Z[3, 1]) + ((Z[1, 2] * har3) / Z[3, 2]) + ((Z[1, 3] * har4) / Z[3, 3]) + ((Z[1, 4] * har5) / Z[3, 4]) + ((Z[1, 5] * har6) / Z[3, 5]) + ((Z[1, 6] * har7) / Z[3, 6]) + ((Z[1, 7] * har8) / Z[3, 7]))
    q4 = q4 + (cmax(q4) * 0.251)
    qv4 = cmax(q4) * 0.251
    q5 = (((Z[1, 0] * har1) / Z[4, 0]) + ((Z[1, 1] * har2) / Z[4, 1]) + ((Z[1, 2] * har3) / Z[4, 2]) + ((Z[1, 3] * har4) / Z[4, 3]) + ((Z[1, 4] * har5) / Z[4, 4]) + ((Z[1, 5] * har6) / Z[4, 5]) + ((Z[1, 6] * har7) / Z[4, 6]) + ((Z[1, 7] * har8) / Z[4, 7]))
    q5 = q5 + (cmax(q5) * 0.251)
    qv5 = cmax(q5) * 0.251
    q6 = (((Z[1, 0] * har1) / Z[5, 0]) + ((Z[1, 1] * har2) / Z[5, 1]) + ((Z[1, 2] * har3) / Z[5, 2]) + ((Z[1, 3] * har4) / Z[5, 3]) + ((Z[1, 4] * har5) / Z[5, 4]) + ((Z[1, 5] * har6) / Z[5, 5]) + ((Z[1, 6] * har7) / Z[5, 6]) + ((Z[1, 7] * har8) / Z[5, 7]))
    q6 = q6 + (cmax(q6) * 0.251)
    qv6 = cmax(q6) * 0.251
    q11 = q5 * abs((1 - T[10, 3]) / (np.exp(r[10, 0] * L[10]) - (T[10, 3] * np.exp(-r[10, 0] * L[10]))))
    qv11 = cmax(q11) * 0.251
    q21 = q11 * abs((1 - T[20, 3]) / (np.exp(r[20, 0] * L[20]) - (T[20, 3] * np.exp(-r[20, 0] * L[20]))))
    qv21 = cmax(q21) * 0.251
    q34 = 0.25 * q21 * abs((1 - T[33, 3]) / (np.exp(r[33, 0] * L[33]) - (T[33, 3] * np.exp(-r[33, 0] * L[33]))))
    qv34 = cmax(q34) * 0.251
    q50 = (((Z[33, 0] * Z[1, 0] * har1) / (Z[4, 0] * Z[49, 0])) + ((Z[33, 1] * Z[1, 1] * har2) / (Z[4, 1] * Z[49, 1])) + ((Z[33, 2] * Z[1, 2] * har3) / (Z[4, 2] * Z[49, 2])) + ((Z[33, 3] * Z[1, 3] * har4) / (Z[4,3] * Z[49, 3])) + ((Z[33, 4] * Z[1, 4] * har5) / (Z[4, 4] * Z[49, 4])) + ((Z[33, 5] * Z[1, 5] * har6) / (Z[4,5] * Z[49, 5])) + ((Z[33, 6] * Z[1, 6] * har7) / (Z[4,6] * Z[49, 6])) + ((Z[33, 7] * Z[1, 7] * har8) / (Z[4,7] * Z[49, 7])))
    q50 = (q50 + (cmax(q50) * 0.251)) * 0.8 * 0.25
    qv50 = cmax(q50) * 0.251
    q65 = (((Z[33, 0] * Z[1, 0] * har1) / (Z[4, 0] * Z[64, 0])) + ((Z[33, 1] * Z[1, 1] * har2) / (Z[4, 1] * Z[64, 1])) + ((Z[33, 2] * Z[1, 2] * har3) / (Z[4, 2] * Z[64, 2])) + ((Z[33, 3] * Z[1, 3] * har4) / (Z[4, 3] * Z[64, 3])) + ((Z[33, 4] * Z[1, 4] * har5) / (Z[4, 4] * Z[64, 4])) + ((Z[33, 5] * Z[1, 5] * har6) / (Z[4, 5] * Z[64, 5])) + ((Z[33, 6] * Z[1, 6] * har7) / (Z[4, 6] * Z[64, 6])) + ((Z[33, 7] * Z[1, 7] * har8) / (Z[4, 7] * Z[64, 7])))
    q65 = (q65 + (cmax(q65) * 0.251)) * 0.8 * 0.25
    qv65 = cmax(q65) * 0.251
    q75 = q65 * abs((1 - T[74, 3]) / (np.exp(r[74, 0] * L[74]) - (T[74, 3] * np.exp(-r[74, 0] * L[74]))))
    qv75 = cmax(q75) * 0.251
    q84 = (((Z[74, 0] * Z[33, 0] * Z[1, 0] * har1) / (Z[4, 0] * Z[64, 0] * Z[83, 0])) + ((Z[74, 1] * Z[33, 1] * Z[1, 1] * har2) / (Z[4, 1] * Z[64, 1] * Z[83, 1])) + ((Z[74, 2] * Z[33, 2] * Z[1, 2] * har3) / (Z[4, 2] * Z[64, 2] * Z[83, 2])) + ((Z[74, 3] * Z[33, 3] * Z[1, 3] * har4) / (Z[4, 3] * Z[64, 3] * Z[83, 3])) + ((Z[74, 4] * Z[33, 4] * Z[1, 4] * har5) / (Z[4, 4] * Z[64, 4] * Z[83, 4])) + ((Z[74, 5] * Z[33, 5] * Z[1, 5] * har6) / (Z[4, 5] * Z[64, 5] * Z[83, 5])) + ((Z[74, 6] * Z[33, 6] * Z[1, 6] * har7) / (Z[4, 6] * Z[64, 6] * Z[83, 6])) + ((Z[74, 7] * Z[33, 7] * Z[1, 7] * har8) / (Z[4, 7] * Z[64, 7] * Z[83, 7])))
    q84 = (q84 + (cmax(q84) * 0.251)) * 0.7232
    qv84 = cmax(q84) * 0.251
    q92 = (((Z[74, 0] * Z[33, 0] * Z[1, 0] * har1) / (Z[4, 0] * Z[64, 0] * Z[91, 0])) + (
                (Z[74, 1] * Z[33, 1] * Z[1, 1] * har2) / (Z[4, 1] * Z[64, 1] * Z[91, 1])) + (
                       (Z[74, 2] * Z[33, 2] * Z[1, 2] * har3) / (Z[4, 2] * Z[64, 2] * Z[91, 2])) + (
                       (Z[74, 3] * Z[33, 3] * Z[1, 3] * har4) / (Z[4, 3] * Z[64, 3] * Z[91, 3])) + (
                       (Z[74, 4] * Z[33, 4] * Z[1, 4] * har5) / (Z[4, 4] * Z[64, 4] * Z[91, 4])) + (
                       (Z[74, 5] * Z[33, 5] * Z[1, 5] * har6) / (Z[4, 5] * Z[64, 5] * Z[91, 5])) + (
                       (Z[74, 6] * Z[33, 6] * Z[1, 6] * har7) / (Z[4, 6] * Z[64, 6] * Z[91, 6])) + (
                       (Z[74, 7] * Z[33, 7] * Z[1, 7] * har8) / (Z[4, 7] * Z[64, 7] * Z[91, 7])))
    q92 = (q92 + (cmax(q92) * 0.251)) * 0.7232
    qv92 = cmax(q92) * 0.251
    q99 = q92 * abs((1 - T[98, 3]) / (np.exp(r[98, 0] * L[98]) - (T[98, 3] * np.exp(-r[98, 0] * L[98]))))
    qv99 = cmax(q99) * 0.251
    q107 = (((Z[98, 0] * Z[74, 0] * Z[33, 0] * Z[1, 0] * har1) / (Z[4, 0] * Z[64, 0] * Z[91, 0] * Z[106, 0])) + (
                (Z[98, 1] * Z[74, 1] * Z[33, 1] * Z[1, 1] * har2) / (Z[4, 1] * Z[64, 1] * Z[91, 1] * Z[106, 1])) + (
                       (Z[98, 2] * Z[74, 2] * Z[33, 2] * Z[1, 2] * har3) / (Z[4, 2] * Z[64, 2] * Z[91, 2] * Z[106, 2])) + (
                       (Z[98, 3] * Z[74, 3] * Z[33, 3] * Z[1, 3] * har4) / (Z[4, 3] * Z[64, 3] * Z[91, 3] * Z[106, 3])) + (
                       (Z[98, 4] * Z[74, 4] * Z[33, 4] * Z[1, 4] * har5) / (Z[4, 4] * Z[64, 4] * Z[91, 4] * Z[106, 4])) + (
                       (Z[98, 5] * Z[74, 5] * Z[33, 5] * Z[1, 5] * har6) / (Z[4, 5] * Z[64, 5] * Z[91, 5] * Z[106, 5])) + (
                       (Z[98, 6] * Z[74, 6] * Z[33, 6] * Z[1, 6] * har7) / (Z[4, 6] * Z[64, 6] * Z[91, 6] * Z[106, 6])) + (
                       (Z[98, 7] * Z[74, 7] * Z[33, 7] * Z[1, 7] * har8) / (Z[4, 7] * Z[64, 7] * Z[91, 7] * Z[106, 7])))
    q107 = (q107 + (cmax(q107) * 0.251)) * 0.7150
    qv107 = cmax(q107) * 0.251
    q110 = q107 * abs((1 - T[109, 3]) / (np.exp(r[109, 0] * L[109]) - (T[109, 3] * np.exp(-r[109, 0] * L[109]))))
    qv110 = cmax(q110) * 0.251
    q112 = q110 * abs((1 - T[111, 3]) / (np.exp(r[111, 0] * L[111]) - (T[111, 3] * np.exp(-r[111, 0] * L[111]))))
    qv112 = cmax(q112) * 0.251
    q114 = 0.8 * q112 * abs((1 - T[113, 3]) / (np.exp(r[113, 0] * L[113]) - (T[113, 3] * np.exp(-r[113, 0] * L[113]))))
    qv114 = cmax(q114) * 0.251
    q118 = (((Z[113, 0] * Z[98, 0] * Z[74, 0] * Z[33, 0] * Z[1, 0] * har1) / (Z[4, 0] * Z[64, 0] * Z[91, 0] * Z[106, 0] * Z[117, 0])) + (
                (Z[113, 1] * Z[98, 1] * Z[74, 1] * Z[33, 1] * Z[1, 1] * har2) / (Z[4, 1] * Z[64, 1] * Z[91, 1] * Z[106, 1] * Z[117, 1])) + (
                       (Z[113, 2] * Z[98, 2] * Z[74, 2] * Z[33, 2] * Z[1, 2] * har3) / (Z[4, 2] * Z[64, 2] * Z[91, 2] * Z[106, 2] * Z[117, 2])) + (
                       (Z[113, 3] * Z[98, 3] * Z[74, 3] * Z[33, 3] * Z[1, 3] * har4) / (Z[4, 3] * Z[64, 3] * Z[91, 3] * Z[106, 3] * Z[117, 3])) + (
                       (Z[113, 4] * Z[98, 4] * Z[74, 4] * Z[33, 4] * Z[1, 4] * har5) / (Z[4, 4] * Z[64, 4] * Z[91, 4] * Z[106, 4] * Z[117, 4])) + (
                       (Z[113, 5] * Z[98, 5] * Z[74, 5] * Z[33, 5] * Z[1, 5] * har6) / (Z[4, 5] * Z[64, 5] * Z[91, 5] * Z[106, 5] * Z[117, 5])) + (
                       (Z[113, 6] * Z[98, 6] * Z[74, 6] * Z[33, 6] * Z[1, 6] * har7) / (Z[4, 6] * Z[64, 6] * Z[91, 6] * Z[106, 6] * Z[117, 6])) + (
                       (Z[113, 7] * Z[98, 7] * Z[74, 7] * Z[33, 7] * Z[1, 7] * har8) / (Z[4, 7] * Z[64, 7] * Z[91, 7] * Z[106, 7] * Z[117, 7])))
    q118 = (q118 + (cmax(q118) * 0.251)) * 0.726
    qv118 = cmax(q118) * 0.251
    q124 = (((Z[117, 0] * Z[113, 0] * Z[98, 0] * Z[74, 0] * Z[33, 0] * Z[1, 0] * har1) / (Z[4, 0] * Z[64, 0] * Z[91, 0] * Z[106, 0] * Z[117, 0] * Z[123, 0])) + (
                (Z[117, 1] * Z[113, 1] * Z[98, 1] * Z[74, 1] * Z[33, 1] * Z[1, 1] * har2) / (Z[4, 1] * Z[64, 1] * Z[91, 1] * Z[106, 1] * Z[117, 1] * Z[123, 1])) + (
                       (Z[117, 2] * Z[113, 2] * Z[98, 2] * Z[74, 2] * Z[33, 2] * Z[1, 2] * har3) / (Z[4, 2] * Z[64, 2] * Z[91, 2] * Z[106, 2] * Z[117, 2] * Z[123, 2])) + (
                       (Z[117, 3] * Z[113, 3] * Z[98, 3] * Z[74, 3] * Z[33, 3] * Z[1, 3] * har4) / (Z[4, 3] * Z[64, 3] * Z[91, 3] * Z[106, 3] * Z[117, 3] * Z[123, 3])) + (
                       (Z[117, 4] * Z[113, 4] * Z[98, 4] * Z[74, 4] * Z[33, 4] * Z[1, 4] * har5) / (Z[4, 4] * Z[64, 4] * Z[91, 4] * Z[106, 4] * Z[117, 4] * Z[123, 4])) + (
                       (Z[117, 5] * Z[113, 5] * Z[98, 5] * Z[74, 5] * Z[33, 5] * Z[1, 5] * har6) / (Z[4, 5] * Z[64, 5] * Z[91, 5] * Z[106, 5] * Z[117, 5] * Z[123, 5])) + (
                       (Z[117, 6] * Z[113, 6] * Z[98, 6] * Z[74, 6] * Z[33, 6] * Z[1, 6] * har7) / (Z[4, 6] * Z[64, 6] * Z[91, 6] * Z[106, 6] * Z[117, 6] * Z[123, 6])) + (
                       (Z[117, 7] * Z[113, 7] * Z[98, 7] * Z[74, 7] * Z[33, 7] * Z[1, 7] * har8) / (Z[4, 7] * Z[64, 7] * Z[91, 7] * Z[106, 7] * Z[117, 7] * Z[123, 7])))
    q124 = 3.3 * (q124 + (cmax(q124) * 0.251)) * 0.726
    qv124 = cmax(q124) * 0.251
    q128 = 0.33 * q124 * abs((1 - T[127, 3]) / (np.exp(r[127, 0] * L[127]) - (T[127, 3] * np.exp(-r[127, 0] * L[127]))))
    qv128 = cmax(q128) * 0.251
    q14 = (((Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[13, 0])) + ((Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[13, 1])) + ((Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[13, 2])) + ((Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[13, 3])) + ((Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[13, 4])) + ((Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[13, 5])) + ((Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[13, 6])) + ((Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[13, 7])))
    q14 = q14 + (cmax(q14) * 0.251) + 13
    qv14 = cmax(q14) * 0.251
    q25 = (((Z[13, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[13, 0] * Z[24, 0])) + (
                (Z[13, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[13, 1] * Z[24, 1])) + (
                       (Z[13, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[13, 2] * Z[24, 2])) + (
                       (Z[13, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[13, 3] * Z[24, 3])) + (
                       (Z[13, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[13, 4] * Z[24, 4])) + (
                       (Z[13, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[13, 5] * Z[24, 5])) + (
                       (Z[13, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[13, 6] * Z[24, 6])) + (
                       (Z[13, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[13, 7] * Z[24, 7])))
    q25 = (q25 + (cmax(q25) * 0.251) + 15)
    qv25 = cmax(q25) * 0.251
    q40 = (((Z[24, 0] * Z[13, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[13, 0] * Z[24, 0] * Z[39, 0])) + (
                (Z[24, 1] * Z[13, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[13, 1] * Z[24, 1] * Z[39, 1])) + (
                       (Z[24, 2] * Z[13, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[13, 2] * Z[24, 2] * Z[39, 2])) + (
                       (Z[24, 3] * Z[13, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[13, 3] * Z[24, 3] * Z[39, 3])) + (
                       (Z[24, 4] * Z[13, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[13, 4] * Z[24, 4] * Z[39, 4])) + (
                       (Z[24, 5] * Z[13, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[13, 5] * Z[24, 5] * Z[39, 5])) + (
                       (Z[24, 6] * Z[13, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[13, 6] * Z[24, 6] * Z[39, 6])) + (
                       (Z[24, 7] * Z[13, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[13, 7] * Z[24, 7] * Z[39, 7])))
    q40 = 0.1 * (q40 + (cmax(q40) * 0.251) + 16)
    qv40 = cmax(q40) * 0.251
    q57 = q40 * abs((1 - T[56, 3]) / (np.exp(r[56, 0] * L[56]) - (T[56, 3] * np.exp(-r[56, 0] * L[56]))))
    qv57 = cmax(q57) * 0.251
    q69 = (((Z[56, 0] * Z[24, 0] * Z[13, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[13, 0] * Z[24, 0] * Z[39, 0] * Z[68, 0])) + (
                (Z[56, 1] * Z[24, 1] * Z[13, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[13, 1] * Z[24, 1] * Z[39, 1] * Z[68, 1])) + (
                       (Z[56, 2] * Z[24, 2] * Z[13, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[13, 2] * Z[24, 2] * Z[39, 2] * Z[68, 2])) + (
                       (Z[56, 3] * Z[24, 3] * Z[13, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[13, 3] * Z[24, 3] * Z[39, 3] * Z[68, 3])) + (
                       (Z[56, 4] * Z[24, 4] * Z[13, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[13, 4] * Z[24, 4] * Z[39, 4] * Z[68, 4])) + (
                       (Z[56, 5] * Z[24, 5] * Z[13, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[13, 5] * Z[24, 5] * Z[39, 5] * Z[68, 5])) + (
                       (Z[56, 6] * Z[24, 6] * Z[13, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[13, 6] * Z[24, 6] * Z[39, 6] * Z[68, 6])) + (
                       (Z[56, 7] * Z[24, 7] * Z[13, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[13, 7] * Z[24, 7] * Z[39, 7] * Z[68, 7])))
    q69 = ((q69 + (cmax(q69) * 0.251)) * 0.9809 +17) * 0.1
    qv69 = cmax(q69) * 0.251
    q79 = (((Z[68, 0] * Z[56, 0] * Z[24, 0] * Z[13, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[13, 0] * Z[24, 0] * Z[39, 0] * Z[68, 0] * Z[78, 0])) + (
                (Z[68, 1] * Z[56, 1] * Z[24, 1] * Z[13, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[13, 1] * Z[24, 1] * Z[39, 1] * Z[68, 1] * Z[78, 1])) + (
                       (Z[68, 2] * Z[56, 2] * Z[24, 2] * Z[13, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[13, 2] * Z[24, 2] * Z[39, 2] * Z[68, 2] * Z[78, 2])) + (
                       (Z[68, 3] * Z[56, 3] * Z[24, 3] * Z[13, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[13, 3] * Z[24, 3] * Z[39, 3] * Z[68, 3] * Z[78, 3])) + (
                       (Z[68, 4] * Z[56, 4] * Z[24, 4] * Z[13, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[13, 4] * Z[24, 4] * Z[39, 4] * Z[68, 4] * Z[78, 4])) + (
                       (Z[68, 5] * Z[56, 5] * Z[24, 5] * Z[13, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[13, 5] * Z[24, 5] * Z[39, 5] * Z[68, 5] * Z[78, 5])) + (
                       (Z[68, 6] * Z[56, 6] * Z[24, 6] * Z[13, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[13, 6] * Z[24, 6] * Z[39, 6] * Z[68, 6] * Z[78, 6])) + (
                       (Z[68, 7] * Z[56, 7] * Z[24, 7] * Z[13, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[13, 7] * Z[24, 7] * Z[39, 7] * Z[68, 7] * Z[78, 7])))
    q79 = ((q79 + (cmax(q79) * 0.251)) * 0.9809 + 18) * 0.1
    qv79 = cmax(q79) * 0.251
    q85 = (((Z[78, 0] * Z[68, 0] * Z[56, 0] * Z[24, 0] * Z[13, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[13, 0] * Z[24, 0] * Z[39, 0] * Z[68, 0] * Z[78, 0] * Z[84, 0])) + (
                (Z[78, 1] * Z[68, 1] * Z[56, 1] * Z[24, 1] * Z[13, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[13, 1] * Z[24, 1] * Z[39, 1] * Z[68, 1] * Z[78, 1] * Z[84, 1])) + (
                       (Z[78, 2] * Z[68, 2] * Z[56, 2] * Z[24, 2] * Z[13, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[13, 2] * Z[24, 2] * Z[39, 2] * Z[68, 2] * Z[78, 2] * Z[84, 2])) + (
                       (Z[78, 3] * Z[68, 3] * Z[56, 3] * Z[24, 3] * Z[13, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[13, 3] * Z[24, 3] * Z[39, 3] * Z[68, 3] * Z[78, 3] * Z[84, 3])) + (
                       (Z[78, 4] * Z[68, 4] * Z[56, 4] * Z[24, 4] * Z[13, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[13, 4] * Z[24, 4] * Z[39, 4] * Z[68, 4] * Z[78, 4] * Z[84, 4])) + (
                       (Z[78, 5] * Z[68, 5] * Z[56, 5] * Z[24, 5] * Z[13, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[13, 5] * Z[24, 5] * Z[39, 5] * Z[68, 5] * Z[78, 5] * Z[84, 5])) + (
                       (Z[78, 6] * Z[68, 6] * Z[56, 6] * Z[24, 6] * Z[13, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[13, 6] * Z[24, 6] * Z[39, 6] * Z[68, 6] * Z[78, 6] * Z[84, 6])) + (
                       (Z[78, 7] * Z[68, 7] * Z[56, 7] * Z[24, 7] * Z[13, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[13, 7] * Z[24, 7] * Z[39, 7] * Z[68, 7] * Z[78, 7] * Z[84, 7])))
    q85 = 0.1 * ((q85 + (cmax(q85) * 0.251)) * 0.9809 + 20)
    qv85 = cmax(q85) * 0.251
    q94 = (((Z[84, 0] * Z[78, 0] * Z[68, 0] * Z[56, 0] * Z[24, 0] * Z[13, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[13, 0] * Z[24, 0] * Z[39, 0] * Z[68, 0] * Z[78, 0] * Z[84, 0] * Z[93, 0])) + (
                (Z[84, 1] * Z[78, 1] * Z[68, 1] * Z[56, 1] * Z[24, 1] * Z[13, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[13, 1] * Z[24, 1] * Z[39, 1] * Z[68, 1] * Z[78, 1] * Z[84, 1] * Z[93, 1])) + (
                       (Z[84, 2] * Z[78, 2] * Z[68, 2] * Z[56, 2] * Z[24, 2] * Z[13, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[13, 2] * Z[24, 2] * Z[39, 2] * Z[68, 2] * Z[78, 2] * Z[84, 2] * Z[93, 2])) + (
                       (Z[84, 3] * Z[78, 3] * Z[68, 3] * Z[56, 3] * Z[24, 3] * Z[13, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[13, 3] * Z[24, 3] * Z[39, 3] * Z[68, 3] * Z[78, 3] * Z[84, 3] * Z[93, 3])) + (
                       (Z[84, 4] * Z[78, 4] * Z[68, 4] * Z[56, 4] * Z[24, 4] * Z[13, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[13, 4] * Z[24, 4] * Z[39, 4] * Z[68, 4] * Z[78, 4] * Z[84, 4] * Z[93, 4])) + (
                       (Z[84, 5] * Z[78, 5] * Z[68, 5] * Z[56, 5] * Z[24, 5] * Z[13, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[13, 5] * Z[24, 5] * Z[39, 5] * Z[68, 5] * Z[78, 5] * Z[84, 5] * Z[93, 5])) + (
                       (Z[84, 6] * Z[78, 6] * Z[68, 6] * Z[56, 6] * Z[24, 6] * Z[13, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[13, 6] * Z[24, 6] * Z[39, 6] * Z[68, 6] * Z[78, 6] * Z[84, 6] * Z[93, 6])) + (
                       (Z[84, 7] * Z[78, 7] * Z[68, 7] * Z[56, 7] * Z[24, 7] * Z[13, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[13, 7] * Z[24, 7] * Z[39, 7] * Z[68, 7] * Z[78, 7] * Z[84, 7] * Z[93, 7])))
    q94 = 0.1 * ((q94 + (cmax(q94) * 0.251)) * 0.9809 + 20)
    qv94 = cmax(q94) * 0.251
    q102 = (((Z[93, 0] * Z[84, 0] * Z[78, 0] * Z[68, 0] * Z[56, 0] * Z[24, 0] * Z[13, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[13, 0] * Z[24, 0] * Z[39, 0] * Z[68, 0] * Z[78, 0] * Z[84, 0] * Z[93, 0] * Z[101, 0])) + (
                (Z[93, 1] * Z[84, 1] * Z[78, 1] * Z[68, 1] * Z[56, 1] * Z[24, 1] * Z[13, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[13, 1] * Z[24, 1] * Z[39, 1] * Z[68, 1] * Z[78, 1] * Z[84, 1] * Z[93, 1] * Z[101, 1])) + (
                       (Z[93, 2] * Z[84, 2] * Z[78, 2] * Z[68, 2] * Z[56, 2] * Z[24, 2] * Z[13, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[13, 2] * Z[24, 2] * Z[39, 2] * Z[68, 2] * Z[78, 2] * Z[84, 2] * Z[93, 2] * Z[101, 2])) + (
                       (Z[93, 3] * Z[84, 3] * Z[78, 3] * Z[68, 3] * Z[56, 3] * Z[24, 3] * Z[13, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[13, 3] * Z[24, 3] * Z[39, 3] * Z[68, 3] * Z[78, 3] * Z[84, 3] * Z[93, 3] * Z[101, 3])) + (
                       (Z[93, 4] * Z[84, 4] * Z[78, 4] * Z[68, 4] * Z[56, 4] * Z[24, 4] * Z[13, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[13, 4] * Z[24, 4] * Z[39, 4] * Z[68, 4] * Z[78, 4] * Z[84, 4] * Z[93, 4] * Z[101, 4])) + (
                       (Z[93, 5] * Z[84, 5] * Z[78, 5] * Z[68, 5] * Z[56, 5] * Z[24, 5] * Z[13, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[13, 5] * Z[24, 5] * Z[39, 5] * Z[68, 5] * Z[78, 5] * Z[84, 5] * Z[93, 5] * Z[101, 5])) + (
                       (Z[93, 6] * Z[84, 6] * Z[78, 6] * Z[68, 6] * Z[56, 6] * Z[24, 6] * Z[13, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[13, 6] * Z[24, 6] * Z[39, 6] * Z[68, 6] * Z[78, 6] * Z[84, 6] * Z[93, 6] * Z[101, 6])) + (
                       (Z[93, 7] * Z[84, 7] * Z[78, 7] * Z[68, 7] * Z[56, 7] * Z[24, 7] * Z[13, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[13, 7] * Z[24, 7] * Z[39, 7] * Z[68, 7] * Z[78, 7] * Z[84, 7] * Z[93, 7] * Z[101, 7])))
    q102 = 0.04 * ((q102 + (cmax(q102) * 0.251)) * 0.9809 + 20)
    qv102 = cmax(q102) * 0.251
    q108 = 0.7 * q102 * abs((1 - T[107, 3]) / (np.exp(r[107, 0] * L[107]) - (T[107, 3] * np.exp(-r[107, 0] * L[107]))))
    qv108 = cmax(q108) * 0.251
    q129 = 0.2 * q108
    qv129 = cmax(q129) * 0.251
    q130 = q129
    q131 = q129
    q132 = q129
    q133 = q129
    q134 = q129
    q135 = q129
    q136 = q129
    q137 = q129
    q138 = q129
    qv130 = qv129
    qv131 = qv129
    qv132 = qv129
    qv133 = qv129
    qv134 = qv129
    qv135 = qv129
    qv136 = qv129
    qv137 = qv129
    qv138 = qv129
    q12 = (((Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[11, 0])) + ((Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[11, 1])) + ((Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[11, 2])) + ((Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[11, 3])) + ((Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[11, 4])) + ((Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[11, 5])) + ((Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[11, 6])) + ((Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[11, 7])))
    q12 = q12 + (cmax(q12) * 0.251)
    qv12 = cmax(q12) * 0.251
    q22 = q12 * abs((1 - T[21, 3]) / (np.exp(r[21, 0] * L[21]) - (T[21, 3] * np.exp(-r[21, 0] * L[21]))))
    qv22 = cmax(q22) * 0.251
    q37 = (((Z[21, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[11, 0] * Z[36, 0])) + (
                (Z[21, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[11, 1] * Z[36, 1])) + (
                       (Z[21, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[11, 2] * Z[36, 2])) + (
                       (Z[21, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[11, 3] * Z[36, 3])) + (
                       (Z[21, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[11, 4] * Z[36, 4])) + (
                       (Z[21, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[11, 5] * Z[36, 5])) + (
                       (Z[21, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[11, 6] * Z[36, 6])) + (
                       (Z[21, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[11, 7] * Z[36, 7])) + 0.7)
    q37 = (q37 + (cmax(q37) * 0.251)) * 0.5477
    qv37 = cmax(q37) * 0.251
    q55 = (((Z[21, 0] * Z[5, 0] * Z[1, 0] * har1) / (Z[5, 0] * Z[11, 0] * Z[54, 0])) + (
                (Z[21, 1] * Z[5, 1] * Z[1, 1] * har2) / (Z[5, 1] * Z[11, 1] * Z[54, 1])) + (
                       (Z[21, 2] * Z[5, 2] * Z[1, 2] * har3) / (Z[5, 2] * Z[11, 2] * Z[54, 2])) + (
                       (Z[21, 3] * Z[5, 3] * Z[1, 3] * har4) / (Z[5, 3] * Z[11, 3] * Z[54, 3])) + (
                       (Z[21, 4] * Z[5, 4] * Z[1, 4] * har5) / (Z[5, 4] * Z[11, 4] * Z[54, 4])) + (
                       (Z[21, 5] * Z[5, 5] * Z[1, 5] * har6) / (Z[5, 5] * Z[11, 5] * Z[54, 5])) + (
                       (Z[21, 6] * Z[5, 6] * Z[1, 6] * har7) / (Z[5, 6] * Z[11, 6] * Z[54, 6])) + (
                       (Z[21, 7] * Z[5, 7] * Z[1, 7] * har8) / (Z[5, 7] * Z[11, 7] * Z[54, 7])) + 0.13)
    q55 = (q55 + (cmax(q55) * 0.251)) * 0.5477
    qv55 = cmax(q55) * 0.251
    q68 = q55 * abs((1 - T[67, 3]) / (np.exp(r[67, 0] * L[67]) - (T[67, 3] * np.exp(-r[67, 0] * L[67]))))
    qv68 = cmax(q68) * 0.251

    qv = np.empty(22)

    qv[0] = ((qv4 + qv12) * 3600).real  # qv_face
    qv[1] = qv[0]  # qv_brain
    qv[2] = qv57.real * 3600  # qv_right_up_arm
    qv[3] = qv[2]  # qv_left_up_arm
    qv[4] = qv85.real * 3600  # qv_right_forearm
    qv[5] = qv[4]  # qv_left_forearm
    qv[6] = qv108.real * 3600  # qv_right_hand
    qv[7] = qv[6]  # qv_left_hand
    qv[8] = qv1.real * 3600  # qv_chest
    qv[9] = qv14.real * 3600  # qv_right_shoulder
    qv[10] = qv50.real * 3600  # qv_abdomen
    qv[11] = qv[9]  # qv_left_shoulder
    qv[12] = qv84.real * 3600  # qv_right_hip
    qv[13] = qv[12]  # qv_left_hip
    qv[14] = qv92.real * 3600  # qv_right_thigh
    qv[15] = qv[14]  # qv_left_thigh
    qv[16] = qv114.real * 3600  # qv_right_calf
    qv[17] = qv[16]  # qv_left_calf
    qv[18] = qv128.real * 3600  # qv_right_foot
    qv[19] = qv[18]  # qv_left_foot
    qv[20] = qv129.real * 3600  # qv_left_finger
    qv[21] = qv[20]  # qv_right_finger

    for i in range(22):
        qv[i] = qv[i].real

    Cb = 4 / 3600
    hv = np.empty(22)
    hv[0] = b_den * Cb * (qv12 + qv4 + qv22 + qv37 + qv55 + qv68).real * 3600 * (1 / 3.14) * 1.5
    hv[1] = hv[0]
    hv[2] = b_den * Cb * (qv57 + qv69 + qv79).real * 3600 * (1 / 3.14) * 1.5
    hv[3] = hv[2]
    hv[4] = b_den * Cb * (qv85 + qv94 + qv102).real * 3600 * (1 / 3.14) * 1.5
    hv[5] = hv[4]
    hv[6] = b_den * Cb * qv108.real * 3600 * (1 / 3.14) * 1.5
    hv[7] = hv[6]
    hv[8] = b_den * Cb * (qv1 + qv2 + qv6 + qv5 + qv11 + qv21 + qv34).real * 3600 * (1 / 3.14) * 0.4
    hv[9] = b_den * Cb * (qv14 + qv25 + qv40).real * 3600 * (1 / 3.14) * 1.5
    hv[10] = b_den * Cb * (qv50 + qv65 + qv75).real * 3600 * (1 / 3.14) * 0.4
    hv[11] = hv[9]
    hv[12] = b_den * Cb * qv84.real * 3600 * (1 / 3.14) * 1.5
    hv[13] = hv[12]
    hv[14] = b_den * Cb * (qv92 + qv99 + qv107 + qv110 + qv112).real * 3600 * (1 / 3.14) * 1.5
    hv[15] = hv[14]
    hv[16] = b_den * Cb * (qv114 + qv118 + qv124).real * 3600 * (1 / 3.14) * 1.5
    hv[17] = hv[16]
    hv[18] = b_den * Cb * qv128.real * 3600 * (1 / 3.14) * 1.5
    hv[19] = hv[18]
    hv[20] = b_den * Cb * qv129.real * 3600 * (1 / 3.14) * 1.5
    hv[21] = hv[20]
    ha = np.empty(22)
    for i in range(22):
        if ti > 0:
            ha[i] = 1.08 * hv[i]
        else:
            ha[i] = hv[i]
    # print(np.trapz((q4 + q12).real, t))
    m_perfusion = 3600 * np.array(
        [
            np.trapz((q4 + q12).real, t), np.trapz((q4 + q12).real, t), np.trapz((q25 - q57).real, t),
            np.trapz((q25 - q57).real, t), np.trapz((q57 - q102).real, t), np.trapz((q57 - q102).real, t),
            np.trapz(q108.real, t), np.trapz(q108.real, t), np.trapz((q1 - q57 - q4 - q12 - q57 - q34).real, t),
            np.trapz((q14 - q25).real, t), np.trapz((q34 - q84).real, t), np.trapz((q14 - q25).real, t),
            np.trapz(q84.real, t), np.trapz(q84.real, t), np.trapz((q92 - q114).real, t), np.trapz((q92 - q114).real, t),
            np.trapz((q114 - q124).real, t), np.trapz((q114 - q124).real, t), np.trapz(q128.real, t), np.trapz(q128.real, t),
            np.trapz((q108.real * 0.2), t), np.trapz((q108.real * 0.2), t),
        ]
    )
    m_perfusion = m_perfusion.real
    return ha, hv, m_perfusion, qv


def cmax(complex_array):
    complex_max = complex_array[0]
    for i in range(len(complex_array)):
        if abs(complex_array[i]) > abs(complex_max):
            complex_max = complex_array[i]
    return complex_max