from typing import Dict

from logzero import logger
from tabulate import tabulate

__all__ = ["last_N_slo_were_met_for_all_services"]


def last_N_slo_were_met_for_all_services(
        value: Dict[str, Dict] = None) -> bool:
    """
    Goes through all the SLO generated by `get_last_N_slos` and
    ensure all of their reports showed each SLO was met recently.

    Fails as soon as one was not met.
    """
    not_ok = {}
    for (service_name, slos) in value.items():
        for slo in slos:
            if not slo["result"]["slo_is_met"]:
                not_ok[service_name] = slo

    headers = [
        "Service", "Type", "SLO", "Objective", "Period", "Current", "Delta"
    ]
    table = []
    for (service_name, slo) in not_ok.items():
        service = service_name.split("/", 1)[0]
        slo_type = slo["type"].title()
        slo_name = slo["name"]
        slo_period = str(slo["period"])
        slo_objective = slo["objective"]
        result = slo["result"]

        table.append([
            service, slo_type, slo_name, slo_period, slo_objective,
            result["actual"], result["delta"]
        ])

    if table:
        logger.info("SLO that were not met during experiment:\n\n{}\n".format(
            tabulate(table, headers=headers, tablefmt="github")))

    return len(not_ok) == 0
