import os
import unittest
from typing import Dict, Optional

import pandas as pd
from pandas.testing import assert_series_equal

from finstmt import FinancialStatements
from finstmt.exc import MismatchingDatesException
from tests.conftest import DEVELOPMENT_MODE, GENERATED_PATH

# Imported for test development purposes
from tests.expectdata.load.load_capiq_cat_annual import LOAD_CAPIQ_CAT_A_INDEX_DATA_DICT

if DEVELOPMENT_MODE:
    from tests.utils.gen.data_load import print_test_data_def, get_keys_for_inc_data_items, get_keys_for_bs_data_items
    inc_keys = get_keys_for_inc_data_items()
    bs_keys = get_keys_for_bs_data_items()


def check_data_items(stmts: FinancialStatements, data_dict: Dict[str, pd.Series]):
    for item_key, item_values in data_dict.items():
        item = getattr(stmts, item_key)
        assert_series_equal(item, item_values, check_dtype=False)


class LoadTest:
    name: str
    a_test_data_dict: Dict[str, pd.Series]
    q_test_data_dict: Dict[str, pd.Series]

    def test_annual(self, stmts: FinancialStatements, data: Optional[Dict[str, pd.Series]] = None):
        if data is None:
            data = self.a_test_data_dict
        if DEVELOPMENT_MODE:
            out_path = os.path.join(GENERATED_PATH, f'{self.name}_annual.py')
            with open(out_path, 'w') as f:
                f.write('import pandas as pd\n\n')
                f.write(
                    print_test_data_def(
                        stmts, inc_keys + bs_keys,
                        f'{self.name.upper()}_A_INDEX',
                        f'{self.name.upper()}_A_INDEX_DATA_DICT',
                        disp=False
                    )
                )
        check_data_items(stmts, data)

    def test_quarterly(self, stmts: FinancialStatements, data: Optional[Dict[str, pd.Series]] = None):
        if data is None:
            data = self.q_test_data_dict
        if DEVELOPMENT_MODE:
            out_path = os.path.join(GENERATED_PATH, f'{self.name}_quarterly.py')
            with open(out_path, 'w') as f:
                f.write('import pandas as pd\n\n')
                f.write(
                    print_test_data_def(
                        stmts, inc_keys + bs_keys,
                        f'{self.name.upper()}_Q_INDEX',
                        f'{self.name.upper()}_Q_INDEX_DATA_DICT',
                        disp=False
                    )
                )
        check_data_items(stmts, data)


class TestLoadStockrowCAT(LoadTest):
    name = 'load_stockrow_cat'
    a_index_str = ["2009-12-31 00:00:00", "2010-12-31 00:00:00", "2011-12-31 00:00:00", "2012-12-31 00:00:00",
                   "2013-12-31 00:00:00", "2014-12-31 00:00:00", "2015-12-31 00:00:00", "2016-12-31 00:00:00",
                   "2017-12-31 00:00:00", "2018-12-31 00:00:00"]
    a_index = [pd.to_datetime(val) for val in a_index_str]
    a_test_data_dict = dict(
        revenue=pd.Series(
            [32396000000.0, 42588000000.0, 60138000000.0, 65875000000.0, 55656000000.0, 55184000000.0, 47011000000.0,
             38537000000.0, 45462000000.0, 54722000000.0],
            index=a_index
        ),
        cogs=pd.Series(
            [24931000000.0, 31281000000.0, 44404000000.0, 47852000000.0, 41454000000.0, 41342000000.0, 34133000000.0,
             28640000000.0, 31906000000.0, 37719000000.0],
            index=a_index
        ),
        gross_profit=pd.Series(
            [7465000000.0, 11307000000.0, 15734000000.0, 18023000000.0, 14202000000.0, 13842000000.0, 12878000000.0,
             9897000000.0, 13556000000.0, 17003000000.0],
            index=a_index
        ),
        rd_exp=pd.Series(
            [1421000000.0, 1905000000.0, 2297000000.0, 2466000000.0, 2046000000.0, 2380000000.0, 2119000000.0,
             1853000000.0, 1842000000.0, 1850000000.0],
            index=a_index
        ),
        sga=pd.Series(
            [3645000000.0, 4248000000.0, 5203000000.0, 5919000000.0, 5547000000.0, 6529000000.0, 4951000000.0,
             4383000000.0, 4999000000.0, 5478000000.0],
            index=a_index
        ),
        dep_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        other_op_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        op_exp=pd.Series(
            [6888000000.0, 7344000000.0, 8581000000.0, 9450000000.0, 8574000000.0, 10528000000.0, 9093000000.0,
             8735000000.0, 9096000000.0, 8710000000.0],
            index=a_index
        ),
        ebit=pd.Series(
            [1014000000.0, 4011000000.0, 7044000000.0, 8676000000.0, 5573000000.0, 3628000000.0, 3935000000.0,
             630000000.0, 4624000000.0, 8249000000.0],
            index=a_index
        ),
        int_exp=pd.Series(
            [389000000.0, 343000000.0, 396000000.0, 467000000.0, 465000000.0, 484000000.0, 507000000.0, 505000000.0,
             531000000.0, 404000000.0],
            index=a_index
        ),
        gain_on_sale_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        gain_on_sale_asset=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        impairment=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        ebt=pd.Series(
            [625000000.0, 3668000000.0, 6648000000.0, 8209000000.0, 5108000000.0, 3144000000.0, 3428000000.0,
             125000000.0, 4093000000.0, 7845000000.0],
            index=a_index
        ),
        tax_exp=pd.Series(
            [270000000.0, 968000000.0, 1720000000.0, 2528000000.0, 1319000000.0, 692000000.0, 916000000.0, 192000000.0,
             3339000000.0, 1698000000.0],
            index=a_index
        ),
        net_income=pd.Series(
            [895000000.0, 2700000000.0, 4928000000.0, 5681000000.0, 3789000000.0, 2452000000.0, 2512000000.0,
             67000000.0, 754000000.0, 6147000000.0],
            index=a_index
        ),
        cash=pd.Series(
            [4867000000, 3592000000, 3057000000, 5490000000, 6081000000, 7341000000, 6460000000, 7168000000, 8261000000,
             7857000000],
            index=a_index
        ),
        st_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        cash_and_st_invest=pd.Series(
            [4867000000, 3592000000, 3057000000, 5490000000, 6081000000, 7341000000, 6460000000, 7168000000, 8261000000,
             7857000000],
            index=a_index
        ),
        receivables=pd.Series(
            [27162000000, 28849000000, 30803000000, 33911000000, 33499000000, 32772000000, 30507000000, 29088000000,
             30725000000, 31899000000],
            index=a_index
        ),
        inventory=pd.Series(
            [6360000000, 9587000000, 14544000000, 15547000000, 12625000000, 12205000000, 9700000000, 8614000000,
             10018000000, 11529000000],
            index=a_index
        ),
        def_tax_st=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        other_current_assets=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        total_current_assets=pd.Series(
            [27217000000, 31810000000, 37900000000, 42138000000, 38335000000, 38867000000, 33508000000, 31967000000,
             36244000000, 38603000000],
            index=a_index
        ),
        gross_ppe=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        dep=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        net_ppe=pd.Series(
            [12386000000, 12539000000, 14395000000, 16461000000, 17075000000, 16577000000, 16090000000, 15322000000,
             14155000000, 13574000000],
            index=a_index
        ),
        goodwill=pd.Series(
            [2734000000, 3419000000, 11448000000, 10958000000, 10552000000, 9770000000, 9436000000, 8369000000,
             8311000000, 8114000000],
            index=a_index
        ),
        lt_invest=pd.Series(
            [105000000, 164000000, 133000000, 272000000, 272000000, 257000000, 0, 0, 0, 0],
            index=a_index
        ),
        def_tax_lt=pd.Series(
            [3930000000, 3424000000, 3737000000, 3558000000, 2147000000, 3143000000, 2489000000, 2790000000, 1693000000,
             1439000000],
            index=a_index
        ),
        other_lt_assets=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        total_non_current_assets=pd.Series(
            [32821000000, 32210000000, 43318000000, 46832000000, 46561000000, 45814000000, 44834000000, 42737000000,
             40718000000, 39906000000],
            index=a_index
        ),
        total_assets=pd.Series(
            [60038000000, 64020000000, 81218000000, 88970000000, 84896000000, 84681000000, 78342000000, 74704000000,
             76962000000, 78509000000],
            index=a_index
        ),
        payables=pd.Series(
            [2993000000, 5856000000, 8161000000, 6753000000, 6560000000, 6515000000, 5023000000, 4614000000, 6487000000,
             7051000000],
            index=a_index
        ),
        st_debt=pd.Series(
            [9784000000, 7981000000, 9648000000, 12391000000, 11031000000, 11501000000, 12844000000, 13965000000,
             11031000000, 11553000000],
            index=a_index
        ),
        current_lt_debt=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        tax_liab_st=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        other_current_liab=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        total_current_liab=pd.Series(
            [18975000000, 22020000000, 28357000000, 29415000000, 27297000000, 27877000000, 26242000000, 26132000000,
             26931000000, 28218000000],
            index=a_index
        ),
        lt_debt=pd.Series(
            [21847000000, 20437000000, 24944000000, 27752000000, 26719000000, 27784000000, 25169000000, 22818000000,
             23847000000, 25000000000],
            index=a_index
        ),
        total_debt=pd.Series(
            [31631000000, 28418000000, 34592000000, 40143000000, 37750000000, 39285000000, 38013000000, 36783000000,
             34878000000, 36553000000],
            index=a_index
        ),
        deferred_rev=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        tax_liab_lt=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        deposit_liab=pd.Series(
            [1217000000, 1831000000, 2487000000, 2638000000, 2360000000, 1697000000, 1146000000, 1167000000, 1426000000,
             1243000000],
            index=a_index
        ),
        other_lt_liab=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        total_non_current_liab=pd.Series(
            [31763000000, 30675000000, 39459000000, 41973000000, 36721000000, 39978000000, 37215000000, 35359000000,
             36265000000, 36211000000],
            index=a_index
        ),
        total_liab=pd.Series(
            [50738000000, 52695000000, 67816000000, 71388000000, 64018000000, 67855000000, 63457000000, 61491000000,
             63196000000, 64429000000],
            index=a_index
        ),
        common_stock=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        other_income=pd.Series(
            [-3764000000, -4051000000, -6328000000, -6433000000, -3898000000, -6431000000, -2035000000, -2039000000,
             -1192000000, -1684000000],
            index=a_index
        ),
        retained_earnings=pd.Series(
            [19711000000, 21384000000, 25219000000, 29558000000, 31854000000, 33887000000, 29246000000, 27377000000,
             26301000000, 30427000000],
            index=a_index
        ),
        minority_interest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        total_equity=pd.Series(
            [8740000000, 10824000000, 12883000000, 17532000000, 20811000000, 16746000000, 14809000000, 13213000000,
             13766000000, 14080000000],
            index=a_index
        ),
        total_liab_and_equity=pd.Series(
            [59478000000.0, 63519000000.0, 80699000000.0, 88920000000.0, 84829000000.0, 84601000000.0, 78266000000.0,
             74704000000.0, 76962000000.0, 78509000000.0],
            index=a_index
        ),
    )

    q_index_str = ["2009-12-31 00:00:00", "2010-03-31 00:00:00", "2010-06-30 00:00:00", "2010-09-30 00:00:00",
                   "2010-12-31 00:00:00", "2011-03-31 00:00:00", "2011-06-30 00:00:00", "2011-09-30 00:00:00",
                   "2011-12-31 00:00:00", "2012-03-31 00:00:00", "2012-06-30 00:00:00", "2012-09-30 00:00:00",
                   "2012-12-31 00:00:00", "2013-03-31 00:00:00", "2013-06-30 00:00:00", "2013-09-30 00:00:00",
                   "2013-12-31 00:00:00", "2014-03-31 00:00:00", "2014-06-30 00:00:00", "2014-09-30 00:00:00",
                   "2014-12-31 00:00:00", "2015-03-31 00:00:00", "2015-06-30 00:00:00", "2015-09-30 00:00:00",
                   "2015-12-31 00:00:00", "2016-03-31 00:00:00", "2016-06-30 00:00:00", "2016-09-30 00:00:00",
                   "2016-12-31 00:00:00", "2017-03-31 00:00:00", "2017-06-30 00:00:00", "2017-09-30 00:00:00",
                   "2017-12-31 00:00:00", "2018-03-31 00:00:00", "2018-06-30 00:00:00", "2018-09-30 00:00:00",
                   "2018-12-31 00:00:00", "2019-03-31 00:00:00", "2019-06-30 00:00:00", "2019-09-30 00:00:00"]
    q_index = [pd.to_datetime(val) for val in q_index_str]
    q_test_data_dict = dict(
        revenue=pd.Series(
            [7898000000.0, 8238000000.0, 10409000000.0, 11134000000.0, 12807000000.0, 12949000000.0, 14230000000.0,
             15716000000.0, 17243000000.0, 15981000000.0, 17374000000.0, 16445000000.0, 16075000000.0, 13210000000.0,
             14621000000.0, 13423000000.0, 14402000000.0, 13241000000.0, 14150000000.0, 13549000000.0, 14244000000.0,
             12702000000.0, 12317000000.0, 10962000000.0, 11030000000.0, 9461000000.0, 10342000000.0, 9160000000.0,
             9574000000.0, 9822000000.0, 11331000000.0, 11413000000.0, 12896000000.0, 12859000000.0, 14011000000.0,
             13510000000.0, 14342000000.0, 13466000000.0, 14432000000.0, 12758000000.0],
            index=q_index
        ),
        cogs=pd.Series(
            [6090000000.0, 6127000000.0, 7606000000.0, 7979000000.0, 9569000000.0, 9260000000.0, 10512000000.0,
             11666000000.0, 12966000000.0, 11441000000.0, 12478000000.0, 11836000000.0, 12097000000.0, 9828000000.0,
             10958000000.0, 9952000000.0, 10716000000.0, 9597000000.0, 10350000000.0, 9791000000.0, 11604000000.0,
             8910000000.0, 8822000000.0, 8014000000.0, 8387000000.0, 6974000000.0, 7567000000.0, 6674000000.0,
             7425000000.0, 6960000000.0, 7978000000.0, 7841000000.0, 9127000000.0, 8732000000.0, 9604000000.0,
             9207000000.0, 10176000000.0, 9193000000.0, 10133000000.0, 8758000000.0],
            index=q_index
        ),
        gross_profit=pd.Series(
            [1808000000.0, 2111000000.0, 2803000000.0, 3155000000.0, 3238000000.0, 3689000000.0, 3718000000.0,
             4050000000.0, 4277000000.0, 4540000000.0, 4896000000.0, 4609000000.0, 3978000000.0, 3382000000.0,
             3663000000.0, 3471000000.0, 3686000000.0, 3644000000.0, 3800000000.0, 3758000000.0, 2640000000.0,
             3792000000.0, 3495000000.0, 2948000000.0, 2643000000.0, 2487000000.0, 2775000000.0, 2486000000.0,
             2149000000.0, 2862000000.0, 3353000000.0, 3572000000.0, 3769000000.0, 4127000000.0, 4407000000.0,
             4303000000.0, 4166000000.0, 4273000000.0, 4299000000.0, 4000000000.0],
            index=q_index
        ),
        rd_exp=pd.Series(
            [355000000.0, 402000000.0, 450000000.0, 510000000.0, 543000000.0, 525000000.0, 584000000.0, 584000000.0,
             604000000.0, 587000000.0, 632000000.0, 634000000.0, 613000000.0, 562000000.0, 548000000.0, 469000000.0,
             467000000.0, 508000000.0, 516000000.0, 533000000.0, 823000000.0, 524000000.0, 510000000.0, 513000000.0,
             572000000.0, 508000000.0, 468000000.0, 453000000.0, 424000000.0, 425000000.0, 458000000.0, 461000000.0,
             498000000.0, 443000000.0, 462000000.0, 479000000.0, 466000000.0, 435000000.0, 441000000.0, 431000000.0],
            index=q_index
        ),
        sga=pd.Series(
            [942000000.0, 932000000.0, 1059000000.0, 1148000000.0, 1109000000.0, 1099000000.0, 1257000000.0,
             1360000000.0, 1487000000.0, 1340000000.0, 1517000000.0, 1471000000.0, 1591000000.0, 1390000000.0,
             1421000000.0, 1319000000.0, 1417000000.0, 1292000000.0, 1437000000.0, 1446000000.0, 2354000000.0,
             1249000000.0, 1318000000.0, 1129000000.0, 1255000000.0, 1088000000.0, 1123000000.0, 992000000.0,
             1180000000.0, 1061000000.0, 1304000000.0, 1254000000.0, 1380000000.0, 1276000000.0, 1440000000.0,
             1299000000.0, 1463000000.0, 1319000000.0, 1309000000.0, 1251000000.0],
            index=q_index
        ),
        dep_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        other_op_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        op_exp=pd.Series(
            [1680000000.0, 1603000000.0, 1826000000.0, 1968000000.0, 1947000000.0, 1856000000.0, 2117000000.0,
             2291000000.0, 2317000000.0, 2217000000.0, 2280000000.0, 2013000000.0, 2940000000.0, 2164000000.0,
             2106000000.0, 2070000000.0, 2234000000.0, 2246000000.0, 2325000000.0, 2366000000.0, 3591000000.0,
             2090000000.0, 2162000000.0, 2023000000.0, 2818000000.0, 1993000000.0, 1990000000.0, 2005000000.0,
             2747000000.0, 2482000000.0, 2169000000.0, 2063000000.0, 2382000000.0, 2019000000.0, 2240000000.0,
             2168000000.0, 2283000000.0, 2066000000.0, 2086000000.0, 1980000000.0],
            index=q_index
        ),
        ebit=pd.Series(
            [229000000.0, 566000000.0, 997000000.0, 1172000000.0, 1276000000.0, 1824000000.0, 1423000000.0,
             1727000000.0, 2070000000.0, 2388000000.0, 2681000000.0, 2581000000.0, 1026000000.0, 1246000000.0,
             1467000000.0, 1372000000.0, 1488000000.0, 1450000000.0, 1538000000.0, 1509000000.0, 869000000.0,
             1895000000.0, 1262000000.0, 904000000.0, 126000000.0, 492000000.0, 864000000.0, 505000000.0, 1231000000.0,
             405000000.0, 1284000000.0, 1647000000.0, 1288000000.0, 2238000000.0, 2299000000.0, 2244000000.0,
             1468000000.0, 2371000000.0, 2288000000.0, 2115000000.0],
            index=q_index
        ),
        int_exp=pd.Series(
            [88000000.0, 102000000.0, 81000000.0, 85000000.0, 75000000.0, 87000000.0, 90000000.0, 112000000.0,
             107000000.0, 113000000.0, 110000000.0, 129000000.0, 115000000.0, 120000000.0, 120000000.0, 116000000.0,
             109000000.0, 110000000.0, 120000000.0, 128000000.0, 126000000.0, 129000000.0, 125000000.0, 127000000.0,
             126000000.0, 129000000.0, 130000000.0, 126000000.0, 120000000.0, 123000000.0, 121000000.0, 118000000.0,
             169000000.0, 101000000.0, 102000000.0, 102000000.0, 99000000.0, 103000000.0, 103000000.0, 103000000.0],
            index=q_index
        ),
        gain_on_sale_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        gain_on_sale_asset=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        impairment=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        ebt=pd.Series(
            [141000000.0, 464000000.0, 916000000.0, 1087000000.0, 1201000000.0, 1737000000.0, 1333000000.0,
             1615000000.0, 1963000000.0, 2275000000.0, 2571000000.0, 2452000000.0, 911000000.0, 1126000000.0,
             1347000000.0, 1256000000.0, 1379000000.0, 1340000000.0, 1418000000.0, 1381000000.0, 995000000.0,
             1766000000.0, 1137000000.0, 777000000.0, 252000000.0, 363000000.0, 734000000.0, 379000000.0, 1351000000.0,
             282000000.0, 1163000000.0, 1529000000.0, 1119000000.0, 2137000000.0, 2197000000.0, 2142000000.0,
             1369000000.0, 2268000000.0, 2185000000.0, 2012000000.0],
            index=q_index
        ),
        tax_exp=pd.Series(
            [91000000.0, 231000000.0, 209000000.0, 295000000.0, 233000000.0, 512000000.0, 318000000.0, 474000000.0,
             416000000.0, 689000000.0, 872000000.0, 753000000.0, 214000000.0, 246000000.0, 387000000.0, 310000000.0,
             376000000.0, 418000000.0, 419000000.0, 364000000.0, 509000000.0, 521000000.0, 335000000.0, 218000000.0,
             158000000.0, 92000000.0, 184000000.0, 96000000.0, 180000000.0, 90000000.0, 361000000.0, 470000000.0,
             2418000000.0, 472000000.0, 490000000.0, 415000000.0, 321000000.0, 387000000.0, 565000000.0, 518000000.0],
            index=q_index
        ),
        net_income=pd.Series(
            [232000000.0, 233000000.0, 707000000.0, 792000000.0, 968000000.0, 1225000000.0, 1015000000.0, 1141000000.0,
             1547000000.0, 1586000000.0, 1699000000.0, 1699000000.0, 697000000.0, 880000000.0, 960000000.0, 946000000.0,
             1003000000.0, 922000000.0, 999000000.0, 1017000000.0, 486000000.0, 1245000000.0, 802000000.0, 559000000.0,
             94000000.0, 271000000.0, 550000000.0, 283000000.0, 1171000000.0, 192000000.0, 802000000.0, 1059000000.0,
             1299000000.0, 1665000000.0, 1707000000.0, 1727000000.0, 1048000000.0, 1881000000.0, 1620000000.0,
             1494000000.0],
            index=q_index
        ),
        cash=pd.Series(
            [4867000000, 3538000000, 3597000000, 2265000000, 3592000000, 4869000000, 10715000000, 3229000000,
             3057000000, 2864000000, 5103000000, 5689000000, 5490000000, 5982000000, 6110000000, 6357000000, 6081000000,
             5345000000, 7927000000, 6082000000, 7341000000, 7563000000, 7821000000, 6046000000, 6460000000, 5886000000,
             6764000000, 6113000000, 7168000000, 9472000000, 10232000000, 9591000000, 8261000000, 7888000000,
             8654000000, 8007000000, 7857000000, 7128000000, 7429000000, 7906000000],
            index=q_index
        ),
        st_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        cash_and_st_invest=pd.Series(
            [4867000000, 3538000000, 3597000000, 2265000000, 3592000000, 4869000000, 10715000000, 3229000000,
             3057000000, 2864000000, 5103000000, 5689000000, 5490000000, 5982000000, 6110000000, 6357000000, 6081000000,
             5345000000, 7927000000, 6082000000, 7341000000, 7563000000, 7821000000, 6046000000, 6460000000, 5886000000,
             6764000000, 6113000000, 7168000000, 9472000000, 10232000000, 9591000000, 8261000000, 7888000000,
             8654000000, 8007000000, 7857000000, 7128000000, 7429000000, 7906000000],
            index=q_index
        ),
        receivables=pd.Series(
            [27162000000, 27070000000, 27169000000, 28250000000, 28849000000, 29348000000, 29997000000, 30034000000,
             30803000000, 31632000000, 32584000000, 33257000000, 33911000000, 34164000000, 33812000000, 33724000000,
             33499000000, 33889000000, 34190000000, 33176000000, 32772000000, 31588000000, 31413000000, 30462000000,
             30507000000, 30852000000, 30396000000, 29453000000, 29088000000, 29587000000, 29732000000, 29836000000,
             30725000000, 31029000000, 31299000000, 31171000000, 31899000000, 31716000000, 32150000000, 31072000000],
            index=q_index
        ),
        inventory=pd.Series(
            [6360000000, 6990000000, 7339000000, 9006000000, 9587000000, 10676000000, 11359000000, 14412000000,
             14544000000, 16511000000, 17344000000, 17550000000, 15547000000, 15074000000, 13889000000, 13392000000,
             12625000000, 12888000000, 13055000000, 13328000000, 12205000000, 12099000000, 11681000000, 11150000000,
             9700000000, 9849000000, 9458000000, 9478000000, 8614000000, 9082000000, 9388000000, 10212000000,
             10018000000, 10947000000, 11255000000, 11814000000, 11529000000, 12340000000, 12007000000, 12180000000],
            index=q_index
        ),
        def_tax_st=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        other_current_assets=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        total_current_assets=pd.Series(
            [27217000000, 26412000000, 27376000000, 28577000000, 31810000000, 34608000000, 40755000000, 36864000000,
             37900000000, 40209000000, 44294000000, 44639000000, 42138000000, 42145000000, 40805000000, 40088000000,
             38335000000, 37968000000, 41276000000, 39042000000, 38867000000, 38491000000, 38227000000, 35280000000,
             33508000000, 33748000000, 33606000000, 31999000000, 31967000000, 35548000000, 36991000000, 37185000000,
             36244000000, 37357000000, 38641000000, 38454000000, 38603000000, 39126000000, 39789000000, 39160000000],
            index=q_index
        ),
        gross_ppe=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        dep=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        net_ppe=pd.Series(
            [12386000000, 12057000000, 11763000000, 12065000000, 12539000000, 12219000000, 12430000000, 13397000000,
             14395000000, 14571000000, 14928000000, 15509000000, 16461000000, 16276000000, 16352000000, 16588000000,
             17075000000, 16716000000, 16690000000, 16431000000, 16577000000, 16277000000, 16136000000, 15955000000,
             16090000000, 15935000000, 15916000000, 15680000000, 15322000000, 14727000000, 14420000000, 14187000000,
             14155000000, 13912000000, 13752000000, 13607000000, 13574000000, 13259000000, 13172000000, 12842000000],
            index=q_index
        ),
        goodwill=pd.Series(
            [2734000000, 2772000000, 2777000000, 3458000000, 3419000000, 3402000000, 3385000000, 12307000000,
             11448000000, 11368000000, 11556000000, 11538000000, 10958000000, 10715000000, 10578000000, 10686000000,
             10552000000, 10495000000, 10367000000, 10011000000, 9770000000, 9383000000, 9413000000, 9387000000,
             9436000000, 9451000000, 9329000000, 9178000000, 8369000000, 8338000000, 8374000000, 8371000000, 8311000000,
             8539000000, 8288000000, 8209000000, 8114000000, 7998000000, 7944000000, 7772000000],
            index=q_index
        ),
        lt_invest=pd.Series(
            [105000000, 133000000, 154000000, 160000000, 164000000, 140000000, 123000000, 121000000, 133000000,
             139000000, 124000000, 199000000, 272000000, 270000000, 288000000, 278000000, 272000000, 266000000,
             259000000, 265000000, 257000000, 230000000, 229000000, 231000000, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0],
            index=q_index
        ),
        def_tax_lt=pd.Series(
            [3930000000, 3711000000, 3505000000, 3916000000, 3424000000, 3307000000, 3366000000, 1928000000, 3737000000,
             3583000000, 3717000000, 3506000000, 3558000000, 3598000000, 3693000000, 3526000000, 2147000000, 2101000000,
             2200000000, 2131000000, 3143000000, 2780000000, 2914000000, 3005000000, 2489000000, 2486000000, 2536000000,
             2579000000, 2790000000, 2940000000, 2866000000, 2845000000, 1693000000, 1687000000, 1626000000, 1288000000,
             1439000000, 1378000000, 1473000000, 1372000000],
            index=q_index
        ),
        other_lt_assets=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        total_non_current_assets=pd.Series(
            [32821000000, 32424000000, 31417000000, 33065000000, 32210000000, 31779000000, 32856000000, 40903000000,
             43318000000, 43699000000, 44544000000, 45902000000, 46832000000, 46598000000, 46570000000, 47184000000,
             46561000000, 46429000000, 46551000000, 45446000000, 45814000000, 43999000000, 44015000000, 43689000000,
             44834000000, 44559000000, 44694000000, 44403000000, 42737000000, 42001000000, 41519000000, 41375000000,
             40718000000, 40657000000, 40346000000, 39755000000, 39906000000, 39600000000, 39398000000, 38833000000],
            index=q_index
        ),
        total_assets=pd.Series(
            [60038000000, 58836000000, 58793000000, 61642000000, 64020000000, 66387000000, 73611000000, 77767000000,
             81218000000, 83908000000, 88838000000, 90541000000, 88970000000, 88743000000, 87375000000, 87272000000,
             84896000000, 84397000000, 87827000000, 84488000000, 84681000000, 82490000000, 82242000000, 78969000000,
             78342000000, 78307000000, 78300000000, 76402000000, 74704000000, 77549000000, 78510000000, 78560000000,
             76962000000, 78014000000, 78987000000, 78209000000, 78509000000, 78726000000, 79187000000, 77993000000],
            index=q_index
        ),
        payables=pd.Series(
            [2993000000, 3431000000, 3975000000, 4970000000, 5856000000, 5990000000, 6858000000, 7524000000, 8161000000,
             8360000000, 8470000000, 7978000000, 6753000000, 6221000000, 6343000000, 6280000000, 6560000000, 6731000000,
             6860000000, 6778000000, 6515000000, 6328000000, 5862000000, 5206000000, 5023000000, 5101000000, 5104000000,
             4713000000, 4614000000, 5302000000, 5778000000, 6113000000, 6487000000, 6938000000, 6831000000, 6788000000,
             7051000000, 7198000000, 7022000000, 6141000000],
            index=q_index
        ),
        st_debt=pd.Series(
            [9784000000, 8622000000, 8927000000, 8271000000, 7981000000, 9691000000, 8456000000, 7508000000, 9648000000,
             10762000000, 12055000000, 13326000000, 12391000000, 13246000000, 13871000000, 13522000000, 11031000000,
             11290000000, 12936000000, 11102000000, 11501000000, 11972000000, 10875000000, 12335000000, 12844000000,
             13893000000, 14343000000, 13488000000, 13965000000, 14557000000, 13377000000, 11089000000, 11031000000,
             12150000000, 12470000000, 10332000000, 11553000000, 11542000000, 11514000000, 12318000000],
            index=q_index
        ),
        current_lt_debt=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        tax_liab_st=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        other_current_liab=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        total_current_liab=pd.Series(
            [18975000000, 18417000000, 19784000000, 20178000000, 22020000000, 23147000000, 23510000000, 25218000000,
             28357000000, 28849000000, 31273000000, 31746000000, 29415000000, 28974000000, 29919000000, 29119000000,
             27297000000, 27295000000, 29741000000, 27589000000, 27877000000, 26566000000, 25606000000, 25833000000,
             26242000000, 26215000000, 27183000000, 25290000000, 26132000000, 27635000000, 28133000000, 25903000000,
             26931000000, 27402000000, 28300000000, 26033000000, 28218000000, 27388000000, 27735000000, 27201000000],
            index=q_index
        ),
        lt_debt=pd.Series(
            [21847000000, 21548000000, 20226000000, 20337000000, 20437000000, 19895000000, 25926000000, 26781000000,
             24944000000, 25191000000, 27261000000, 26526000000, 27752000000, 27240000000, 25680000000, 26015000000,
             26719000000, 26801000000, 27307000000, 28180000000, 27784000000, 26803000000, 27445000000, 25208000000,
             25169000000, 24470000000, 23980000000, 23622000000, 22818000000, 23725000000, 23815000000, 24835000000,
             23847000000, 23165000000, 23699000000, 25441000000, 25000000000, 24240000000, 24764000000, 25588000000],
            index=q_index
        ),
        total_debt=pd.Series(
            [31631000000, 30170000000, 29153000000, 28608000000, 28418000000, 29586000000, 34382000000, 34289000000,
             34592000000, 35953000000, 39316000000, 39852000000, 40143000000, 40486000000, 39551000000, 39537000000,
             37750000000, 38091000000, 40243000000, 39282000000, 39285000000, 38775000000, 38320000000, 37543000000,
             38013000000, 38363000000, 38323000000, 37110000000, 36783000000, 38282000000, 37192000000, 35924000000,
             34878000000, 35315000000, 36169000000, 35773000000, 36553000000, 35782000000, 36278000000, 37906000000],
            index=q_index
        ),
        deferred_rev=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        tax_liab_lt=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        deposit_liab=pd.Series(
            [1217000000, 1367000000, 1404000000, 1470000000, 1831000000, 1728000000, 1823000000, 2745000000, 2487000000,
             2921000000, 3132000000, 3035000000, 2638000000, 2920000000, 2738000000, 2699000000, 2360000000, 2500000000,
             2344000000, 2165000000, 1697000000, 1636000000, 1754000000, 1610000000, 1146000000, 1328000000, 1259000000,
             1161000000, 1167000000, 1383000000, 1533000000, 1510000000, 1426000000, 1399000000, 1378000000, 1491000000,
             1243000000, 1354000000, 1263000000, 1309000000],
            index=q_index
        ),
        other_lt_liab=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        total_non_current_liab=pd.Series(
            [31763000000, 30945000000, 29305000000, 31132000000, 30675000000, 30109000000, 36205000000, 37845000000,
             39459000000, 39588000000, 41584000000, 40859000000, 41973000000, 41451000000, 39778000000, 39976000000,
             36721000000, 36733000000, 37163000000, 38003000000, 39978000000, 38886000000, 39475000000, 37168000000,
             37215000000, 36339000000, 35814000000, 35397000000, 35359000000, 36254000000, 36298000000, 36960000000,
             36265000000, 35340000000, 35745000000, 36286000000, 36211000000, 35860000000, 36574000000, 35799000000],
            index=q_index
        ),
        total_liab=pd.Series(
            [50738000000, 49362000000, 49089000000, 51310000000, 52695000000, 53256000000, 59715000000, 63063000000,
             67816000000, 68437000000, 72857000000, 72605000000, 71388000000, 70425000000, 69697000000, 69095000000,
             64018000000, 64028000000, 66904000000, 65592000000, 67855000000, 65452000000, 65081000000, 63001000000,
             63457000000, 62554000000, 62997000000, 60687000000, 61491000000, 63889000000, 64431000000, 62863000000,
             63196000000, 62742000000, 64045000000, 62319000000, 64429000000, 63248000000, 64309000000, 63000000000],
            index=q_index
        ),
        common_stock=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        other_income=pd.Series(
            [-3764000000, -3886000000, -4045000000, -4412000000, -4051000000, -3724000000, -3544000000, -4019000000,
             -6328000000, -6033000000, -6150000000, -5988000000, -6433000000, -6669000000, -6698000000, -6247000000,
             -3898000000, -3801000000, -3683000000, -4357000000, -6431000000, -7101000000, -6729000000, -6843000000,
             -2035000000, -1493000000, -1633000000, -1527000000, -2039000000, -1827000000, -1471000000, -1233000000,
             -1192000000, -1016000000, -1496000000, -1568000000, -1684000000, -1588000000, -1499000000, -1783000000],
            index=q_index
        ),
        retained_earnings=pd.Series(
            [19711000000, 19941000000, 20133000000, 20955000000, 21384000000, 22640000000, 23081000000, 24251000000,
             25219000000, 26815000000, 27842000000, 29541000000, 29558000000, 30438000000, 30668000000, 31614000000,
             31854000000, 32775000000, 32961000000, 33977000000, 33887000000, 34998000000, 34823000000, 35191000000,
             29246000000, 29517000000, 29167000000, 29450000000, 27377000000, 27584000000, 27471000000, 28530000000,
             26301000000, 27929000000, 28657000000, 30384000000, 30427000000, 32435000000, 32981000000, 34477000000],
            index=q_index
        ),
        minority_interest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        total_equity=pd.Series(
            [8740000000, 8942000000, 9185000000, 9837000000, 10824000000, 12629000000, 13391000000, 14162000000,
             12883000000, 14969000000, 15926000000, 17884000000, 17532000000, 18274000000, 17621000000, 18110000000,
             20811000000, 20305000000, 20856000000, 18823000000, 16746000000, 16962000000, 17092000000, 15896000000,
             14809000000, 15676000000, 15232000000, 15645000000, 13213000000, 13660000000, 14079000000, 15697000000,
             13766000000, 15272000000, 14942000000, 15890000000, 14080000000, 15478000000, 14878000000, 14993000000],
            index=q_index
        ),
        total_liab_and_equity=pd.Series(
            [59478000000.0, 58304000000.0, 58274000000.0, 61147000000.0, 63519000000.0, 65885000000.0, 73106000000.0,
             77225000000.0, 80699000000.0, 83406000000.0, 88783000000.0, 90489000000.0, 88920000000.0, 88699000000.0,
             87318000000.0, 87205000000.0, 84829000000.0, 84333000000.0, 87760000000.0, 84415000000.0, 84601000000.0,
             82414000000.0, 82173000000.0, 78897000000.0, 78266000000.0, 78230000000.0, 78229000000.0, 76332000000.0,
             74704000000.0, 77549000000.0, 78510000000.0, 78560000000.0, 76962000000.0, 78014000000.0, 78987000000.0,
             78209000000.0, 78509000000.0, 78726000000.0, 79187000000.0, 77993000000.0],
            index=q_index
        ),
    )

    def test_annual(self, annual_stockrow_stmts_cat: FinancialStatements):
        super().test_annual(annual_stockrow_stmts_cat)

    def test_quarterly(self, quarterly_stockrow_stmts_cat: FinancialStatements):
        super().test_quarterly(quarterly_stockrow_stmts_cat)


class TestLoadStockrowMAR(LoadTest):
    name = 'load_stockrow_mar'
    a_index_str = ["2009-12-31 00:00:00", "2010-12-31 00:00:00", "2011-12-31 00:00:00", "2012-12-31 00:00:00",
                   "2013-12-31 00:00:00", "2014-12-31 00:00:00", "2015-12-31 00:00:00", "2016-12-31 00:00:00",
                   "2017-12-31 00:00:00", "2018-12-31 00:00:00"]
    a_index = [pd.to_datetime(val) for val in a_index_str]
    a_test_data_dict = dict(
        revenue=pd.Series(
            [10908000000.0, 11691000000.0, 12317000000.0, 11814000000.0, 12784000000.0, 13796000000.0, 14486000000.0,
             15407000000.0, 20452000000.0, 20758000000.0],
            index=a_index
        ),
        cogs=pd.Series(
            [10287000000.0, 10216000000.0, 11039000000.0, 10190000000.0, 11020000000.0, 11830000000.0, 12363000000.0,
             12735000000.0, 16639000000.0, 17084000000.0],
            index=a_index
        ),
        gross_profit=pd.Series(
            [621000000.0, 1475000000.0, 1278000000.0, 1624000000.0, 1764000000.0, 1966000000.0, 2123000000.0,
             2672000000.0, 3813000000.0, 3674000000.0],
            index=a_index
        ),
        rd_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        sga=pd.Series(
            [722000000.0, 780000000.0, 752000000.0, 582000000.0, 649000000.0, 659000000.0, 634000000.0, 743000000.0,
             921000000.0, 927000000.0],
            index=a_index
        ),
        dep_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        other_op_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        op_exp=pd.Series(
            [722000000.0, 780000000.0, 752000000.0, 582000000.0, 649000000.0, 659000000.0, 634000000.0, 743000000.0,
             921000000.0, 927000000.0],
            index=a_index
        ),
        ebit=pd.Series(
            [152000000.0, 695000000.0, 526000000.0, 940000000.0, 988000000.0, 1159000000.0, 1350000000.0, 1424000000.0,
             2504000000.0, 2366000000.0],
            index=a_index
        ),
        int_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        gain_on_sale_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        gain_on_sale_asset=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        impairment=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        ebt=pd.Series(
            [418000000.0, 551000000.0, 356000000.0, 849000000.0, 897000000.0, 1088000000.0, 1255000000.0, 1239000000.0,
             2982000000.0, 2345000000.0],
            index=a_index
        ),
        tax_exp=pd.Series(
            [65000000.0, 93000000.0, 158000000.0, 278000000.0, 271000000.0, 335000000.0, 396000000.0, 431000000.0,
             1523000000.0, 438000000.0],
            index=a_index
        ),
        net_income=pd.Series(
            [353000000.0, 458000000.0, 198000000.0, 571000000.0, 626000000.0, 753000000.0, 859000000.0, 808000000.0,
             1459000000.0, 1907000000.0],
            index=a_index
        ),
        cash=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        st_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        cash_and_st_invest=pd.Series(
            [115000000.0, 505000000.0, 102000000.0, 88000000.0, 126000000.0, 104000000.0, 96000000.0, 858000000.0,
             383000000.0, 316000000.0],
            index=a_index
        ),
        receivables=pd.Series(
            [838000000.0, 938000000.0, 875000000.0, 1028000000.0, 1081000000.0, 1100000000.0, 1103000000.0,
             1695000000.0, 1973000000.0, 2133000000.0],
            index=a_index
        ),
        inventory=pd.Series(
            [1444000000.0, 1489000000.0, 11000000.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            index=a_index
        ),
        def_tax_st=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        other_current_assets=pd.Series(
            [131000000.0, 123000000.0, 22000000.0, 377000000.0, 342000000.0, 108000000.0, 588000000.0, 149000000.0,
             8000000.0, 0.0],
            index=a_index
        ),
        total_current_assets=pd.Series(
            [2851000000.0, 3382000000.0, 1324000000.0, 1475000000.0, 1903000000.0, 1610000000.0, 1384000000.0,
             3371000000.0, 2740000000.0, 2706000000.0],
            index=a_index
        ),
        gross_ppe=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        dep=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        net_ppe=pd.Series(
            [1362000000.0, 1307000000.0, 1168000000.0, 1539000000.0, 1543000000.0, 1460000000.0, 1029000000.0,
             2335000000.0, 1793000000.0, 1956000000.0],
            index=a_index
        ),
        goodwill=pd.Series(
            [1606000000.0, 1643000000.0, 1721000000.0, 1989000000.0, 2005000000.0, 2245000000.0, 2394000000.0,
             16868000000.0, 17751000000.0, 17419000000.0],
            index=a_index
        ),
        lt_invest=pd.Series(
            [249000000.0, 250000000.0, 265000000.0, 216000000.0, 222000000.0, 224000000.0, 165000000.0, 728000000.0,
             734000000.0, 732000000.0],
            index=a_index
        ),
        def_tax_lt=pd.Series(
            [1020000000.0, 932000000.0, 873000000.0, 676000000.0, 647000000.0, 819000000.0, 672000000.0, 116000000.0,
             93000000.0, 171000000.0],
            index=a_index
        ),
        other_lt_assets=pd.Series(
            [393000000.0, 205000000.0, 261000000.0, 267000000.0, 332000000.0, 260000000.0, 223000000.0, 477000000.0,
             593000000.0, 587000000.0],
            index=a_index
        ),
        total_non_current_assets=pd.Series(
            [5082000000.0, 5601000000.0, 4586000000.0, 4867000000.0, 4891000000.0, 5223000000.0, 4698000000.0,
             20769000000.0, 21106000000.0, 20990000000.0],
            index=a_index
        ),
        total_assets=pd.Series(
            [7933000000.0, 8983000000.0, 5910000000.0, 6342000000.0, 6794000000.0, 6833000000.0, 6082000000.0,
             24140000000.0, 23846000000.0, 23696000000.0],
            index=a_index
        ),
        payables=pd.Series(
            [562000000.0, 548000000.0, 569000000.0, 557000000.0, 605000000.0, 593000000.0, 687000000.0, 783000000.0,
             767000000.0, 0.0],
            index=a_index
        ),
        st_debt=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        current_lt_debt=pd.Series(
            [64000000.0, 138000000.0, 355000000.0, 407000000.0, 6000000.0, 324000000.0, 300000000.0, 309000000.0,
             398000000.0, 833000000.0],
            index=a_index
        ),
        tax_liab_st=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        other_current_liab=pd.Series(
            [1142000000.0, 1037000000.0, 1005000000.0, 1052000000.0, 1295000000.0, 677000000.0, 952000000.0,
             1866000000.0, 2121000000.0, 2529000000.0],
            index=a_index
        ),
        total_current_liab=pd.Series(
            [2287000000.0, 2501000000.0, 2558000000.0, 2773000000.0, 2675000000.0, 3038000000.0, 3233000000.0,
             5147000000.0, 5807000000.0, 6437000000.0],
            index=a_index
        ),
        lt_debt=pd.Series(
            [2234000000.0, 2691000000.0, 1816000000.0, 2528000000.0, 3147000000.0, 3447000000.0, 3807000000.0,
             8197000000.0, 7840000000.0, 8514000000.0],
            index=a_index
        ),
        total_debt=pd.Series(
            [2234000000.0, 2691000000.0, 1816000000.0, 2528000000.0, 3147000000.0, 3447000000.0, 3807000000.0,
             8197000000.0, 7840000000.0, 8514000000.0],
            index=a_index
        ),
        deferred_rev=pd.Series(
            [583000000, 731000000, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        tax_liab_lt=pd.Series(
            [16000000.0, 1020000000.0, 605000000.0, 485000000.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            index=a_index
        ),
        deposit_liab=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        other_lt_liab=pd.Series(
            [2270000000.0, 2206000000.0, 2317000000.0, 2326000000.0, 2387000000.0, 2548000000.0, 2616000000.0,
             4419000000.0, 5429000000.0, 5304000000.0],
            index=a_index
        ),
        total_non_current_liab=pd.Series(
            [4504000000.0, 4897000000.0, 4133000000.0, 4854000000.0, 5534000000.0, 5995000000.0, 6439000000.0,
             13636000000.0, 14457000000.0, 15034000000.0],
            index=a_index
        ),
        total_liab=pd.Series(
            [6791000000.0, 7398000000.0, 6691000000.0, 7627000000.0, 8209000000.0, 9033000000.0, 9672000000.0,
             18783000000.0, 20264000000.0, 21471000000.0],
            index=a_index
        ),
        common_stock=pd.Series(
            [5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0,
             5000000.0],
            index=a_index
        ),
        other_income=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        retained_earnings=pd.Series(
            [3103000000.0, 3286000000.0, 3212000000.0, 3509000000.0, 3837000000.0, 4286000000.0, 4878000000.0,
             6501000000.0, 7242000000.0, 8982000000.0],
            index=a_index
        ),
        minority_interest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=a_index
        ),
        total_equity=pd.Series(
            [1142000000.0, 1585000000.0, 781000000.0, 1285000000.0, 1415000000.0, 2200000000.0, 3590000000.0,
             5357000000.0, 3582000000.0, 2225000000.0],
            index=a_index
        ),
        total_liab_and_equity=pd.Series(
            [7933000000.0, 8983000000.0, 7472000000.0, 8912000000.0, 9624000000.0, 11233000000.0, 13262000000.0,
             24140000000.0, 23846000000.0, 23696000000.0],
            index=a_index
        ),
    )
    q_index_str = ["2009-12-31 00:00:00", "2010-03-31 00:00:00", "2010-06-30 00:00:00", "2010-09-30 00:00:00",
                   "2010-12-31 00:00:00", "2011-03-31 00:00:00", "2011-06-30 00:00:00", "2011-09-30 00:00:00",
                   "2011-12-31 00:00:00", "2012-03-31 00:00:00", "2012-06-30 00:00:00", "2012-09-30 00:00:00",
                   "2012-12-31 00:00:00", "2013-03-31 00:00:00", "2013-06-30 00:00:00", "2013-09-30 00:00:00",
                   "2013-12-31 00:00:00", "2014-03-31 00:00:00", "2014-06-30 00:00:00", "2014-09-30 00:00:00",
                   "2014-12-31 00:00:00", "2015-03-31 00:00:00", "2015-06-30 00:00:00", "2015-09-30 00:00:00",
                   "2015-12-31 00:00:00", "2016-03-31 00:00:00", "2016-06-30 00:00:00", "2016-09-30 00:00:00",
                   "2016-12-31 00:00:00", "2017-03-31 00:00:00", "2017-06-30 00:00:00", "2017-09-30 00:00:00",
                   "2017-12-31 00:00:00", "2018-03-31 00:00:00", "2018-06-30 00:00:00", "2018-09-30 00:00:00",
                   "2018-12-31 00:00:00", "2019-03-31 00:00:00", "2019-06-30 00:00:00", "2019-09-30 00:00:00"]
    q_index = [pd.to_datetime(val) for val in q_index_str]
    q_test_data_dict = dict(
        revenue=pd.Series(
            [3380000000.0, 2630000000.0, 2771000000.0, 2648000000.0, 3642000000.0, 2778000000.0, 2972000000.0,
             2874000000.0, 3693000000.0, 2552000000.0, 2776000000.0, 2729000000.0, 3757000000.0, 3142000000.0,
             3263000000.0, 3160000000.0, 3219000000.0, 3293000000.0, 3484000000.0, 3460000000.0, 3559000000.0,
             3513000000.0, 3689000000.0, 3578000000.0, 3706000000.0, 3772000000.0, 3902000000.0, 3942000000.0,
             3791000000.0, 4912000000.0, 5211000000.0, 5078000000.0, 5251000000.0, 5009000000.0, 5409000000.0,
             5051000000.0, 5289000000.0, 5012000000.0, 5305000000.0, 5284000000.0],
            index=q_index
        ),
        cogs=pd.Series(
            [2943000000.0, 2312000000.0, 2403000000.0, 2332000000.0, 3169000000.0, 2428000000.0, 2581000000.0,
             2838000000.0, 3192000000.0, 2230000000.0, 2373000000.0, 2384000000.0, 3203000000.0, 2727000000.0,
             2791000000.0, 2734000000.0, 2768000000.0, 2855000000.0, 2962000000.0, 2957000000.0, 3056000000.0,
             2992000000.0, 3136000000.0, 3059000000.0, 3176000000.0, 3211000000.0, 3315000000.0, 3346000000.0,
             2863000000.0, 4052000000.0, 4141000000.0, 4001000000.0, 4445000000.0, 4144000000.0, 4298000000.0,
             4170000000.0, 4472000000.0, 4217000000.0, 4438000000.0, 4396000000.0],
            index=q_index
        ),
        gross_profit=pd.Series(
            [437000000.0, 318000000.0, 368000000.0, 316000000.0, 473000000.0, 350000000.0, 391000000.0, 36000000.0,
             501000000.0, 322000000.0, 403000000.0, 345000000.0, 554000000.0, 415000000.0, 472000000.0, 426000000.0,
             451000000.0, 438000000.0, 522000000.0, 503000000.0, 503000000.0, 521000000.0, 553000000.0, 519000000.0,
             530000000.0, 561000000.0, 587000000.0, 596000000.0, 928000000.0, 860000000.0, 1070000000.0, 1077000000.0,
             806000000.0, 865000000.0, 1111000000.0, 881000000.0, 817000000.0, 795000000.0, 867000000.0, 888000000.0],
            index=q_index
        ),
        rd_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        sga=pd.Series(
            [215000000.0, 138000000.0, 142000000.0, 149000000.0, 351000000.0, 159000000.0, 159000000.0, 180000000.0,
             254000000.0, 147000000.0, 160000000.0, 132000000.0, 143000000.0, 164000000.0, 160000000.0, 147000000.0,
             178000000.0, 148000000.0, 159000000.0, 172000000.0, 180000000.0, 145000000.0, 152000000.0, 149000000.0,
             188000000.0, 155000000.0, 154000000.0, 161000000.0, 273000000.0, 212000000.0, 234000000.0, 205000000.0,
             270000000.0, 247000000.0, 217000000.0, 221000000.0, 242000000.0, 222000000.0, 229000000.0, 220000000.0],
            index=q_index
        ),
        dep_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        other_op_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        op_exp=pd.Series(
            [215000000.0, 138000000.0, 142000000.0, 149000000.0, 351000000.0, 159000000.0, 159000000.0, 180000000.0,
             254000000.0, 147000000.0, 160000000.0, 132000000.0, 143000000.0, 164000000.0, 160000000.0, 147000000.0,
             178000000.0, 148000000.0, 159000000.0, 172000000.0, 180000000.0, 145000000.0, 152000000.0, 149000000.0,
             188000000.0, 155000000.0, 154000000.0, 161000000.0, 273000000.0, 212000000.0, 234000000.0, 205000000.0,
             270000000.0, 247000000.0, 217000000.0, 221000000.0, 242000000.0, 222000000.0, 229000000.0, 220000000.0],
            index=q_index
        ),
        ebit=pd.Series(
            [215000000.0, 180000000.0, 226000000.0, 167000000.0, 122000000.0, 191000000.0, 232000000.0, 144000000.0,
             247000000.0, 175000000.0, 243000000.0, 213000000.0, 309000000.0, 226000000.0, 279000000.0, 245000000.0,
             238000000.0, 254000000.0, 316000000.0, 298000000.0, 291000000.0, 332000000.0, 369000000.0, 339000000.0,
             310000000.0, 367000000.0, 389000000.0, 171000000.0, 497000000.0, 546000000.0, 744000000.0, 790000000.0,
             424000000.0, 530000000.0, 818000000.0, 596000000.0, 422000000.0, 510000000.0, 409000000.0, 607000000.0],
            index=q_index
        ),
        int_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        gain_on_sale_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        gain_on_sale_asset=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        impairment=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        ebt=pd.Series(
            [174000000.0, 129000000.0, 184000000.0, 128000000.0, 110000000.0, 152000000.0, 201000000.0, 199000000.0,
             202000000.0, 147000000.0, 209000000.0, 222000000.0, 271000000.0, 201000000.0, 263000000.0, 223000000.0,
             210000000.0, 231000000.0, 285000000.0, 290000000.0, 282000000.0, 307000000.0, 355000000.0, 309000000.0,
             284000000.0, 326000000.0, 344000000.0, 131000000.0, 438000000.0, 494000000.0, 716000000.0, 738000000.0,
             1034000000.0, 532000000.0, 874000000.0, 594000000.0, 345000000.0, 432000000.0, 314000000.0, 527000000.0],
            index=q_index
        ),
        tax_exp=pd.Series(
            [68000000.0, 46000000.0, 65000000.0, 45000000.0, 63000000.0, 51000000.0, 66000000.0, 20000000.0, 61000000.0,
             43000000.0, 66000000.0, 79000000.0, 90000000.0, 65000000.0, 84000000.0, 63000000.0, 59000000.0, 59000000.0,
             93000000.0, 98000000.0, 85000000.0, 100000000.0, 115000000.0, 99000000.0, 82000000.0, 107000000.0,
             97000000.0, 61000000.0, 166000000.0, 123000000.0, 227000000.0, 253000000.0, 920000000.0, 112000000.0,
             207000000.0, 91000000.0, 28000000.0, 57000000.0, 82000000.0, 140000000.0],
            index=q_index
        ),
        net_income=pd.Series(
            [106000000.0, 83000000.0, 119000000.0, 83000000.0, 47000000.0, 101000000.0, 135000000.0, 179000000.0,
             141000000.0, 104000000.0, 143000000.0, 143000000.0, 181000000.0, 136000000.0, 179000000.0, 160000000.0,
             151000000.0, 172000000.0, 192000000.0, 192000000.0, 197000000.0, 207000000.0, 240000000.0, 210000000.0,
             202000000.0, 219000000.0, 247000000.0, 70000000.0, 272000000.0, 371000000.0, 489000000.0, 485000000.0,
             114000000.0, 420000000.0, 667000000.0, 503000000.0, 317000000.0, 375000000.0, 232000000.0, 387000000.0],
            index=q_index
        ),
        cash=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        st_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        cash_and_st_invest=pd.Series(
            [115000000.0, 118000000.0, 100000000.0, 223000000.0, 505000000.0, 144000000.0, 117000000.0, 220000000.0,
             102000000.0, 290000000.0, 105000000.0, 105000000.0, 88000000.0, 221000000.0, 108000000.0, 144000000.0,
             126000000.0, 184000000.0, 192000000.0, 150000000.0, 104000000.0, 120000000.0, 140000000.0, 95000000.0,
             96000000.0, 99000000.0, 679000000.0, 1078000000.0, 858000000.0, 738000000.0, 498000000.0, 508000000.0,
             383000000.0, 701000000.0, 366000000.0, 373000000.0, 316000000.0, 258000000.0, 284000000.0, 276000000.0],
            index=q_index
        ),
        receivables=pd.Series(
            [838000000.0, 946000000.0, 956000000.0, 931000000.0, 938000000.0, 944000000.0, 1006000000.0, 1011000000.0,
             875000000.0, 863000000.0, 939000000.0, 841000000.0, 1028000000.0, 1080000000.0, 1006000000.0, 972000000.0,
             1081000000.0, 1035000000.0, 1057000000.0, 1030000000.0, 1100000000.0, 1172000000.0, 1188000000.0,
             1144000000.0, 1103000000.0, 1143000000.0, 1152000000.0, 1705000000.0, 1695000000.0, 1752000000.0,
             1804000000.0, 1914000000.0, 1973000000.0, 2098000000.0, 2174000000.0, 2175000000.0, 2133000000.0,
             2218000000.0, 2328000000.0, 2394000000.0],
            index=q_index
        ),
        inventory=pd.Series(
            [1444000000.0, 1503000000.0, 1467000000.0, 1455000000.0, 1489000000.0, 1459000000.0, 1365000000.0,
             1029000000.0, 11000000.0, 12000000.0, 12000000.0, 11000000.0, 11000000.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
             0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            index=q_index
        ),
        def_tax_st=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        other_current_assets=pd.Series(
            [131000000.0, 84000000.0, 93000000.0, 155000000.0, 123000000.0, 107000000.0, 120000000.0, 99000000.0,
             7000000.0, 70000000.0, 10000000.0, 22000000.0, 11000000.0, 24000000.0, 256000000.0, 377000000.0,
             52000000.0, 28000000.0, 111000000.0, 342000000.0, 161000000.0, 201000000.0, 169000000.0, 108000000.0,
             103000000.0, 57000000.0, 836000000.0, 588000000.0, 400000000.0, 285000000.0, 297000000.0, 149000000.0,
             121000000.0, 13000000.0, 13000000.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            index=q_index
        ),
        total_current_assets=pd.Series(
            [2851000000.0, 3015000000.0, 2951000000.0, 3061000000.0, 3382000000.0, 2999000000.0, 2938000000.0,
             2678000000.0, 1324000000.0, 1464000000.0, 1355000000.0, 1158000000.0, 1475000000.0, 1626000000.0,
             1404000000.0, 1646000000.0, 1903000000.0, 1565000000.0, 1537000000.0, 1552000000.0, 1610000000.0,
             1828000000.0, 1906000000.0, 1723000000.0, 1384000000.0, 1424000000.0, 1959000000.0, 3787000000.0,
             3371000000.0, 3121000000.0, 2830000000.0, 2944000000.0, 2740000000.0, 3152000000.0, 2817000000.0,
             2803000000.0, 2706000000.0, 2735000000.0, 2869000000.0, 2900000000.0],
            index=q_index
        ),
        gross_ppe=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        dep=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        net_ppe=pd.Series(
            [1362000000.0, 1339000000.0, 1335000000.0, 1327000000.0, 1307000000.0, 1412000000.0, 1483000000.0,
             1480000000.0, 1168000000.0, 1367000000.0, 1383000000.0, 1421000000.0, 1539000000.0, 1574000000.0,
             1634000000.0, 1489000000.0, 1543000000.0, 1569000000.0, 1647000000.0, 1736000000.0, 1460000000.0,
             1446000000.0, 987000000.0, 985000000.0, 1029000000.0, 1042000000.0, 1056000000.0, 2411000000.0,
             2335000000.0, 2109000000.0, 2102000000.0, 1894000000.0, 1793000000.0, 1791000000.0, 1958000000.0,
             1967000000.0, 1956000000.0, 1961000000.0, 1945000000.0, 1960000000.0],
            index=q_index
        ),
        goodwill=pd.Series(
            [1606000000.0, 1618000000.0, 1610000000.0, 1623000000.0, 1643000000.0, 1677000000.0, 1719000000.0,
             1716000000.0, 1721000000.0, 1727000000.0, 1720000000.0, 1772000000.0, 1989000000.0, 1993000000.0,
             1991000000.0, 1989000000.0, 2005000000.0, 2000000000.0, 2228000000.0, 2236000000.0, 2245000000.0,
             2289000000.0, 2414000000.0, 2420000000.0, 2394000000.0, 2414000000.0, 2420000000.0, 17223000000.0,
             16868000000.0, 17125000000.0, 17351000000.0, 17940000000.0, 17751000000.0, 17864000000.0, 17543000000.0,
             17470000000.0, 17419000000.0, 17655000000.0, 17681000000.0, 17540000000.0],
            index=q_index
        ),
        lt_invest=pd.Series(
            [249000000.0, 243000000.0, 245000000.0, 245000000.0, 250000000.0, 313000000.0, 278000000.0, 271000000.0,
             265000000.0, 268000000.0, 223000000.0, 217000000.0, 216000000.0, 230000000.0, 228000000.0, 228000000.0,
             222000000.0, 222000000.0, 229000000.0, 234000000.0, 224000000.0, 217000000.0, 216000000.0, 174000000.0,
             165000000.0, 169000000.0, 179000000.0, 830000000.0, 728000000.0, 745000000.0, 762000000.0, 720000000.0,
             734000000.0, 754000000.0, 638000000.0, 689000000.0, 732000000.0, 584000000.0, 580000000.0, 580000000.0],
            index=q_index
        ),
        def_tax_lt=pd.Series(
            [1020000000.0, 1091000000.0, 1074000000.0, 1050000000.0, 932000000.0, 919000000.0, 910000000.0,
             1005000000.0, 873000000.0, 849000000.0, 846000000.0, 851000000.0, 676000000.0, 673000000.0, 665000000.0,
             671000000.0, 647000000.0, 627000000.0, 617000000.0, 597000000.0, 819000000.0, 469000000.0, 397000000.0,
             450000000.0, 672000000.0, 620000000.0, 586000000.0, 92000000.0, 116000000.0, 119000000.0, 119000000.0,
             110000000.0, 93000000.0, 172000000.0, 171000000.0, 171000000.0, 171000000.0, 171000000.0, 171000000.0,
             169000000.0],
            index=q_index
        ),
        other_lt_assets=pd.Series(
            [393000000.0, 219000000.0, 202000000.0, 188000000.0, 205000000.0, 224000000.0, 336000000.0, 213000000.0,
             261000000.0, 273000000.0, 283000000.0, 275000000.0, 267000000.0, 257000000.0, 303000000.0, 320000000.0,
             332000000.0, 541000000.0, 352000000.0, 277000000.0, 260000000.0, 400000000.0, 247000000.0, 247000000.0,
             223000000.0, 234000000.0, 223000000.0, 419000000.0, 477000000.0, 439000000.0, 442000000.0, 400000000.0,
             593000000.0, 604000000.0, 596000000.0, 611000000.0, 587000000.0, 1516000000.0, 1550000000.0, 1554000000.0],
            index=q_index
        ),
        total_non_current_assets=pd.Series(
            [5082000000.0, 5778000000.0, 5696000000.0, 5728000000.0, 5601000000.0, 5761000000.0, 5853000000.0,
             5787000000.0, 4586000000.0, 4707000000.0, 4652000000.0, 4707000000.0, 4867000000.0, 4897000000.0,
             4973000000.0, 4834000000.0, 4891000000.0, 5100000000.0, 5293000000.0, 5295000000.0, 5223000000.0,
             4975000000.0, 4415000000.0, 4430000000.0, 4698000000.0, 4697000000.0, 4691000000.0, 21215000000.0,
             20769000000.0, 20804000000.0, 21051000000.0, 21292000000.0, 21106000000.0, 21332000000.0, 21027000000.0,
             21030000000.0, 20990000000.0, 22011000000.0, 22024000000.0, 21912000000.0],
            index=q_index
        ),
        total_assets=pd.Series(
            [7933000000.0, 8793000000.0, 8647000000.0, 8789000000.0, 8983000000.0, 8760000000.0, 8791000000.0,
             8465000000.0, 5910000000.0, 6171000000.0, 6007000000.0, 5865000000.0, 6342000000.0, 6523000000.0,
             6377000000.0, 6480000000.0, 6794000000.0, 6665000000.0, 6830000000.0, 6847000000.0, 6833000000.0,
             6803000000.0, 6321000000.0, 6153000000.0, 6082000000.0, 6121000000.0, 6650000000.0, 25002000000.0,
             24140000000.0, 23925000000.0, 23881000000.0, 24236000000.0, 23846000000.0, 24484000000.0, 23844000000.0,
             23833000000.0, 23696000000.0, 24746000000.0, 24893000000.0, 24812000000.0],
            index=q_index
        ),
        payables=pd.Series(
            [562000000.0, 499000000.0, 475000000.0, 509000000.0, 557000000.0, 607000000.0, 619000000.0, 548000000.0,
             579000000.0, 592000000.0, 594000000.0, 569000000.0, 517000000.0, 554000000.0, 496000000.0, 557000000.0,
             616000000.0, 617000000.0, 612000000.0, 605000000.0, 604000000.0, 640000000.0, 577000000.0, 593000000.0,
             597000000.0, 621000000.0, 667000000.0, 687000000.0, 661000000.0, 650000000.0, 733000000.0, 783000000.0,
             766000000.0, 834000000.0, 759000000.0, 767000000.0, 745000000.0, 927000000.0, 813000000.0, 0.0],
            index=q_index
        ),
        st_debt=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        current_lt_debt=pd.Series(
            [64000000.0, 145000000.0, 142000000.0, 126000000.0, 138000000.0, 137000000.0, 482000000.0, 480000000.0,
             355000000.0, 755000000.0, 407000000.0, 408000000.0, 407000000.0, 49000000.0, 50000000.0, 52000000.0,
             6000000.0, 7000000.0, 7000000.0, 7000000.0, 324000000.0, 325000000.0, 614000000.0, 615000000.0,
             300000000.0, 300000000.0, 303000000.0, 316000000.0, 309000000.0, 309000000.0, 402000000.0, 398000000.0,
             398000000.0, 988000000.0, 616000000.0, 617000000.0, 833000000.0, 231000000.0, 229000000.0, 227000000.0],
            index=q_index
        ),
        tax_liab_st=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        other_current_liab=pd.Series(
            [1142000000.0, 1126000000.0, 1136000000.0, 1147000000.0, 1037000000.0, 1094000000.0, 1154000000.0,
             1266000000.0, 1005000000.0, 970000000.0, 1010000000.0, 1020000000.0, 1052000000.0, 1094000000.0,
             1093000000.0, 1141000000.0, 1295000000.0, 1241000000.0, 1295000000.0, 1360000000.0, 677000000.0,
             1342000000.0, 1457000000.0, 1538000000.0, 952000000.0, 1013000000.0, 992000000.0, 1860000000.0,
             1866000000.0, 1948000000.0, 1948000000.0, 1959000000.0, 2121000000.0, 2238000000.0, 2177000000.0,
             2305000000.0, 2529000000.0, 2625000000.0, 2157000000.0, 2185000000.0],
            index=q_index
        ),
        total_current_liab=pd.Series(
            [2287000000.0, 2313000000.0, 2351000000.0, 2496000000.0, 2501000000.0, 2423000000.0, 2917000000.0,
             3100000000.0, 2558000000.0, 2906000000.0, 2642000000.0, 2690000000.0, 2773000000.0, 2358000000.0,
             2467000000.0, 2422000000.0, 2675000000.0, 2596000000.0, 2690000000.0, 2738000000.0, 3038000000.0,
             3030000000.0, 3517000000.0, 3509000000.0, 3233000000.0, 3247000000.0, 3244000000.0, 5229000000.0,
             5147000000.0, 5223000000.0, 5363000000.0, 5608000000.0, 5807000000.0, 6469000000.0, 5904000000.0,
             6051000000.0, 6437000000.0, 5848000000.0, 5624000000.0, 5738000000.0],
            index=q_index
        ),
        lt_debt=pd.Series(
            [2234000000.0, 3124000000.0, 2769000000.0, 2600000000.0, 2691000000.0, 2720000000.0, 2440000000.0,
             2623000000.0, 1816000000.0, 1772000000.0, 2153000000.0, 2101000000.0, 2528000000.0, 3206000000.0,
             3037000000.0, 3104000000.0, 3147000000.0, 3295000000.0, 3397000000.0, 3521000000.0, 3447000000.0,
             3703000000.0, 3303000000.0, 3689000000.0, 3807000000.0, 3859000000.0, 4057000000.0, 8506000000.0,
             8197000000.0, 8161000000.0, 7911000000.0, 8271000000.0, 7840000000.0, 7858000000.0, 8375000000.0,
             8710000000.0, 8514000000.0, 10914000000.0, 11078000000.0, 11420000000.0],
            index=q_index
        ),
        total_debt=pd.Series(
            [2234000000.0, 3124000000.0, 2769000000.0, 2600000000.0, 2691000000.0, 2720000000.0, 2440000000.0,
             2623000000.0, 1816000000.0, 1772000000.0, 2153000000.0, 2101000000.0, 2528000000.0, 3206000000.0,
             3037000000.0, 3104000000.0, 3147000000.0, 3295000000.0, 3397000000.0, 3521000000.0, 3447000000.0,
             3703000000.0, 3303000000.0, 3689000000.0, 3807000000.0, 3859000000.0, 4057000000.0, 8506000000.0,
             8197000000.0, 8161000000.0, 7911000000.0, 8271000000.0, 7840000000.0, 7858000000.0, 8375000000.0,
             8710000000.0, 8514000000.0, 10914000000.0, 11078000000.0, 11420000000.0],
            index=q_index
        ),
        deferred_rev=pd.Series(
            [583000000, 645000000, 645000000, 632000000, 731000000, 743000000, 768000000, 799000000, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_index
        ),
        tax_liab_lt=pd.Series(
            [16000000.0, 1308000000.0, 1020000000.0, 891000000.0, 994000000.0, 927000000.0, 605000000.0, 634000000.0,
             567000000.0, 551000000.0, 485000000.0, 441000000.0, 359000000.0, 291000000.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
             0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            index=q_index
        ),
        deposit_liab=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        other_lt_liab=pd.Series(
            [2270000000.0, 2286000000.0, 2303000000.0, 2338000000.0, 2206000000.0, 2231000000.0, 2298000000.0,
             2312000000.0, 2317000000.0, 2341000000.0, 2336000000.0, 2370000000.0, 2326000000.0, 2336000000.0,
             2366000000.0, 2363000000.0, 2387000000.0, 2399000000.0, 2463000000.0, 2430000000.0, 2548000000.0,
             2607000000.0, 2534000000.0, 2544000000.0, 2616000000.0, 2682000000.0, 2811000000.0, 4118000000.0,
             4419000000.0, 4482000000.0, 4712000000.0, 4918000000.0, 5429000000.0, 5312000000.0, 5512000000.0,
             5565000000.0, 5304000000.0, 5197000000.0, 5824000000.0, 5726000000.0],
            index=q_index
        ),
        total_non_current_liab=pd.Series(
            [4504000000.0, 5410000000.0, 5072000000.0, 4938000000.0, 4897000000.0, 4951000000.0, 4738000000.0,
             4935000000.0, 4133000000.0, 4113000000.0, 4489000000.0, 4471000000.0, 4854000000.0, 5542000000.0,
             5403000000.0, 5467000000.0, 5534000000.0, 5694000000.0, 5860000000.0, 5951000000.0, 5995000000.0,
             6310000000.0, 5837000000.0, 6233000000.0, 6439000000.0, 6541000000.0, 6868000000.0, 13932000000.0,
             13636000000.0, 13534000000.0, 13617000000.0, 14116000000.0, 14457000000.0, 14449000000.0, 15099000000.0,
             15458000000.0, 15034000000.0, 17295000000.0, 18029000000.0, 18236000000.0],
            index=q_index
        ),
        total_liab=pd.Series(
            [6791000000.0, 7723000000.0, 7423000000.0, 7434000000.0, 7398000000.0, 7374000000.0, 7655000000.0,
             8035000000.0, 6691000000.0, 7019000000.0, 7131000000.0, 7161000000.0, 7627000000.0, 7900000000.0,
             7870000000.0, 7889000000.0, 8209000000.0, 8290000000.0, 8550000000.0, 8689000000.0, 9033000000.0,
             9340000000.0, 9354000000.0, 9742000000.0, 9672000000.0, 9788000000.0, 10112000000.0, 19161000000.0,
             18783000000.0, 18757000000.0, 18980000000.0, 19724000000.0, 20264000000.0, 20918000000.0, 21003000000.0,
             21509000000.0, 21471000000.0, 23143000000.0, 23653000000.0, 23974000000.0],
            index=q_index
        ),
        common_stock=pd.Series(
            [5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0,
             5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0,
             5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0,
             5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0, 5000000.0,
             5000000.0, 5000000.0, 5000000.0, 5000000.0],
            index=q_index
        ),
        other_income=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        retained_earnings=pd.Series(
            [3103000000.0, 3038000000.0, 3130000000.0, 3191000000.0, 3286000000.0, 3310000000.0, 3401000000.0,
             3165000000.0, 3212000000.0, 3249000000.0, 3335000000.0, 3434000000.0, 3509000000.0, 3549000000.0,
             3662000000.0, 3763000000.0, 3837000000.0, 3917000000.0, 4030000000.0, 4156000000.0, 4286000000.0,
             4401000000.0, 4573000000.0, 4740000000.0, 4878000000.0, 5025000000.0, 5185000000.0, 6375000000.0,
             6501000000.0, 6750000000.0, 7040000000.0, 7310000000.0, 7242000000.0, 7898000000.0, 8363000000.0,
             8705000000.0, 8982000000.0, 9219000000.0, 9292000000.0, 9522000000.0],
            index=q_index
        ),
        minority_interest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0],
            index=q_index
        ),
        total_equity=pd.Series(
            [1142000000.0, 1070000000.0, 1224000000.0, 1355000000.0, 1585000000.0, 1386000000.0, 1136000000.0,
             430000000.0, 781000000.0, 848000000.0, 1124000000.0, 1296000000.0, 1285000000.0, 1377000000.0,
             1493000000.0, 1409000000.0, 1415000000.0, 1625000000.0, 1720000000.0, 1842000000.0, 2200000000.0,
             2537000000.0, 3033000000.0, 3589000000.0, 3590000000.0, 3667000000.0, 3462000000.0, 5841000000.0,
             5357000000.0, 5168000000.0, 4901000000.0, 4512000000.0, 3582000000.0, 3566000000.0, 2841000000.0,
             2324000000.0, 2225000000.0, 1603000000.0, 1240000000.0, 838000000.0],
            index=q_index
        ),
        total_liab_and_equity=pd.Series(
            [7933000000.0, 8793000000.0, 8647000000.0, 8789000000.0, 8983000000.0, 8760000000.0, 8791000000.0,
             8465000000.0, 7472000000.0, 7867000000.0, 8255000000.0, 8457000000.0, 8912000000.0, 9277000000.0,
             9363000000.0, 9298000000.0, 9624000000.0, 9915000000.0, 10270000000.0, 10531000000.0, 11233000000.0,
             11877000000.0, 12387000000.0, 13331000000.0, 13262000000.0, 13455000000.0, 13574000000.0, 25002000000.0,
             24140000000.0, 23925000000.0, 23881000000.0, 24236000000.0, 23846000000.0, 24484000000.0, 23844000000.0,
             23833000000.0, 23696000000.0, 24746000000.0, 24893000000.0, 24812000000.0],
            index=q_index
        ),
    )

    def test_annual(self, annual_stockrow_stmts_mar: FinancialStatements):
        super().test_annual(annual_stockrow_stmts_mar)

    def test_quarterly(self, quarterly_stockrow_stmts_mar: FinancialStatements):
        super().test_quarterly(quarterly_stockrow_stmts_mar)


class TestLoadCapitalIQ(LoadTest):
    name = 'load_capiq_cat'
    a_test_data_dict = LOAD_CAPIQ_CAT_A_INDEX_DATA_DICT

    q_inc_index_str = ["2012-03-31 00:00:00", "2012-06-30 00:00:00", "2012-09-30 00:00:00", "2012-12-31 00:00:00",
                       "2013-03-31 00:00:00", "2013-06-30 00:00:00", "2013-09-30 00:00:00", "2013-12-31 00:00:00",
                       "2014-03-31 00:00:00", "2014-06-30 00:00:00", "2014-09-30 00:00:00", "2014-12-31 00:00:00",
                       "2015-03-31 00:00:00", "2015-06-30 00:00:00", "2015-09-30 00:00:00", "2015-12-31 00:00:00",
                       "2016-03-31 00:00:00", "2016-06-30 00:00:00", "2016-09-30 00:00:00", "2016-12-31 00:00:00",
                       "2017-03-31 00:00:00", "2017-06-30 00:00:00", "2017-09-30 00:00:00", "2017-12-31 00:00:00",
                       "2018-03-31 00:00:00", "2018-06-30 00:00:00", "2018-09-30 00:00:00", "2018-12-31 00:00:00",
                       "2019-03-31 00:00:00", "2019-06-30 00:00:00", "2019-09-30 00:00:00"]
    q_inc_index = [pd.to_datetime(val) for val in q_inc_index_str]
    q_inc_test_data_dict = dict(
        revenue=pd.Series(
            [15981, 17374, 16445, 16075, 13210, 14621, 13423, 14402, 13241, 14150, 13549, 14244, 12702, 12317, 10962,
             11030, 9461, 10342, 9160, 9574, 9822, 11331, 11413, 12896, 12859, 14011, 13510, 14342, 13466, 14432,
             12758],
            index=q_inc_index
        ),
        cogs=pd.Series(
            [11237, 12280, 11639, 11899, 9639, 10773, 9774, 10541, 9437, 10197, 9634, 10499, 8760, 8674, 7872, 8240,
             6822, 7419, 6527, 7541, 6801, 7816, 7678, 8965, 8566, 9422, 9022, 9987, 9003, 9941, 8569],
            index=q_inc_index
        ),
        gross_profit=pd.Series(
            [4744, 5094, 4806, 4176, 3571, 3848, 3649, 3861, 3804, 3953, 3915, 3745, 3942, 3643, 3090, 2790, 2639, 2923,
             2633, 2033, 3021, 3515, 3735, 3931, 4293, 4589, 4488, 4355, 4463, 4491, 4189],
            index=q_inc_index
        ),
        rd_exp=pd.Series(
            [587, 632, 634, 613, 562, 548, 469, 467, 508, 516, 533, 578, 524, 510, 513, 572, 508, 468, 453, 522, 425,
             458, 461, 498, 443, 462, 479, 466, 435, 441, 431],
            index=q_inc_index
        ),
        sga=pd.Series(
            [1213, 1367, 1318, 1403, 1261, 1271, 1163, 1286, 1146, 1278, 1274, 1364, 1116, 1146, 990, 1111, 949, 976,
             854, 1334, 935, 1165, 1081, 1244, 1001, 1131, 1045, 1372, 1101, 1076, 1064],
            index=q_inc_index
        ),
        dep_exp=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_inc_index
        ),
        other_op_exp=pd.Series(
            [74, 129, 91, 233, 7, 4, 34, 3, 1, 54, 25, 14, 17, 33, 2, 25, 23, 4, 15, 37, 9, 16, 0, 17, 43, 43, 5, 2, 23,
             44, 33],
            index=q_inc_index
        ),
        op_exp=pd.Series(
            [1874.0, 2128.0, 2043.0, 2249.0, 1830.0, 1823.0, 1666.0, 1756.0, 1655.0, 1848.0, 1832.0, 1956.0, 1657.0,
             1689.0, 1505.0, 1708.0, 1480.0, 1448.0, 1322.0, 1893.0, 1369.0, 1639.0, 1542.0, 1759.0, 1487.0, 1636.0,
             1529.0, 1840.0, 1559.0, 1561.0, 1528.0],
            index=q_inc_index
        ),
        ebit=pd.Series(
            [2287, 2628, 2333, 1769, 1225, 1442, 1441, 1522, 1562, 1604, 1482, 1168, 1736, 1401, 1025, 496, 616, 908,
             758, 397, 1063, 1308, 1587, 1589, 2225, 2335, 2301, 1732, 2296, 2295, 2063],
            index=q_inc_index
        ),
        int_exp=pd.Series(
            [143, 104, 121, 137, 119, 120, 115, 224, 109, 118, 146, 151, 129, 125, 127, 126, 129, 130, 126, 120, 123,
             121, 118, 169, 101, 102, 102, 99, 103, 103, 103],
            index=q_inc_index
        ),
        gain_on_sale_invest=pd.Series(
            [0, -1, 0, 3, 1, -1, 0, 20, 14, 0, 21, 1, 124, 1, 31, 20, 0, 31, 9, 7, 0, 85, 36, 66, -2, 7, 13, -33, 39,
             14, 2],
            index=q_inc_index
        ),
        gain_on_sale_asset=pd.Series(
            [0, 0, 276, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_inc_index
        ),
        impairment=pd.Series(
            [0, 0, 0, 580, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 595, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_inc_index
        ),
        ebt=pd.Series(
            [2300, 2581, 2455, 914, 1128, 1352, 1260, 1382, 1343, 1421, 1385, 942, 1769, 1138, 780, 248, 364, 737, 379,
             1347, 284, 1164, 1531, 1119, 2139, 2195, 2142, 1370, 2271, 2184, 2012],
            index=q_inc_index
        ),
        tax_exp=pd.Series(
            [689, 872, 753, 214, 246, 387, 310, 376, 418, 419, 364, 179, 521, 335, 218, 158, 92, 184, 96, 180, 90, 361,
             470, 2418, 472, 490, 415, 321, 387, 565, 518],
            index=q_inc_index
        ),
        net_income=pd.Series(
            [1586, 1699, 1699, 697, 880, 960, 946, 1003, 922, 999, 1017, 757, 1245, 802, 559, 94, 271, 550, 283, 1171,
             192, 802, 1059, 1299, 1665, 1707, 1727, 1048, 1881, 1620, 1494],
            index=q_inc_index
        ),
    )
    q_bs_index_str = ["2012-03-31 00:00:00", "2012-06-30 00:00:00", "2012-09-30 00:00:00", "2012-12-31 00:00:00",
                      "2013-03-31 00:00:00", "2013-06-30 00:00:00", "2013-09-30 00:00:00", "2013-12-31 00:00:00",
                      "2014-03-31 00:00:00", "2014-06-30 00:00:00", "2014-09-30 00:00:00", "2014-12-31 00:00:00",
                      "2015-03-31 00:00:00", "2015-06-30 00:00:00", "2015-09-30 00:00:00", "2015-12-31 00:00:00",
                      "2016-03-31 00:00:00", "2016-06-30 00:00:00", "2016-09-30 00:00:00", "2016-12-31 00:00:00",
                      "2017-03-31 00:00:00", "2017-06-30 00:00:00", "2017-09-30 00:00:00", "2017-12-31 00:00:00",
                      "2018-03-31 00:00:00", "2018-06-30 00:00:00", "2018-09-30 00:00:00", "2018-12-31 00:00:00",
                      "2019-03-31 00:00:00", "2019-06-30 00:00:00", "2019-09-30 00:00:00"]
    q_bs_index = [pd.to_datetime(val) for val in q_bs_index_str]
    q_bs_test_data_dict = dict(
        cash=pd.Series(
            [2864, 2810, 3363, 3306, 3594, 3974, 4108, 4597, 4057, 6615, 4805, 6317, 6218, 6466, 4808, 5340, 4744, 5579,
             4894, 5257, 8223, 8926, 8736, 7381, 7034, 7786, 7189, 6968, 6358, 6528, 6380],
            index=q_bs_index
        ),
        st_invest=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_bs_index
        ),
        cash_and_st_invest=pd.Series(
            [2864, 2810, 3363, 3306, 3594, 3974, 4108, 4597, 4057, 6615, 4805, 6317, 6218, 6466, 4808, 5340, 4744, 5579,
             4894, 5257, 8223, 8926, 8736, 7381, 7034, 7786, 7189, 6968, 6358, 6528, 6380],
            index=q_bs_index
        ),
        receivables=pd.Series(
            [10370, 10376, 9713, 9617, 9733, 9268, 8533, 8326, 8499, 7993, 7643, 7699, 7554, 7185, 6757, 6677, 6784,
             6267, 5762, 5919, 6480, 6583, 6612, 7376, 7828, 7922, 7863, 8714, 8881, 8903, 8167],
            index=q_bs_index
        ),
        inventory=pd.Series(
            [16511, 17344, 17550, 15547, 15074, 13889, 13392, 12625, 12888, 13055, 13328, 12205, 12099, 11681, 11150,
             9700, 9849, 9458, 9478, 8614, 9082, 9388, 10212, 10018, 10947, 11255, 11814, 11529, 12340, 12007, 12180],
            index=q_bs_index
        ),
        def_tax_st=pd.Series(
            [1501, 1621, 1563, 1501, 1449, 1492, 1501, 1511, 1347, 1410, 1318, 1644, 1372, 1386, 1400, 0, 0, 0, 0, 0, 0,
             0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_bs_index
        ),
        other_current_assets=pd.Series(
            [0, 41, 160, 90, 110, 135, 154, 73, 65, 48, 40, 27, 20, 20, 22, 14, 49, 53, 33, 23, 25, 46, 37, 48, 52, 23,
             18, 19, 27, 35, 27],
            index=q_bs_index
        ),
        total_current_assets=pd.Series(
            [40209, 44294, 44639, 42138, 42145, 40805, 40088, 38335, 37968, 41276, 39042, 38867, 38491, 38227, 35280,
             33508, 33748, 33606, 31999, 31967, 35548, 36991, 37185, 36244, 37357, 38641, 38454, 38603, 39126, 39789,
             39160],
            index=q_bs_index
        ),
        gross_ppe=pd.Series(
            [0, 0, 0, 29932, 0, 0, 0, 31316, 0, 0, 0, 31572, 0, 0, 0, 31977, 0, 0, 0, 31940, 0, 0, 0, 31538, 0, 0, 0,
             29781, 0, 0, 0],
            index=q_bs_index
        ),
        dep=pd.Series(
            [0, 0, 0, 13471, 0, 0, 0, 14241, 0, 0, 0, 14995, 0, 0, 0, 15887, 0, 0, 0, 16618, 0, 0, 0, 17383, 0, 0, 0,
             16207, 0, 0, 0],
            index=q_bs_index
        ),
        net_ppe=pd.Series(
            [14571, 11832, 12217, 16461, 12854, 12814, 12852, 17075, 12773, 12578, 12350, 16577, 12156, 11970, 11816,
             16090, 11668, 11432, 11194, 15322, 10427, 10071, 9851, 14155, 9486, 9244, 9106, 13574, 9561, 9327, 9131],
            index=q_bs_index
        ),
        goodwill=pd.Series(
            [7081, 7303, 7355, 6925, 6823, 6797, 6951, 6939, 6969, 6952, 6784, 6677, 6476, 6533, 6529, 6598, 6693, 6660,
             6708, 6003, 6034, 6125, 6179, 6183, 6359, 6249, 6233, 6217, 6191, 6211, 6142],
            index=q_bs_index
        ),
        lt_invest=pd.Series(
            [139, 124, 199, 272, 270, 288, 278, 272, 266, 259, 265, 257, 230, 229, 231, 246, 246, 246, 247, 249, 0, 0,
             0, 0, 0, 0, 0, 0, 1225, 1280, 1328],
            index=q_bs_index
        ),
        def_tax_lt=pd.Series(
            [2082, 2032, 1873, 1863, 2112, 2153, 1985, 456, 700, 737, 761, 1267, 1342, 1473, 1559, 2367, 2486, 2536,
             2579, 2683, 2940, 2866, 2845, 1569, 1579, 1521, 1190, 1363, 1257, 1361, 1254],
            index=q_bs_index
        ),
        other_lt_assets=pd.Series(
            [2208, 860, 864, 584, 365, 381, 409, 577, 434, 370, 381, 528, 398, 386, 1740, 1874, 1755, 1798, 1783, 1529,
             561, 600, 627, 950, 974, 953, 927, 962, 1349, 1370, 1400],
            index=q_bs_index
        ),
        total_non_current_assets=pd.Series(
            [26081.0, 22151.0, 22508.0, 26105.0, 22424.0, 22433.0, 22475.0, 25319.0, 21142.0, 20896.0, 20541.0,
             25306.0, 20602.0, 20591.0, 21875.0, 27175.0, 22848.0, 22672.0, 22511.0, 25786.0, 19962.0, 19662.0,
             19502.0, 22857.0, 18398.0, 17967.0, 17456.0, 22116.0, 19583.0, 19549.0, 19255.0],
            index=q_bs_index
        ),
        total_assets=pd.Series(
            [83908, 88838, 90541, 88970, 88743, 87375, 87272, 84896, 84397, 87827, 84488, 84681, 82490, 82242, 78969,
             78342, 78307, 78300, 76402, 74704, 77549, 78510, 78560, 76962, 78014, 78987, 78209, 78509, 78726, 79187,
             77993],
            index=q_bs_index
        ),
        payables=pd.Series(
            [8360, 8470, 7978, 6753, 6221, 6343, 6280, 6560, 6731, 6860, 6778, 6515, 6328, 5862, 5206, 5023, 5101, 5104,
             4713, 4614, 5302, 5778, 6113, 6487, 6938, 6831, 6788, 7051, 7198, 7022, 6141],
            index=q_bs_index
        ),
        st_debt=pd.Series(
            [696, 592, 607, 636, 667, 392, 290, 16, 18, 20, 7, 9, 8, 14, 12, 9, 13, 263, 263, 209, 436, 5, 11, 1, 7, 35,
             59, 0, 4, 0, 0],
            index=q_bs_index
        ),
        current_lt_debt=pd.Series(
            [555, 1269, 1266, 1113, 1111, 1112, 1110, 760, 797, 509, 509, 510, 513, 12, 516, 517, 568, 1055, 553, 507,
             505, 5, 5, 6, 0, 0, 0, 10, 13, 13, 25],
            index=q_bs_index
        ),
        tax_liab_st=pd.Series(
            [0, 0, 0, 484, 0, 0, 0, 447, 0, 0, 0, 414, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_bs_index
        ),
        other_current_liab=pd.Series(
            [4927, 5687, 5232, 4756, 4973, 5039, 4609, 4545, 4368, 4580, 4121, 4004, 3582, 4054, 2909, 2955, 2598, 2941,
             2411, 3377, 2606, 3298, 2760, 3424, 3299, 3859, 3449, 3735, 3543, 3837, 3364],
            index=q_bs_index
        ),
        total_current_liab=pd.Series(
            [28849, 31273, 31746, 29415, 28974, 29919, 29119, 27297, 27295, 29741, 27589, 27877, 26566, 25606, 25833,
             26242, 26215, 27183, 25290, 26132, 27635, 28133, 25903, 26931, 27402, 28300, 26033, 28218, 27388, 27735,
             27201],
            index=q_bs_index
        ),
        lt_debt=pd.Series(
            [8411, 9169, 9010, 8593, 8668, 7949, 7951, 7902, 7998, 9495, 9498, 9408, 9495, 9497, 8997, 8883, 8914, 8434,
             8432, 8368, 8804, 8815, 8820, 7492, 7980, 7982, 7991, 7549, 7650, 7657, 9134],
            index=q_bs_index
        ),
        total_debt=pd.Series(
            [35953, 39325, 39865, 40152, 40495, 39557, 39543, 37756, 38135, 40251, 39291, 39293, 38781, 38325, 37547,
             38017, 38366, 38327, 37112, 36784, 38283, 37193, 35925, 34880, 35318, 36172, 35776, 36593, 36541, 37016,
             38607],
            index=q_bs_index
        ),
        deferred_rev=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_bs_index
        ),
        tax_liab_lt=pd.Series(
            [0, 0, 0, 484, 0, 0, 0, 447, 0, 0, 0, 414, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_bs_index
        ),
        deposit_liab=pd.Series(
            [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
            index=q_bs_index
        ),
        other_lt_liab=pd.Series(
            [3582, 3697, 3625, 2652, 3212, 3232, 3176, 2582, 3217, 3259, 3284, 2817, 3206, 3271, 2869, 3203, 2820, 2838,
             2812, 3184, 3238, 3235, 3152, 3657, 3518, 3520, 3367, 3319, 3236, 3448, 3330],
            index=q_bs_index
        ),
        total_non_current_liab=pd.Series(
            [11993.0, 12866.0, 12635.0, 11729.0, 11880.0, 11181.0, 11127.0, 10931.0, 11215.0, 12754.0, 12782.0,
             12639.0, 12701.0, 12768.0, 11866.0, 12086.0, 11734.0, 11272.0, 11244.0, 11552.0, 12042.0, 12050.0,
             11972.0, 11149.0, 11498.0, 11502.0, 11358.0, 10868.0, 10886.0, 11105.0, 12464.0],
            index=q_bs_index
        ),
        total_liab=pd.Series(
            [68437, 72857, 72605, 71388, 70425, 69697, 69095, 64018, 64028, 66904, 65592, 67855, 65452, 65081, 63001,
             63457, 62554, 62997, 60687, 61491, 63889, 64431, 62863, 63196, 62742, 64045, 62319, 64429, 63248, 64309,
             63000],
            index=q_bs_index
        ),
        common_stock=pd.Series(
            [4351, 4373, 4449, 4481, 4510, 4591, 4657, 4709, 4773, 4890, 4968, 5016, 5101, 5142, 5190, 5238, 5247, 5277,
             5266, 5277, 5222, 5316, 5460, 5593, 5640, 5746, 5715, 5827, 5804, 5822, 5951],
            index=q_bs_index
        ),
        other_income=pd.Series(
            [-6033, -6150, -5988, -6433, -6669, -6698, -6247, -3898, -3801, -3683, -4357, -6431, -7101, -6729, -6843,
             -2035, -1493, -1633, -1527, -2039, -1827, -1471, -1233, -1192, -1016, -1496, -1568, -1684, -1588, -1499,
             -1783],
            index=q_bs_index
        ),
        retained_earnings=pd.Series(
            [26815, 27842, 29541, 29558, 30438, 30668, 31614, 31854, 32775, 32961, 33977, 33887, 34998, 34823, 35191,
             29246, 29517, 29167, 29450, 27377, 27584, 27471, 28530, 26301, 27929, 28657, 30384, 30427, 32435, 32981,
             34477],
            index=q_bs_index
        ),
        minority_interest=pd.Series(
            [502, 55, 52, 50, 44, 57, 67, 67, 64, 67, 73, 80, 76, 69, 72, 76, 77, 71, 70, 76, 72, 70, 70, 69, 66, 63,
             40, 41, 41, 41, 41],
            index=q_bs_index
        ),
        total_equity=pd.Series(
            [15471, 15981, 17936, 17582, 18318, 17678, 18177, 20878, 20369, 20923, 18896, 16826, 17038, 17161, 15968,
             14885, 15753, 15303, 15715, 13213, 13660, 14079, 15697, 13766, 15272, 14942, 15890, 14080, 15478, 14878,
             14993],
            index=q_bs_index
        ),
        total_liab_and_equity=pd.Series(
            [83908, 88838, 90541, 88970, 88743, 87375, 87272, 84896, 84397, 87827, 84488, 84681, 82490, 82242, 78969,
             78342, 78307, 78300, 76402, 74704, 77549, 78510, 78560, 76962, 78014, 78987, 78209, 78509, 78726, 79187,
             77993],
            index=q_bs_index
        ),
    )
    q_test_data_dict = q_bs_test_data_dict.copy()
    q_test_data_dict.update(q_inc_test_data_dict)

    def test_annual(self, annual_capiq_stmts: FinancialStatements):
        super().test_annual(annual_capiq_stmts)

    def test_quarterly(self, quarterly_capiq_stmts: FinancialStatements):
        super().test_quarterly(quarterly_capiq_stmts)
