# Copyright (c) 2021 Emanuele Bellocchia
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

"""Module for Substrate address computation."""

# Imports
from typing import Any, Union
from bip_utils.addr.iaddr_encoder import IAddrEncoder
from bip_utils.addr.utils import AddrUtils
from bip_utils.ecc import IPublicKey
from bip_utils.ss58 import SS58Encoder


class SubstrateEd25519Addr(IAddrEncoder):
    """
    Substrate address class based on ed25519 keys.
    It allows the Substrate address generation.
    """

    @staticmethod
    def EncodeKey(pub_key: Union[bytes, IPublicKey],
                  **kwargs: Any) -> str:
        """
        Get address in Substrate format.

        Args:
            pub_key (bytes or IPublicKey): Public key bytes or object

        Other Parameters:
            ss58_format (int): SS58 format

        Returns:
            str: Address string

        Raised:
            ValueError: If the public key is not valid
        """
        ss58_format = kwargs["ss58_format"]

        pub_key_obj = AddrUtils.ValidateAndGetEd25519Key(pub_key)

        return SS58Encoder.Encode(pub_key_obj.RawCompressed().ToBytes()[1:], ss58_format)


class SubstrateSr25519Addr(IAddrEncoder):
    """
    Substrate address class based on sr25519 keys.
    It allows the Substrate address generation.
    """

    @staticmethod
    def EncodeKey(pub_key: Union[bytes, IPublicKey],
                  **kwargs: Any) -> str:
        """
        Get address in Substrate format.

        Args:
            pub_key (bytes or IPublicKey): Public key bytes or object

        Other Parameters:
            ss58_format (int): SS58 format

        Returns:
            str: Address string

        Raised:
            ValueError: If the public key is not valid
        """
        ss58_format = kwargs["ss58_format"]

        pub_key_obj = AddrUtils.ValidateAndGetSr25519Key(pub_key)

        return SS58Encoder.Encode(pub_key_obj.RawCompressed().ToBytes(), ss58_format)
