# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['djangocms_xliff',
 'djangocms_xliff.management',
 'djangocms_xliff.management.commands']

package_data = \
{'': ['*'],
 'djangocms_xliff': ['locale/de/LC_MESSAGES/*',
                     'templates/djangocms_xliff/*',
                     'templates/djangocms_xliff/export/*',
                     'templates/djangocms_xliff/import/*']}

install_requires = \
['Django>=3.2,<4.1', 'defusedxml>=0.7', 'django-cms>=3.9', 'requests>=2.20']

setup_kwargs = {
    'name': 'djangocms-xliff',
    'version': '1.0.0',
    'description': 'XLIFF Import and Export for the Django CMS',
    'long_description': '# djangocms-xliff\n\n[![Tests](https://github.com/energie360/djangocms-xliff/actions/workflows/tests.yaml/badge.svg?branch=main)](https://github.com/energie360/djangocms-xliff/actions/workflows/tests.yaml)\n\nXLIFF (XML Localization Interchange File Format) is an XML-based format created\nto standardize the way localizable data are passed between and among tools during a localization process.\n\nWith djangocms-xliff it is possible to export all text objects from a page into an XLIFF-compatible file and re-import\nthe file at the end of the translation process.\n\n## Installation\n\nBefore the installation you need to make sure, that your\nlocalization / internalization are set up properly\nfor [Django](https://docs.djangoproject.com/en/dev/topics/i18n/translation/)\nand [Django-CMS](https://docs.django-cms.org/en/latest/topics/i18n.html)\n\n### Setup\n\ndjangocms-xliff is available on [PyPI](https://pypi.org/project/djangocms-xliff/):\n\n```shell\n$ pip install djangocms-xliff\n```\n\nAdd `djangocms_xliff` to your `INSTALLED_APPS`.\n\n```python\nINSTALLED_APPS = (\n    ...,\n    "djangocms_xliff"\n)\n```\n\nAdd the views for `djangocms_xliff` to your `urls.py`\n\n```python\nurlpatterns = [\n    path("xliff/", include("djangocms_xliff.urls"))\n]\n```\n\n## Documentation\n\nTo make the process fail-safe there are some Django CMS related restrictions:\n\n* You can only import the file to the same page and language that you exported from before.\n* It is not possible to export a file for one language and import it to another language.\n* It is not possible to add fields during the translation process. (Missing fields will be ignored.)\n\nThis is because the reference for an entity is the unique ID of the Django CMS plugin, and each plugin has its own\nunique ID for each page and language.\n\nTherefore, you need to follow these steps to work with djangocms-xliff.\n\n### Step-by-step\n\nIf the page does not exist yet in the target language, create it and copy the plugins from the page with the source\nplugins.\n\nGo to the page in the target language.\n\nExport the XLIFF file at Language > Export as XLIFF…\n\n![Export](docs/screenshots/export.png)\n\nThis will generate an XLIFF file in the following format:\n\n```xml\n<?xml version="1.0" encoding="utf-8" standalone="no"?>\n<xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">\n    <file original="verbund/meilen/projekt" datatype="plaintext" source-language="fr" target-language="en">\n        <tool tool-id="96" tool-name="djangocms_xliff" tool-company-name="Energie 360°"/>\n        <body>\n            <trans-unit id="5872__title" resname="5872__title" maxwidth="60" size-unit="char"\n                        extype="django.db.models.fields.CharField">\n                <source><![CDATA[The project in short]]></source>\n                <target><![CDATA[]]></target>\n                <note>CarouselTripleBlockWrapperPlugin</note>\n                <note>Carousel Triple Block</note>\n                <note>Titel</note>\n                <note>Max characters: 60</note>\n            </trans-unit>\n            <trans-unit id="5874__title" resname="5874__title" maxwidth="35" size-unit="char"\n                        extype="django.db.models.fields.CharField">\n                <source><![CDATA[Practical Solution]]></source>\n                <target><![CDATA[]]></target>\n                <note>CarouselTripleBlockSlidePlugin</note>\n                <note>Slide</note>\n                <note>Titel</note>\n                <note>Max characters: 35</note>\n            </trans-unit>\n        </body>\n    </file>\n</xliff>\n```\n\nEdit the file in the XLIFF editor of your choice.\n\nImport the XLIFF to the same page in the same language you exported from with Languages > Import from XLIFF\n\n![Import](docs/screenshots/import.png)\n\nYou will get a preview of the import that needs to be confirmed.\n\n![Preview](docs/screenshots/preview.png)\n\nThe translations are now imported, and you can publish the page.\n\n## Settings\n\nBy default, djangocms-xliff searches for the following django model fields: `CharField, TextField, URLField` in your\nplugins.\nThe texts from these fields will be used for the XLIFF import and export.\n\nIf you want to add additional or 3rd party app fields, you can define the following settings in your `settings.py`,\nto integrate them into the XLIFF package:\n\n```python\n# A list of fields, that will be searched for while exporting.\nDJANGOCMS_XLIFF_FIELDS = (\n    "djangocms_text_ckeditor.fields.HTMLField",\n)\n```\n\n```python\n# List of tuples with field and custom function for the export\nDJANGOCMS_XLIFF_FIELD_EXTRACTORS = (\n    ("third_party.models.LinkField", "your_module.xliff.link_field_extractor"),\n)\n\n\n# The signature of the extractor function must be the following:\n# The source parameter is the same as getattr(instance, field.name)\ndef link_field_extractor(instance: CMSPlugin, field: LinkField, source: Any) -> List[djangocms_xliff.types.Unit]:\n    # example:\n    from djangocms_xliff.utils import get_type_with_path\n\n    text = source.find_text()\n    return [\n        Unit(\n            plugin_id=instance.id,\n            plugin_type=instance.plugin_type,\n            plugin_name=instance.get_plugin_name(),\n            field_name=field.name,\n            field_type=get_type_with_path(field),\n            field_verbose_name=field.verbose_name,\n            source=text,\n            max_length=field.max_length,\n        )\n    ]\n```\n\n```python\n# List of tuples with field and custom function for the import\nDJANGOCMS_XLIFF_FIELD_IMPORTERS = (\n    ("third_party.models.LinkField", "your_module.xliff.link_field_importer"),\n)\n\n\n# The signature of the importer function must be the following:\ndef link_field_importer(instance: CMSPlugin, unit: djangocms_xliff.types.Unit) -> CMSPlugin:\n    # example:\n    field = getattr(instance, unit.field_name)\n    field.body = unit.target\n    return instance\n```\n\n```python\n# List of custom validators for fields that need to be ignored or included in the export\nDJANGOCMS_XLIFF_VALIDATORS = ("your_module.xliff.is_not_background",)\n\n\n# The signature of the validator function must be the following:\ndef is_not_background(field: django.db.models.Field, instance: CMSPlugin) -> bool:\n    # example:\n    return field.name != "background"\n```\n\n## Contribute\n\nIssues and pull requests are welcomed.\n\nYou can find a documentation on how to set up your project\nin [here](docs/contribute.md)\n',
    'author': 'Energie 360',
    'author_email': 'onlineservice@energie360.ch',
    'maintainer': 'Energie 360',
    'maintainer_email': 'onlineservice@energie360.ch',
    'url': 'https://energie360.ch',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9',
}


setup(**setup_kwargs)
