# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['edgescan',
 'edgescan.api',
 'edgescan.cli',
 'edgescan.cli.command_groups',
 'edgescan.data',
 'edgescan.data.types']

package_data = \
{'': ['*']}

install_requires = \
['click>=8.1.3,<9.0.0',
 'inflection>=0.5.1,<0.6.0',
 'python-dateutil>=2.8.2,<3.0.0',
 'requests>=2.28.0,<3.0.0',
 'urllib3>=1.26.9,<2.0.0']

entry_points = \
{'console_scripts': ['edgescan = edgescan.cli:cli']}

setup_kwargs = {
    'name': 'edgescan',
    'version': '2.0.8',
    'description': 'An API client for EdgeScan',
    'long_description': '# An API client for Edgescan\n\n---\n\n## Overview\n\n### Edgescan\n\n[Edgescan](https://www.edgescan.com/) is a vulnerability management solution that allows you to identify both network and application layer vulnerabilities across a wide variety of systems.\n\n### Data model\n\n![Edgescan\'s data model](resources/images/edgescan-data-model.png)\n\nEdgescan\'s data model includes:\n- **Assets**: define which _hosts_ to scan for _vulnerabilities_;\n- **Hosts**: represent individual computer systems (physical or virtual); and \n- **Vulnerabilities**: represent known attack vectors that may be exploited by an adversary.\n\n### Caveats\n\n- Since hosts and vulnerabilities are linked by location (i.e. by IP address or hostname) rather than using stronger factors such as UUIDs, any two hosts that have the same IP address or hostname will have the same vulnerabilities.\n\n---\n\n## API client\n\nThis API client is written in Python and includes a command line interface that you can use to list, count, and retrieve objects.\n\n### Features\n\nYou can use this package to:\n\n- List, count, and retrieve assets, hosts, licenses, and vulnerabilities; and\n- Export data from Edgescan in JSONL format\n\n### Disclaimers\n\n- This package is not maintained by, or supported by Edgescan.\n\n### Warnings\n\n- The latest version of all hosts, licenses, and vulnerabilities will be written to the system\'s temporary directory by default <sub>1</sub>.\n\n<sub>1. You can disable this behaviour by setting `EDGESCAN_ENABLE_CACHE=false`</sub>\n\n---\n\n### Installation\n\nTo install the `edgescan-api-client` (requires [`poetry`](https://github.com/python-poetry/poetry)):\n\n```shell\n$ git clone https://github.com/whitfieldsdad/edgescan-api-client.git\n$ cd edgescan-api-client\n$ make install\n```\n\n#### Environment variables\n\n| Name                    | Description                                    | Default           | Required |\n|-------------------------|------------------------------------------------|-------------------|----------|\n| `EDGESCAN_HOST`         | Address of Edgescan API                        | live.edgescan.com | false    |\n| `EDGESCAN_API_KEY`      | Edgescan API key                               | n/a               | true     |\n| `EDGESCAN_ENABLE_CACHE` | Enable/disable caching to temporary directory. | n/a               | false    |\n\n---\n\n### Testing\n\nTo run the unit tests and integration tests:\n\n```shell\n$ make test\n```\n\n#### Code coverage\n\nCode coverage reports will automatically be created in two different formats:\n- HTML: `coverage/html/index.html`\n- JSON: `coverage/json/coverage.json`\n\nTo read the HTML-formatted code coverage report:\n\n```shell\n$ open coverage/html/index.html\n```\n\nTo read the JSON-formatted code coverage report:\n\n```shell\n$ cat coverage/json/coverage.json | jq\n```\n\n---\n\n### Docker\n\nThis repository has been packaged as a Docker container! ✨📦🐋✨\n\n#### Building the container\n\nTo build the container:\n\n```shell\n$ make build-container\n```\n\n#### Exporting the container to a file\n\nTo export the container to a file named `edgescan-api-client.tar.gz`:\n\n```shell\n$ make export-container\n$ du -sh edgescan-api-client.tar.gz\n176M    edgescan-api-client.tar.gz\n```\n\n---\n\n### General usage\n\nThere are two different ways to interact with this module:\n- Via the command line; or\n- Via Python\n\n#### Command line interface\n\nUsing `poetry`:\n\n```shell\n$ poetry run edgescan\nUsage: edgescan [OPTIONS] COMMAND [ARGS]...\n\nOptions:\n  --edgescan-host TEXT\n  --edgescan-api-key TEXT\n  --help                   Show this message and exit.\n\nCommands:\n  assets           Query or count assets.\n  hosts            Query or count hosts.\n  licenses         Query or count licenses.\n  vulnerabilities  Query or count vulnerabilities.\n```\n\n![Command line interface features](resources/images/command-line-interface.png)\n\n##### Search for assets\n\nYou can search for assets by:\n- Asset ID;\n- Name;\n- Tag;\n- Create time; and/or\n- Update time\n\n```shell\n$ poetry run edgescan assets get-assets --help\nUsage: edgescan assets get-assets [OPTIONS]\n\n  Search for assets.\n\nOptions:\n  --asset-ids TEXT\n  --names TEXT\n  --tags TEXT\n  --min-create-time TEXT\n  --max-create-time TEXT\n  --min-update-time TEXT\n  --max-update-time TEXT\n  --limit INTEGER\n  --help                  Show this message and exit.\n```\n\n#### Search for hosts\n\nYou can search for hosts by:\n\n- Asset ID;\n- Host ID;\n- Location (i.e. by IP address or hostname);\n- Status (i.e. whether they\'re "dead" or "alive");\n- Create time; and/or\n- Update time.\n\n```shell\n$ poetry run edgescan hosts get-hosts --help\nUsage: edgescan hosts get-hosts [OPTIONS]\n\n  Search for hosts.\n\nOptions:\n  --asset-ids TEXT\n  --host-ids TEXT\n  --locations TEXT\n  --alive / --dead\n  --min-create-time TEXT\n  --max-create-time TEXT\n  --min-update-time TEXT\n  --max-update-time TEXT\n  --limit INTEGER\n  --help                  Show this message and exit.\n```\n\n##### Count active vs. inactive hosts\n\nYou can count active hosts like this:\n\n```shell\n$ poetry run edgescan hosts count-hosts --alive\n123\n```\n\nAnd inactive hosts like this:\n\n```shell\n$ poetry run edgescan hosts count-hosts --dead\n456\n```\n\n#### Search for vulnerabilities\n\nYou can search for vulnerabilities by:\n- Vulnerability ID;\n- CVE ID;\n- Asset ID;\n- Host ID;\n- Location (i.e. by IP address or hostname);\n- Status (i.e. whether the host is "dead" or "alive");\n- Layer (i.e. "app" layer or "network" layer);\n- Create time;\n- Update time\n\n```shell\n$ poetry run edgescan vulnerabilities get-vulnerabilities --help\nUsage: edgescan vulnerabilities get-vulnerabilities [OPTIONS]\n\n  List vulnerabilities.\n\nOptions:\n  --vulnerability-ids TEXT\n  --cve-ids TEXT\n  --asset-ids TEXT\n  --host-ids TEXT\n  --locations TEXT\n  --alive / --dead\n  --include-application-layer-vulnerabilities / --exclude-application-layer-vulnerabilities\n  --include-network-layer-vulnerabilities / --exclude-network-layer-vulnerabilities\n  --min-create-time TEXT\n  --max-create-time TEXT\n  --min-update-time TEXT\n  --max-update-time TEXT\n  --limit INTEGER\n  --help                          Show this message and exit.\n```\n\n#### Search for licenses\n\nYou can search for licenses by:\n- License ID;\n- License name; and/or\n- Whether the license is expired.\n\n```shell\n$ poetry run edgescan licenses get-licenses --help\nUsage: edgescan licenses get-licenses [OPTIONS]\n\n  List licenses.\n\nOptions:\n  --license-ids TEXT\n  --license-names TEXT\n  --expired / --not-expired\n  --limit INTEGER\n  --help                     Show this message and exit.\n```\n\n---\n\n#### Python\n\n##### Search for assets\n\nTo look up all assets and list their names:\n\n```python\nfrom edgescan import Client\n\napi = Client()\nassets = api.iter_assets()\n\nnames = {asset[\'name\'] for asset in assets}\nprint(\', \'.join(sorted(names)))\n```\n\n##### Search for hosts\n\nTo look up all active hosts and list their locations (i.e. IP addresses and hostnames):\n\n```python\nfrom edgescan import Client\n\napi = Client()\nlocations = set()\nfor host in api.iter_hosts(alive=True):\n    locations |= {host[\'location\']} | set(host[\'hostnames\'])\n\nprint(\', \'.join(sorted(locations)))\n```\n\nTo look up all active hosts and their OS versions:\n\n```python\nfrom edgescan import Client\n\napi = Client()\nos_versions = {host[\'os_name\'] for host in api.iter_hosts(alive=True) if host[\'os_name\']}\nprint(\', \'.join(sorted(os_versions)))\n```\n\nTo count active hosts by OS type and OS version:\n\n```python\nfrom edgescan import Client\nfrom collections import OrderedDict\n\nimport edgescan.platforms\nimport collections\nimport json\n\napi = Client()\n\ntally = collections.defaultdict(lambda: collections.defaultdict(int))\nfor host in api.iter_hosts(alive=True):\n    os_version = host[\'os_name\']\n    if os_version:\n        os_type = edgescan.platforms.parse_os_type(os_version)\n        tally[os_type][os_version] += 1\n\n#: Sort by rate of occurrence.\ntally = OrderedDict(sorted(tally.items(), key=lambda e: e[1], reverse=True))\nprint(json.dumps(tally, indent=4))\n```\n\n##### Search for licenses\n\nTo search for licenses and list their names:\n\n```python\nfrom edgescan import Client\n\napi = Client()\nnames = {row[\'name\'] for row in api.iter_licenses()}\nprint(\', \'.join(sorted(names)))\n```\n\n##### Search for vulnerabilities\n\nTo count vulnerabilities on active hosts:\n\n```python\nfrom edgescan import Client\n\napi = Client()\ntotal = api.count_vulnerabilities(host_is_alive=True)\nprint(total)\n```\n\nTo count vulnerabilities on active hosts by CVE ID:\n\n```python\nfrom edgescan import Client\nfrom collections import OrderedDict\n\nimport collections\nimport json\n\napi = Client()\n\ntally = collections.defaultdict(int)\nfor vulnerability in api.iter_vulnerabilities(host_is_alive=True):\n    for cve_id in vulnerability[\'cves\']:\n        tally[cve_id] += 1\n\n#: Sort by rate of occurrence.\ntally = OrderedDict(sorted(tally.items(), key=lambda e: e[1], reverse=True))\nprint(json.dumps(tally, indent=4))\n```\n',
    'author': 'Tyler Fisher',
    'author_email': 'tylerfisher@tylerfisher.ca',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/whitfieldsdad/edgescan-api-client',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
