from typing import Any, Optional, Union

from easytxt import text

from easydata.parsers.base import BaseData
from easydata.parsers.text import Text
from easydata.queries.base import QuerySearchBase

__all__ = (
    "Bool",
    "IBool",
    "Has",
    "IHas",
)


class Bool(BaseData):
    def parse_value(
        self,
        value: Any,
        data: Any,
    ):

        return bool(value)


class IBool(BaseData):
    def parse_value(
        self,
        value: Any,
        data: Any,
    ):

        return not super(IBool, self).parse_value(value, data)


class Has(Text):
    def __init__(
        self,
        *args,
        contains: Optional[Union[list, str]] = None,
        ccontains: Optional[Union[list, str]] = None,
        contains_query: Optional[QuerySearchBase] = None,
        has_value: bool = False,
        empty_as_false: bool = True,
        **kwargs,
    ):

        self._contains = ccontains or contains
        self._contains_query = contains_query
        self._has_value = has_value
        self._contains_case = bool(ccontains)
        self._empty_as_false = empty_as_false

        if "default" not in kwargs and empty_as_false:
            kwargs["default"] = False

        super().__init__(
            *args,
            **kwargs,
        )

    def parse_value(
        self,
        value: Any,
        data: Any,
    ):

        if isinstance(value, bool):
            return value

        if isinstance(value, (float, int)):
            return bool(value)

        if not value:
            return False

        value = super(Has, self).parse_value(value, data)

        if not value:

            return self._default

        if self._contains:
            return self._contains and text.contains(
                text=value,
                keys=self._contains,
                case_sensitive=self._contains_case,
            )

        if self._contains_query:
            contains_values = self._parse_query(
                query=self._contains_query,
                data=data,
                source=self.source,
                parent_data=data,
            )

            return contains_values and text.contains(
                text=value,
                keys=contains_values,
            )

        if self._has_value:
            return bool(value)

        if value and isinstance(value, str):
            if "true" == value.lower():
                return True
            elif "false" == value.lower():
                return False

        return False


class IHas(Has):
    def parse_value(
        self,
        value: Any,
        data: Any,
    ):
        value = super(IHas, self).parse_value(value, data)

        if value is None:
            return value

        return False if value else True
