# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pytest_docker_tools',
 'pytest_docker_tools.factories',
 'pytest_docker_tools.wrappers']

package_data = \
{'': ['*'], 'pytest_docker_tools': ['contexts/scratch/*']}

install_requires = \
['docker>=4.3.1,<5.0.0', 'pytest>=6.0.1,<7.0.0']

entry_points = \
{'pytest11': ['docker_tools = pytest_docker_tools.plugin']}

setup_kwargs = {
    'name': 'pytest-docker-tools',
    'version': '1.0.2',
    'description': 'Docker integration tests for pytest',
    'long_description': '# pytest-docker-tools\n\nYou have written a software application (in any language) and have packaged in as a Docker image. Now you want to smoke test the built image or do some integration testing with other containers before releasing it. You:\n\n * want to reason about your environment in a similar way to a `docker-compose.yml`\n * want the environment to be automatically created and destroyed as tests run\n * don\'t want to have to write loads of boilerplate code for creating the test environment\n * want to be able to run the tests in parallel\n * want the tests to be reliable\n\n`pytest-docker-tools` is a set of opinionated helpers for creating `py.test` fixtures for your smoke testing and integration testing needs. It strives to keep your environment definition declarative, like a docker-compose.yml. It embraces py.test fixture overloading. It tries not to be too magical.\n\nThe main interface provided by this library is a set of \'fixture factories\'. It provides a \'best in class\' implementation of a fixture, and then allows you to treat it as a template - injecting your own configuration declaratively. You can define your fixtures in your `conftest.py` and access them from all your tests, and you can override them as needed in individual test modules.\n\nThe API is straightforward and implicitly captures the interdependencies in the specification. For example, here is how it might look if you were building out a microservice and wanted to point its DNS and a mock DNS server:\n\n```python\n# conftest.py\n\nfrom http.client import HTTPConnection\n\nimport pytest\nfrom pytest_docker_tools import build, container\n\nfakedns_image = build(\n    path=\'examples/resolver-service/dns\',\n)\n\nfakedns = container(\n    image=\'{fakedns_image.id}\',\n    environment={\n        \'DNS_EXAMPLE_COM__A\': \'127.0.0.1\',\n    }\n)\n\napiserver_image = build(\n    path=\'examples/resolver-service/api\',\n)\n\napiserver = container(\n    image=\'{apiserver_image.id}\',\n    ports={\n        \'8080/tcp\': None,\n    },\n    dns=[\'{fakedns.ips.primary}\']\n)\n\n\n@pytest.fixture\ndef apiclient(apiserver):\n    port = apiserver.ports[\'8080/tcp\'][0]\n    return HTTPConnection(f\'localhost:{port}\')\n```\n\nYou can now create a test that exercises your microservice:\n\n```python\n# test_smoketest.py\n\nimport socket\n\ndef test_my_frobulator(apiserver):\n    sock = socket.socket()\n    sock.connect((\'127.0.0.1\', apiserver.ports[\'8080/tcp\'][0]))\n\n\ndef test_my_frobulator_works_after_restart(apiserver):\n    apiserver.restart()\n\n    sock = socket.socket()\n    sock.connect((\'127.0.0.1\', apiserver.ports[\'8080/tcp\'][0]))\n```\n\nIn this example all the dependencies will be resolved in order and once per session:\n\n * The latest `redis:latest` will be fetched\n * A container image will be build from the `Dockerfile` in the `db` folder.\n\nThen once per test:\n\n * A new volume will be created\n * A new \'backend\' container will be created from `redis:latest`. It will be attached to the new volume.\n * A new \'frontend\' container will be created from the freshly built container. It will be given the IP if the backend via an environment variable. Port 3679 in the container will be exposed as an ephemeral port on the host.\n\nThe test can then run and access the container via its ephemeral high port. At the end of the test the environment will be thrown away.\n\nIf the test fails the `docker logs` output from each container will be captured and added to the test output.\n\nIn the example you\'ll notice we defined an `apiclient` fixture. Of course if you use that it will implicitly pull in both of the server fixtures and \'just work\':\n\n```python\n# test_smoketest.py\n\nimport json\n\n\ndef test_api_server(apiclient):\n    apiclient.request(\'GET\', \'/\')\n    response = apiclient.getresponse()\n    assert response.status == 200\n    assert json.loads(response.read()) == {\'result\': \'127.0.0.1\'}\n```\n\n\n## Scope\n\nAll of the fixture factories take the `scope` keyword. Fixtures created with these factories will behave like any py.test fixture with that scope.\n\nIn this example we create a memcache that is `session` scoped and another that is `module` scoped.\n\n```python\n# conftest.py\n\nfrom pytest_docker_tools import container, fetch\n\nmemcache_image = fetch(repository=\'memcached:latest\')\n\nmemcache_session = container(\n    image=\'{memcache_image.id}\',\n    scope=\'session\',\n    ports={\n        \'11211/tcp\': None,\n    },\n)\n\nmemcache_module = container(\n    image=\'{memcache_image.id}\',\n    scope=\'module\',\n    ports={\n        \'11211/tcp\': None,\n    },\n)\n```\n\nWhen `test_scope_1.py` runs neither container is running so a new instance of each is started. Their scope is longer than a single `function` so they are kept alive for the next test that needs them.\n\n```python\n# test_scope_1.py\n\nimport socket\n\ndef test_session_1(memcache_session):\n    sock = socket.socket()\n    sock.connect((\'127.0.0.1\', memcache_session.ports[\'11211/tcp\'][0]))\n    sock.sendall(b\'set mykey 0 600 4\\r\\ndata\\r\\n\')\n    sock.sendall(b\'get mykey\\r\\n\')\n    assert sock.recv(1024) == b\'STORED\\r\\nVALUE mykey 0 4\\r\\ndata\\r\\nEND\\r\\n\'\n    sock.close()\n\ndef test_session_2(memcache_session):\n    sock = socket.socket()\n    sock.connect((\'127.0.0.1\', memcache_session.ports[\'11211/tcp\'][0]))\n    sock.sendall(b\'set mykey 0 600 4\\r\\ndata\\r\\n\')\n    sock.sendall(b\'get mykey\\r\\n\')\n    assert sock.recv(1024) == b\'STORED\\r\\nVALUE mykey 0 4\\r\\ndata\\r\\nEND\\r\\n\'\n    sock.close()\n\ndef test_module_1(memcache_module):\n    sock = socket.socket()\n    sock.connect((\'127.0.0.1\', memcache_module.ports[\'11211/tcp\'][0]))\n    sock.sendall(b\'set mykey 0 600 4\\r\\ndata\\r\\n\')\n    sock.sendall(b\'get mykey\\r\\n\')\n    assert sock.recv(1024) == b\'STORED\\r\\nVALUE mykey 0 4\\r\\ndata\\r\\nEND\\r\\n\'\n    sock.close()\n\ndef test_module_2(memcache_module):\n    sock = socket.socket()\n    sock.connect((\'127.0.0.1\', memcache_module.ports[\'11211/tcp\'][0]))\n    sock.sendall(b\'set mykey 0 600 4\\r\\ndata\\r\\n\')\n    sock.sendall(b\'get mykey\\r\\n\')\n    assert sock.recv(1024) == b\'STORED\\r\\nVALUE mykey 0 4\\r\\ndata\\r\\nEND\\r\\n\'\n    sock.close()\n```\n\nWhen `test_scope_2.py` runs the `session` scoped container is still running, so it will be reused. But we are now in a new module now so the `module` scoped container will have been destroyed. A new empty instance will be created.\n\n```python\n# test_scope_2.py\n\nimport socket\n\ndef test_session_3(memcache_session):\n    sock = socket.socket()\n    sock.connect((\'127.0.0.1\', memcache_session.ports[\'11211/tcp\'][0]))\n    sock.sendall(b\'get mykey\\r\\n\')\n    assert sock.recv(1024) == b\'VALUE mykey 0 4\\r\\ndata\\r\\nEND\\r\\n\'\n    sock.close()\n\ndef test_module_3(memcache_module):\n    sock = socket.socket()\n    sock.connect((\'127.0.0.1\', memcache_module.ports[\'11211/tcp\'][0]))\n    sock.sendall(b\'get mykey\\r\\n\')\n    assert sock.recv(1024) == b\'END\\r\\n\'\n    sock.close()\n```\n\n\n## Parallelism\n\nIntegration and smoke tests are often slow, but a lot of time is spent waiting. So running tests in parallel is a great way to speed them up. `pytest-docker-tools` avoids creating resource names that could collide. It also makes it easy to not care what port your service is bound to. This means its a great fit for use with `pytest-xdist`.\n\nHere is a bare minimum example that just tests creating and destroying 100 instances of a redis fixture that runs under xdist. Create a `test_xdist.py` plugin:\n\n```python\n\nimport pytest\nfrom pytest_docker_tools import container, fetch\n\nmy_redis_image = fetch(repository=\'redis:latest\')\n\nmy_redis = container(\n    image=\'{my_redis_image.id}\',\n)\n\n\n@pytest.mark.parametrize("i", list(range(100)))\ndef test_xdist(i, my_redis):\n    assert my_redis.status == "running"\n```\n\nAnd invoke it with:\n\n```bash\npytest test_xdist.py -n auto\n```\n\nIt will create a worker per core and run the tests in parallel:\n\n```\n===================================== test session starts ======================================\nplatform darwin -- Python 3.6.5, pytest-3.6.3, py-1.5.4, pluggy-0.6.0\nrootdir: ~/pytest-docker-tools, inifile:\nplugins: xdist-1.22.2, forked-0.2, docker-tools-0.0.2\ngw0 [100] / gw1 [100] / gw2 [100] / gw3 [100] / gw4 [100] / gw5 [100] / gw6 [100] / gw7 [100]\nscheduling tests via LoadScheduling\n......................................................................................... [ 82%]\n...........                                                                              [100%]\n================================= 100 passed in 70.08 seconds ==================================\n```\n\n\n## Factories Reference\n\n### Containers\n\nTo create a container in your tests use the `container` fixture factory.\n\n```python\nfrom pytest_docker_tools import container\n\nmy_microservice_backend = container(image=\'redis:latest\')\n```\n\nThe default scope for this factory is `function`. This means a new container will be created for each test.\n\nThe `container` fixture factory supports all parameters that can be passed to the docker-py `run` method. See [here](https://docker-py.readthedocs.io/en/stable/containers.html#docker.models.containers.ContainerCollection.run) for them all.\n\nAny string variables are interpolated against other defined fixtures. This means that a fixture can depend on other fixtures, and they will be built and run in order.\n\nFor example:\n\n```python\nfrom pytest_docker_tools import container, fetch\n\nredis_image = fetch(repository=\'redis:latest\')\nredis = container(image=\'{redis_image.id}\')\n\n\ndef test_container_starts(redis):\n    assert redis.status == "running"\n```\n\nThis will fetch the latest `redis:latest` first, and then run a container from the exact image that was pulled. Note that if you don\'t use `build` or `fetch` to prepare a Docker image then the tag or hash that you specify must already exist on the host where you are running the tests. There is no implicit fetching of Docker images.\n\nThe container will be ready when the test is started, and will be automatically deleted after the test has finished.\n\nIf for some reason it\'s not ready in the timeout period (30 seconds by default) the test will fail.\n\n`timeout` can be passed to the `container` factory:\n\n```python\nfrom pytest_docker_tools import container, fetch\n\nredis_image = fetch(repository=\'redis:latest\')\nredis = container(image=\'{redis_image.id}\', timeout=30)\n\ndef test_container_starts(redis):\n    assert redis.status == "running"\n```\n\nTo create a container defining its `Dockerfile` in code:\n\n```python\nimport io\n\nfrom pytest_docker_tools import build, container\n\ndockerfile = io.BytesIO(b"""\nFROM alpine:3.12\nRUN apk --no-cache add python3\nCMD ["tail", "-f", "/dev/null"]\n""")\n\nimage = build(fileobj=dockerfile)\ncontainer = container(image=\'{image.id}\')\n\ndef test_container_starts(container):\n    assert container.status == "running"\n```\n\n#### Ip Addresses\n\nIf your container is only attached to a single network you can get its Ip address through a helper property. Given this environment:\n\n```python\n# conftest.py\n\nfrom pytest_docker_tools import container, fetch, network\n\nredis_image = fetch(repository=\'redis:latest\')\nbackend_network = network()\n\nredis = container(\n  image=\'{redis_image.id}\',\n  network=\'{backend_network.name}\',\n)\n```\n\nYou can access the IP via the container helper:\n\n```python\nimport ipaddress\n\ndef test_get_service_ip(redis):\n    # This will raise a ValueError if not a valid ip\n    ipaddress.ip_address(redis.ips.primary)\n```\n\nIf you want to look up its ip address by network you can also access it more specifically:\n\n```python\nimport ipaddress\n\ndef test_get_service_ip(backend_network, redis):\n    ipaddress.ip_address(redis.ips[backend_network])\n```\n\n#### Ports\n\nThe factory takes the same port arguments as the official Python Docker API. We recommend using the ephemeral high ports syntax:\n\n```python\n# conftest.py\n\nfrom pytest_docker_tools import container\n\napiserver = container(\n  image=\'{apiserver_image.id}\',\n  ports={\'8080/tcp\': None}\n)\n```\n\nDocker will map port 8080 in the container to a random port on your host. In order to access it from your tests you can get the bound port from the container instance:\n\n```python\ndef test_connect_my_service(apiserver):\n    assert apiserver.ports[\'8080/tcp\'][0] != 8080\n```\n\n\n#### Logs\n\nYou can inspect the logs of your container with the logs method:\n\n```python\nfrom pytest_docker_tools import container, fetch\n\n\nredis_image = fetch(repository=\'redis:latest\')\nredis = container(\n    image=\'{redis_image.id}\',\n    ports={\'6379/tcp\': None},\n)\n\ndef test_logs(redis):\n    assert \'oO0OoO0OoO0Oo Redis is starting oO0OoO0OoO0Oo\' in redis.logs()\n```\n\n\n### Images\n\nTo pull an image from your default repository use the `fetch` fixture factory. To build an image from local source use the `build` fixture factory. If you are smoke testing an artifact already built locally you can use the `image` fixture factory to reference it.\n\n```python\nfrom pytest_docker_tools import build, image, fetch\n\nmy_image = fetch(repository=\'redis:latest\')\n\nmy_image_2 = build(\n  path=\'db\'\n)\n```\n\nThe `build` fixture factory supports all parameters that can be passed to the docker-py `build` method. See [here](https://docker-py.readthedocs.io/en/stable/images.html#docker.models.images.ImageCollection.build) for them all. The `fetch` fixture factory supports all parameters that can be passed to the docker-py `pull` method. See [here](https://docker-py.readthedocs.io/en/stable/images.html#docker.models.images.ImageCollection.pull) for them all.\n\nThe default scope for this factory is `session`. This means the fixture will only build or fetch once per py.test invocation. The fixture will not be triggered until a test (or other fixture) tries to use it. This means you won\'t waste time building an image if you aren\'t running the test that uses it.\n\n\n### Networks\n\nBy default any containers you create with the `container()` fixture factory will run on your default docker network. You can create a dedicated network for your test with the `network()` fixture factory.\n\n```python\nfrom pytest_docker_tools import network\n\nfrontend_network = network()\n```\n\nThe `network` fixture factory supports all parameters that can be passed to the docker-py network `create` method. See [here](https://docker-py.readthedocs.io/en/stable/networks.html#docker.models.networks.NetworkCollection.create) for them all.\n\nThe default scope for this factory is `function`. This means a new network will be created for each test that is executed.\n\nThe network will be removed after the test using it has finished.\n\n\n### Volumes\n\nIn the ideal case a Docker container instance is read only. No data inside the container is written to, if it is its to a volume. If you are testing that your service can run read only you might want to mount a rw volume. You can use the `volume()` fixture factory to create a Docker volume with a lifecycle tied to your tests.\n\n```python\nfrom pytest_docker_tools import volume\n\nbackend_storage = volume()\n```\n\nThe `volume` fixture factory supports all parameters that can be passed to the docker-py volume `create` method. See [here](https://docker-py.readthedocs.io/en/stable/volumes.html#docker.models.volumes.VolumeCollection.create) for them all.\n\nIn addition you can specify a `initial_content` dictionary. This allows you to seed a volume with a small set of initial state. In the following example we\'ll preseed a minio service with 2 buckets and 1 object in 1 of those buckets.\n\n```python\nfrom pytest_docker_tools import container, fetch, volume\n\n\nminio_image = fetch(repository=\'minio/minio:latest\')\n\nminio_volume = volume(\n    initial_content={\n        \'bucket-1\': None,\n        \'bucket-2/example.txt\': b\'Test file 1\',\n    }\n)\n\nminio = container(\n    image=\'{minio_image.id}\',\n    command=[\'server\', \'/data\'],\n    volumes={\n        \'{minio_volume.name}\': {\'bind\': \'/data\'},\n    },\n    environment={\n        \'MINIO_ACCESS_KEY\': \'minio\',\n        \'MINIO_SECRET_KEY\': \'minio123\',\n    },\n)\n\ndef test_volume_is_seeded(minio):\n    files = minio.get_files(\'/data\')\n    assert files[\'data/bucket-2/example.txt\'] == b\'Test file 1\'\n    assert files[\'data/bucket-1\'] == None\n```\n\nThe `minio_volume` container will be created with an empty folder (`bucket-1`) and a text file called `example.txt` in a seperate folder called `bucket-2`.\n\nThe default scope for this factory is `function`. This means a new volume will be created for each test that is executed. The volume will be removed after the test using it has finished.\n\n\n## Fixtures\n\n### docker_client\n\nThe `docker_client` fixture returns an instance of the official docker client.\n\n```python\ndef test_container_created(docker_client, fakedns):\n    for c in docker_client.containers.list(ignore_removed=True):\n        if c.id == fakedns.id:\n            # Looks like we managed to start one!\n            break\n    else:\n        assert False, \'Looks like we failed to start a container\'\n```\n\nTake care when using the `docker_client` directly:\n\n * Obviously resources created imperatively via the API won\'t be removed at the end of the test automatically\n * It\'s easy to break xdist compatibility\n   * Always use `ignore_removed` with `docker_client.containers.list()` - it is racy without\n   * It\'s easy to find other instances of the resources you are working with (created in other workers). Be mindful of this!\n * Don\'t take destructive action - someone could be running tests on a machine with other (non-test) containers running, collateral damage is easy and should be avoided.\n\nThis is the fixture used by our fixture factories. This means if you define a `docker_client` fixture of your own then the tests will use that instead.\n\n\n## Tips and tricks\n\n### Testing build artifacts\n\nWe often find ourselves using a set of tests against a container we\'ve built at test time (with `build()`) but then wanting to use the same tests with an artifact generated on our CI platform (with `image()`). This ended up looking like this:\n\n```\nif not os.environ.get(\'IMAGE_ID\', \'\'):\n    image = build(path=\'examples/resolver-service/dns\')\nelse:\n    image = image(name=os.environ[\'IMAGE_ID\'])\n```\n\nBut now you can just do:\n\n```python\nfrom pytest_docker_tools import image_or_build\n\nimage = image_or_build(\n    environ_key=\'IMAGE_ID\',\n    path=\'examples/resolver-service/dns\',\n)\n\ndef test_image(image):\n    assert image.attrs[\'Os\'] == \'linux\'\n```\n\n\n### Network differences between dev env and CI\n\nAnother common difference between your dev environment and your CI environment might be that your tests end up running in Docker on your CI. If you bind-mount your `docker.sock` then your tests might end up running on the same container network as the containers you are testing, and unable to access any port you are mapping to the host box. In otherwords:\n\n * On your dev machine your tests might access locahost:8000 to access your test instance (ports mapped to host)\n * On your CI machine they might need to access 172.16.0.5:8000 to access your test instance\n\nThe container object has a `get_addr` helper which will return the right thing depending on the environment it is in.\n\n```python\nfrom pytest_docker_tools import container\n\napiserver = container(\n  image=\'{apiserver_image.id}\',\n  ports={\'8080/tcp\': None}\n)\n\ndef test_connect_my_service(apiserver):\n    ip, port = apiserver.get_addr(\'8080/tcp\')\n    # ... connect to ip:port ...\n```\n\n### Dynamic scope\n\nThe pytest fixture decorator now lets you specify a callback to determine the scope of a fixture This is called [dynamic scope](https://docs.pytest.org/en/stable/fixture.html#dynamic-scope). You can use this to make it a runtime option whether to have a container instance per test or per test run. For example:\n\n```python\n# conftest.py\nfrom pytest_docker_tools import container, fetch\n\ndef determine_scope(fixture_name, config):\n    if config.getoption("--keep-containers", None):\n        return "session"\n    return "function"\n\nmemcache_image = fetch(repository=\'memcached:latest\')\n\nmemcache = container(\n    image=\'{memcache_image.id}\',\n    scope=determine_scope,\n    ports={\n        \'11211/tcp\': None,\n    },\n)\n```\n\nYour tests can look exactly the same as before:\n\n```python\ndef test_connect_my_service(memcache):\n    ip, port = memcache.get_addr(\'11211/tcp\')\n    # ... connect to ip:port ...\n```\n\n\n### Client fixtures\n\nYou will probably want to create an API client for the service you are testing. Although we\'ve already done this in the README, its worth calling it out. You can define a client fixture, have it depend on your docker containers, and then only have to reference the client from your tests.\n\n```python\n# conftest.py\n\nfrom http.client import HTTPConnection\n\nimport pytest\nfrom pytest_docker_tools import build, container\n\nfakedns_image = build(\n    path=\'examples/resolver-service/dns\',\n)\n\nfakedns = container(\n    image=\'{fakedns_image.id}\',\n    environment={\n        \'DNS_EXAMPLE_COM__A\': \'127.0.0.1\',\n    }\n)\n\napiserver_image = build(\n    path=\'examples/resolver-service/api\',\n)\n\napiserver = container(\n    image=\'{apiserver_image.id}\',\n    ports={\n        \'8080/tcp\': None,\n    },\n    dns=[\'{fakedns.ips.primary}\']\n)\n\n\n@pytest.fixture\ndef apiclient(apiserver):\n    port = apiserver.ports[\'8080/tcp\'][0]\n    return HTTPConnection(f\'localhost:{port}\')\n```\n\nAnd then reference it from your tests:\n\n```python\n# test_the_test_client.py\n\nimport json\n\n\ndef test_api_server(apiclient):\n    apiclient.request(\'GET\', \'/\')\n    response = apiclient.getresponse()\n    assert response.status == 200\n    result = json.loads(response.read())\n    assert result[\'result\'] == \'127.0.0.1\'\n```\n\nIn this example, any test that uses the `hpfeeds_client` fixture will get a properly configure client connected to a broker running in a Docker container on an ephemeral high port. When the test finishes the client will cleanly disconnect, and the docker container will be thrown away.\n\n\n### Fixture overloading\n\nComplicated environments can be defined with fixture factories. They form a directed acyclic graph. By using fixture overloading it is possible to (in the context of a single test module) replace a node in that dependency graph without having to redefine the entire environment.\n\n#### Replacing a container fixture without having to redefine its dependents\n\nYou can define a fixture in your `conftest.py`:\n\n```python\n# conftest.py\n\nfrom http.client import HTTPConnection\n\nimport pytest\nfrom pytest_docker_tools import build, container\n\nfakedns_image = build(\n    path=\'examples/resolver-service/dns\',\n)\n\nfakedns = container(\n    image=\'{fakedns_image.id}\',\n    environment={\n        \'DNS_EXAMPLE_COM__A\': \'127.0.0.1\',\n    }\n)\n\napiserver_image = build(\n    path=\'examples/resolver-service/api\',\n)\n\napiserver = container(\n    image=\'{apiserver_image.id}\',\n    ports={\n        \'8080/tcp\': None,\n    },\n    dns=[\'{fakedns.ips.primary}\']\n)\n\n\n@pytest.fixture\ndef apiclient(apiserver):\n    port = apiserver.ports[\'8080/tcp\'][0]\n    return HTTPConnection(f\'localhost:{port}\')\n```\n\nYou can then overload these fixtures in your test modules. For example, if redis had a magic replication feature and you want to test for an edge case with your API you could in your `test_smoketest_alternate.py`:\n\n```python\n# test_smoketest_alternate.py\n\nimport json\n\nfrom pytest_docker_tools import container\n\nfakedns = container(\n    image=\'{fakedns_image.id}\',\n    environment={\n        \'DNS_EXAMPLE_COM__A\': \'192.168.192.168\',\n    }\n)\n\ndef test_api_server(apiclient):\n    apiclient.request(\'GET\', \'/\')\n    response = apiclient.getresponse()\n    assert response.status == 200\n    result = json.loads(response.read())\n    assert result[\'result\'] == \'192.168.192.168\'\n```\n\nHere we have redefined the fakedns container locally in `test_smoketest_alternate`. It is able to use the `fakedns_image` fixture we defined in `conftest.py`. More crucially though, in `test_smoketest_alternate.py` when we use the core `apiclient` fixture it actually pulls in the local definition of `fakedns` and not the one from `conftest.py`! You don\'t have to redefine anything else. It just works.\n\n\n#### Injecting fixture configuration through fixtures\n\nYou can pull in normal py.test fixtures from your fixture factory too. This means we can use fixture overloading and pass in config. In your `conftest.py`:\n\n```python\n# conftest.py\n\nfrom http.client import HTTPConnection\n\nimport pytest\nfrom pytest_docker_tools import build, container\n\nfakedns_image = build(\n    path=\'examples/resolver-service/dns\',\n)\n\nfakedns = container(\n    image=\'{fakedns_image.id}\',\n    environment={\n        \'DNS_EXAMPLE_COM__A\': \'{example_com_a}\',\n    }\n)\n\napiserver_image = build(\n    path=\'examples/resolver-service/api\',\n)\n\napiserver = container(\n    image=\'{apiserver_image.id}\',\n    ports={\n        \'8080/tcp\': None,\n    },\n    dns=[\'{fakedns.ips.primary}\']\n)\n\n\n@pytest.fixture\ndef apiclient(apiserver):\n    port = apiserver.ports[\'8080/tcp\'][0]\n    return HTTPConnection(f\'localhost:{port}\')\n\n\n@pytest.fixture\ndef example_com_a():\n    return \'127.0.0.1\'\n\n```\n\nWhen a test uses the apiclient fixture now they will get the fakedns container configured as normal. However you can redefine the fixture in your test module - and the other fixtures will still respect it. For example:\n\n```python\n# test_smoketest_alternate.py\n\nimport json\n\nimport pytest\n\n\n@pytest.fixture\ndef example_com_a():\n    return \'192.168.192.168\'\n\n\ndef test_api_server(apiclient):\n    apiclient.request(\'GET\', \'/\')\n    response = apiclient.getresponse()\n    assert response.status == 200\n    result = json.loads(response.read())\n    assert result[\'result\'] == \'192.168.192.168\'\n```\n\nYour `api_server` container (and its `redis` backend) will be built as normal, only in this one test module it will use its sqlite backend.\n\n\n### Fixture parameterisation\n\nYou can create parameterisation fixtures. Perhaps you wan to run all your `api_server` tests against both of your authentication backends. Perhaps you have a fake that you want to test multiple configurations of.\n\nIn your `conftest.py`:\n\n```python\n# conftest.py\n\nfrom http.client import HTTPConnection\n\nimport pytest\nfrom pytest_docker_tools import build, container\n\nfakedns_image = build(\n    path=\'examples/resolver-service/dns\',\n)\n\nfakedns_localhost = container(\n    image=\'{fakedns_image.id}\',\n    environment={\n        \'DNS_EXAMPLE_COM__A\': \'127.0.0.1\',\n    }\n)\n\nfakedns_alternate = container(\n    image=\'{fakedns_image.id}\',\n    environment={\n        \'DNS_EXAMPLE_COM__A\': \'192.168.192.168\',\n    }\n)\n\n@pytest.fixture(scope=\'function\', params=[\'fakedns_localhost\', \'fakedns_alternate\'])\ndef fakedns(request):\n      return request.getfixturevalue(request.param)\n\napiserver_image = build(\n    path=\'examples/resolver-service/api\',\n)\n\napiserver = container(\n    image=\'{apiserver_image.id}\',\n    ports={\n        \'8080/tcp\': None,\n    },\n    dns=[\'{fakedns.ips.primary}\']\n)\n\n\n@pytest.fixture\ndef apiclient(apiserver):\n    port = apiserver.ports[\'8080/tcp\'][0]\n    return HTTPConnection(f\'localhost:{port}\')\n```\n\nThe test is the same as the first example, only now it will be tested against 2 different fake configurations.\n\n```python\n# test_smoketest.py\n\nimport ipaddress\nimport json\n\n\ndef test_api_server(apiclient):\n    apiclient.request(\'GET\', \'/\')\n    response = apiclient.getresponse()\n    assert response.status == 200\n    result = json.loads(response.read())\n    ipaddress.ip_address(result[\'result\'])\n```\n\nThis test will be invoked twice - once against the memory backend, and once against the sqlite backend.\n\n\n### Fixture wrappers\n\nYou can wrap your fixtures with a `wrapper_class`. This allows you to add helper methods to fixtures for use in your tests. In the case of the `container` fixture factory you can also implement `ready()` to add additional container readyness checks.\n\nIn previous tests we\'ve created an entire test client fixture. With `wrapper_class` we could hang this convenience method off the fixture itself instead:\n\n```python\n# test_fixture_wrappers.py\n\nimport ipaddress\nimport json\nimport random\n\nfrom http.client import HTTPConnection\n\nimport pytest\nfrom pytest_docker_tools import build, container\nfrom pytest_docker_tools import wrappers\n\n\nclass Container(wrappers.Container):\n\n    def ready(self):\n        # This is called until it returns True - its a great hook for e.g.\n        # waiting until a log message appears or a pid file is created etc\n        if super().ready():\n            return random.choice([True, False])\n        return False\n\n    def client(self):\n        port = self.ports[\'8080/tcp\'][0]\n        return HTTPConnection(f\'localhost:{port}\')\n\n\nfakedns_image = build(\n    path=\'examples/resolver-service/dns\',\n)\n\nfakedns = container(\n    image=\'{fakedns_image.id}\',\n    environment={\n        \'DNS_EXAMPLE_COM__A\': \'127.0.0.1\',\n    }\n)\n\napiserver_image = build(\n    path=\'examples/resolver-service/api\',\n)\n\napiserver = container(\n    image=\'{apiserver_image.id}\',\n    ports={\n        \'8080/tcp\': None,\n    },\n    dns=[\'{fakedns.ips.primary}\'],\n    wrapper_class=Container,\n)\n\n\ndef test_container_wrapper_class(apiserver):\n    client = apiserver.client()\n    client.request(\'GET\', \'/\')\n    response = client.getresponse()\n    assert response.status == 200\n    result = json.loads(response.read())\n    ipaddress.ip_address(result[\'result\'])\n\n```\n\n## Hacking\n\nSet up a virtualenv for development:\n\n```\n$ python3 -m venv venv\n$ . venv/bin/activate\n$ pip install -U pip\n$ pip install flit -r requirements.txt\n$ flit install --symlink\n```\n\nRun tests and check coverage:\n\n```\n$ coverage run --source pytest_docker_tools -m py.test\n$ coverage html\n$ $BROWSER htmlcov/index.html\n```\n\nCheck style:\n\n```\n$ flake8 pytest_docker_tools tests\n```\n',
    'author': 'John Carr',
    'author_email': 'john.carr@unrouted.co.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Jc2k/pytest-docker-tools',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
