#!/usr/bin/env python3
"""
Kenning CLI - Report Command Implementation

This module implements the `report` command that generates comprehensive Markdown
summaries of Kenning's risk analysis findings. The report command transforms
structured JSON data from scans into human-readable documentation suitable for
sharing with teams, management, and compliance auditors.

Key Features:
    - Professional Markdown report generation with structured sections
    - Executive summaries with key metrics and risk breakdowns
    - Detailed findings organized by priority and risk type
    - Actionable remediation recommendations
    - Support for multiple input and output file paths

Architecture:
    - Loads scan results from JSON files generated by the scan command
    - Converts JSON data back into RiskItem objects for processing
    - Integrates with the markdown report writer for document generation
    - Handles various edge cases including empty results and file errors
    - Provides clear user feedback and error messaging

For detailed usage patterns and examples, see technical_docs/cli/report.md
"""

import click
import json
import logging
import sys
from typing import List, Dict, Any
from datetime import datetime

# Import risk models and report generation
from audit.models import RiskItem
from reports.markdown_writer import generate_markdown_report

# Configure logging
logger = logging.getLogger(__name__)


@click.command()
@click.option("--input-file", default="kenning-report.json", help="Path to the input JSON report.")
@click.option(
    "--output-file", default="kenning-summary.md", help="Path to save the output Markdown report."
)
def report(input_file: str, output_file: str):
    """
    Generate comprehensive Markdown reports from scan results.

    This command transforms Kenning's structured risk analysis data into professional
    documentation that can be shared with stakeholders, included in compliance reports,
    or used for team collaboration. The generated report includes executive summaries,
    detailed findings, and actionable remediation guidance.

    The report includes:
    - Executive summary with key metrics and risk distribution
    - Prioritized findings starting with compound risks
    - Detailed tables of security and cost issues
    - Recommended remediation actions organized by priority
    - Next steps for deeper analysis and tracking progress

    Args:
        input_file (str): Path to JSON file containing scan results from 'kenning scan'
                         (default: kenning-report.json)
        output_file (str): Path where the Markdown report should be saved
                          (default: kenning-summary.md)

    Examples:
        kenning report                                    # Use default file paths
        kenning report --output-file team-review.md      # Custom output file
        kenning report --input-file prod-scan.json --output-file prod-report.md

    Output:
        Professional Markdown report including:
        - Executive summary with risk statistics
        - Compound risks (highest priority)
        - Security vulnerabilities
        - Cost optimization opportunities
        - Remediation recommendations
        - Next steps and action items

    Raises:
        FileNotFoundError: When the input file cannot be found
        ValueError: When the JSON file format is invalid
        PermissionError: When unable to write to the output location
    """
    try:
        # Load and parse JSON data from input file
        click.echo(f"📖 Loading scan results from: {input_file}")

        try:
            with open(input_file, "r") as f:
                scan_data = json.load(f)
        except FileNotFoundError:
            error_msg = f"Input file not found: {input_file}"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            click.echo("💡 Tip: Run 'kenning scan' first to generate scan results.")
            sys.exit(1)
        except json.JSONDecodeError as e:
            error_msg = f"Invalid JSON in input file: {e}"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            sys.exit(1)

        # Validate that scan_data is a list
        if not isinstance(scan_data, list):
            error_msg = "Invalid scan file format: expected a list of risks"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            sys.exit(1)

        click.echo(f"📊 Loaded {len(scan_data)} risks from scan results")

        # Handle empty results
        if len(scan_data) == 0:
            click.echo("🎉 No risks found in scan results!")
            # Generate a report for empty results
            empty_report_content = f"""# Kenning Security & Cost Analysis Report

**Generated:** {datetime.now().strftime("%Y-%m-%d %H:%M:%S UTC")}  
**Analysis Engine:** Kenning CLI v1.0.0  
**Total Findings:** 0 risks

---

## 🎉 Excellent News!

Your AWS environment scan completed successfully with **zero security vulnerabilities or cost optimization opportunities** detected.

### What This Means
- Your infrastructure configuration follows security best practices
- No obvious cost optimization opportunities were identified
- All scanned resources appear to be properly configured

### Recommendations
1. **Maintain Current Standards** - Continue following the practices that resulted in this clean scan
2. **Regular Monitoring** - Schedule periodic scans to catch any configuration drift
3. **Expand Scope** - Consider scanning additional regions or resource types

---

*This report was generated by Kenning CLI. Consider running scans regularly to maintain this excellent security and cost posture.*
"""
            with open(output_file, "w", encoding="utf-8") as f:
                f.write(empty_report_content)

            click.echo(f"📄 Generated clean report: {output_file}")
            return

        # Convert scan data back to RiskItem objects
        click.echo("🔄 Converting scan data to risk objects...")
        risk_items = []

        for i, risk_data in enumerate(scan_data):
            try:
                if not isinstance(risk_data, dict):
                    logger.warning(f"Skipping invalid risk item at index {i}: not a dictionary")
                    continue

                # Create RiskItem with proper error handling for missing fields
                risk_item = RiskItem(
                    resource_type=risk_data.get("resource_type", "Unknown"),
                    resource_id=risk_data.get("resource_id", f"unknown-{i}"),
                    resource_region=risk_data.get("resource_region", "unknown"),
                    risk_type=risk_data.get("risk_type", "Unknown"),
                    risk_description=risk_data.get("risk_description", "No description available"),
                    resource_metadata=risk_data.get("resource_metadata", {}),
                    discovered_at=None,  # Will be set by RiskItem.__post_init__
                )
                risk_items.append(risk_item)

            except Exception as e:
                logger.warning(f"Error processing risk item at index {i}: {e}")
                continue

        click.echo(f"✅ Successfully processed {len(risk_items)} risk items")

        if len(risk_items) == 0:
            error_msg = "No valid risk items could be processed from the scan data"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            click.echo("💡 The scan data might be corrupted or in an unexpected format.")
            sys.exit(1)

        # Generate the Markdown report
        click.echo(f"📝 Generating Markdown report...")
        try:
            generate_markdown_report(risk_items, output_file)
        except Exception as e:
            error_msg = f"Error generating report: {e}"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            sys.exit(1)

        # Success message with summary
        click.echo()
        click.echo("✅ Report generated successfully!")
        click.echo(f"📄 Report saved to: {output_file}")

        # Provide helpful summary statistics
        risk_stats = {
            "total": len(risk_items),
            "security": len([r for r in risk_items if r.risk_type == "Security"]),
            "cost": len([r for r in risk_items if r.risk_type == "Cost"]),
            "compound": len(
                [
                    r
                    for r in risk_items
                    if r.risk_type == "Both" or "compound" in r.risk_description.lower()
                ]
            ),
        }

        click.echo()
        click.echo("📊 Report Summary:")
        click.echo(f"  • Total Risks: {risk_stats['total']}")
        click.echo(f"  • Security Risks: {risk_stats['security']}")
        click.echo(f"  • Cost Risks: {risk_stats['cost']}")
        click.echo(f"  • Compound Risks: {risk_stats['compound']}")

        click.echo()
        click.echo("💡 Next Steps:")
        click.echo(f"  • Review the generated report: {output_file}")
        click.echo("  • Share with your team for remediation planning")
        if risk_stats["compound"] > 0:
            click.echo("  • Focus on compound risks first (highest impact)")
        click.echo(f"  • Use 'kenning explain' for detailed analysis of specific risks")

    except KeyboardInterrupt:
        click.echo("\n⚠️ Operation cancelled by user.")
        sys.exit(1)

    except Exception as e:
        logger.error(f"Unexpected error in report command: {e}")
        click.echo(f"❌ Unexpected error: {e}")
        click.echo("💡 Use --help for usage information or check the logs for details.")
        sys.exit(1)
