#!/usr/bin/env python3
"""
Kenning CLI - Explain Command Implementation

This module implements the `explain` command that provides detailed, AI-powered
explanations for individual risks from completed scans. The explain command bridges
the gap between technical risk analysis and human understanding by leveraging
Kenning's AI explanation engine.

Key Features:
    - AI-powered explanation generation for compound risks
    - Support for explaining specific risks by resource ID
    - Structured output with executive summaries and technical analysis
    - Actionable remediation steps for each explained risk
    - Graceful error handling when risks are not found or AI services fail

Architecture:
    - Loads scan results from JSON files generated by the scan command
    - Integrates with Kenning's AI explanation engine (AIExplainer)
    - Processes RiskItem objects through AI analysis pipeline
    - Formats explanations for optimal readability in terminal output
    - Handles edge cases and provides clear error messages

For detailed usage patterns and examples, see technical_docs/cli/explain.md
"""

import click
import json
import logging
import sys
from typing import List, Dict, Any, Optional

# Import risk models and AI explanation engine
from audit.models import RiskItem
from agent.explainer import AIExplainer
from agent.models import AIExplanation

# Configure logging
logger = logging.getLogger(__name__)


@click.command()
@click.option(
    "--input-file", required=True, help="Path to the kenning-report.json file from a scan."
)
@click.option("--risk-id", required=True, help="The resource_id of the risk you want to explain.")
def explain(input_file: str, risk_id: str):
    """
    Generate AI-powered explanations for specific risks.

    This command takes a risk identified during a scan and generates a detailed,
    human-readable explanation using Kenning's AI analysis engine. The explanation
    includes why the risk is dangerous, how it could be exploited, and specific
    steps to remediate the issue.

    The command loads scan results from a JSON file (generated by 'kenning scan')
    and searches for the specified risk by its resource_id. Once found, it creates
    a RiskItem object and passes it to the AI explanation engine for analysis.

    Args:
        input_file (str): Path to the JSON file containing scan results
                         (typically kenning-report.json)
        risk_id (str): The resource_id of the specific risk to explain
                      (e.g., "i-1234567890abcdef0" for EC2 instances)

    Examples:
        kenning explain --input-file kenning-report.json --risk-id i-1234567890abcdef0
        kenning explain --input-file /path/to/scan-results.json --risk-id my-bucket-name

    Output:
        Formatted Markdown explanation including:
        - Risk title and severity score
        - Executive summary for non-technical stakeholders
        - Technical amplification analysis
        - Actionable remediation steps
        - Context about why this combination is dangerous

    Raises:
        FileNotFoundError: When the input file cannot be found
        ValueError: When the JSON file format is invalid
        RuntimeError: When the specified risk_id is not found in the scan results
        Exception: When AI explanation generation fails
    """
    try:
        # Load and parse JSON data from input file
        click.echo(f"📖 Loading scan results from: {input_file}")

        try:
            with open(input_file, "r") as f:
                scan_data = json.load(f)
        except FileNotFoundError:
            error_msg = f"Input file not found: {input_file}"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            click.echo("💡 Tip: Run 'kenning scan' first to generate scan results.")
            sys.exit(1)
        except json.JSONDecodeError as e:
            error_msg = f"Invalid JSON in input file: {e}"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            sys.exit(1)

        # Validate that scan_data is a list
        if not isinstance(scan_data, list):
            error_msg = "Invalid scan file format: expected a list of risks"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            sys.exit(1)

        click.echo(f"📊 Loaded {len(scan_data)} risks from scan results")

        # Find the specific risk by resource_id
        target_risk_data = None
        for risk_data in scan_data:
            if isinstance(risk_data, dict) and risk_data.get("resource_id") == risk_id:
                target_risk_data = risk_data
                break

        # Check if risk was found
        if target_risk_data is None:
            available_ids = [
                risk.get("resource_id", "unknown") for risk in scan_data if isinstance(risk, dict)
            ]
            error_msg = f"Risk with resource_id '{risk_id}' not found in scan results"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            click.echo(f"\n📋 Available resource IDs:")
            for rid in sorted(set(available_ids))[:10]:  # Show first 10 unique IDs
                click.echo(f"  • {rid}")
            if len(available_ids) > 10:
                click.echo(f"  ... and {len(available_ids) - 10} more")
            sys.exit(1)

        click.echo(f"🎯 Found risk: {target_risk_data.get('risk_description', 'Unknown risk')}")

        # Create RiskItem object from the found data
        try:
            # Handle potential missing fields with defaults
            risk_item = RiskItem(
                resource_type=target_risk_data.get("resource_type", "Unknown"),
                resource_id=target_risk_data.get("resource_id", risk_id),
                resource_region=target_risk_data.get("resource_region", "unknown"),
                risk_type=target_risk_data.get("risk_type", "Unknown"),
                risk_description=target_risk_data.get(
                    "risk_description", "No description available"
                ),
                resource_metadata=target_risk_data.get("resource_metadata", {}),
                discovered_at=None,  # Will be set by RiskItem.__post_init__
            )
        except Exception as e:
            error_msg = f"Error creating RiskItem from data: {e}"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            click.echo("💡 The scan data might be corrupted or in an unexpected format.")
            sys.exit(1)

        # Initialize AI Explainer
        click.echo("🤖 Initializing AI explanation engine...")
        try:
            explainer = AIExplainer()
        except Exception as e:
            error_msg = f"Failed to initialize AI explainer: {e}"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            click.echo(
                "💡 Make sure Ollama is installed and running, or check your AI configuration."
            )
            sys.exit(1)

        # Generate explanation
        click.echo("🧠 Generating AI-powered explanation...")
        try:
            explanation = explainer.explain(risk_item)
        except Exception as e:
            error_msg = f"AI explanation generation failed: {e}"
            logger.error(error_msg)
            click.echo(f"❌ Error: {error_msg}")
            click.echo("💡 This might be due to AI service issues or model availability.")
            sys.exit(1)

        # Format and display the explanation
        click.echo("\n" + "=" * 80)
        click.echo("🔍 KENNING AI RISK ANALYSIS")
        click.echo("=" * 80)
        click.echo()

        # Risk title and severity
        severity_emoji = (
            "🔥"
            if explanation.severity_score >= 8
            else "⚠️" if explanation.severity_score >= 5 else "💡"
        )
        click.echo(f"## {severity_emoji} {explanation.risk_title}")
        click.echo()
        click.echo(f"**Severity Score:** {explanation.severity_score}/10")
        click.echo()

        # Executive summary
        click.echo("### 📊 Executive Summary")
        click.echo()
        click.echo(f"*{explanation.executive_summary}*")
        click.echo()

        # Technical analysis
        click.echo("### 🔬 Technical Analysis")
        click.echo()
        click.echo(explanation.amplification_analysis)
        click.echo()

        # Remediation steps
        click.echo("### 🛠️ Remediation Steps")
        click.echo()
        for i, step in enumerate(explanation.remediation_steps, 1):
            click.echo(f"{i}. {step}")
        click.echo()

        # Additional context if available
        if hasattr(target_risk_data, "resource_metadata") and target_risk_data.get(
            "resource_metadata"
        ):
            metadata = target_risk_data["resource_metadata"]
            if "correlation_details" in metadata:
                click.echo("### 🔗 Risk Correlation Context")
                click.echo()
                click.echo(metadata["correlation_details"])
                click.echo()

        click.echo("=" * 80)
        click.echo("✅ Explanation generated successfully!")
        click.echo()
        click.echo("💡 **Next Steps:**")
        click.echo("   • Review the remediation steps above")
        click.echo("   • Assign tasks to appropriate team members")
        click.echo(f"   • Generate a full report with: kenning report --input-file {input_file}")

    except KeyboardInterrupt:
        click.echo("\n⚠️ Operation cancelled by user.")
        sys.exit(1)

    except Exception as e:
        logger.error(f"Unexpected error in explain command: {e}")
        click.echo(f"❌ Unexpected error: {e}")
        click.echo("💡 Use --help for usage information or check the logs for details.")
        sys.exit(1)
