"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityConfiguration = exports.JobBookmarksEncryptionMode = exports.CloudWatchEncryptionMode = exports.S3EncryptionMode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const kms = require("@aws-cdk/aws-kms");
const cdk = require("@aws-cdk/core");
const glue_generated_1 = require("./glue.generated");
/**
 * Encryption mode for S3.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_S3Encryption.html#Glue-Type-S3Encryption-S3EncryptionMode
 */
var S3EncryptionMode;
(function (S3EncryptionMode) {
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    S3EncryptionMode["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    S3EncryptionMode["KMS"] = "SSE-KMS";
})(S3EncryptionMode = exports.S3EncryptionMode || (exports.S3EncryptionMode = {}));
/**
 * Encryption mode for CloudWatch Logs.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_CloudWatchEncryption.html#Glue-Type-CloudWatchEncryption-CloudWatchEncryptionMode
 */
var CloudWatchEncryptionMode;
(function (CloudWatchEncryptionMode) {
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    CloudWatchEncryptionMode["KMS"] = "SSE-KMS";
})(CloudWatchEncryptionMode = exports.CloudWatchEncryptionMode || (exports.CloudWatchEncryptionMode = {}));
/**
 * Encryption mode for Job Bookmarks.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_JobBookmarksEncryption.html#Glue-Type-JobBookmarksEncryption-JobBookmarksEncryptionMode
 */
var JobBookmarksEncryptionMode;
(function (JobBookmarksEncryptionMode) {
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    JobBookmarksEncryptionMode["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(JobBookmarksEncryptionMode = exports.JobBookmarksEncryptionMode || (exports.JobBookmarksEncryptionMode = {}));
/**
 * A security configuration is a set of security properties that can be used by AWS Glue to encrypt data at rest.
 *
 * The following scenarios show some of the ways that you can use a security configuration.
 * - Attach a security configuration to an AWS Glue crawler to write encrypted Amazon CloudWatch Logs.
 * - Attach a security configuration to an extract, transform, and load (ETL) job to write encrypted Amazon Simple Storage Service (Amazon S3) targets and encrypted CloudWatch Logs.
 * - Attach a security configuration to an ETL job to write its jobs bookmarks as encrypted Amazon S3 data.
 * - Attach a security configuration to a development endpoint to write encrypted Amazon S3 targets.
 */
class SecurityConfiguration extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityConfigurationName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_SecurityConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SecurityConfiguration);
            }
            throw error;
        }
        if (!props.s3Encryption && !props.cloudWatchEncryption && !props.jobBookmarksEncryption) {
            throw new Error('One of cloudWatchEncryption, jobBookmarksEncryption or s3Encryption must be defined');
        }
        const kmsKeyCreationRequired = (props.s3Encryption && props.s3Encryption.mode === S3EncryptionMode.KMS && !props.s3Encryption.kmsKey) ||
            (props.cloudWatchEncryption && !props.cloudWatchEncryption.kmsKey) ||
            (props.jobBookmarksEncryption && !props.jobBookmarksEncryption.kmsKey);
        const autoCreatedKmsKey = kmsKeyCreationRequired ? new kms.Key(this, 'Key') : undefined;
        let cloudWatchEncryption;
        if (props.cloudWatchEncryption) {
            this.cloudWatchEncryptionKey = props.cloudWatchEncryption.kmsKey || autoCreatedKmsKey;
            cloudWatchEncryption = {
                cloudWatchEncryptionMode: props.cloudWatchEncryption.mode,
                kmsKeyArn: this.cloudWatchEncryptionKey?.keyArn,
            };
        }
        let jobBookmarksEncryption;
        if (props.jobBookmarksEncryption) {
            this.jobBookmarksEncryptionKey = props.jobBookmarksEncryption.kmsKey || autoCreatedKmsKey;
            jobBookmarksEncryption = {
                jobBookmarksEncryptionMode: props.jobBookmarksEncryption.mode,
                kmsKeyArn: this.jobBookmarksEncryptionKey?.keyArn,
            };
        }
        let s3Encryptions;
        if (props.s3Encryption) {
            if (props.s3Encryption.mode === S3EncryptionMode.KMS) {
                this.s3EncryptionKey = props.s3Encryption.kmsKey || autoCreatedKmsKey;
            }
            // NOTE: CloudFormations errors out if array is of length > 1. That's why the props don't expose an array
            s3Encryptions = [{
                    s3EncryptionMode: props.s3Encryption.mode,
                    kmsKeyArn: this.s3EncryptionKey?.keyArn,
                }];
        }
        const resource = new glue_generated_1.CfnSecurityConfiguration(this, 'Resource', {
            name: props.securityConfigurationName,
            encryptionConfiguration: {
                cloudWatchEncryption,
                jobBookmarksEncryption,
                s3Encryptions,
            },
        });
        this.securityConfigurationName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Creates a Connection construct that represents an external security configuration.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param securityConfigurationName name of external security configuration.
     */
    static fromSecurityConfigurationName(scope, id, securityConfigurationName) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.securityConfigurationName = securityConfigurationName;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecurityConfiguration = SecurityConfiguration;
_a = JSII_RTTI_SYMBOL_1;
SecurityConfiguration[_a] = { fqn: "@aws-cdk/aws-glue.SecurityConfiguration", version: "1.166.1" };
//# sourceMappingURL=data:application/json;base64,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