# SPDX-License-Identifier: MPL-2.0
# Copyright 2020-2022 John Mille <john@compose-x.io>

"""
Module for the XStack SQS
"""
from __future__ import annotations

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from ecs_composex.common.settings import ComposeXSettings
    from ecs_composex.mods_manager import XResourceModule

from troposphere import GetAtt, Ref

from ecs_composex.common import add_update_mapping, build_template
from ecs_composex.common.stacks import ComposeXStack
from ecs_composex.compose.x_resources.api_x_resources import ApiXResource
from ecs_composex.compose.x_resources.helpers import (
    set_lookup_resources,
    set_new_resources,
    set_resources,
)
from ecs_composex.sqs.sqs_ecs_scaling import handle_service_scaling
from ecs_composex.sqs.sqs_helpers import resolve_lookup
from ecs_composex.sqs.sqs_params import SQS_ARN, SQS_KMS_KEY, SQS_NAME, SQS_URL
from ecs_composex.sqs.sqs_template import render_new_queues


class Queue(ApiXResource):
    """
    Class to represent a SQS Queue
    """

    def __init__(
        self,
        name: str,
        definition: dict,
        module: XResourceModule,
        settings: ComposeXSettings,
    ):
        super().__init__(name, definition, module, settings)
        self.kms_arn_attr = SQS_KMS_KEY
        self.arn_parameter = SQS_ARN
        self.ref_parameter = SQS_URL
        self.predefined_resource_service_scaling_function = handle_service_scaling

    def init_outputs(self):
        """
        Init output properties for a new resource
        """
        self.output_properties = {
            SQS_URL: (self.logical_name, self.cfn_resource, Ref, None, "Url"),
            SQS_ARN: (
                f"{self.logical_name}{SQS_ARN.return_value}",
                self.cfn_resource,
                GetAtt,
                SQS_ARN.return_value,
                "Arn",
            ),
            SQS_NAME: (
                f"{self.logical_name}{SQS_NAME.return_value}",
                self.cfn_resource,
                GetAtt,
                SQS_NAME.return_value,
                "QueueName",
            ),
        }


class XStack(ComposeXStack):
    """
    Class to handle SQS Root stack related actions
    """

    def __init__(
        self, title, settings: ComposeXSettings, module: XResourceModule, **kwargs
    ):
        """
        :param str title: Name of the stack
        :param ecs_composex.common.settings.ComposeXSettings settings:
        :param dict kwargs:
        """
        set_resources(settings, Queue, module)
        x_resources = settings.compose_content[module.res_key].values()
        lookup_resources = set_lookup_resources(x_resources)
        if lookup_resources:
            resolve_lookup(lookup_resources, settings, module)
        new_resources = set_new_resources(x_resources, True)
        if new_resources:
            template = build_template("SQS template generated by ECS Compose-X")
            if lookup_resources:
                add_update_mapping(
                    template, module.mapping_key, settings.mappings[module.mapping_key]
                )
            super().__init__(title, stack_template=template, **kwargs)
            render_new_queues(settings, new_resources, x_resources, self, template)
        else:
            self.is_void = True
        for resource in settings.compose_content[module.res_key].values():
            resource.stack = self
