import { BytesLiteral } from "./micheline";
import { MichelsonData, MichelsonType } from "./michelson-types";
declare type AddressType = "ED25519PublicKeyHash" | "SECP256K1PublicKeyHash" | "P256PublicKeyHash" | "ContractHash";
export interface Address {
    type: AddressType;
    hash: number[] | Uint8Array;
    entryPoint?: string;
}
export declare type PublicKeyType = "ED25519PublicKey" | "SECP256K1PublicKey" | "P256PublicKey";
export interface PublicKey {
    type: PublicKeyType;
    publicKey: number[] | Uint8Array;
}
/**
 * Serializes any value of packable type to its optimized binary representation
 * identical to the one used by PACK and UNPACK Michelson instructions.
 * Without a type definition (not recommended) the data will be encoded as a binary form of a generic Michelson expression.
 * Type definition allows some types like `timestamp` and `address` and other base58 representable types to be encoded to
 * corresponding optimized binary forms borrowed from the Tezos protocol
 *
 * ```typescript
 * const data: MichelsonData = {
 *     string: "KT1RvkwF4F7pz1gCoxkyZrG1RkrxQy3gmFTv%foo"
 * };
 *
 * const typ: MichelsonType = {
 *     prim: "address"
 * };
 *
 * const packed = packData(data, typ);
 *
 * // 050a0000001901be41ee922ddd2cf33201e49d32da0afec571dce300666f6f
 * ```
 *
 * Without a type definition the base58 encoded address will be treated as a string
 * ```typescript
 * const data: MichelsonData = {
 *     string: "KT1RvkwF4F7pz1gCoxkyZrG1RkrxQy3gmFTv%foo"
 * };
 *
 * const packed = packData(data);
 *
 * // 0501000000284b543152766b7746344637707a3167436f786b795a724731526b7278517933676d46547625666f6f
 * ```
 * @param d Data object
 * @param t Optional type definition
 * @returns Binary representation as numeric array
 */
export declare function packData(d: MichelsonData, t?: MichelsonType): number[];
/**
 * Serializes any value of packable type to its optimized binary representation
 * identical to the one used by PACK and UNPACK Michelson instructions.
 * Same as {@link packData} but returns a `bytes` Michelson data literal instead of an array
 *
 * ```typescript
 * const data: MichelsonData = {
 *     string: "2019-09-26T10:59:51Z"
 * };
 *
 * const typ: MichelsonType = {
 *     prim: "timestamp"
 * };
 *
 * const packed = packDataBytes(data, typ);
 *
 * // { bytes: "0500a7e8e4d80b" }
 * ```
 * @param d Data object
 * @param t Optional type definition
 * @returns Binary representation as a bytes literal
 */
export declare function packDataBytes(d: MichelsonData, t?: MichelsonType): BytesLiteral;
/**
 * Deserialize a byte array into the corresponding Michelson value.
 * Without a type definition (not recommended) the binary data will be treated as a binary form of a generic Michelson expression and returned as is.
 * Type definition allows some types like `timestamp` and `address` and other types usually encoded in optimized binary forms to be transformed
 * back to their string representations like base58 and ISO timestamps.
 *
 * ```typescript
 * const src = [0x05, 0x00, 0xa7, 0xe8, 0xe4, 0xd8, 0x0b];
 *
 * const typ: MichelsonType = {
 *     prim: "timestamp"
 * };
 *
 * const data = unpackData(src, typ);
 *
 * // { string: "2019-09-26T10:59:51Z" }
 * ```
 *
 * Same binary data without a type definition
 * ```typescript
 * const src = [0x05, 0x00, 0xa7, 0xe8, 0xe4, 0xd8, 0x0b];
 *
 * const data = unpackData(src);
 *
 * // { int: "1569495591" }
 * ```
 * @param src Byte array
 * @param t Optional type definition
 * @returns Deserialized data
 */
export declare function unpackData(src: number[] | Uint8Array, t?: MichelsonType): MichelsonData;
/**
 * Deserialize a byte array into the corresponding Michelson value.
 * Same as {@link unpackData} but takes a `bytes` Michelson data literal instead of an array
 *
 * ```typescript
 * const src = { bytes: "0500a7e8e4d80b" };
 *
 * const typ: MichelsonType = {
 *     prim: "timestamp"
 * };
 *
 * const data = unpackDataBytes(src, typ);
 *
 * // { string: "2019-09-26T10:59:51Z" }
 * ```
 * @param src Bytes object
 * @param t Optional type definition
 * @returns Deserialized data
 */
export declare function unpackDataBytes(src: BytesLiteral, t?: MichelsonType): MichelsonData;
export declare function decodeAddressBytes(b: BytesLiteral): Address;
export declare function decodePublicKeyHashBytes(b: BytesLiteral): Address;
export declare function decodePublicKeyBytes(b: BytesLiteral): PublicKey;
export {};
