"""
A one cli for all the anime.
"""

import os
import random
import sys

import click
from rich.align import Align
from rich.style import Style
from rich.text import Text
from rich.traceback import install

from .core import __version__
from .core.cli.commands import download, grab, schedule, search, stream, update
from .core.cli.helpers import stream_handlers

install(show_locals=True, suppress=[click])

commands = {
    "download": download.ap0dl_download,
    "grab": grab.ap0dl_grab,
    "schedule": schedule.ap0dl_schedule,
    "search": search.ap0dl_search,
    "update": update.ap0dl_update,
    "stream": stream.ap0dl_stream,
}


@click.group(commands=commands, invoke_without_command=True)
@click.version_option(__version__.__core__, "--version")
@click.option("-x", "--disable-update", is_flag=True, help="Disable update check.")
@click.pass_context
def __ap0dl_cli__(ctx: click.Context, disable_update):
    from .core.cli.helpers import constants

    sys.stderr = sys.__stderr__
    console = stream_handlers.get_console()

    if ctx.invoked_subcommand is None:
        console.print(
            "The project is operational. Use --help flag for help & be sure read the documentation!"
        )

    author, repository_name = constants.SOURCE_REPOSITORY

    console.print(
        Align(
            Text(
                f"{author}/{repository_name} v{__version__.__core__}",
                style="magenta",
            ),
            align="center",
        )
    )

    with stream_handlers.context_raiser(
        console,
        Text(
            f"{random.choice(('Hey', 'Hello', 'Welcome'))}, {os.getenv('USERNAME', 'buddy')}.",
            style="bold white",
        ),
    ):
        for greeting in stream_handlers.iter_greetings():
            console.print(greeting, style="white")

        if not disable_update:
            from .core.cli.http_client import client
            from .core.codebase.helpers.optopt import regexlib

            branch, version_file = constants.VERSION_FILE_PATH

            upstream_version = regexlib.search(
                r'__core__ = "(.*?)"',
                client.get(
                    f"https://raw.githubusercontent.com/{author}/{repository_name}/{branch}/{version_file}"
                ).text,
            ).group(1)

            tuplisied_upstream, tuplised_current_version = tuple(
                map(int, upstream_version.split("."))
            ), tuple(map(int, __version__.__core__.split(".")))

            if tuplisied_upstream > tuplised_current_version:
                with stream_handlers.context_raiser(
                    console, "[cyan]An update is available![/]"
                ):
                    console.print(
                        f"↑ {upstream_version} ↓ {__version__.__core__}, to update, use: ap0dl update",
                        style=Style(color="magenta"),
                    )
                    console.print(
                        f"Staying up-to-date resolves bugs and security vulnerabilities!",
                        style="dim i",
                    )


if __name__ == "__main__":
    __ap0dl_cli__()
