import inspect
from abc import ABC, abstractmethod
import pathlib

from jinja2 import Environment, PackageLoader, select_autoescape


class Writer(ABC):

    def __init__(self, app=None, name=None, data=None, template_filenames=None, env=None):
        self.output = None
        self.name = name
        self.data = data or {}
        self.template_filenames = template_filenames or self.default_template_filenames
        self.env = env or Environment(
            loader=PackageLoader("coworks", "templates"),
            autoescape=select_autoescape(['html', 'xml'])
        )

        if app is not None:
            self.app = app
            self.init_app(app)

    def init_app(self, app):
        if 'writers' not in app.extensions:
            app.extensions['writers'] = {}
        app.extensions['writers'][self.name] = self

    def export(self, output=None, **kwargs):
        self.output = None
        if output is not None:
            self.output = open(output, 'w+') if type(output) is str else output

        self._export_header(**kwargs)
        self._export_content(**kwargs)

    @property
    @abstractmethod
    def default_template_filenames(self):
        ...

    @abstractmethod
    def _export_header(self, **kwargs):
        ...

    def _export_content(self, module_name='app', handler_name='app', **kwargs):
        module_path = module_name.split('.')
        data = {
            'writer': self,
            'module': module_name,
            'module_path': pathlib.PurePath(*module_path),
            'module_dir': pathlib.PurePath(*module_path[:-1]),
            'module_file': module_path[-1],
            'handler': handler_name,
            'app': self.app,
            'app_name': self.app.app_name,
        }
        data.update(self.data)
        for template_filename in self.template_filenames:
            template = self.env.get_template(template_filename)
            print(template.render(**data), file=self.output)


class ListWriter(Writer):

    def __init__(self, app=None, name='list', **kwargs):
        super().__init__(app, name, **kwargs)

    @property
    def default_template_filenames(self):
        return ['list.j2']

    def _export_header(self, **kwargs):
        pass


class OpenApiWriter(Writer):
    """Export the microservice in swagger format."""

    def __init__(self, app=None, name='openapi', **kwargs):
        super().__init__(app, name, **kwargs)

    @property
    def default_template_filenames(self):
        return ['openapi.yml']

    def _export_header(self, **kwargs):
        print("# Do NOT edit this file as it is auto-generated by cws\n", file=self.output)

    @property
    def paths(self):
        paths = {}
        for route, entry in self.app.routes.items():
            entries = {}
            for method, infos in entry.items():
                func = self.app.entries[route]
                args = inspect.getfullargspec(func).args[1:]
                defaults = inspect.getfullargspec(func).defaults or []
                len_defaults = len(defaults)
                entry = {
                    'description': func.__doc__,
                    'parameters': []
                }
                for index, arg in enumerate(args[1:-len_defaults]):
                    entry['parameters'].append({
                        'name': f"_{index}:{arg}",
                        'in': 'path',
                    })
                for arg in args[-len_defaults:]:
                    entry['parameters'].append({
                        'name': arg,
                        'in': 'query',
                    })
                for varkw in inspect.getfullargspec(func).varkw or []:
                    entry['parameters'].append({
                        'name': varkw,
                        'in': 'query',
                    })
                entries[method.lower()] = entry
            paths[route] = entries
        return paths


class TerraformWriter(Writer):

    def __init__(self, app=None, name='terraform', **kwargs):
        super().__init__(app=app, name=name, **kwargs)

    @property
    def default_template_filenames(self):
        return ['terraform.j2']

    def _export_header(self, **kwargs):
        print("// Do NOT edit this file as it is auto-generated by cws\n", file=self.output)

    @property
    def entries(self):

        def uid(_path):
            if _path:
                return f"{self.app.app_name}_{_path.replace('{', '').replace('}', '')}"
            return None

        def combine(_prev, _path):
            return f"{_prev}_{_path}" if _prev else _path

        all_pathes_id = {}
        for route, entry in self.app.routes.items():
            prev_path_id = None
            splited_route = route[1:].split('/')

            # special root case
            if splited_route == ['']:
                all_pathes_id[uid('root')] = (None, None, entry.keys())
                continue

            # creates intermediate resources
            last = splited_route[-1:][0]
            for path in splited_route[:-1]:
                current_path_id = combine(prev_path_id, path)
                if current_path_id not in all_pathes_id:
                    all_pathes_id[uid(current_path_id)] = (uid(prev_path_id), path, None)
                prev_path_id = current_path_id

            # set entryes keys for last entry
            current_path_id = combine(prev_path_id, last)
            all_pathes_id[uid(current_path_id)] = (uid(prev_path_id), last, entry.keys())

        return all_pathes_id


def reduce_not_none(data):
    """Remove all entries where value is None."""
    return dict((k, v) for k, v in data.items() if v is not None)
