'''
Created on Sep. 25, 2019

@author: KJNETHER

This module uses mostly the bcdc_dataset_schema module as an interface to the
scheming rules provided via the bcdc scheming extension.

This module will use the scheming data to return actual data that will be used
in the testing.
'''
import datetime
import inspect
import json
import logging
import os.path
import random
import re
import shutil

import randomwordgenerator.randomwordgenerator

import bcdc_apitests.config.testConfig as testConfig
# import bcdc_apitests.helpers.data_config as data_config
from bcdc_apitests.helpers.file_utils import FileUtils

LOGGER = logging.getLogger(__name__)
WORDS = []

# pylint: disable=logging-fstring-interpolation, logging-not-lazy


def undefined_prefix(fld):

    msg = f'prefix is set to: {fld.preset}.  There is no code to ' + \
        'to accomodate this type.  Need to add a method definition for ' + \
        'that type to this class'
    raise UndefinedPreset(msg)


class DataPopulation():
    '''
    This class will consume a data definition from a ckan scheming end point.
    The BCDC scheming end point currently contains:
     - resource_fields
     - dataset_fields

    This class uses DataPopulationResource methods to get return datasets with
    randomized data that comply with the rules defined the scheming end points for
    use in testing.

    Every attempt has been made to keep this class generic, so that it doesn't
    matter what data definition is provided it will still be able to follow the
    rules defined in the schema and return a randomized dataset that complies
    with the data rules defined in the schema.

    All data is returned is returned as a DataSetIterator allowing for single or
    multiple datasets to be returned.

    Randomized data generated by this class is also cached to allow for re-use
    of the same data in different tests.  Current behaviour is always use
    the cached version of data if it exists.  Cache files are defined based
    on the method names that generate them.
    '''

    def __init__(self, fields_schema, data_type):
        self.schema = fields_schema
        self.data_type = data_type
        self.pop_resource = DataPopulationResource(fields_schema)
        self.fields_schema = fields_schema

        # calculate cache file
        # self.cache_dir = FileUtils().get_test_data_dir()
        self.cache = None

        # this property can be set to disable cache usage
        self.disable_cache = False

    def populate_randomized(self, overrides=None):
        '''
        Returns a single dataset, fields are all randomly populated, fields will
        obey domains, and dependencies, however their actual values are completely
        randomized.

        :param overrides: A dictionary of keys for values in the dataset that
            should be overriden, by static values vs dynamic / random values
            common values that you may want to override include:
              - name
              - title
              - org
              - owner_org (org id)
        '''
        # name of the cache file will be defined either by the test config
        # in the property 'test_data'.  'test_data' is a list of the methods
        # to use to generate dynamic data sets for use in a test.  All the
        # method names are members of this class.  This method is an example
        # of a method that can be referenced in the test config file
        # 'testParams.json'
        file_name = f'{inspect.currentframe().f_code.co_name}_{self.data_type}.json'
        self.cache = DataCache(file_name)

        data_iterable = None
        if self.cache.cache_exists() and not self.disable_cache:
            LOGGER.debug("loading a cached dataset")
            data_iterable = self.cache.load_cache_data(overrides)
        if data_iterable is None:
            LOGGER.debug("generating a new dynamic dataset")
            dataset = self.pop_resource.populate_all(overrides=overrides)
            data_iterable = DataSetIterator(dataset)
            self.cache.write_cache_data(data_iterable)
        LOGGER.debug(f"iterable: {data_iterable}")
        return data_iterable

    def populate_required_fields_failure(self, overrides=None):
        '''
        :return: an iterable where each iteration will return a dataset that is
                 missing a different field that is defined as "required" in the
                 data schema.
        :rtype: DataSetIterator
        '''
        # TODO: could add a metaclass that does the caching automatically
        file_name = f'{inspect.currentframe().f_code.co_name}.json'
        self.cache = DataCache(file_name)

        core_data = self.pop_resource.populate_all(overrides=overrides)

        # get list of required field names
        required = []
        for fld in self.schema:
            if fld.required:
                required.append(fld.field_name)

        if self.cache.cache_exists() and not self.disable_cache:
            data_iterable = self.cache.load_cache_data()
        else:
            data_iterable = DataSetIterator(core_data)
            data_iterable.flds_to_remove(required)
            self.cache.write_cache_data(data_iterable)
        return data_iterable

    def populate_bcdc_types(self, overrides=None):
        '''
        :return: an iterator that returns each of the different datasets
        '''
        if overrides is None:
            overrides = {}
        # file_name = f'{inspect.currentframe().f_code.co_name}_{self.data_type}.json'
        # self.cache = DataCache(file_name)

        # cache isn't being used

        if self.data_type != 'resource_fields':
            msg = 'bcdc_types is a field for resources not datasets, the ' + \
                  f'dataset this object is currently describing is a: {self.data_type}'
            raise ValueError(msg)
        bcdc_type_def = self.fields_schema.get_field('bcdc_type')
        choices = bcdc_type_def.choices
        bcdc_type_choices = choices.values
        LOGGER.debug(f"bcdc_types options: {bcdc_type_choices}")
        dataset_list = []
        for bcdc_type in bcdc_type_choices:
            overrides['bcdc_type'] = bcdc_type
            LOGGER.debug(f"bcdc_type overrides: {overrides}")
            dataset = self.pop_resource.populate_all(overrides=overrides)
            dataset_list.append(dataset)
        data_iterable = DataSetIterator(dataset_list)
        return data_iterable


class DataCache():
    '''
    used to create and load from cached data files.  Dynamic data gets generated
    and cached so that it can be re-used for different tests.  This class
    contains the logic to easily write and read from caches
    '''

    def __init__(self, cache_file):
        '''
        :param cache_file: just the file name to use to cache the data in a json
            file, any directory info will be ignored
        '''
        cache_file = os.path.basename(cache_file)

        self.cache_dir_name = 'bcdc_apitest_cache'
        self.set_cache_paths()
        self.cache_file = os.path.join(self.cache_dir, cache_file)
        LOGGER.debug(f"cache file: {self.cache_file}")

    def set_cache_paths(self):
        '''
        The path to be used for temp files works through this order:
           a) is there a TEMP or a TMP env var.
           b) is there a /tmp folder
           c) FAIL
        '''
        default_tmp_dir = r'/tmp'
        if 'TEMP' in os.environ:
            self.cache_dir = os.path.join(os.environ['TEMP'],
                                          self.cache_dir_name)
        elif 'TMP' in os.environ:
            self.cache_dir = os.path.join(os.environ['TMP'],
                                          self.cache_dir_name)
        elif os.path.exists(default_tmp_dir):
            self.cache_dir = os.path.join(default_tmp_dir,
                              self.cache_dir_name)
        else:
            msg = 'unable to find a temp directory to cache data to.'
            raise EnvironmentError(msg)

    def cache_exists(self):
        '''
        :return: boolean indicating if the cache exists or not
        '''
        ret_val = True if os.path.exists(self.cache_file) else False
        LOGGER.debug(f"cache file exists? {ret_val}")
        return ret_val

    def write_cache_data(self, iter_obj):
        '''
        :param iter_obj:
        :type iter_obj: DataPopulation
        '''
        # converts the iterable DataPopulation object to a list then
        # writes
        if not os.path.exists(self.cache_dir):
            LOGGER.info(f"creating the cache dir: {self.cache_dir}")
            os.makedirs(self.cache_dir)

        LOGGER.debug(f"iter type: {type(iter_obj)}")
        bcdc_dataets = []
        for dataset in iter_obj:
            LOGGER.debug(f'dataset in iterable: {dataset}')
            bcdc_dataets.append(dataset)

        try:
            with open(self.cache_file, 'w', encoding='utf8') as file_hand:
                json.dump(bcdc_dataets, file_hand, ensure_ascii=False)
            LOGGER.debug(f"cache has been written to: {self.cache_file}")
        except PermissionError:
            msg = f'cannot write to the location: {self.cache_file}, going to try tmp'
            LOGGER.warning(msg)
            raise

    def load_cache_data(self, overrides=None):
        '''
        loads data from cached file and returns as a DataPopulation
        object

        :return: a data population object of the data loaded from the cache
        :rtype: DataPopulation
        '''
        LOGGER.debug(f"Cache file is being read from: {self.cache_file}")
        bcdc_dataset = None
        if self.cache_exists():
            LOGGER.debug(f"Cache file exists")
            with open(self.cache_file, 'r') as file_hand:
                data_struct_list = json.load(file_hand)
            LOGGER.debug("loading from cache file")
            if overrides:
                updated = []
                for struct in data_struct_list:
                    struct.update(overrides)
                    updated.append(struct)
                bcdc_dataset = updated
            bcdc_dataset = DataSetIterator(data_struct_list)
        return bcdc_dataset

    def delete_cache(self):
        '''
        Delete the cache file if it exists.
        '''
        if self.cache_exists():
            LOGGER.info(f"removing the data cache file: {self.cache_file}")
            os.remove(self.cache_file)

    def delete_all_caches(self):
        '''
        removes the cache directory
        '''
        if os.path.exists(self.cache_dir):
            LOGGER.info(f"removing the existing cache dir: {self.cache_dir}")
            shutil.rmtree(self.cache_dir)


class DataSetIterator():
    '''
    class used to provide an iterable that returns different data for each
    iteration.

    A generic /configurable iteration class that is going to get re-used.

    :ivar core_data: the base dataset that each iteration will be built off of

    '''

    def __init__(self, coredata):
        # if core data is a dict then stuff it in a list
        LOGGER.debug(f"coredata: {coredata}")
        self.core_data = coredata
        if isinstance(coredata, dict):
            self.core_data = [coredata]
        LOGGER.debug(f"coredata: {coredata}")
        self.remove_flds = []
        self.ds_cnt = 0  # dataset count
        self.fld_cnt = 0  # field count

    def add_dataset(self, dataset_dict):
        self.core_data.append(dataset_dict)

    def flds_to_remove(self, fldname_list):
        '''
        :param fldname_list: a list of fields for each iteration one of these
                             fields will be removed from the resulting data
                             set
        '''
        self.remove_flds = fldname_list

    def __iter__(self):
        return self

    def reset(self):
        '''
        resets the iterator back to the start.
        '''
        self.ds_cnt = 0
        self.fld_cnt = 0

    def __next__(self):
        '''
        Iterator set up to look for:
         - fields to remove, returns new dataset with one of the remove fields
           removed with each iteration.
         - more later!
        '''
        if self.ds_cnt >= len(self.core_data):
            LOGGER.debug(f"core data: {self.core_data}")
            LOGGER.debug(f"cnt: {self.ds_cnt}")
            raise StopIteration

        # dataset = self.core_data.copy()
        return_dataset = self.core_data[self.ds_cnt].copy()
        LOGGER.debug(f"return_dataset: {return_dataset}")
        # if remvoe_flds have been defined then iterate over each
        # dataset, and for each dataset remove the core fields thus
        # the total number of iterations ends up being datasets * remove fields
        if self.remove_flds:
            if self.fld_cnt >= len(self.remove_flds):
                # iterate to next dataset
                self.ds_cnt += 1
                self.fld_cnt = 0
            else:

                fld_to_drop = self.remove_flds[self.fld_cnt]
                if fld_to_drop in return_dataset:
                    del return_dataset[fld_to_drop]
                else:
                    LOGGER.warning(f'The required field {fld_to_drop} does not ' +
                                   'exist in the current dataset')
                self.fld_cnt += 1

        self.ds_cnt += 1
        return return_dataset


class DataPopulationResource():
    '''
    This class contains re-usable functionality that can be used to assemble
    methods in the DataPopulation class.

    Idea is the publicly accessible methods in DataPopulation will be included
    in the parameterization.  Those methods will use functionality in this class
    to accomplish what they need to do.
    '''

    def __init__(self, fields):
        # TODO: may want to pass in the organization to put the data under
        self.fields = fields
        LOGGER.debug(f"type of fields: {type(fields)}")
        # TODO: use isinstance to enforce type here maybe... should be Fields or
        # subclass of.
        self.datastruct = {}
        self.rand = RandomWords()
        self.deferred = []

    def has_conditional(self, fld):  # pylint: disable=no-self-use
        '''
        :return: true if the field has a conditional property
        '''
        has_cond = False
        if fld.conditional_field:
            has_cond = True
        return has_cond

    def conditional_field_exists(self, fld):
        '''
        :return: true if the field has a conditional field and the conditional
                 field exists in the fields that have already been processed.
        '''
        cond_exists = False
        if (self.has_conditional) and fld.conditional_field in self.datastruct:
            cond_exists = True
        return cond_exists

    def conditional_satisfied(self, fld):
        '''
        :param fld: a Field object.

        * does the fld have a condition, if not then return True, proceed.

        * otherwise...

        * if conditional_field exists, then determine the value and populate_random
          field of value matches conditional value.
            * if condition is met then True
            * otherwise False

        * if the conditional_field does not exist, then add to the deferred
          processing.
            * return False
        '''
        proceed = False
        # field exists in the populated dataset, and the value of the field
        # in the populated dataset matches the condition
        if (self.conditional_field_exists(fld)) and \
           self.datastruct[fld.conditional_field] in fld.conditional_values:
            LOGGER.debug(f"conditional satisfied: {fld.conditional_field}, " +
                         f"{self.datastruct[fld.conditional_field]}, " +
                         f"{fld.conditional_values}")
            proceed = True
        LOGGER.debug(f"conditional_satisfied return: {proceed}")
        return proceed

    def select(self, fld, override=None):
        '''
        used for preset "select", grabs a random value from the choices option.

        Future: some fields may need the ability to select multiple values.

        :param fld: A Field object
        '''
        LOGGER.debug(f"Calling Select on fld: {fld}")

        if fld.choices:
            LOGGER.debug(f" number of choices: {len(fld.choices)}")
            values = fld.choices.values
            if (override) and override not in values:
                msg = f'A static/override value of {override[fld.field_name]}' + \
                      f'was specified for the field {fld.field_name} ' + \
                      'however that value is not defined in the domain: ' + \
                      f'{values}'
                raise ValueError(msg)
            elif override:
                # if previous conditional didn't raise an error then the override
                # does not violate the domain, therefor use it, otherwise
                # randomly populate
                value = override
            else:
                LOGGER.debug(f" values: {values}")
                value = values[random.randint(0, len(values) - 1)]
        elif (fld.choices_helper) and fld.choices_helper == 'edc_orgs_form':
            # TODO, example of this is the subfield for contacts...
            #   field_name = org
            # add a edc_org used for the package
            value = self.dataset_organization(fld)
        else:
            msg = f'malformed select prefix for the field: {fld}'
            raise ValueError(msg)
        return value

    def title(self, fld, override=None):  # pylint: disable=no-self-use
        '''
        sets the title for the data set, going to hard code this as
        test_data
        '''
        LOGGER.debug(f"{fld.field_name}: {testConfig.TEST_PACKAGE_TITLE}")
        return testConfig.TEST_PACKAGE_TITLE

    def dataset_slug(self, fld, override=None):  # pylint: disable=no-self-use
        '''
        This is currently configured for the name of the dataset to just returning
        the name of the dataset.
        '''
        LOGGER.debug(f"{fld.field_name}: {testConfig.TEST_PACKAGE}")
        return testConfig.TEST_PACKAGE if not override else override

    def dataset_organization(self, fld, override=None):  # pylint: disable=no-self-use
        '''
        :returns: retrieves the name of the organization that is going to be
                 used by the testing and returns it
        '''
        ds_org = testConfig.TEST_ORGANIZATION if override is None else override
        LOGGER.debug(f"{fld.field_name}: {testConfig.TEST_ORGANIZATION}")
        return ds_org

    def string(self, fld, override=None):
        '''
        :return: a random string for the field
        '''
        word = self.rand.getword()
        # does the field name end with _email?
        if override:
            word = override
        elif re.match(r'^\w+_+email$', fld.field_name):
            domain = self.rand.getword()
            email = f'{word}@{domain}.com'
            word = email
        # type
        elif  fld.field_name == 'type':
            word = 'bcdc_dataset'
        LOGGER.debug(f"random word assigned to the field {fld.field_name}: {word}")
        return word

    def tag_string_autocomplete(self, fld, override=None):
        '''
        Not sure if this should be referencing existing tags... for now
        just making it random text.
        '''
        return self.string(fld, override)

    def composite_repeating(self, fld, flds2gen=None, override=None):  # pylint: disable=no-self-use
        '''
        This type of field is a list made up of a bunch of subfields, this
        call will make a couple calls to this Datapopulation class to generate
        new subfields
        '''
        subfields_values = []
        if flds2gen is None:
            flds2gen = random.randint(1, 3)

        if override:
            LOGGER.warning(f'override values supplied for the field: {fld.field_name} ' +
                           'are being ignored.  Currently functionality to set ' +
                           'overrides of subfields does not exist')

        # configured to generate randomly between 1 and 3 subfields the actual
        # range is never actually, just an easy way to create a loop
        for iterval in range(0, flds2gen):  # pylint: unused-variable
            LOGGER.debug(f"flds2gen type: {type(fld.subfields)}")
            population = DataPopulationResource(fld.subfields)
            subfield_data = population.populate_all()
            subfields_values.append(subfield_data)
        LOGGER.debug(f"subfields_values: {subfields_values}")
        subfield_json_str = json.dumps(subfields_values)
        LOGGER.debug(f"subfields_values as stringify json: {subfield_json_str}")
        return subfield_json_str

    def multiple_checkbox(self, fld, override=None):
        '''
        can select multiple values from the choices.
        '''
        return self.select(fld, override)

    def date(self, fld, override=None):  # pylint: disable=unused-argument
        '''
        :return: a random date.  will be some time between now and 10 years
                 ago.
        '''
        date_1 = datetime.datetime.now()
        delta = datetime.timedelta(days=365 * 10)
        date_2 = date_1 - delta
        rand_date = self.random_date(date_2, date_1)
        return override if override else rand_date.strftime('%Y-%m-%d')

    def random_date(self, start, end):
        '''
        Generates a random date string in between the start and end dates
        :param start: the start datetime
        :param end: the end datetime
        :return: a randomly selected date some time in between the two dates.
        '''
        delta = end - start
        LOGGER.debug(f"delta: {delta}")
        int_delta = (delta.days * 24 * 60 * 60) + delta.seconds
        LOGGER.debug(f"int_delta: {int_delta}")
        random_second = random.randrange(int_delta)
        LOGGER.debug(f"random_second: {random_second}")
        return start + datetime.timedelta(seconds=random_second)

    def resource_url_upload(self, fld, override=None):
        '''
        :return: gets a random string and then assembles into a url by appending
            https:// and .com
        '''
        random_string = self.string(fld)
        url = f'https://{random_string}.com' if not override else override
        return url

    def json_object(self, fld, override=None):  # pylint: disable=no-self-use, unused-argument
        '''
        right now returning a static json text
        '''
        dummyjson = '{"schema": { "fields":[ { "mode": "nullable", "name": ' + \
                    '"placeName", "type": "string"  },  { "mode": "nullable' + \
                    '", "name": "kind", "type": "string"  }  ] } }'

        return dummyjson if override is None else override

    def composite(self, fld, override=None):
        '''
        :return: right now just treating the same as composite_repeating but specify
        to only return a single subfield
        '''
        return self.composite_repeating(fld, flds2gen=1, override=override)

    def autocomplete(self, fld, override=None):
        '''
        example of what an autocomplete json snippet looks like:
        {
          "field_name": "iso_topic_string",
          "label": "ISO Topic Category",
          "preset": "autocomplete",
          "conditional_field": "bcdc_type",
          "conditional_values": ["geographic"],
          "validators": "conditional_required scheming_multiple_choice",
          "choices": [
            {
              "value": "farming",
              "label": "Farming"
            },
            {
              "value": "biota",
              "label": "Biota"
            },
            {
              "value": "boundaries",
              "label": "Boundaries"
              ...

        Interpretation: if bcdc_type = 'geographic' then fill out this field otherwise
                        don't populate_random.  verfied with John.

        Method will see if bcdc_type has already been populated, if it has then it
        will process otherwise the processing of this record gets deferred, and is
        tried again after all other records have been processed.

        :return:
        '''
        return self.select(fld, override)

    def populate_overrides(self, overrides):
        '''
        When override / hard coded values are provided they get added to the
        output dataset / resource using this method.  This method will
         - verify that the overrides do not violate any of the data rules
         - populate the overrides first, to address any possible conditional
           validators that are dependent on these hard coded values.
        '''

        def undefined_prefix(fld):
            msg = f'prefix is set to: {fld.preset}.  There is no code to ' + \
                  'to accomodate this type.  Need to add a method definition for ' + \
                  'that type to this class'
            raise UndefinedPreset(msg)

        for override_field_name in overrides:
            # validate field def
            LOGGER.debug(f"override_field_name: {override_field_name}")
            override_fld_def = self.fields.get_field(override_field_name)
            LOGGER.debug(f"override_fld_def: {override_fld_def}")
            if override_fld_def is None:
                fld_names = self.fields.get_field_names()
                msg = f'an override property: "{override_field_name}" was specified ' + \
                      'that does not exist in the schema definitions.  Possible ' + \
                      f'values include: {fld_names}'
                raise ValueError(msg)

            # now test that the value complies with the field def.
            LOGGER.debug(f"preset for override field {override_field_name} " +
                         f"is {override_fld_def.preset}")
            func = getattr(self, override_fld_def.preset, undefined_prefix)
            field_value = func(override_fld_def, override=overrides[override_field_name])
            LOGGER.debug(f'override value: {overrides[override_field_name]}, {field_value}')
            if field_value != overrides[override_field_name]:
                msg = f'override value was not correctly set for {override_field_name} '
                msg = msg + f' and the corresponding value: {overrides[override_field_name]}'
                LOGGER.error(msg)
                raise ValueError(msg)
            self.datastruct[override_fld_def.field_name] = field_value

    def populate_all(self, overrides=None):
        '''
        iterates over the fields object populating it with data.

        data population will be different for different preset types.

        Current list of presets:
            * autocomplete - treated same as choice
            * composite    - has subfields, a list of dicts
            * composite_repeating - has subfields, the schema repeats, for the
                                    list
            * dataset_organization - the reference to an existing org, for now
                                    hard coding test-organization
            * dataset_slug - used for name of the package
            * date - looks like YY-MM-DD 2019-06-13
            * json_object - not required, skipping for now
            * resource_url_upload - not required, not sure skip
            * select -  another choice... select one.
            * tag_string_autocomplete - might come from tags?
            * title - just text

        This method will populate_random all the fields defined in the schema, except
        types noted above
        '''
        # datastruct = {}
        # set the scope for this variable for this method
        fld = None
        if overrides is None:
            overrides = {}

        # reset the dataset for a new dataset in case was previously
        # populated.
        self.datastruct = {}
        self.deferred = []

        # populate the overrides first, that way if there are dependencies on the
        # overrides they will be processed second.
        LOGGER.debug(f"overrides are set to: {overrides}")
        self.populate_overrides(overrides)

        def undefined_prefix(fld):
            msg = f'prefix is set to: {fld.preset}.  There is no code to ' + \
                  'to accomodate this type.  Need to add a method definition for ' + \
                  'that type to this class'
            raise UndefinedPreset(msg)

        LOGGER.debug('populate_random all called')
        for fld in self.fields:
            # population is based on the preset, if no preset available then
            # assume str
            LOGGER.debug(f'Fld is: {fld}')
            LOGGER.debug(f'Fld preset is: {fld.preset}')

            # has field already been populated?
            if fld.field_name in self.datastruct:
                LOGGER.info("field: {fld.field_name} populated by override")
            # Does the field have a conditional property AND the conditional resolves as False...
            elif self.has_conditional(fld) and not self.conditional_satisfied(fld):
                # condition is not met, add to deferred
                LOGGER.debug(f"adding to the deferred list: {fld}")
                self.deferred.append(fld)
            else:
                field_override = overrides[fld.field_name] if fld.field_name in overrides else None

                if fld.preset is None:
                    field_value = self.string(fld, override=field_override)
                else:
                    # the name of the method to call is contained in the property: preset,
                    # turning the value of preset into a method call
                    func = getattr(self, fld.preset, undefined_prefix)
                    field_value = func(fld, override=field_override)
                self.datastruct[fld.field_name] = field_value
        if self.deferred:
            self.process_deferred_fields(overrides=overrides)
            # shortcut for now... should really continuously iterate through deferred
            # fields until they are all removed
        return self.datastruct

    def process_deferred_fields(self, overrides=None):
        '''
        Some fields like
        '''
        LOGGER.debug(f"deferred: {len(self.deferred)} {self.deferred}")
        if self.deferred:
            to_remove = []
            for defer_cnt in range(0, len(self.deferred)):
                defer_fld = self.deferred[defer_cnt]
                LOGGER.debug(f"processing deferred field: {defer_fld}")
                # conditional exists, therefore can retrieve its value
                if self.conditional_field_exists(defer_fld):
                    LOGGER.debug(f"conditional field exists: {defer_fld}")
                    # is the condition satisfied
                    if self.conditional_satisfied(defer_fld):
                        field_override = overrides[defer_fld.field_name] if \
                            defer_fld.field_name in overrides else None
                        if defer_fld.preset is None:
                            field_value = self.string(defer_fld,
                                                      override=field_override)
                        else:
                            # the name of the method to call is contained in the
                            # property: preset, turning the value of preset into
                            # a method call
                            func = getattr(self, defer_fld.preset)
                            field_value = func(defer_fld, override=field_override)
                        self.datastruct[defer_fld.field_name] = field_value
                        to_remove.append(defer_fld)
                    else:
                        # conditional field exists, but the condition is not
                        # satisfied, don't populate_random, and remove from deferred list
                        if defer_fld.field_name in self.datastruct:
                            del self.datastruct[defer_fld.field_name]
                        to_remove.append(defer_fld)
            # after iteration remove the deferred fields, then try again
            for remove_rec in to_remove:
                self.deferred.remove(remove_rec)
            self.process_deferred_fields()


class RandomWords():
    '''
    The module I was using for random words makes a network call, which is slow
    so wrapping it with this module so that it makes one network call and caches
    100 random words, once the 100 are used up it will grab another 100, hopeuflly
    speeding things up.
    '''

    def __init__(self, cache_size=500):
        self.cache_size = cache_size
        # self.words = []

    def getword(self):
        '''
        :return: a random word that can be used for various fields.
        '''
        global WORDS  # pylint: disable=global-statement
        if not WORDS:
            LOGGER.debug("getting words from network")
            self.get_words_from_network()

        # loop and check words for non alphanumerics
        while True:
            # get word from list and remove via pop
            word = WORDS.pop()
            # check if string contains non alphanumeric characters
            if word.isalnum():
                LOGGER.debug('word is alphanumeric: ' + word)
                break
            else:
                LOGGER.debug('word is NOT alphanumeric: ' + word)

        LOGGER.debug('getword will return: ' + word)
        return word

    def get_words_from_network(self):
        '''
        Goes to the network, and reloads the cache with the number of words
        defined in the cache_size.
        '''
        LOGGER.info(f"getting another {self.cache_size} random words" +  # pylint: disable=logging-not-lazy
                    "from generator..")
        # can't think of a way around this, want to make the WORDS available to
        # all instances of the module
        global WORDS  # pylint: disable=global-statement
        WORDS = randomwordgenerator.randomwordgenerator.generate_random_words(
            self.cache_size)
        LOGGER.debug(f"WORDS: {WORDS}")


class UndefinedPreset(AttributeError):
    '''
    Error for when a preset is encountered that has not been coded for.
    '''

    def __init__(self, message, errors=None):  # pylint: disable=unused-argument

        # Call the base class constructor with the parameters it needs
        super().__init__(message)
