# pylint: disable=no-init
"""Реализации справочников и моделей контингента."""

from collections import (
    namedtuple,
)

from m3.db import (
    BaseEnumerate,
)

from educommon.contingent.base import (
    BaseCatalogVirtualModel,
    BaseEnumerateProductSpecific,
    BaseModelView,
    load_values,
)


class Okfs(BaseEnumerate):
    """Справочник ОКФС."""

    values = {
        11: 'Государственная собственность',
        49: 'Иная смешанная российская собственность',
        20: 'Иностранная собственность',
        14: 'Муниципальная собственность',
        10: 'Российская собственность',
        27: 'Смешанная иностранная собственность',
        17: 'Смешанная российская собственность',
        40: 'Смешанная российская собственность с долей государственной собственности',
        42: 'Смешанная российская собственность с долей собственности субъектов Российской Федерации',
        41: 'Смешанная российская собственность с долей федеральной собственности',
        43: 'Смешанная российская собственность с долями федеральной '
        'собственности и собственности субъектов Российской Федерации',
        50: 'Собственность благотворительных организаций',
        61: 'Собственность государственных корпораций',
        22: 'Собственность иностранных государств',
        24: 'Собственность иностранных граждан и лиц без гражданства',
        23: 'Собственность иностранных юридических лиц',
        21: 'Собственность международных организаций',
        15: 'Собственность общественных и религиозных организаций (объединений)',
        53: 'Собственность общественных объединений',
        51: 'Собственность политических общественных объединений',
        19: 'Собственность потребительской кооперации',
        52: 'Собственность профессиональных союзов',
        54: 'Собственность религиозных объединений',
        18: 'Собственность российских граждан, постоянно проживающих за границей',
        13: 'Собственность субъектов Российской Федерации',
        33: 'Совместная муниципальная и иностранная собственность',
        30: 'Совместная российская и иностранная собственность',
        35: 'Совместная собственность общественных и религиозных '
        'организаций (объединений) и  иностранная собственность',
        32: 'Совместная собственность субъектов Российской Федерации и иностранная собственность',
        31: 'Совместная федеральная и иностранная собственность',
        34: 'Совместная частная и иностранная собственность',
        12: 'Федеральная собственность',
        16: 'Частная собственность',
    }


class Citizenship(BaseEnumerate):
    """Гражданство."""

    RF_CITIZEN = 1
    DOUBLE_CITIZEN = 2
    EXTERNAL_CITIZEN = 3
    WITHOUT_CITIZEN = 4

    values = {
        RF_CITIZEN: 'Гражданин Российской Федерации',
        DOUBLE_CITIZEN: 'Гражданин Российской Федерации и иностранного государства (двойное гражданство)',
        EXTERNAL_CITIZEN: 'Иностранный гражданин',
        WITHOUT_CITIZEN: 'Лицо без гражданства',
    }

    # соотнесение значений из КО с тем, как представлено гражданство в РИС
    # основной справочник, в `values`, не соответствует тому, что
    #   указано в confluence по этому полю:
    # https://conf.bars.group/pages/viewpage.action?pageId=15292952 п. 2.1.6
    values_received_from_contingent = {0: RF_CITIZEN, 1: EXTERNAL_CITIZEN, 2: WITHOUT_CITIZEN, 3: DOUBLE_CITIZEN}


class PhysicalCultureGroups(BaseEnumerate):
    """Физкультурные группы."""

    MAIN = 21
    PREPARATORY = 22
    SPECIAL_A = 23
    SPECIAL_B = 24

    values = {MAIN: 'Основная', PREPARATORY: 'Подготовительная', SPECIAL_A: 'Специальная А', SPECIAL_B: 'Специальная Б'}


class HealthGroups(BaseEnumerate):
    """Справочник "Группы здоровья"."""

    HEALTHY = 11
    SLIGHT_DEVIATIONS = 12
    CHRONIC_DISEASES_AND_WELLNESS = 13
    CHRONIC_DISEASES_AND_POOR_HEALTH = 14
    CHRONIC_DISEASE_AND_OBSERVED = 15

    FIRST_GROUP_18_YEARS_OLD = 16
    SECOND_GROUP_18_YEARS_OLD = 17
    THIRD_GROUP_18_YEARS_OLD = 18

    BEFORE_18_YEAR_CODES = [
        HEALTHY,
        SLIGHT_DEVIATIONS,
        CHRONIC_DISEASES_AND_WELLNESS,
        CHRONIC_DISEASES_AND_POOR_HEALTH,
        CHRONIC_DISEASE_AND_OBSERVED,
    ]

    AFTER_18_YEAR_CODES = [FIRST_GROUP_18_YEARS_OLD, SECOND_GROUP_18_YEARS_OLD, THIRD_GROUP_18_YEARS_OLD]

    values = {
        HEALTHY: 'Группа 1 - здоровые',
        SLIGHT_DEVIATIONS: 'Группа 2 - с незначительными отклонениями',
        CHRONIC_DISEASES_AND_WELLNESS: (
            'Группа 3 - с хроническими заболеваниями и хорошим самочувствием, '
            'либо с временными отклонениями в состоянии здоровья'
        ),
        CHRONIC_DISEASES_AND_POOR_HEALTH: 'Группа 4 - с хроническими заболеваниями и плохим самочувствием',
        CHRONIC_DISEASE_AND_OBSERVED: 'Группа 5 - с хроническими заболеваниями и наблюдаются в специальных лечебницах',
        FIRST_GROUP_18_YEARS_OLD: 'Группа 1 (18 лет и старше)',
        SECOND_GROUP_18_YEARS_OLD: 'Группа 2 (18 лет и старше)',
        THIRD_GROUP_18_YEARS_OLD: 'Группа 3 (18 лет и старше)',
    }


class DisabilityGroups(BaseEnumerate):
    """Группы инвалидности."""

    FIRST_GROUP = 11
    SECOND_GROUP = 12
    THIRD_GROUP = 13
    DISABLED_CHILD = 14

    values = {
        FIRST_GROUP: 'Первая группа',
        SECOND_GROUP: 'Вторая группа',
        THIRD_GROUP: 'Третья группа',
        DISABLED_CHILD: 'Ребенок-инвалид (для лиц до 18 лет)',
    }


class CertainCategoriesOfDisability(BaseEnumerate):
    """Отдельные категории инвалидности."""

    DISABLED_SINCE_CHILDHOOD = 21
    ARMY_INVALID = 22

    values = {
        DISABLED_SINCE_CHILDHOOD: 'Инвалид с детства',
        ARMY_INVALID: 'Инвалид вследствие военной травмы или заболевания, '
        'полученного в период прохождения военной службы',
    }


class OrganizationKindByFounder(BaseEnumerate):
    """Вид организации по учредителю."""

    RF = 1
    RF_SUBJECT = 2
    MO = 3
    RF_COMMERCIAL_ORGANIZATION = 4
    EXTERNAL_COMMERCIAL_ORGANIZATION = 5
    RF_NON_COMMERCIAL_ORGANIZATION = 6
    EXTERNAL_NON_COMMERCIAL_ORGANIZATION = 7
    RF_RELIGION_ORGANIZATION = 8
    EXTERNAL_RELIGION_ORGANIZATION = 9
    RF_CITIZENSHIP = 10
    EXTERNAL_CITIZENSHIP = 11

    values = {
        RF: 'Российская Федерация',
        RF_SUBJECT: 'Субъект Российской Федерации',
        MO: 'Муниципальное образование',
        RF_COMMERCIAL_ORGANIZATION: 'Российские коммерческие организации',
        EXTERNAL_COMMERCIAL_ORGANIZATION: 'Иностранные коммерческие организации',
        RF_NON_COMMERCIAL_ORGANIZATION: 'Российские некоммерческие организации',
        EXTERNAL_NON_COMMERCIAL_ORGANIZATION: 'Иностранные некоммерческие организации',
        RF_RELIGION_ORGANIZATION: 'Российские религиозные организации',
        EXTERNAL_RELIGION_ORGANIZATION: 'Иностранные религиозные организации',
        RF_CITIZENSHIP: 'Граждане Российской Федерации',
        EXTERNAL_CITIZENSHIP: 'Иностранные граждане',
    }


class OrganizationStatus(BaseEnumerate):
    """Статусы организаций."""

    FUNCTIONS = 1
    CAPITAL_REPAIRS = 2
    RECONSTRUCTION = 3
    ACTIVITY_STOPPED = 4
    CONTINGENT_MISSING = 5
    PENDING_THE_OPENING = 6
    LIQUIDATED = 7
    CLOSED = 8
    JOINED_OTHER_ORGANIZATIONS = 9

    values = {
        FUNCTIONS: 'Функционирует',
        CAPITAL_REPAIRS: 'Капитальный ремонт',
        RECONSTRUCTION: 'Реконструкция',
        ACTIVITY_STOPPED: 'Деятельность приостановлена',
        CONTINGENT_MISSING: 'Контингент отсутствует',
        PENDING_THE_OPENING: 'Ожидает открытия',
        LIQUIDATED: 'Ликвидирована',
        CLOSED: 'Закрыта',
        JOINED_OTHER_ORGANIZATIONS: 'Присоединена к другой организации',
    }


class EduProgramKind(BaseEnumerateProductSpecific):
    """
    Справочник образовательных программ.

    Содержит все виды обр.программ по УФТТ
    """

    JUNIOR_COMMON = 12
    DEFAULT_COMMON = 13
    MIDDLE_COMMON = 14

    values = {
        1: 'Основная общеобразовательная программа',
        11: 'Образовательная программа дошкольного образования',
        JUNIOR_COMMON: 'Образовательная программа начального общего образования',
        DEFAULT_COMMON: 'Образовательная программа основного общего образования',
        MIDDLE_COMMON: 'Образовательная программа среднего общего образования',
        15: 'Адаптированная основная общеобразовательная программа',
        2: 'Основная профессиональная образовательная программа',
        21: 'Образовательная программа среднего профессионального образования',
        211: 'Программа подготовки квалифицированных рабочих, служащих',
        212: 'Программа подготовки специалистов среднего звена',
        23: 'Образовательная программа, адаптированная для обучения лиц с ограниченными возможностями здоровья',
        3: 'Основная программа профессионального обучения',
        31: 'Программа профессиональной подготовки по профессиям рабочих, должностям служащих',
        32: 'Программа переподготовки рабочих, служащих,',
        33: 'Программа повышения квалификации рабочих, служащих.',
        4: 'Дополнительная образовательная программа',
        41: 'Дополнительная общеобразовательная программа',
        411: 'Дополнительная общеразвивающая программа',
        412: 'Дополнительная предпрофессиональная программа',
        42: 'Дополнительная профессиональная программа',
        421: 'Программа повышения квалификации',
        422: 'Программа профессиональной переподготовки',
        5: 'Адаптированная образовательная программа',
        51: 'Адаптированная основная общеобразовательная программа',
        52: 'Образовательная программа, адаптированная для обучения лиц с ограниченными возможностями здоровья',
    }

    # коды, соответствующие видам обр.программ в Электронной Школе
    WEBEDU_CODES = (1, 11, 12, 13, 14, 15, 211, 212, 4, 41, 411, 412)

    # коды, соответствующие видам обр.программ в Электронном Детском Саду
    KINDER_CODES = ()

    # коды, соответствующие видам обр.программ в Электронном Колледже
    SSUZ_CODES = (211, 212)

    # коды, соответствующие видам обр.программ в Электронном Доп.образовании
    EXTEDU_CODES = ()


class EduDocumentOnEducationTypes(BaseEnumerate):
    """Справочник 'Типы документов об образовании'."""

    values = {
        11: 'Аттестат о среднем общем образовании',
        12: 'Аттестат об основном общем образовании',
        4: 'Справка об обучении',
        41: 'Свидетельство об обучении для лиц с ограниченными возможностями '
        'здоровья (с различными формами умственной отсталости)',
        6: 'Другой документ об образовании',
        7: 'Другой документ об обучении',
    }


class DifficultSituations(BaseEnumerateProductSpecific):
    """Трудные жизненные ситуации."""

    DISABILITY = 2
    LIMITED_ABILITIES = 3
    DEAF = 301
    BLINDLY = 302
    MUTE = 303
    IMMOBILIZE = 304
    PARALYSIS = 305
    INTELLIGENT_VIOLATION = 306
    AUTISTIC = 307
    IMPAIRED_MENTAL = 308
    MENTAL_ILLNESS = 309
    NERVOUS_SYSTEM_DISEASE = 310
    COMPLEX_DEFECTS = 311

    values = {
        101: 'Дети, оставшиеся без попечения родителей по причине смерти родителей',
        102: 'Дети, оставшиеся без попечения родителей по причине лишения родителей родительских прав',
        103: 'Дети, оставшиеся без попечения родителей по причине ограничения родителей в родительских правах',
        104: 'Дети, оставшиеся без попечения родителей по причине признания родителей недееспособными',
        105: 'Дети, оставшиеся без попечения родителей по причине болезни родителей',
        106: 'Дети, оставшиеся без попечения родителей по причине длительного отсутствия родителей',
        DISABILITY: 'Дети-инвалиды',
        LIMITED_ABILITIES: 'Дети с ограниченными возможностями здоровья',
        DEAF: 'Дети с нарушениями слуха',
        BLINDLY: 'Дети с нарушениями зрения',
        MUTE: 'Дети с тяжелыми нарушениями речи',
        IMMOBILIZE: 'Дети с нарушением опорно-двигательного аппарата',
        PARALYSIS: 'Дети с детским церебральным параличом',
        INTELLIGENT_VIOLATION: 'Дети с интеллектуальными нарушениями',
        AUTISTIC: 'Дети с расстройством аутистического спектра',
        IMPAIRED_MENTAL: 'Дети с задержкой психического развития',
        MENTAL_ILLNESS: 'Дети с психическими заболеваниями',
        NERVOUS_SYSTEM_DISEASE: 'Дети с заболеваниями нервной системы',
        COMPLEX_DEFECTS: 'Дети со сложной структурой дефекта',
        4: 'Дети - жертвы вооруженных и межнациональных конфликтов, '
        'экологических и техногенных катастроф, стихийных бедствий',
        5: 'Дети - жертвы экологических и техногенных катастроф, стихийных бедствий',
        6: 'Дети из семей беженцев и вынужденных переселенцев',
        7: 'Дети, оказавшиеся в экстремальных условиях',
        8: 'Дети - жертвы насилия',
        9: 'Дети, отбывающие наказание в виде лишения свободы в воспитательных колониях',
        10: 'Дети, с девиантным (общественно опасным) поведением',
        11: 'Дети, проживающие в малоимущих семьях',
        12: 'Дети с отклонениями в поведении',
        13: 'Дети, жизнедеятельность которых объективно нарушена в результате сложившихся обстоятельств',
    }
    # коды, соответствующие типам законного представителя в
    # Электронном Детском Саду
    KINDER_CODES = tuple(code for code in values if code != 9)

    # Коды тяжелых жизненных ситуаций, которые требуют обучение с ОВЗ.
    NEED_OVZ_LEARNING = (
        DEAF,
        BLINDLY,
        MUTE,
        IMMOBILIZE,
        PARALYSIS,
        INTELLIGENT_VIOLATION,
        AUTISTIC,
        IMPAIRED_MENTAL,
        MENTAL_ILLNESS,
        NERVOUS_SYSTEM_DISEASE,
        COMPLEX_DEFECTS,
    )


class LegalDelegateTypes(BaseEnumerateProductSpecific):
    """Тип законного представителя."""

    PARENT = 1
    CURATOR = 2
    TRUSTEE = 3
    AGENCY_OF_TRUSTEE = 4
    ADOPTIVE_PARENT = 5
    EDUCATION_AGENCY_LEADER = 6
    LEGAL_REPRESENTATIVE = 7

    values = {
        PARENT: 'Родитель',
        CURATOR: 'Опекун',
        TRUSTEE: 'Попечитель',
        AGENCY_OF_TRUSTEE: 'Орган опеки и попечительства',
        ADOPTIVE_PARENT: 'Приемный родитель',
        EDUCATION_AGENCY_LEADER: (
            'Руководитель воспитательного, лечебного и иного учреждения, '
            'в котором ребенок находится на полном государственном обеспечении'
        ),
        LEGAL_REPRESENTATIVE: 'Законный представитель',
    }

    # коды, соответствующие типам законного представителя в Электронной Школе
    WEBEDU_CODES = ()

    # коды, соответствующие типам законного представителя в
    # Электронном Детском Саду
    KINDER_CODES = (2, 3, 4, 5, 6)

    # коды, соответствующие типам законного представителя в
    # Электронном Колледже
    SSUZ_CODES = (1, 2, 3, 4, 5, 6)

    # коды, соответствующие типам законного представителя в
    # Электронном Доп.образовании
    EXTEDU_CODES = ()


class OkoguVirtualModel(BaseCatalogVirtualModel):
    """Виртуальная модель справочника ОКОГУ."""

    id_field = 'code'

    data = load_values('okogu.json')

    def __str__(self):
        """Строковое представление записи в виртуальной модели."""
        return self.full_name

    class Meta:
        verbose_name = 'ОКОГУ'


class Okogu(BaseModelView):
    """
    Справочник ОКОГУ.

    ОКОГУ - Общероссийский классификатор органов государственной власти и
    управления.
    """

    model = OkoguVirtualModel

    value_field = 'code'
    display_field = 'full_name'

    KINDER_CODES = (
        '2300223',
        '3300000',
        '3300100',
        '3300200',
        '3300300',
        '3300400',
        '3300500',
        '3500000',
        '3500100',
        '3500200',
        '3500300',
        '3500400',
        '3500500',
        '4210007',
        '4210008',
        '4210009',
        '4210014',
        '4210015',
    )


class Okopf(BaseEnumerateProductSpecific):
    """Справочник ОКОПФ."""

    values = {
        '1 00 00': 'ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ЮРИДИЧЕСКИХ ЛИЦ, '
        'ЯВЛЯЮЩИХСЯ КОММЕРЧЕСКИМИ КОРПОРАТИВНЫМИ ОРГАНИЗАЦИЯМИ',
        '1 10 00': 'Хозяйственные товарищества',
        '1 10 51': 'Полные товарищества',
        '1 10 64': 'Товарищества на вере (коммандитные товарищества)',
        '1 20 00': 'Хозяйственные общества',
        '1 22 00': 'Акционерные общества',
        '1 22 47': 'Публичные акционерные общества',
        '1 22 67': 'Непубличные акционерные общества',
        '1 23 00': 'Общества с ограниченной ответственностью',
        '1 30 00': 'Хозяйственные партнерства',
        '1 40 00': 'Производственные кооперативы (артели)',
        '1 41 00': 'Сельскохозяйственные производственные кооперативы',
        '1 41 53': 'Сельскохозяйственные артели (колхозы)',
        '1 41 54': 'Рыболовецкие артели (колхозы)',
        '1 41 55': 'Кооперативные хозяйства (коопхозы)',
        '1 42 00': 'Производственные кооперативы (кроме сельскохозяйственных производственных кооперативов)',
        '1 53 00': 'Крестьянские (фермерские) хозяйства',
        '1 90 00': 'Прочие юридические лица, являющиеся коммерческими организациями',
        '2 00 00': 'ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ЮРИДИЧЕСКИХ ЛИЦ, '
        'ЯВЛЯЮЩИХСЯ НЕКОММЕРЧЕСКИМИ КОРПОРАТИВНЫМИ ОРГАНИЗАЦИЯМИ',
        '2 01 00': 'Потребительские кооперативы',
        '2 01 01': 'Гаражные и гаражно-строительные кооперативы',
        '2 01 02': 'Жилищные или жилищно-строительные кооперативы',
        '2 01 03': 'Жилищные накопительные кооперативы',
        '2 01 04': 'Кредитные потребительские кооперативы',
        '2 01 05': 'Кредитные потребительские кооперативы граждан',
        '2 01 06': 'Кредитные кооперативы второго уровня',
        '2 01 07': 'Потребительские общества',
        '2 01 08': 'Общества взаимного страхования',
        '2 01 09': 'Сельскохозяйственные потребительские перерабатывающие кооперативы',
        '2 01 10': 'Сельскохозяйственные потребительские сбытовые (торговые) кооперативы',
        '2 01 11': 'Сельскохозяйственные потребительские обслуживающие кооперативы',
        '2 01 12': 'Сельскохозяйственные потребительские снабженческие кооперативы',
        '2 01 13': 'Сельскохозяйственные потребительские садоводческие кооперативы',
        '2 01 14': 'Сельскохозяйственные потребительские огороднические кооперативы',
        '2 01 15': 'Сельскохозяйственные потребительские животноводческие кооперативы',
        '2 01 20': 'Садоводческие, огороднические или дачные потребительские кооперативы',
        '2 01 21': 'Фонды проката',
        '2 02 00': 'Общественные организации',
        '2 02 01': 'Политические партии',
        '2 02 02': 'Профсоюзные организации',
        '2 02 10': 'Общественные движения',
        '2 02 11': 'Органы общественной самодеятельности',
        '2 02 17': 'Территориальные общественные самоуправления',
        '2 06 00': 'Ассоциации (союзы)',
        '2 06 01': 'Ассоциации (союзы) экономического взаимодействия субъектов Российской Федерации',
        '2 06 03': 'Советы муниципальных образований субъектов Российской Федерации',
        '2 06 04': 'Союзы (ассоциации) кредитных кооперативов',
        '2 06 05': 'Союзы (ассоциации) кооперативов',
        '2 06 06': 'Союзы (ассоциации) общественных объединений',
        '2 06 07': 'Союзы (ассоциации) общин малочисленных народов',
        '2 06 08': 'Союзы потребительских обществ',
        '2 06 09': 'Адвокатские палаты',
        '2 06 10': 'Нотариальные палаты',
        '2 06 11': 'Торгово-промышленные палаты',
        '2 06 12': 'Объединения работодателей',
        '2 06 13': 'Объединения фермерских хозяйств',
        '2 06 14': 'Некоммерческие партнерства',
        '2 06 15': 'Адвокатские бюро',
        '2 06 16': 'Коллегии адвокатов',
        '2 06 17': 'Садоводческие, огороднические или дачные некоммерческие партнерства',
        '2 06 18': 'Ассоциации (союзы) садоводческих, огороднических и дачных некоммерческих объединений',
        '2 06 19': 'Саморегулируемые организации',
        '2 06 20': 'Объединения (ассоциации и союзы) благотворительных организаций',
        '2 07 00': 'Товарищества собственников недвижимости',
        '2 07 01': 'Садоводческие, огороднические или дачные некоммерческие товарищества',
        '2 07 16': 'Товарищества собственников жилья',
        '2 11 00': 'Казачьи общества, внесенные в государственный реестр казачьих обществ в Российской Федерации',
        '2 12 00': 'Общины коренных малочисленных народов Российской Федерации',
        '3 00 00': 'ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ОРГАНИЗАЦИЙ, СОЗДАННЫХ БЕЗ ПРАВ ЮРИДИЧЕСКОГО ЛИЦА',
        '3 00 01': 'Представительства юридических лиц',
        '3 00 02': 'Филиалы юридических лиц',
        '3 00 03': 'Обособленные подразделения юридических лиц',
        '3 00 04': 'Структурные подразделения обособленных подразделений юридических лиц',
        '3 00 05': 'Паевые инвестиционные фонды',
        '3 00 06': 'Простые товарищества',
        '3 00 08': 'Районные суды, городские суды, межрайонные суды (районные суды)',
        '4 00 00': 'ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ '
        'МЕЖДУНАРОДНЫХ ОРГАНИЗАЦИЙ, ОСУЩЕСТВЛЯЮЩИХ ДЕЯТЕЛЬНОСТЬ '
        'НА ТЕРРИТОРИИ РОССИЙСКОЙ ФЕДЕРАЦИИ',
        '4 00 01': 'Межправительственные международные организации',
        '4 00 02': 'Неправительственные международные организации',
        '5 00 00': 'ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ДЛЯ ДЕЯТЕЛЬНОСТИ ГРАЖДАН (ФИЗИЧЕСКИХ ЛИЦ)',
        '5 01 00': 'Организационно-правовые формы для коммерческой деятельности граждан',
        '5 01 01': 'Главы крестьянских (фермерских) хозяйств',
        '5 01 02': 'Индивидуальные предприниматели',
        '5 02 00': 'Организационно-правовые формы для деятельности граждан, не отнесенной к предпринимательству',
        '5 02 01': 'Адвокаты, учредившие адвокатский кабинет',
        '5 02 02': 'Нотариусы, занимающиеся частной практикой',
        '6 00 00': 'ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ЮРИДИЧЕСКИХ ЛИЦ, ЯВЛЯЮЩИХСЯ КОММЕРЧЕСКИМИ УНИТАРНЫМИ ОРГАНИЗАЦИЯМИ',
        '6 50 00': 'Унитарные предприятия',
        '6 51 00': 'Унитарные предприятия, основанные на праве оперативного управления (казенные предприятия)',
        '6 51 41': 'Федеральные казенные предприятия',
        '6 51 42': 'Казенные предприятия субъектов Российской Федерации',
        '6 51 43': 'Муниципальные казенные предприятия',
        '6 52 00': 'Унитарные предприятия, основанные на праве хозяйственного ведения',
        '6 52 41': 'Федеральные государственные унитарные предприятия',
        '6 52 42': 'Государственные унитарные предприятия субъектов Российской Федерации',
        '6 52 43': 'Муниципальные унитарные предприятия',
        '7 00 00': 'ОРГАНИЗАЦИОННО-ПРАВОВЫЕ ФОРМЫ ЮРИДИЧЕСКИХ ЛИЦ, ЯВЛЯЮЩИХСЯ НЕКОММЕРЧЕСКИМИ УНИТАРНЫМИ ОРГАНИЗАЦИЯМИ',
        '7 04 00': 'Фонды',
        '7 04 01': 'Благотворительные фонды',
        '7 04 02': 'Негосударственные пенсионные фонды',
        '7 04 03': 'Общественные фонды',
        '7 04 04': 'Экологические фонды',
        '7 14 00': 'Автономные некоммерческие организации',
        '7 15 00': 'Религиозные организации',
        '7 16 00': 'Публично-правовые компании',
        '7 16 01': 'Государственные корпорации',
        '7 16 02': 'Государственные компании',
        '7 16 10': 'Отделения иностранных некоммерческих неправительственных организаций',
        '7 50 00': 'Учреждения',
        '7 51 00': 'Учреждения, созданные Российской Федерацией',
        '7 51 01': 'Федеральные государственные автономные учреждения',
        '7 51 03': 'Федеральные государственные бюджетные учреждения',
        '7 51 04': 'Федеральные государственные казенные учреждения',
        '7 52 00': 'Учреждения, созданные субъектами Российской Федерации',
        '7 52 01': 'Государственные автономные учреждения субъектов Российской Федерации',
        '7 52 03': 'Государственные бюджетные учреждения субъектов Российской Федерации',
        '7 52 04': 'Государственные казенные учреждения субъектов Российской Федерации',
        '7 53 00': 'Государственные академии наук',
        '7 54 00': 'Учреждения, созданные муниципальными образованиями',
        '7 54 01': 'Муниципальные автономные учреждения',
        '7 54 03': 'Муниципальные бюджетные учреждения',
        '7 54 04': 'Муниципальные казенные учреждения',
        '7 55 00': 'Частные учреждения',
        '7 55 02': 'Благотворительные учреждения',
        '7 55 05': 'Общественные учреждения',
    }
    # коды, соответствующие типам законного представителя в
    # Электронном Детском Саду
    KINDER_CODES = (
        '3 00 02',
        '5 01 02',
        '7 50 00',
        '7 51 00',
        '7 51 01',
        '7 51 03',
        '7 51 04',
        '7 52 00',
        '7 52 01',
        '7 52 03',
        '7 52 04',
        '7 54 00',
        '7 54 01',
        '7 54 03',
        '7 54 04',
        '7 55 00',
        '7 55 05',
    )


class DocumentConfirmingTypes(BaseEnumerate):
    """Справочник "Тип документа, подтверждающего права" """

    ADOPTION = 401
    BIRTH_CERT = 402
    TRUSTEE = 403
    OTHER_DOC = 499

    values = {
        ADOPTION: 'Решение суда об установлении усыновления',
        BIRTH_CERT: 'Свидетельство о рождении',
        TRUSTEE: 'Назначение органом опеки опекуна',
        OTHER_DOC: 'Другой документ, подтверждающий представительство',
    }


class IdentityDocumentsTypes(BaseEnumerate):
    """Справочник "Типы документов удостоверяющих личность".

    Ключи, это id из таблицы certificate,
    из которой перенесены данные в этот справочник.

    Значения это кортежи в которых, первое значение это значение по ключу,
    а второе (не обязательное), код УФТТ.
    """

    BIRTH_CERT = 1
    PASSPORT = 2
    DOC = 3
    TEMP_CARD_IDENT_RF = 4
    DOM_PERM_RF = 5
    PASSPORT_ALIEN = 6
    FOREIGN_PASSPORT = 7
    ARMY_TICK = 8
    DIP_PASS_RF = 11
    PASSPORT_USSR = 13
    PASSPORT_MINMOR = 15
    PASSPORT_NAUTA = 16
    TEMP_RF = 17
    REFUGEE_RF = 18
    BIRTH_CERT_INTR = 19
    FROM_PRISON = 20
    REFUGEE = 21
    OFFICER = 22
    MILLITARY_RF = 23
    TEMP_MILLITARY = 24
    WITHOUT_CITIZEN_RF = 25
    DOC_CITIZEN_RF = 26
    DOC_REFUGEE_RF = 27
    DOC_TEMP_RF = 28
    DOC_PERM_RF = 29
    CERTIFICATE_PERM_RF = 30
    SERVICE_PASSPORT = 31
    FOREIGN_PASSPORT_USSR = 32
    PERSONAL_ELECTRONIC_CARD = 33
    FOREIGN_STATE_DOCUMENT = 34

    # Перечень ключей словаря values, содержащие
    # типы документов, которые могут не иметь серии
    SERIESLESS_DOCS = (3, 4, 6, 15, 18, 19, 20, 21, 24, 25, 26, 27, 28, 33)

    Value = namedtuple('Value', ['value', 'uftt_code'])

    values = {
        BIRTH_CERT: Value('Свидетельство о рождении', 11),
        PASSPORT: Value('Паспорт гражданина РФ', 12),
        DOC: Value('Другой документ, удостоверяющий личность', 4),
        TEMP_CARD_IDENT_RF: Value('Временное удостоверение личности гражданина РФ', 17),
        PASSPORT_ALIEN: Value('Паспорт иностранного гражданина', 21),
        FOREIGN_PASSPORT: Value('Загранпаспорт гражданина РФ', 13),
        ARMY_TICK: Value('Военный билет', 15),
        DIP_PASS_RF: Value('Дипломатический паспорт гражданина Российской Федерации', None),
        PASSPORT_USSR: Value('Паспорт гражданина СССР', 4),
        PASSPORT_MINMOR: Value('Паспорт Минморфлота', 4),
        PASSPORT_NAUTA: Value('Паспорт моряка', 4),
        TEMP_RF: Value('Разрешение на временное проживание в Российской Федерации', 28),
        REFUGEE_RF: Value(
            'Свидетельство о рассмотрении ходатайства о признании  беженцем на территории Российской Федерации', 29
        ),
        BIRTH_CERT_INTR: Value(
            'Свидетельство о рождении, выданное уполномоченным органом иностранного государства', 31
        ),
        FROM_PRISON: Value('Справка об освобождении из места лишения свободы', 4),
        REFUGEE: Value('Удостоверение личности лица, признанного беженцем', 24),
        OFFICER: Value('Удостоверение личности офицера', 4),
        MILLITARY_RF: Value('Удостоверение личности военнослужащего РФ', 14),
        TEMP_MILLITARY: Value('Временное удостоверение, выданное взамен военного билета', 16),
        WITHOUT_CITIZEN_RF: Value('Удостоверение личности лица без гражданства в РФ', 22),
        DOC_CITIZEN_RF: Value(
            'Удостоверение личности отдельных категорий лиц, находящихся на '
            'территории РФ, подавших заявление о признании гражданами РФ или '
            'о приеме в гражданство РФ',
            23,
        ),
        DOC_REFUGEE_RF: Value(
            'Удостоверение личности лица, ходатайствующего о признании беженцем на территории РФ', 25
        ),
        DOC_TEMP_RF: Value('Удостоверение личности лица, получившего временное убежище на территории РФ', 26),
        DOC_PERM_RF: Value('Вид на жительство в Российской Федерации', 27),
        CERTIFICATE_PERM_RF: Value(
            'Свидетельство о предоставлении временного убежища на территории Российской Федерации', 30
        ),
        SERVICE_PASSPORT: Value('Служебный паспорт', None),
        FOREIGN_PASSPORT_USSR: Value('Загранпаспорт гражданина бывшего СССР', None),
        PERSONAL_ELECTRONIC_CARD: Value('Персональная электронная карта', None),
        FOREIGN_STATE_DOCUMENT: Value(
            'Документ, выданный иностранным государством и признаваемый '
            'в соответствии с международным договором РФ в качестве документа, '
            'удостоверяющего личность лица без гражданства',
            None,
        ),
    }

    @classmethod
    def get_full_items(cls):
        """Возвращает все значения справочника."""
        result = []

        for item in cls.get_items():
            cls.__append_to_result(item, result)

        return result

    @classmethod
    def get_uftt_items(cls):
        """Возвращает значения справочника, у которых есть код УФТТ."""
        result = []

        for item in cls.get_items():
            if item[1].uftt_code is None:
                continue
            cls.__append_to_result(item, result)

        return result

    @classmethod
    def get_items_without_uftt(cls):
        """Возвращает значения справочника, у которых нет кода УФТТ."""
        result = []

        for item in cls.get_items():
            if item[1].uftt_code is not None:
                continue
            cls.__append_to_result(item, result)

        return result

    @classmethod
    def get_value_by_id(cls, key):
        """Возвращает значение по ключу, если ключа нет возвращает None."""
        try:
            return cls.values[key].value
        except KeyError:
            return

    @classmethod
    def __append_to_result(cls, item, result):
        """Добавляет значения к результирующему списку.

        :param item: tuple, например (ключ1, ("значение", УФТТ код))
        :param result: [(key1, value1), (key2, value2), .. (keyN, valueN)]
        :rtype : list
        """
        key = item[0]
        value = item[1].value
        result.append((key, value))


class AdaptationType(BaseEnumerate):
    """Вид адаптированности."""

    HALF_BLIND = 201
    BLIND = 202
    HALF_DEAF = 203
    DEAF = 204
    HALF_DEAF_HARD = 205
    WEAK = 206
    SPEECH_PATHOLOGY = 207
    DELAYED = 208
    RETARDED = 209
    RETARDED_HARD = 210
    OTHER = 211
    AUTISTIC_DISORDER = 212
    HEARING_IMPAIRED = 213
    MENTAL_RETARDATION = 214

    values = {
        HALF_BLIND: 'Для слабовидящих детей',
        BLIND: 'Для слепых детей',
        HALF_DEAF: 'Для слабослышащих',
        DEAF: 'Для глухих детей',
        HALF_DEAF_HARD: 'Для слабослышащих обучающихся, имеющих сложную структуру '
        'дефекта (нарушение слуха и задержка психического развития)',
        WEAK: 'Для детей с нарушениями опорно-двигательного аппарата',
        SPEECH_PATHOLOGY: 'Для детей с тяжёлыми нарушениями речи',
        DELAYED: 'Для детей с задержкой психического развития',
        RETARDED: 'Для обучающихся с умственной отсталостью',
        RETARDED_HARD: 'Для обучающихся с умственной отсталостью, имеющих сложную структуру дефекта',
        OTHER: 'Для обучающихся с иными ограничениями здоровья',
        AUTISTIC_DISORDER: 'Для детей с расстройством аутистического спектра',
        HEARING_IMPAIRED: 'Для слабослышащих и позднооглохших детей',
        MENTAL_RETARDATION: 'Для детей с умственной отсталостью, нарушением интеллекта',
    }

    epgu_codes = {
        DEAF: 1,
        HEARING_IMPAIRED: 2,
        BLIND: 3,
        HALF_BLIND: 4,
        SPEECH_PATHOLOGY: 5,
        WEAK: 6,
        DELAYED: 7,
        AUTISTIC_DISORDER: 8,
        MENTAL_RETARDATION: 9,
    }

    @classmethod
    def get_epgu_code(cls, adaptation_type):
        """Возвращает код ЕПГУ для заданного типа адаптации, если он существует."""
        return cls.epgu_codes.get(adaptation_type)


class TypeTrainingLongTermTreatment(BaseEnumerate):
    """Справочник "Виды обучения при длительном лечении"."""

    IS_HOME_STUDY = 1
    IS_MED_ORG_STUDY = 2
    IS_HEALING_ORG_STUDY = 3

    values = {
        IS_HOME_STUDY: 'Обучение на дому',
        IS_MED_ORG_STUDY: 'Обучение в медицинской организации',
        IS_HEALING_ORG_STUDY: 'Обучение в организации, осуществляющей лечение, оздоровление и (или) отдых',
    }


class EducationOrganization(BaseEnumerateProductSpecific):
    """Образовательная организация."""

    PROFESSIONAL_ORGANIZATION_CODE = 14
    EDUCATION_ORGANIZATION_CODE = 1
    INDIVIDUAL_BUSINESS = 3
    FOR_THE_DEAF = 1301
    FOR_THE_HEARING_IMPAIRED = 1302
    FOR_LATE_DEAFENED = 1303
    FOR_THE_BLIND = 1304
    FOR_THE_VISUALLY_IMPAIRED = 1305
    WITH_SEVERE_SPEECH_DISORDERS = 1306
    WITH_DISORDERS_OF_MS = 1307
    WITH_MENTAL_RETARDATION = 1308
    WITH_AUTISM_DISORDERS = 1310
    WITH_SEVERE_DEFECTS = 1311
    FOR_OTHERS_WITH_DISORDERS = 1312

    values = {
        EDUCATION_ORGANIZATION_CODE: 'Образовательная организация',
        11: 'Дошкольная образовательная организация',
        12: 'Общеобразовательная организация',
        13: 'ОО, с деятельностью по адаптированным программам',
        FOR_THE_DEAF: 'ОО, с деятельностью по адаптированным программам (для глухих)',
        FOR_THE_HEARING_IMPAIRED: 'ОО, с деятельностью по адаптированным программам (для слабослышащих)',
        FOR_LATE_DEAFENED: 'ОО, с деятельностью по адаптированным программам (для позднооглохших)',
        FOR_THE_BLIND: 'ОО, с деятельностью по адаптированным программам (для слепых)',
        FOR_THE_VISUALLY_IMPAIRED: 'ОО, с деятельностью по адаптированным программам (для слабовидящих)',
        WITH_SEVERE_SPEECH_DISORDERS: (
            'ОО, с деятельностью по адаптированным программам (с тяжелыми нарушениями речи)'
        ),
        WITH_DISORDERS_OF_MS: (
            'ОО, с деятельностью по адаптированным программам (с нарушениями опорно-двигательного аппарата)'
        ),
        WITH_MENTAL_RETARDATION: (
            'ОО, с деятельностью по адаптированным программам (с задержкой психического развития)'
        ),
        1309: 'ОО, с деятельностью по адаптированным программам (с умственной отсталостью)',
        WITH_AUTISM_DISORDERS: (
            'ОО, с деятельностью по адаптированным программам (с расстройствами аутистического спектра)'
        ),
        WITH_SEVERE_DEFECTS: 'ОО, с деятельностью по адаптированным программам (со сложными дефектами)',
        FOR_OTHERS_WITH_DISORDERS: 'ОО, с деятельностью по адаптированным программам (для других обучающихся с ОВЗ)',
        PROFESSIONAL_ORGANIZATION_CODE: 'Профессиональная образовательная организация',
        15: 'Образовательная организация высшего образования',
        16: 'Организация дополнительного образования',
        17: 'Организация дополнительного профессионального образования',
        2: 'Организация, осуществляющая обучение',
        21: 'Научная организация',
        22: 'Центр психолого-педагогической, медицинской и социальной помощи',
        23: 'Организация для детей-сирот и детей, оставшихся без попечения родителей',
        24: 'Организация, осуществляющая лечение, оздоровление и (или) отдых',
        25: 'Организация, осуществляющая социальное обслуживание',
        26: 'Дипломатическое представительство РФ',
        27: 'Консульское учреждение РФ',
        28: 'Представительство РФ при международных (межгосударственных, межправительственных) организациях',
        29: 'Иные юридические лица',
        INDIVIDUAL_BUSINESS: 'Индивидуальный предприниматель, осуществляющий образовательную деятельность',
    }
    # коды, соответствующие типам законного представителя в Электронной Школе
    WEBEDU_CODES = (
        1,
        2,
        3,
        12,
        13,
        23,
        24,
        25,
        26,
        27,
        28,
        29,
        1301,
        1302,
        1303,
        1304,
        1305,
        1306,
        1307,
        1308,
        1309,
        1310,
        1311,
        1312,
    )

    SSUZ_CODES = (14, 3)


class OksmVirtialModel(BaseCatalogVirtualModel):
    """Виртуальная модель справочника ОКСМ."""

    rf_code = '643'

    id_field = 'code'
    data = load_values('oksm.json')

    fields_to_serialize = ['id', 'shortname']

    def __str__(self):
        """Строковое представление записи в виртуальной модели."""
        return self.shortname

    class Meta:
        verbose_name = 'ОКСМ'


class Oksm(BaseModelView):
    """
    Справочник ОКСМ.

    ОКСМ - Общероссийский классификатор стран мира.
    """

    model = OksmVirtialModel
    value_field = 'code'
    display_field = 'shortname'


class CrippleGroups(BaseEnumerate):
    """Справочник "Группы инвалидности"."""

    FIRST_GROUP = 11
    SECOND_GROUP = 12
    THIRD_GROUP = 13
    DISABLED_CHILD_GROUP = 14

    values = {
        FIRST_GROUP: '1 группа',
        SECOND_GROUP: '2 группа',
        THIRD_GROUP: '3 группа',
        DISABLED_CHILD_GROUP: 'Ребенок-инвалид',
    }


class CrippleCategories(BaseEnumerate):
    """Справочник "Категории инвалидности"."""

    values = {
        21: 'Инвалид с детства',
        22: 'Инвалид вследствие военной травмы или заболевания, полученного в период прохождения военной службы',
    }


class EducationMethods(BaseEnumerate):
    """Справочник "Формы образования"."""

    FULL_TIME_STUDY = 11
    CORRESPONDENCE_STUDY = 12
    FULL_TIME_AND_CORRESPONDENCE_STUDY = 13
    FAMILY_STUDY = 21
    SELF_EDUCATION_STUDY = 22

    values = {
        FULL_TIME_STUDY: 'Очная',
        CORRESPONDENCE_STUDY: 'Заочная',
        FULL_TIME_AND_CORRESPONDENCE_STUDY: 'Очно-заочная',
        FAMILY_STUDY: 'вне организации (семейное образование)',
        SELF_EDUCATION_STUDY: 'вне организации (самообразование)',
    }


class EducationTypeOVZ(BaseEnumerate):
    """Справочник "Вид обучения для детей с ОВЗ"."""

    COMMON = 1
    SEPARATE_CLASS = 2
    SEPARATE_UNIT = 3

    values = {
        COMMON: 'Совместно с другими обучающимися',
        SEPARATE_CLASS: 'В отдельных классах, группах',
        SEPARATE_UNIT: 'В отдельных организациях, осуществляющих образовательную деятельность',
    }


class RealizationForms(BaseEnumerateProductSpecific):
    """Справочник "Формы реализации образовательной программы"."""

    NETWORK = 1
    ELECTRON = 2
    REMOTE = 3
    HOME = 4
    ELECTRON_ONLY = 5
    REMOTE_ONLY = 6
    FOREIGN_NETWORK = 7

    values = {
        NETWORK: 'Сетевая форма',
        ELECTRON: 'С применением электронного обучения',
        REMOTE: 'С применением дистанционных образовательных технологий',
        HOME: 'Надомная форма получения образования',
        ELECTRON_ONLY: 'С применением исключительно электронного обучения',
        REMOTE_ONLY: 'С применением исключительно дистанционных образовательных технологий',
        FOREIGN_NETWORK: 'Сетевая форма с использованием ресурсов иностранных организаций',
    }

    current_kind = (1, 2, 3, 4)

    # коды, соответствующие видам форм образовательных программ
    # в Электронной Школе
    WEBEDU_CODES = (1, 2, 3, 4, 5, 6)

    # коды, соответствующие видам форм образовательных программ
    # в Электронном Детском Саду
    KINDER_CODES = (1, 2, 3, 4)

    # коды, соответствующие видам форм образовательных программ
    # в Электронном Колледже
    SSUZ_CODES = (1, 2, 3, 4, 5, 6, 7)

    # коды, соответствующие видам форм образовательных программ
    # в Электронном Доп.образовании
    EXTEDU_CODES = (1, 2, 3, 4)


class DismissalReasons(BaseEnumerate):
    """Справочник "Причины увольнения"."""

    TRANSFER_TO_ORG = 1
    TRANSFER_TO_IND = 2
    DISMISS = 3
    PENSION = 4
    REDUCTION = 5
    AGREEMENT_END = 6
    DEATH = 7
    RELOCATION = 8
    HEALTH = 9
    WILL = 10
    ARMY = 11
    EDUCATION = 12

    values = {
        TRANSFER_TO_ORG: 'Перевод в другую образовательную организацию',
        TRANSFER_TO_IND: 'Перевод в другую отрасль',
        DISMISS: 'Уволен по решению суда, за прогул, за нарушение дисциплины',
        PENSION: 'Уход на пенсию',
        REDUCTION: 'Сокращение штатов',
        AGREEMENT_END: 'Прекращение договора',
        DEATH: 'Смерть',
        RELOCATION: 'Смена места жительства',
        HEALTH: 'По состоянию здоровья',
        WILL: 'По собственному желанию',
        ARMY: 'Призыв в армию',
        EDUCATION: 'Дневное обучение',
    }


class IcaoVirtialModel(BaseCatalogVirtualModel):
    """Виртуальная модель справочника ICAO."""

    rf_code = 'RUS'
    rf_shortname = 'РОССИЯ'

    id_field = 'code'
    data = load_values('icao.json')

    fields_to_serialize = ['id', 'shortname']

    def __str__(self):
        """Строковое представление записи в виртуальной модели."""
        return self.shortname

    class Meta:
        verbose_name = 'ICAO'
