# -*- coding: utf-8 -*-
"""MX Google class file."""

import os
import sys


class Google(object):
    """Google class."""

    def __init__(self, update):
        """Initialize a class instance."""
        self.update = update
        self.mx = update.mx
        self.verbose = update.mx.verbose

        # define the name of the google transport file
        # self.transportdest = 'aspmx.l.google.com'
        self.transportdest = 'ASPMX.L.GOOGLE.com'
        self.transportfile = 'transport.google'

        # define the path to the gnarwl puppet file
        self.puppetfile = '%s/%s%s' % (
            self.mx.transports_puppetdir,
            self.transportfile,
            self.mx.extension,
        )

        # get program name
        self.program = os.path.basename(sys.argv[0])

    def generate(self):
        """Return the contents of the puppetfile."""
        output = ''
        # define header
        output = '# This file is autogenerated by the bits-mx %s script.\n' % (
            self.program
        )
        output += '# Do not make local modifications to this file.\n'

        # get autoreplies
        people_transports = self.get_people_transports()
        other_accounts_transports = self.get_other_accounts_transports()

        # users without a forward
        output += '\n# People Gmail transports\n'
        output += '\n'.join(people_transports)

        # users with a forward
        output += '\n\n# Other Accounts Gmail transports\n'
        output += '\n'.join(other_accounts_transports)

        return output

    def get_people_transports(self):
        """Return People transports."""
        transports = []
        for username in sorted(self.update.people):
            # skip terminated people
            if self.update.people[username]['terminated']:
                continue
            # add broadinstitute.org transport
            transports.append('%s@broadinstitute.org\tsmtp:[%s]' % (
                username,
                self.transportdest,
            ))
            # add broadintitute.com transport
            transports.append('%s@broadinstitute.com\tsmtp:[%s]' % (
                username,
                self.transportdest,
            ))
        return transports

    def get_other_accounts_transports(self):
        """Return Other Accounts transports."""
        transports = []
        for email in sorted(self.update.google_other_accounts, key=lambda x: x or ''):
            if not email:
                continue
            # add broadinstitute.org transport
            transports.append('%s\tsmtp:[%s]' % (
                email,
                self.transportdest,
            ))

            # skip non-broadinstitute.org accounts
            if 'broadinstitute.org' not in email:
                continue

            # add broadintitute.com transport
            transports.append('%s\tsmtp:[%s]' % (
                email.replace('broadinstitute.org', 'broadinstitute.com'),
                self.transportdest,
            ))
        return transports
