"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = exports.KeyLength = void 0;
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("cdk-iam-floyd");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
/**
 * @stability stable
 */
var KeyLength;
(function (KeyLength) {
    KeyLength[KeyLength["L2048"] = 2048] = "L2048";
    KeyLength[KeyLength["L4096"] = 4096] = "L4096";
})(KeyLength = exports.KeyLength || (exports.KeyLength = {}));
/**
 * An EC2 Key Pair.
 *
 * @stability stable
 */
class KeyPair extends cdk.Construct {
    /**
     * Defines a new EC2 Key Pair.
     *
     * The private key will be stored in AWS Secrets Manager
     *
     * @stability stable
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager.
         *
         * @stability stable
         */
        this.arn = '';
        /**
         * Name of the Key Pair.
         *
         * @stability stable
         */
        this.name = '';
        /**
         * @stability stable
         */
        this.prefix = '';
        if (props.removePrivateKeyAfterDays &&
            (props.removePrivateKeyAfterDays < 0 ||
                (props.removePrivateKeyAfterDays > 0 &&
                    props.removePrivateKeyAfterDays < 7) ||
                props.removePrivateKeyAfterDays > 30)) {
            scope.node.addError(`Parameter removePrivateKeyAfterDays must be 0 or between 7 and 30. Got ${props.removePrivateKeyAfterDays}`);
        }
        const stack = cdk.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        this.lambda = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag('CreatedBy', ID);
        if (props.tags) {
            for (const [key, value] of Object.entries(props.tags)) {
                this.tags.setTag(key, value);
            }
        }
        const key = new cfn.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(this.lambda),
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KeyLength: props.keyLength || KeyLength.L2048,
                Kms: ((_a = props.kms) === null || _a === void 0 ? void 0 : _a.keyArn) || 'alias/aws/secretsmanager',
                RemovePrivateKeyAfterDays: props.removePrivateKeyAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-private-key/',
                StackName: stack,
                Tags: cdk.Lazy.anyValue({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(this.lambda.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(this.lambda.role);
        }
        this.arn = key.getAttString('PrivateKeyARN');
        this.name = key.getAttString('KeyPairName');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new statement.Ec2()
                    .allow()
                    .toDescribeKeyPairs()
                    .toCreateKeyPair()
                    .toDeleteKeyPair(),
                new statement.Secretsmanager().allow().toListSecrets(),
                new statement.Secretsmanager()
                    .allow()
                    .toCreateSecret()
                    .toTagResource()
                    .ifAwsRequestTag('CreatedBy', ID),
                new statement.Secretsmanager()
                    .allow()
                    .allMatchingActions('/^(Describe|Delete|Put|Update)/')
                    .toGetResourcePolicy()
                    .toRestoreSecret()
                    .toListSecretVersionIds()
                    .toUntagResource()
                    .ifResourceTag('CreatedBy', ID),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager.
     *
     * @stability stable
     */
    grantRead(grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [this.arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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