import sys

from django.core import checks
from django.apps import apps as django_apps
from django.core.exceptions import ObjectDoesNotExist, ImproperlyConfigured
from django.db import models
from django.db.models.deletion import PROTECT, SET_NULL
from django.urls.base import reverse
from django.utils.safestring import mark_safe
from edc_base import get_utcnow
from edc_base.model_managers import HistoricalRecords
from edc_base.model_mixins import BaseUuidModel
from edc_base.sites import CurrentSiteManager as BaseCurrentSiteManager, SiteModelMixin
from edc_constants.constants import NEW
from edc_identifier.model_mixins import NonUniqueSubjectIdentifierFieldMixin

from ..admin_site import edc_action_item_admin
from ..choices import ACTION_STATUS, PRIORITY
from ..identifiers import ActionIdentifier
from ..site_action_items import site_action_items
from .action_type import ActionType


class ActionItemUpdatesRequireFollowup(Exception):
    pass


class SubjectDoesNotExist(Exception):
    pass


class CurrentSiteManager(BaseCurrentSiteManager):

    use_in_migrations = True

    def get_by_natural_key(self, action_identifier):
        return self.get(action_identifier=action_identifier)


class ActionItemManager(models.Manager):

    use_in_migrations = True

    def get_by_natural_key(self, action_identifier):
        return self.get(action_identifier=action_identifier)


class ActionItem(NonUniqueSubjectIdentifierFieldMixin, SiteModelMixin,
                 BaseUuidModel):

    subject_identifier_model = 'edc_registration.registeredsubject'

    action_identifier = models.CharField(
        max_length=25,
        unique=True)

    report_datetime = models.DateTimeField(
        default=get_utcnow)

    action_type = models.ForeignKey(
        ActionType, on_delete=PROTECT,
        related_name='action_type',
        verbose_name='Action')

    reference_model = models.CharField(
        max_length=50,
        null=True)

    linked_to_reference = models.BooleanField(
        default=False,
        editable=False,
        help_text=(
            'True if this action is linked to it\'s reference_model.'
            'Initially False if this action is created before reference_model.'
            'Always True when reference_model creates the action.'
            'Set to True when reference_model is created and "links" to this action.'
            '(Note: reference_model looks for actions where '
            'linked_to_reference is False before attempting to '
            'create a new ActionItem).'))

    related_reference_model = models.CharField(
        max_length=100,
        null=True,
        editable=False)

    related_action_identifier = models.CharField(
        max_length=50,
        null=True,
        blank=True,
        help_text=('May be left blank. e.g. action identifier from '
                   'source model that opened the item.'))

    parent_reference_model = models.CharField(
        max_length=100,
        null=True,
        editable=False)

    parent_action_identifier = models.CharField(
        max_length=50,
        null=True,
        blank=True,
        help_text=('May be left blank. e.g. action identifier from '
                   'reference model that opened the item (parent).'))

    priority = models.CharField(
        max_length=25,
        choices=PRIORITY,
        null=True,
        blank=True,
        help_text='Leave blank to use default for this action type.')

    parent_action_item = models.ForeignKey(
        'self', on_delete=PROTECT,
        null=True,
        blank=True,
        editable=False)

    status = models.CharField(
        max_length=25,
        default=NEW,
        choices=ACTION_STATUS)

    instructions = models.TextField(
        null=True,
        blank=True,
        help_text='populated by action class')

    auto_created = models.BooleanField(
        default=False)

    auto_created_comment = models.CharField(
        max_length=25,
        null=True,
        blank=True)

    emailed = models.BooleanField(default=False)

    emailed_datetime = models.DateTimeField(null=True)

    on_site = CurrentSiteManager()

    objects = ActionItemManager()

    history = HistoricalRecords()

    def __str__(self):
        return (f'{self.action_type.display_name} {self.action_identifier[-9:]} '
                f'({self.get_status_display()})')

    def save(self, *args, **kwargs):
        """See also signals and action_cls.
        """
        if not self.id:
            # a new persisted action item always has
            # a unique action identifier
            self.action_identifier = ActionIdentifier().identifier
            # subject_identifier
            subject_identifier_model_cls = django_apps.get_model(
                self.subject_identifier_model)
            try:
                subject_identifier_model_cls.objects.get(
                    subject_identifier=self.subject_identifier)
            except ObjectDoesNotExist:
                raise SubjectDoesNotExist(
                    f'Invalid subject identifier. Subject does not exist '
                    f'in \'{self.subject_identifier_model}\'. '
                    f'Got \'{self.subject_identifier}\'.')
            self.priority = self.priority or self.action_type.priority
            self.reference_model = self.action_type.reference_model
            self.related_reference_model = self.action_type.related_reference_model
            self.instructions = self.action_type.instructions
        super().save(*args, **kwargs)

    def natural_key(self):
        return (self.action_identifier, )

    @property
    def last_updated(self):
        return None if self.status == NEW else self.modified

    @property
    def user_last_updated(self):
        return None if self.status == NEW else self.user_modified or self.user_created

    @property
    def action_cls(self):
        """Returns the action_cls.
        """
        return site_action_items.get(self.action_type.name)

    @property
    def action(self):
        """Returns the instantiated action_cls.
        """
        return self.action_cls(action_identifier=self.action_identifier)

    @property
    def reference_model_cls(self):
        return django_apps.get_model(self.reference_model)

    @property
    def reference_obj(self):
        return self.reference_model_cls.objects.get(
            action_identifier=self.action_identifier)

    @property
    def parent_reference_model_cls(self):
        return django_apps.get_model(self.parent_reference_model)

    @property
    def parent_reference_obj(self):
        """Returns the parent reference model instance.

        Links this action item to the immediate previous
        action_item in the chain.
        """
        return self.parent_reference_model_cls.objects.get(
            action_identifier=self.parent_action_identifier)

    @property
    def related_reference_obj(self):
        """Returns the related reference model instance
        or raises ObjectDoesNotExist.

        Links this action item to all previous
        action_items in the chain.

        The related reference FK points to the related reference
        object that links this action item to the initial
        reference model instance.
        """
        return django_apps.get_model(self.related_reference_model).objects.get(
            action_identifier=self.related_action_identifier)

    @property
    def related_reference_model_cls(self):
        """Returns the related reference model instance.
        """
        return django_apps.get_model(self.related_reference_model)

    @property
    def identifier(self):
        """Returns a shortened action identifier.
        """
        return self.action_identifier[-9:]

    @property
    def reference(self):
        """Returns a shortened action_identifier which in
        most cases is the reference model's action identifier.
        """
        if self.action_identifier:
            return self.action_identifier[-9:]
        return None

    @property
    def parent_reference(self):
        """Returns a shortened parent_action_identifier of the
        parent model reference which in most cases is the
        parent reference model's action identifier.
        """
        try:
            parent_reference = self.parent_action_item.action_identifier
        except AttributeError:
            parent_reference = None
        if parent_reference:
            return parent_reference[-9:]
        return None

    @property
    def related_reference(self):
        """Returns a shortened related_action_identifier of the
        parent model reference which in most cases is the
        parent reference model's action identifier.
        """
        try:
            related_reference = self._related_action_identifier
        except AttributeError:
            related_reference = None
        if related_reference:
            return related_reference[-9:]
        return None

    @classmethod
    def check(cls, **kwargs):
        errors = super().check(**kwargs)
        if ('test' not in sys.argv
                and 'makemigrations' not in sys.argv
                and 'migrate' not in sys.argv):
            for obj in cls.objects.all():
                if (obj.action_cls.related_reference_fk_attr
                        and not obj.related_action_identifier):
                    errors.append(
                        checks.Error(
                            f'ActionItem.related_action_identifier cannot be '
                            f'None if related_reference_fk_attr is specified. '
                            f'Got ActionItem.action_identifier={obj.action_identifier}. '
                            f'Expected the \'action_identifier\' from an instance of '
                            f'{obj.action_cls.related_reference_model}',
                            hint=(f'update {obj.__class__.__name__}.'
                                  f'related_action_identifier '
                                  f'or delete the {obj.__class__.__name__}.'),
                            obj=obj,
                            id='edc_action_item.E001'))
            for action_cls in site_action_items.registry.values():
                if not action_cls.reference_model:
                    raise ImproperlyConfigured(
                        f'Attribute reference_model cannot be None. See {action_cls}')
                for obj in action_cls.reference_model_cls().objects.all():
                    try:
                        ActionItem.objects.get(
                            action_identifier=obj.action_identifier)
                    except ObjectDoesNotExist:
                        errors.append(
                            checks.Error(
                                f'Model refers to non-existent action item.\n '
                                f'Got {action_cls.reference_model} where '
                                f'action_identifier={obj.action_identifier}.\n',
                                hint=f'Set action_identifier=None and re-save the object.',
                                obj=obj,
                                id='edc_action_item.E002'))
        return errors

    class Meta:
        verbose_name = 'Action Item'
        verbose_name_plural = 'Action Items'
