# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['image_pattern', 'image_pattern.contrib', 'image_pattern.elements']

package_data = \
{'': ['*']}

install_requires = \
['pillow>=7.0,<8.0', 'pydantic>=1.4,<2.0']

setup_kwargs = {
    'name': 'image-pattern',
    'version': '0.0.18',
    'description': 'Package for creating images, such as social media sharing images and avatars, from declarative templates.',
    'long_description': "Image Pattern\n=============\n\n|ImagePattern|\n\n| image\\_pattern - package for creating image templates and\n| generation of images from these templates using changing content.\n\nWhat's all this for?\n--------------------\n\n| It is often necessary to generate the same type of images with changing content.\n| For example, images to be shared to social networks or to create a user avatar with the name\n| (if no picture for the avatar is specified).\n\n| I wanted to have a declarative way to describe image templates, and simple generation of images from them.\n| I hope I made it.\n\nInstallation\n------------\n\n``shell script pip3 install image-pattern``\n\nLet's start\n-----------\n\n| First we need to create the pattern. We'll create an image pattern of the avatar.\n| The image pattern is defined with the\\ ``Pattern`` as follows:\n\n.. code:: python\n\n    from image_pattern import Pattern\n\n\n    class Avatar(Pattern):\n        pass\n\nEach image pattern consists of two components:\n\n-  canvas - the image canvas settings, such as size;\n-  layers - layers that contain image content;\n\n.. code:: python\n\n    from image_pattern import (\n        Pattern,\n        Canvas,\n    )\n\n\n    class Avatar(Pattern):\n        canvas = Canvas(size=(200, 200))\n\nIn this case we created an empty pattern for 200x200 image.\n\nNow, we can generate an image from it with the code:\n\n.. code:: python\n\n    avatar_pattern = Avatar()\n    image = avatar_pattern.render()\n    image.save('avatar.jpg', 'JPEG')\n\nAnd we get the next image:\n\n|Blank image for avatar|\n\n| Now we need to set content.\n| We'll start by adding the first layer:\n\n.. code:: python\n\n    from __future__ import annotations\n    from typing import List\n    from image_pattern import (\n        Pattern,\n        Canvas,\n        Layer,\n        Rectangle,\n        Point,\n    )\n\n\n    class Avatar(Pattern):\n        canvas = Canvas(\n            size=(200, 200),\n        )\n        layers: List[Layer] = [\n            Layer(\n                Rectangle(\n                    size=(100, 200),\n                    point=Point(x=0, y=0),\n                    background_color=(51, 204, 255),\n                ),\n                Rectangle(\n                    size=(100, 200),\n                    point=Point(x=100, y=0),\n                    background_color=(255, 51, 153),\n                ),\n            ),\n        ]\n\n\n    avatar_pattern = Avatar()\n    image = avatar_pattern.render()\n    image.save('avatar.jpg', 'JPEG')\n\nLet's run the script and get the image:\n\n|Avatar with background|\n\nWhat's going on here:\n\n-  We added a list of layers that contains one layer.\n-  This layer contains two rectangles;\n-  The ``size`` property specifies the size of the rectangle in pixels;\n-  The ``point`` property specifies a point on the canvas indicating the\n   upper left corner of the element;\n-  The ``background_color`` property specifies the color of the\n   rectangle in the *RGB* system.\n\nMore information about the elements and their properties can be found in\n*API*.\n\n| After reviewing this example, a reasonable question may arise: Why do we need layers?\n| So that no overlapping of elements occurs within one layer.\n| For example, if we move the right rectangle by 50 pixels to the left inside one layer to make it run over the left rectangle:\n\n.. code:: python\n\n    ...\n            Layer(\n                Rectangle(\n                    size=(100, 200),\n                    point=Point(x=0, y=0),\n                    background_color=(51, 204, 255),\n                ),\n                Rectangle(\n                    size=(100, 200),\n                    point=Point(x=50, y=0),\n                    background_color=(255, 51, 153),\n                ),\n            ),\n    ...\n\nwe'll generate the next image:\n\n|Image with rectangular offset|\n\n| As we can see, the right rectangle is missing from the image.\n| If we increase the height of the image, we can see where it disappeared:\n\n.. code:: python\n\n    ...\n    class Avatar(Pattern):\n        canvas = Canvas(\n            size=(200, 400),\n        )\n    ...\n\n|Image with rectangle shift and height increase|\n\n| As we can see, the right rectangle has shifted to down so as not to intersect with the left rectangle in the same layer.\n| The rule of shifting elements is very simple - elements are always shifted down.\n| This is especially useful for working with texts. When the length of text is unknown.\n\nBut if we return the height of 200 pixels and place the rectangles in\ndifferent layers:\n\n.. code:: python\n\n    ...\n        layers: List[Layer] = [\n            Layer(\n                Rectangle(\n                    size=(100, 200),\n                    point=Point(x=0, y=0),\n                    background_color=(51, 204, 255),\n                ),\n            ),\n            Layer(\n                Rectangle(\n                    size=(100, 200),\n                    point=Point(x=50, y=0),\n                    background_color=(255, 51, 153),\n                ),\n            ),\n        ]\n    ...\n\nthen we can generate the next image:\n\n|Picture with layers overlapping|\n\n| As we can see, the rectangles are superimposed on each other.\n| That's because the layers serve to specifically\n| to put elements on top of each other.\n\n| Okay, now we need to write a text in our image\n| For this we will need a text element and a new layer:\n\n.. code:: python\n\n    ...\n            Layer(\n                Text(\n                    text='Image Pattern',\n                    font=FONT_PATH,\n                    font_color=(255, 255, 255),\n                    font_size=42,\n                    point=Point(x=0, y=0),\n                    margin=Position(\n                        top=20,\n                        left=20,\n                        right=20,\n                    ),\n                )\n            ),\n    ...\n\n| You can read more about the element in the *API* section.\n| By running the script, we get the image:\n\n|Picture with long text|\n\n| Template takes into account the transfer of text by words, if there is not enough space for the text.\n| We can also specify fonts, alignment and indents.\n\nLet's put the first letters of words in the center of the image:\n\n.. code:: python\n\n    ...\n    from image_pattern import (\n        ...\n        HorizontalAlignment,\n        VerticalAlignment,\n    )\n    ...\n            Layer(\n                Text(\n                    text='IP',\n                    font=FONT_PATH,\n                    font_color=(255, 255, 255),\n                    font_size=102,\n                    point=Point(x=100, y=100),\n                    margin=Position(\n                        left=20,\n                        right=20,\n                    ),\n                    horizontal_alignment=HorizontalAlignment.CENTER,\n                    vertical_alignment=VerticalAlignment.CENTER,\n                ),\n            ),\n    ...\n\nWe got the image:\n\n|Avatar with text in the center|\n\n| Hmm... We have the feeling that the text is not in center of the image.\n| However, that's not entirely true. This is because the image template is used to align the width of the entire line.\n| And since the width of the letter ``I`` is less than the width of the letter ``P``, the text looks as if shifted.\n\nTo get rid of this effect, we will try to place each letter in the\ncenter of its rectangle as follows.\n\n.. code:: python\n\n    ...\n                Text(\n                    text='I',\n                    font=FONT_PATH,\n                    font_color=(255, 255, 255),\n                    font_size=102,\n                    point=Point(x=50, y=100),\n                    margin=Position(\n                        left=20,\n                        right=20,\n                    ),\n                    horizontal_alignment=HorizontalAlignment.CENTER,\n                    vertical_alignment=VerticalAlignment.CENTER,\n                ),\n                Text(\n                    text='P',\n                    font=FONT_PATH,\n                    font_color=(255, 255, 255),\n                    font_size=102,\n                    point=Point(x=150, y=100),\n                    margin=Position(\n                        left=20,\n                        right=20,\n                    ),\n                    horizontal_alignment=HorizontalAlignment.CENTER,\n                    vertical_alignment=VerticalAlignment.CENTER,\n                ),\n    ...\n\nHaving executed the script, we will get the following image:\n\n|Аватар|\n\nSuccess!\n\n| But what do we do if we need to change the text in the image often?\n| For this purpose, the template provides a context:\n\n.. code:: python\n\n    ...\n    from image_pattern import (\n        ...\n        Context,\n    )\n    ...\n    class AvatarContext(Context):\n        first_char: str\n        second_char: str\n    ...\n                Text(\n                    text=AvatarContext.var('first_char'),\n                    ...\n                ),\n                Text(\n                    text=AvatarContext.var('second_char'),\n                    ...\n                ),\n    ...\n    context = AvatarContext(\n        first_char='I',\n        second_char='P',\n    )\n    avatar_pattern = Avatar(context=context)\n    ...\n\n| Now we can change the content of the generated image without changing the template itself.\n| See the full code of the example in ``./examples/avatar.py``.\n\nAPI\n---\n\nAll classes are inherited from ``pydantic.BaseModel`` to validate passed\narguments, which imposes certain specifics when working with api.\n\nPattern\n~~~~~~~\n\n| Basic template class.\n| The template must be inherited from it.\n| To create a template, you need to override the following attributes:\n\n-  canvas - attribute of the ``Canvas`` type. Sets the properties of the\n   canvas.\n-  layers - attribute of the ``List[Layer]`` type. Sets a list of\n   layers.\n\nThe object constructor accepts the following arguments:\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n-  context - an argument of the ``Context`` type that will be passed to\n   the elements to form their properties.\n\nMethods of the object:\n^^^^^^^^^^^^^^^^^^^^^^\n\n-  render - returns the generated image object of the ``PIL.Image`` type;\n-  render\\_to\\_blob(\\*\\*save_kwargs) - returns the generated image object of the ``io.BytesIO`` type. Accepts the parameters passed to the method ``PIL.Image.save()``,such as ``quality`` and etc. You cannot pass the image format, as it is saved in ``JPEG``. Made simply for easy use of the generation results.\n\n\nCanvas\n~~~~~~\n\nThe object describing the properties of the canvas.\n\nThe object constructor accepts the following arguments:\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n-  size - is the size of the canvas. It can be set as\n   ``Tuple[int, int]`` as well as context variable.\n\nLayer\n~~~~~\n\n| An object that describes a layer containing image content.\n| The layers are overlapping, starting from the first layer in the list - the bottom layer, and ending with the last layer in the list - the top layer.\n\nThe object constructor accepts the following arguments:\n^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n\n-  \\*elements - a list of items ``Recatngle`` or ``Text`` to add to the image.\n\nContext\n~~~~~~~\n\n| An object that describes the context of an image. The context is described by the object attributes.\n| The context is an heir to ``pydantic.BaseModel``, so it requires a description of the types to perform the validation.\n\nMethods of the object:\n^^^^^^^^^^^^^^^^^^^^^^\n\n-  var(attribute\\_name: str) - indicates which context variable to use for this attribute.\n\nPositionable elements\n~~~~~~~~~~~~~~~~~~~~~\n\nRectangle\n^^^^^^^^^\n\nAn object that adds rectangles to an image.\n\nThe object constructor accepts the following arguments:\n'''''''''''''''''''''''''''''''''''''''''''''''''''''''\n\n-  point - ``Point`` object, which points to the upper left corner of\n   the element in the image;\n-  horizontal\\_alignment - one of the values of the enumeration\n   ``HorizontalAlignment``, to specify the horizontal alignment. Can be\n   set from a context variable. By default -\n   ``HorizontalAlignment.LEFT``;\n-  vertical\\_alignment - one of the values of the enumeration\n   ``VerticalAlignment``, to specify the vertical alignment. Can be set\n   from a context variable. By default - ``VerticalAlignment.TOP``;\n-  size - element size. It can be set as ``Tuple[int, int]`` as well as\n   context variable;\n-  brightness - element brightness. Optional argument. It ca be set as\n   ``float`` from 0 to 1 or context variable;\n-  background\\_image - sets the background image for the element.\n   Optional argument. Must set the path to the image. Can be set from a\n   context variable. The background image is scaled to the same extent\n   as set in css - ``background-size: cover;``.\n-  background\\_color - sets the color of background of the element.\n   Optional argument if set ``background_image``. Used when generating\n   an element only when the property ``background_image`` is not set.\n   It can be set as RGB ``Tuple[int, int, int]`` or RGBA\n   ``Tuple[int, int, int]``. Can be set from a context variable.\n-  alpha - alpha assignment. Optional argument. It can be set as ``int``\n   from 0 to 255. Can be set from a context variable.\n\nText\n^^^^\n\nAn object that adds text to the image.\n\nThe object constructor accepts the following arguments:\n'''''''''''''''''''''''''''''''''''''''''''''''''''''''\n\n-  point - ``Point`` object, which points to the upper left corner of\n   the element in the image;\n-  horizontal\\_alignment - one of the values of the enumeration\n   ``HorizontalAlignment``, to specify the horizontal alignment. Can be\n   set from a context variable. By default -\n   ``HorizontalAlignment.LEFT``;\n-  vertical\\_alignment - one of the values of the enumeration\n   ``VerticalAlignment``, to specify the vertical alignment. Can be set\n   from a context variable. By default - ``VerticalAlignment.TOP``;\n-  font - specifies the font to be used for text. Presented as a path to\n   OpenType or TrueType font. Can be set from a context variable.\n-  font\\_size - sets the font size. It can be represented by a ``int``\n   or context variable. By default - ``12``;\n-  font\\_color - sets the font color as RGB ``Tuple[int, int, int]``.\n   Can be set from context variable. By default - ``(0, 0, 0)``;\n-  text - specifies, directly, the text to be added to the image. It can\n   be set as ``str`` or context variable;\n-  line\\_height - sets the height of the line. Optional argument. It can\n   be set as ``int``\\ or context variable;\n-  margin - sets the indents for the text relative to the canvas.\n   Optional argument. It can be set as ``Position`` or context variable;\n\nSupport objects\n~~~~~~~~~~~~~~~\n\nPoint\n^^^^^\n\nDescribes the point on the canvas.\n\nThe object constructor accepts the following arguments:\n'''''''''''''''''''''''''''''''''''''''''''''''''''''''\n\n-  x - x coordinate as ``int``;\n-  y - y coordinate as ``int``.\n\nPosition\n^^^^^^^^\n\nDescribes the position of the element relative to the sides of the\ncanvas. For example, indents for text.\n\n-  top - ``int`` indented from the top edge of the canvas;\n-  right - ``int`` indented from the right edge of the canvas;\n-  bottom - ``int`` indented from the bottom edge of the canvas;\n-  left - ``int`` indented from the left edge of the canvas.\n\nEnums\n~~~~~\n\nHorizontalAlignment\n^^^^^^^^^^^^^^^^^^^\n\nProvides horizontal alignment options.\n\nValues\n''''''\n\n-  HorizontalAlignment.LEFT - left edge alignment;\n-  HorizontalAlignment.CENTER - center alignment;\n-  HorizontalAlignment.RIGHT - right edge alignment.\n\nVerticalAlignment\n^^^^^^^^^^^^^^^^^\n\nProvides vertical alignment options.\n\nValues\n''''''\n\n-  VerticalAlignment.TOP - top edge alignment;\n-  VerticalAlignment.CENTER - center alignment;\n-  VerticalAlignment.BOTTOM - bottom edge alignment.\n\nIntegrations\n~~~~~~~~~~~~\n\nDjango\n^^^^^^\n\n| For integration with django, the package provides the ``image_pattern.cotrib.ImagePatternField`` field inherited from ``django.db.models.ImageField``.\n| The field has a preset element ``blank = True`` relative to ``ImageField`` and a number of new arguments:\n\n-  pattern - image pattern;\n-  context - ``callback`` object method that returns the context for\n   generating the image. Optional argument.\n   if the method is not specified, the object method\n   ``get_image_pattern_context`` will be used;\n-  should\\_be\\_created - ``callback`` object method, indicating the need\n   to generate an image. Optional argument. The method is not specified, the object method ``image_pattern_should_be_created`` will be used.\n\n| The image is generated if the field is empty and ``should_be_created`` returns ``True``.\n| For more information ``ImagePatternField``\\ see the example project in ``./django_example``.\n\nTODO\n~~~~\n\n-  [ ] Make it possible to change the image format.\n-  [ ] Do something with the autocomplete to create objects (Since all\n   objects are inherited from pydantic.BaseModel, they do not contain\n   meta information for the autocomplete. Perhaps should manually write\n   all the constructors.).\n-  [ ] Think about using context. Using Context.var() with a string name\n   does not seem to be the best way.\n-  [ ] Make it possible to shift within the layer not only to down, but\n   also to the right.\n-  [ ] Setting the center of the background image.\n-  [ ] Test refactoring and bringing coverage to 100%.\n-  [ ] Setup linter.\n-  [ ] Check with mypy.\n-  [ ] Setup github actions.\n\n.. |ImagePattern| image:: https://raw.githubusercontent.com/brogency/image-pattern/master/assets/avatar-finally.jpg\n.. |Blank image for avatar| image:: https://raw.githubusercontent.com/brogency/image-pattern/master/assets/empty-avatar.jpg\n.. |Avatar with background| image:: https://raw.githubusercontent.com/brogency/image-pattern/master/assets/avatar-with-background.jpg\n.. |Image with rectangular offset| image:: ./assets/avatar-with-offset.jpg\n.. |Image with rectangle shift and height increase| image:: https://raw.githubusercontent.com/brogency/image-pattern/master/assets/avatar-with-offset-height.jpg\n.. |Picture with layers overlapping| image:: https://raw.githubusercontent.com/brogency/image-pattern/master/assets/avata-rwith-offset-layers.jpg\n.. |Picture with long text| image:: https://raw.githubusercontent.com/brogency/image-pattern/master/assets/avatar-long-text.jpg\n.. |Avatar with text in the center| image:: https://raw.githubusercontent.com/brogency/image-pattern/master/assets/avatar-center-text.jpg\n.. |Аватар| image:: https://raw.githubusercontent.com/brogency/image-pattern/master/assets/avatar-finally.jpg\n",
    'author': 'acrius',
    'author_email': 'acrius@mail.ru',
    'url': 'https://github.com/brogency/image-pattern',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
