"""Integration tests for end-to-end workflow resolution pipeline.

Tests the complete flow from workflow JSON → pyproject.toml using auto-strategies.
"""

import sys
from pathlib import Path

# Import helpers
sys.path.insert(0, str(Path(__file__).parent.parent))
from comfydock_core.strategies.auto import AutoModelStrategy, AutoNodeStrategy
from conftest import simulate_comfyui_save_workflow
from helpers.model_index_builder import ModelIndexBuilder
from helpers.pyproject_assertions import PyprojectAssertions
from helpers.workflow_builder import WorkflowBuilder, make_minimal_workflow


class TestWorkflowResolutionPipeline:
    """Test complete workflow resolution pipeline with maximum code coverage."""

    def test_end_to_end_resolution_with_auto_strategies(self, test_env, test_workspace):
        """Test full pipeline: workflow JSON → analysis → resolution → pyproject.toml.

        This single test covers ~60% of the critical workflow resolution path:
        - WorkflowDependencyParser.analyze_dependencies()
        - GlobalNodeResolver.resolve_single_node_with_context()
        - ModelResolver.resolve_model() (all strategies)
        - WorkflowManager.apply_resolution()
        - PyprojectManager writes (models + nodes)
        - Progressive write behavior
        """
        # ARRANGE: Populate model index
        model_builder = ModelIndexBuilder(test_workspace)
        model_builder.add_model(
            filename="sd15.safetensors",
            relative_path="checkpoints",
            category="checkpoints"
        )
        model_builder.index_all()

        # Create workflow with: 1 builtin node + 1 custom node + 1 model ref
        workflow = (
            WorkflowBuilder()
            .add_checkpoint_loader("sd15.safetensors")  # Model reference
            .add_builtin_node("KSampler", [123456, "fixed", 20, 8.0])  # Should be ignored
            .add_custom_node("CR_AspectRatioSD15", ["1:1 square"])  # Custom node
            .build()
        )

        simulate_comfyui_save_workflow(test_env, "test_workflow", workflow)

        # ACT: Resolve workflow with auto-strategies
        result = test_env.resolve_workflow(
            name="test_workflow",
            node_strategy=AutoNodeStrategy(),
            model_strategy=AutoModelStrategy()
        )

        # ASSERT: Model resolution succeeded
        assert len(result.models_resolved) == 1, \
            f"Should resolve 1 model, got {len(result.models_resolved)}"

        # Node may be unresolved if not in registry (that's OK for this test)
        # The important part is that the model was resolved and written to pyproject

        # ASSERT: Pyproject.toml state (the core validation)
        assertions = PyprojectAssertions(test_env)

        # Check workflow entry exists with correct structure
        workflow_check = assertions.has_workflow("test_workflow").has_model_count(1)

        # Validate model details - use the actual hash from resolution
        actual_hash = result.models_resolved[0].resolved_model.hash

        (
            workflow_check
            .has_model_with_hash(actual_hash)
            .has_model_with_filename("sd15.safetensors")
            .has_status("resolved")
            .has_criticality("flexible")  # Default for checkpoints
            .has_category("checkpoints")
        )

        # Check global models table
        (
            assertions
            .has_global_model(actual_hash)
            .has_filename("sd15.safetensors")
            .has_relative_path("checkpoints/sd15.safetensors")
            .has_category("checkpoints")
        )

    def test_model_path_reconstruction_for_builtin_nodes(self, test_env, test_workspace):
        """Test that builtin nodes get path reconstruction + stripping.

        Covers:
        - ModelConfig.reconstruct_model_path()
        - WorkflowManager.update_workflow_model_paths()
        - _strip_base_directory_for_node()
        """
        # ARRANGE: Model exists as "checkpoints/SD1.5/model.safetensors"
        model_builder = ModelIndexBuilder(test_workspace)
        model_builder.add_model(
            filename="model.safetensors",
            relative_path="checkpoints/SD1.5",
            category="checkpoints"
        )
        model_builder.index_all()

        # Workflow references without prefix: "SD1.5/model.safetensors"
        workflow = (
            WorkflowBuilder()
            .add_checkpoint_loader("SD1.5/model.safetensors")  # Missing "checkpoints/"
            .build()
        )

        simulate_comfyui_save_workflow(test_env, "test_path", workflow)

        # ACT: Resolve
        result = test_env.resolve_workflow(
            name="test_path",
            node_strategy=AutoNodeStrategy(),
            model_strategy=AutoModelStrategy()
        )

        # ASSERT: Resolution succeeded via reconstruction
        assert not result.has_issues
        assert len(result.models_resolved) == 1

        # Check match type
        resolved_model = result.models_resolved[0]
        assert resolved_model.match_type == "reconstructed", \
            "Should use path reconstruction for builtin nodes"

        # Check workflow JSON was updated (stripped path)
        from comfydock_core.repositories.workflow_repository import WorkflowRepository
        updated_workflow = WorkflowRepository.load(
            test_env.comfyui_path / "user" / "default" / "workflows" / "test_path.json"
        )

        # Builtin nodes should have base directory stripped
        checkpoint_node = updated_workflow.nodes["1"]
        assert checkpoint_node.widgets_values[0] == "SD1.5/model.safetensors", \
            "Builtin node widget should have stripped path (no 'checkpoints/' prefix)"

    def test_context_based_resolution_idempotency(self, test_env, test_workspace):
        """Test that re-resolving same workflow uses cached context.

        Covers:
        - ModelResolutionContext.previous_resolutions lookup
        - ModelResolver._try_context_resolution()
        - Idempotency (no duplicate entries)
        """
        # ARRANGE: Setup model
        model_builder = ModelIndexBuilder(test_workspace)
        model_builder.add_model(
            filename="sd15.safetensors",
            relative_path="checkpoints"
        )
        model_builder.index_all()

        workflow = make_minimal_workflow("sd15.safetensors")
        simulate_comfyui_save_workflow(test_env, "test_cache", workflow)

        # ACT: Resolve TWICE
        result1 = test_env.resolve_workflow(
            name="test_cache",
            node_strategy=AutoNodeStrategy(),
            model_strategy=AutoModelStrategy()
        )

        result2 = test_env.resolve_workflow(
            name="test_cache",
            node_strategy=AutoNodeStrategy(),
            model_strategy=AutoModelStrategy()
        )

        # ASSERT: Both succeed
        assert not result1.has_issues
        assert not result2.has_issues

        # Check 2nd resolution used context (should be instant)
        assert len(result2.models_resolved) == 1
        assert result2.models_resolved[0].match_type == "workflow_context", \
            "Second resolution should use context cache"

        # Verify no duplication in pyproject
        assertions = PyprojectAssertions(test_env)
        assertions.has_workflow("test_cache").has_model_count(1)

    def test_ambiguous_model_with_auto_strategy(self, test_env, test_workspace):
        """Test that AutoModelStrategy picks first candidate for ambiguous models.

        Covers:
        - ModelResolver returning multiple candidates
        - AutoModelStrategy disambiguation logic
        """
        # ARRANGE: Create TWO models with same filename, different paths
        model_builder = ModelIndexBuilder(test_workspace)
        model_builder.add_model(
            filename="model.safetensors",
            relative_path="checkpoints/SD1.5",
            size_mb=4
        )
        model_builder.add_model(
            filename="model.safetensors",
            relative_path="checkpoints/SD2.1",
            size_mb=5
        )
        model_builder.index_all()

        # Workflow references just filename (ambiguous)
        workflow = make_minimal_workflow("model.safetensors")
        simulate_comfyui_save_workflow(test_env, "test_ambig", workflow)

        # ACT: Resolve with auto-strategy
        result = test_env.resolve_workflow(
            name="test_ambig",
            node_strategy=AutoNodeStrategy(),
            model_strategy=AutoModelStrategy()
        )

        # ASSERT: Should succeed (auto-picks first)
        assert not result.has_issues
        assert len(result.models_resolved) == 1

        # Verify first model was selected
        resolved = result.models_resolved[0]
        assert resolved.resolved_model is not None, \
            "AutoModelStrategy should pick first candidate"

    def test_multiple_model_references_same_node(self, test_env, test_workspace):
        """Test CheckpointLoader with multiple model widgets.

        Covers:
        - WorkflowDependencyParser._extract_model_node_refs() special case
        - Multiple WorkflowNodeWidgetRef entries from same node
        """
        # ARRANGE: Create checkpoint and config models
        model_builder = ModelIndexBuilder(test_workspace)
        model_builder.add_model("checkpoint.safetensors", "checkpoints")
        model_builder.add_model("config.yaml", "configs")
        model_builder.index_all()

        # Create CheckpointLoader with TWO model references
        workflow = {
            "nodes": [
                {
                    "id": "1",
                    "type": "CheckpointLoader",  # Special multi-model node
                    "pos": [100, 100],
                    "size": [315, 98],
                    "flags": {},
                    "order": 0,
                    "mode": 0,
                    "outputs": [],
                    "properties": {},
                    "widgets_values": [
                        "checkpoint.safetensors",  # Widget 0: checkpoint
                        "config.yaml"              # Widget 1: config
                    ]
                }
            ],
            "links": [],
            "groups": [],
            "config": {},
            "extra": {},
            "version": 0.4
        }

        simulate_comfyui_save_workflow(test_env, "test_multi", workflow)

        # ACT: Analyze dependencies
        analysis = test_env.workflow_manager.analyze_workflow("test_multi")

        # ASSERT: Should extract 2 model refs from same node
        assert len(analysis.found_models) == 2, \
            f"CheckpointLoader should yield 2 model refs, got {len(analysis.found_models)}"

        # Verify both refs have same node_id but different widget_index
        refs = analysis.found_models
        assert refs[0].node_id == "1" and refs[1].node_id == "1"
        assert refs[0].widget_index == 0 and refs[1].widget_index == 1
        assert refs[0].widget_value == "checkpoint.safetensors"
        assert refs[1].widget_value == "config.yaml"
