"""High-level Git workflow manager for ComfyDock environments.

This module provides higher-level git workflows that combine multiple git operations
with business logic. It builds on top of the low-level git utilities in git.py.
"""
from __future__ import annotations

import os
import socket
from pathlib import Path
from typing import TYPE_CHECKING

from ..logging.logging_config import get_logger
from ..models.environment import GitStatus

if TYPE_CHECKING:
    from .pyproject_manager import PyprojectManager

from ..utils.git import (
    get_uncommitted_changes,
    git_checkout,
    git_commit,
    git_config_get,
    git_config_set,
    git_diff,
    git_history,
    git_init,
    git_ls_files,
    git_ls_tree,
    git_show,
    git_status_porcelain,
)

logger = get_logger(__name__)


class GitManager:
    """Manages high-level git workflows for environment tracking."""

    def __init__(self, repo_path: Path):
        """Initialize GitManager for a specific repository.

        Args:
            repo_path: Path to the git repository (usually .cec directory)
        """
        self.repo_path = repo_path
        self.gitignore_content = """# Staging area
staging/

# Staging metadata
metadata/

# logs
logs/

# Python cache
__pycache__/
*.pyc

# Temporary files
*.tmp
*.bak

# Runtime marker (created after successful environment initialization)
.complete
"""

    def ensure_git_identity(self) -> None:
        """Ensure git has a user identity configured for commits.

        Sets up local git config (not global) with sensible defaults.
        """
        # Check if identity is already configured
        existing_name = git_config_get(self.repo_path, "user.name")
        existing_email = git_config_get(self.repo_path, "user.email")

        # If both are set, we're good
        if existing_name and existing_email:
            return

        # Determine git identity using fallback chain
        git_name = self._get_git_identity()
        git_email = self._get_git_email()

        # Set identity locally for this repository only
        git_config_set(self.repo_path, "user.name", git_name)
        git_config_set(self.repo_path, "user.email", git_email)

        logger.info(f"Set local git identity: {git_name} <{git_email}>")

    def _get_git_identity(self) -> str:
        """Get a suitable git user name with smart fallbacks."""
        # Try environment variables first
        git_name = os.environ.get("GIT_AUTHOR_NAME")
        if git_name:
            return git_name

        # Try to get system username as fallback for name
        try:
            import pwd
            git_name = (
                pwd.getpwuid(os.getuid()).pw_gecos or pwd.getpwuid(os.getuid()).pw_name
            )
            if git_name:
                return git_name
        except Exception:
            pass

        try:
            git_name = os.getlogin()
            if git_name:
                return git_name
        except Exception:
            pass

        return "ComfyDock User"

    def _get_git_email(self) -> str:
        """Get a suitable git email with smart fallbacks."""
        # Try environment variables first
        git_email = os.environ.get("GIT_AUTHOR_EMAIL")
        if git_email:
            return git_email

        # Try to construct from username and hostname
        try:
            hostname = socket.gethostname()
            username = os.getlogin()
            return f"{username}@{hostname}"
        except Exception:
            pass

        return "user@comfydock.local"

    def initialize_environment_repo(
        self, initial_message: str = "Initial environment setup"
    ) -> None:
        """Initialize a new environment repository with proper setup.

        This combines:
        - Git init
        - Identity setup
        - Gitignore creation
        - Initial commit

        Args:
            initial_message: Message for the initial commit
        """
        # Initialize git repository
        git_init(self.repo_path)

        # Ensure git identity is configured
        self.ensure_git_identity()

        # Create standard .gitignore
        self._create_gitignore()

        # Initial commit (if there are files to commit)
        if any(self.repo_path.iterdir()):
            git_commit(self.repo_path, initial_message)
            logger.info(f"Created initial commit: {initial_message}")

    def commit_with_identity(self, message: str, add_all: bool = True) -> None:
        """Commit changes ensuring identity is set up.

        Args:
            message: Commit message
            add_all: Whether to stage all changes first
        """
        # Ensure identity before committing
        self.ensure_git_identity()

        # Perform the commit
        git_commit(self.repo_path, message, add_all)

    def _get_files_in_commit(self, commit_hash: str) -> set[str]:
        """Get all tracked file paths in a specific commit.

        Args:
            commit_hash: Git commit hash

        Returns:
            Set of file paths that exist in the commit
        """
        result = git_ls_tree(self.repo_path, commit_hash, recursive=True)
        if not result.strip():
            return set()

        return {line for line in result.splitlines() if line}

    def _get_tracked_files(self) -> set[str]:
        """Get all currently tracked file paths in working tree.

        Returns:
            Set of file paths currently tracked by git
        """
        result = git_ls_files(self.repo_path)
        if not result.strip():
            return set()

        return {line for line in result.splitlines() if line}

    def apply_version(self, version: str, leave_unstaged: bool = True) -> None:
        """Apply files from a specific version to working directory.

        This is a high-level rollback operation that:
        - Resolves version identifiers (v1, v2, etc.) to commits
        - Applies files from that commit
        - Deletes files that don't exist in target commit
        - Optionally leaves them unstaged for review

        Args:
            version: Version identifier (e.g., "v1", "v2") or commit hash
            leave_unstaged: If True, files are left as uncommitted changes

        Raises:
            ValueError: If version doesn't exist
        """
        # Resolve version to commit hash
        commit_hash = self.resolve_version(version)

        logger.info(f"Applying files from version {version} (commit {commit_hash[:8]})")

        # Phase 1: Get file lists
        target_files = self._get_files_in_commit(commit_hash)
        current_files = self._get_tracked_files()
        files_to_delete = current_files - target_files

        # Phase 2: Restore files from target commit
        git_checkout(self.repo_path, commit_hash, files=["."], unstage=leave_unstaged)

        # Phase 3: Delete files that don't exist in target version
        if files_to_delete:
            from ..utils.common import run_command

            for file_path in files_to_delete:
                full_path = self.repo_path / file_path
                if full_path.exists():
                    full_path.unlink()
                    logger.info(f"Deleted {file_path} (not in target version)")

            # Stage only the specific deletions (not all modifications)
            # git add <file> will stage the deletion when file doesn't exist
            for file_path in files_to_delete:
                run_command(["git", "add", file_path], cwd=self.repo_path, check=True)

            # If leave_unstaged, unstage the deletions again
            if leave_unstaged:
                run_command(["git", "reset", "HEAD"] + list(files_to_delete),
                          cwd=self.repo_path, check=True)

    def discard_uncommitted(self) -> None:
        """Discard all uncommitted changes in the repository."""
        logger.info("Discarding uncommitted changes")
        git_checkout(self.repo_path, "HEAD", files=["."])

    def get_version_history(self, limit: int = 10) -> list[dict]:
        """Get simplified version history with v1, v2 labels.

        Args:
            limit: DEPRECATED - Now always shows all commits for version stability.
                   Parameter kept for API compatibility but is ignored.

        Returns:
            List of version info dicts with stable version numbers
        """
        # Always get ALL commits to ensure version numbers remain stable
        # Pagination can be added post-MVP if needed
        return self._get_commit_versions(limit=1000)

    def resolve_version(self, version: str) -> str:
        """Resolve a version identifier to a commit hash.

        Args:
            version: Version identifier (e.g., "v1", "v2") or commit hash

        Returns:
            Full commit hash

        Raises:
            ValueError: If version doesn't exist
        """
        return self._resolve_version_to_commit(version)

    def get_pyproject_diff(self) -> str:
        """Get the git diff specifically for pyproject.toml.

        Returns:
            Diff output or empty string
        """
        pyproject_path = Path("pyproject.toml")
        return git_diff(self.repo_path, pyproject_path) or ""

    def get_pyproject_from_version(self, version: str) -> str:
        """Get pyproject.toml content from a specific version.

        Args:
            version: Version identifier or commit hash

        Returns:
            File content as string

        Raises:
            ValueError: If version or file doesn't exist
        """
        commit_hash = self.resolve_version(version)
        return git_show(self.repo_path, commit_hash, Path("pyproject.toml"))

    def commit_all(self, message: str | None = None) -> None:
        """Commit all changes in the repository.

        Args:
            message: Commit message

        Raises:
            OSError: If git commands fail

        """
        if message is None:
            message = "Committing all changes"
        return git_commit(self.repo_path, message, add_all=True)

    def get_workflow_git_changes(self) -> dict[str, str]:
        """Get git status for workflow files specifically.

        Returns:
            Dict mapping workflow names to their git status:
            - 'modified' for modified files
            - 'added' for new/untracked files
            - 'deleted' for deleted files
        """
        status_entries = git_status_porcelain(self.repo_path)
        workflow_changes = {}

        for index_status, working_status, filename in status_entries:
            logger.debug(f"index status: {index_status}, working status: {working_status}, filename: {filename}")

            # Only process workflow files
            if filename.startswith('workflows/') and filename.endswith('.json'):
                # Extract workflow name from path (keep spaces as-is)
                workflow_name = Path(filename).stem
                logger.debug(f"Workflow name: {workflow_name}")

                # Determine status (prioritize working tree status)
                if working_status == 'M' or index_status == 'M':
                    workflow_changes[workflow_name] = 'modified'
                elif working_status == 'D' or index_status == 'D':
                    workflow_changes[workflow_name] = 'deleted'
                elif working_status == '?' or index_status == 'A':
                    workflow_changes[workflow_name] = 'added'

        logger.debug(f"Workflow changes: {str(workflow_changes)}")
        return workflow_changes

    def has_uncommitted_changes(self) -> bool:
        """Check if there are any uncommitted changes.

        Returns:
            True if there are uncommitted changes
        """
        return bool(get_uncommitted_changes(self.repo_path))

    def _create_gitignore(self) -> None:
        """Create standard .gitignore for environment tracking."""
        gitignore_path = self.repo_path / ".gitignore"
        gitignore_path.write_text(self.gitignore_content)

    def _get_commit_versions(self, limit: int = 10) -> list[dict]:
        """Get simplified version list from git history.

        Returns commits with simple identifiers instead of full hashes.

        Args:
            limit: Maximum number of commits to return

        Returns:
            List of commit info dicts with keys: version, hash, message, date

        Raises:
            OSError: If git command fails
        """
        result = git_history(self.repo_path, max_count=limit, pretty="format:%H|%s|%ai")

        commits = []
        for line in result.strip().split('\n'):
            if line:
                hash_val, message, date = line.split('|', 2)
                commits.append({
                    'hash': hash_val,
                    'message': message,
                    'date': date
                })

        # Reverse so oldest commit is first (chronological order)
        commits.reverse()

        # Now assign version numbers: oldest = v1, newest = v<highest>
        for i, commit in enumerate(commits):
            commit['version'] = f"v{i + 1}"

        return commits

    def _resolve_version_to_commit(self, version: str) -> str:
        """Resolve a simple version identifier to a git commit hash.
        
        Args:
            repo_path: Path to git repository
            version: Version identifier (e.g., "v1", "v2")
            
        Returns:
            Full commit hash
            
        Raises:
            ValueError: If version doesn't exist
            OSError: If git command fails
        """
        # If it's already a commit hash, return as-is
        if len(version) >= 7 and all(c in '0123456789abcdef' for c in version.lower()):
            return version

        commits = self._get_commit_versions(limit=100)

        for commit in commits:
            if commit['version'] == version:
                return commit['hash']

        raise ValueError(f"Version '{version}' not found")

    def get_status(self, pyproject_manager: PyprojectManager | None = None) -> GitStatus:
        """Get complete git status with optional change parsing.
        
        Args:
            pyproject_manager: Optional PyprojectManager for parsing changes
            
        Returns:
            GitStatus with all git information encapsulated
        """
        # Get basic git information
        workflow_changes = self.get_workflow_git_changes()
        pyproject_has_changes = bool(self.get_pyproject_diff().strip())
        has_changes = pyproject_has_changes or bool(workflow_changes)

        # Create status object
        status = GitStatus(
            has_changes=has_changes,
            # diff=diff,
            workflow_changes=workflow_changes
        )

        # Parse changes if we have them and a pyproject manager
        if has_changes and pyproject_manager:
            from ..analyzers.git_change_parser import GitChangeParser
            parser = GitChangeParser(self.repo_path)
            current_config = pyproject_manager.load()

            # The parser updates the status object directly
            parser.update_git_status(status, current_config)

        return status

    def create_checkpoint(self, description: str | None = None) -> str:
        """Create a version checkpoint of the current state.

        Args:
            description: Optional description for the checkpoint

        Returns:
            Version identifier (e.g., "v3")
        """
        # Generate automatic message if not provided
        if not description:
            from datetime import datetime

            description = f"Checkpoint created at {datetime.now().isoformat()}"

        # Commit current state
        self.commit_with_identity(description)

        # Get the new version number
        versions = self.get_version_history(limit=1)
        if versions:
            return versions[-1]["version"]
        return "v1"

    def rollback_to(self, version: str, safe: bool = False, force: bool = False) -> None:
        """Rollback environment to a previous version.

        Args:
            version: Version to rollback to
            safe: If True, leaves changes unstaged for review (default: False for clean state)
            force: If True, discard uncommitted changes without error

        Raises:
            ValueError: If version doesn't exist
            CDEnvironmentError: If uncommitted changes exist and force=False
        """
        from comfydock_core.models.exceptions import CDEnvironmentError

        # Check for uncommitted changes
        if self.has_uncommitted_changes():
            if not force:
                raise CDEnvironmentError(
                    "Cannot rollback with uncommitted changes.\n"
                    "Options:\n"
                    "  • Commit: comfydock commit -m '<message>'\n"
                    "  • Force discard: comfydock rollback --force <version>\n"
                    "  • See changes: comfydock status"
                )
            logger.warning("Discarding uncommitted changes (--force flag used)")
            self.discard_uncommitted()

        # Apply the target version (clean state by default)
        self.apply_version(version, leave_unstaged=safe)

        logger.info(f"Rolled back to {version}")

    def get_version_summary(self) -> dict:
        """Get a summary of the version state.

        Returns:
            Dict with current version, has_changes, total_versions
        """
        versions = self.get_version_history(limit=100)
        has_changes = self.has_uncommitted_changes()

        current_version = versions[-1]["version"] if versions else None

        return {
            "current_version": current_version,
            "has_uncommitted_changes": has_changes,
            "total_versions": len(versions),
            "latest_message": versions[-1]["message"] if versions else None,
        }

    # =============================================================================
    # Pull/Push/Remote Operations
    # =============================================================================

    def pull(self, remote: str = "origin", branch: str | None = None, ff_only: bool = False) -> dict:
        """Pull from remote (fetch + merge).

        Args:
            remote: Remote name (default: origin)
            branch: Branch to pull (default: current branch)
            ff_only: Only allow fast-forward merges (default: False)

        Returns:
            Dict with keys: 'fetch_output', 'merge_output', 'branch'

        Raises:
            ValueError: If no remote, detached HEAD, or merge conflicts
            OSError: If fetch/merge fails
        """
        from ..utils.git import git_pull

        logger.info(f"Pulling {remote}/{branch or 'current branch'}")

        result = git_pull(self.repo_path, remote, branch, ff_only=ff_only)

        return result

    def push(self, remote: str = "origin", branch: str | None = None, force: bool = False) -> str:
        """Push commits to remote.

        Args:
            remote: Remote name (default: origin)
            branch: Branch to push (default: current branch)
            force: Use --force-with-lease (default: False)

        Returns:
            Push output

        Raises:
            ValueError: If no remote or detached HEAD
            OSError: If push fails
        """
        from ..utils.git import git_push, git_current_branch

        # Get current branch if not specified
        if not branch:
            branch = git_current_branch(self.repo_path)

        logger.info(f"Pushing to {remote}/{branch}" + (" (force)" if force else ""))

        return git_push(self.repo_path, remote, branch, force=force)

    def add_remote(self, name: str, url: str) -> None:
        """Add a git remote.

        Args:
            name: Remote name (e.g., "origin")
            url: Remote URL

        Raises:
            OSError: If remote already exists
        """
        from ..utils.git import git_remote_add

        logger.info(f"Adding remote '{name}': {url}")
        git_remote_add(self.repo_path, name, url)

    def remove_remote(self, name: str) -> None:
        """Remove a git remote.

        Args:
            name: Remote name (e.g., "origin")

        Raises:
            ValueError: If remote doesn't exist
        """
        from ..utils.git import git_remote_remove

        logger.info(f"Removing remote '{name}'")
        git_remote_remove(self.repo_path, name)

    def list_remotes(self) -> list[tuple[str, str, str]]:
        """List all git remotes.

        Returns:
            List of tuples: [(name, url, type), ...]
        """
        from ..utils.git import git_remote_list

        return git_remote_list(self.repo_path)

    def has_remote(self, name: str = "origin") -> bool:
        """Check if a remote exists.

        Args:
            name: Remote name (default: origin)

        Returns:
            True if remote exists
        """
        from ..utils.git import git_remote_get_url

        url = git_remote_get_url(self.repo_path, name)
        return bool(url)