# core/status_scanner.py
from __future__ import annotations

import subprocess
from pathlib import Path
from typing import TYPE_CHECKING

from comfydock_core.constants import CUSTOM_NODES_BLACKLIST

from ..logging.logging_config import get_logger
from ..models.environment import (
    EnvironmentComparison,
    PackageSyncStatus,
    NodeState,
    EnvironmentState,
)
from ..models.exceptions import UVCommandError
from .node_git_analyzer import get_node_git_info

if TYPE_CHECKING:
    from ..managers.pyproject_manager import PyprojectManager
    from ..managers.uv_project_manager import UVProjectManager

logger = get_logger(__name__)


class StatusScanner:
    """Scans environment to get current state."""

    def __init__(
        self,
        uv: UVProjectManager,
        pyproject: PyprojectManager,
        venv_path: Path,
        comfyui_path: Path,
    ):
        self._uv = uv
        self._pyproject = pyproject
        self._venv_path = venv_path
        self._comfyui_path = comfyui_path

    def get_full_comparison(self) -> EnvironmentComparison:
        """Get complete environment comparison with all details.

        Args:
            comfyui_path: Path to ComfyUI directory
            venv_path: Path to virtual environment
            uv: UV interface for package operations
            pyproject: PyprojectManager instance

        Returns:
            Complete environment comparison
        """
        # Scan current and expected states
        current = self.scan_environment()
        expected = self.scan_manifest()

        # Get basic comparison
        comparison = self.compare_states(current, expected)

        # Skip package sync check for performance (100-500ms saved)
        # Rationale:
        #   - Users rarely manually modify .venv/
        #   - Operations like 'run', 'repair', 'node add' auto-sync before executing
        #   - Status is high-frequency with workflow caching - needs to be fast
        #   - Package drift self-corrects on next sync operation
        # If thorough check needed, use 'comfydock repair --dry-run' (future)
        # TODO: Add package sync status
        # package_status = self.check_packages_sync()
        comparison.packages_in_sync = True #package_status.in_sync
        comparison.package_sync_message = "" #package_status.message

        return comparison

    def scan_environment(self) -> EnvironmentState:
        """Scan the environment for its current state.

        Args:
            comfyui_path: Path to ComfyUI directory
            venv_path: Path to virtual environment
            uv: UV interface for package operations

        Returns:
            Current environment state
        """
        # Scan custom nodes
        custom_nodes = self._scan_custom_nodes()

        # Get installed packages
        # packages = self._scan_packages()

        # Get Python version
        # python_version = self._get_python_version()

        return EnvironmentState(
            custom_nodes=custom_nodes, 
            packages=None,#packages, 
            python_version=None,#python_version
        )

    def _scan_custom_nodes(self) -> dict[str, NodeState]:
        """Scan the custom_nodes directory."""
        nodes = {}
        custom_nodes_path = self._comfyui_path / "custom_nodes"

        if not custom_nodes_path.exists():
            logger.debug("custom_nodes directory not found")
            return nodes

        # Skip these directories
        skip_dirs = CUSTOM_NODES_BLACKLIST

        # TODO: Support .comfydock_ignore
        for node_dir in custom_nodes_path.iterdir():
            if not node_dir.is_dir() or node_dir.name in skip_dirs:
                continue

            # Skip hidden directories and disabled nodes
            if node_dir.name.startswith(".") or node_dir.name.endswith(".disabled"):
                continue

            try:
                node_state = self._scan_single_node(node_dir)
                nodes[node_dir.name] = node_state
            except Exception as e:
                logger.debug(f"Error scanning node {node_dir.name}: {e}")
                # Still record it as present but with minimal info
                nodes[node_dir.name] = NodeState(
                    name=node_dir.name,
                    path=node_dir,
                    disabled=node_dir.name.endswith(".disabled"),
                )

        return nodes

    def _scan_single_node(self, node_dir: Path) -> NodeState:
        """Scan a single custom node directory."""
        state = NodeState(name=node_dir.name, path=node_dir)

        # Check if is disabled (has .disabled appended to dir name)
        if node_dir.name.endswith(".disabled"):
            state.disabled = True

        # Check for git info
        if (node_dir / ".git").exists():
            git_info = get_node_git_info(node_dir)
            if git_info:
                state.git_commit = git_info.commit
                state.git_branch = git_info.branch
                state.version = git_info.tag  # Use git tag as version
                state.is_dirty = git_info.is_dirty

        # If no version from git, check pyproject.toml
        if not state.version:
            pyproject_path = node_dir / "pyproject.toml"
            if pyproject_path.exists():
                try:
                    import tomlkit

                    with open(pyproject_path, encoding='utf-8') as f:
                        data = tomlkit.load(f)

                    # Try different locations for version
                    project_section = data.get("project")
                    if project_section and isinstance(project_section, dict):
                        version = project_section.get("version")
                        if isinstance(version, str):
                            state.version = version

                    if not state.version:
                        tool_section = data.get("tool")
                        if tool_section and isinstance(tool_section, dict):
                            poetry_section = tool_section.get("poetry")
                            if poetry_section and isinstance(poetry_section, dict):
                                version = poetry_section.get("version")
                                if isinstance(version, str):
                                    state.version = version
                except Exception:
                    pass  # Ignore parse errors

        return state

    def _scan_packages(self) -> dict[str, str]:
        """Get installed packages using UV."""
        packages = {}

        python_path = self._uv.python_executable

        if not python_path.exists():
            logger.warning(f"Python not found in venv: {self._venv_path}")
            return packages

        # TODO: Make this more robust
        try:
            output = self._uv.freeze_packages(python=python_path)
            if output:
                for line in output.strip().split("\n"):
                    if line and "==" in line and not line.startswith("#"):
                        if line.startswith("-e "):
                            # Skip editable installs for now
                            continue
                        name, version = line.split("==", 1)
                        packages[name.strip().lower()] = version.strip()
        except Exception as e:
            logger.warning(f"Failed to get packages: {e}")

        return packages

    def _get_python_version(self) -> str:
        """Get Python version from venv."""
        python_path = self._uv.python_executable

        if not python_path.exists():
            return "unknown"

        try:
            result = subprocess.run(
                [str(python_path), "--version"],
                capture_output=True,
                text=True,
                timeout=5,
            )
            if result.returncode == 0:
                # Parse "Python 3.11.5" -> "3.11.5"
                return result.stdout.strip().split()[-1]
        except Exception as e:
            logger.debug(f"Failed to get Python version: {e}")

        return "unknown"

    def scan_manifest(self) -> EnvironmentState:
        """Scan expected state from pyproject.toml.

        Args:
            pyproject: PyprojectManager instance

        Returns:
            Expected environment state from configuration
        """
        config = self._pyproject.load()

        # Get expected custom nodes from pyproject
        node_infos = self._pyproject.nodes.get_existing()

        expected_nodes = {}
        for _, node_info in node_infos.items():
            expected_nodes[node_info.name] = NodeState(
                name=node_info.name,
                path=Path("custom_nodes") / node_info.name,
                version=node_info.version,
                source=node_info.source,
            )

        # # Get expected packages from dependency groups
        # expected_packages = {}
        # dependencies = config.get("dependency-groups", {})

        # for _, deps in dependencies.items():
        #     for dep in deps:
        #         if isinstance(dep, str):
        #             # Parse package spec like "torch>=2.0.0"
        #             if "==" in dep:
        #                 name, version = dep.split("==", 1)
        #                 expected_packages[name.strip().lower()] = version.strip()
        #             elif ">=" in dep or "<=" in dep or ">" in dep or "<" in dep:
        #                 # For now, just record that the package should be present
        #                 name = dep.split(">")[0].split("<")[0].split("=")[0]
        #                 expected_packages[name.strip().lower()] = "*"
        #             else:
        #                 expected_packages[dep.strip().lower()] = "*"

        # # Get Python version from project settings
        # python_version = (
        #     config.get("project", {}).get("requires-python", "").strip(">=")
        # )
        # if not python_version:
        #     python_version = "3.11"  # Default

        return EnvironmentState(
            custom_nodes=expected_nodes,
            packages=None,#expected_packages,
            python_version=None,#python_version,
        )

    def compare_states(
        self, current: EnvironmentState, expected: EnvironmentState
    ) -> EnvironmentComparison:
        """Compare current and expected environment states.

        Args:
            current: Current environment state
            expected: Expected environment state

        Returns:
            Comparison results
        """
        comparison = EnvironmentComparison()

        # Compare custom nodes
        current_nodes = set(current.custom_nodes.keys())
        expected_nodes = set(expected.custom_nodes.keys())

        comparison.missing_nodes = list(expected_nodes - current_nodes)
        comparison.extra_nodes = list(current_nodes - expected_nodes)

        # Check version mismatches (skip development nodes)
        for name in current_nodes & expected_nodes:
            current_node = current.custom_nodes[name]
            expected_node = expected.custom_nodes[name]

            # Skip version comparison for development nodes - they're user-managed
            if expected_node.source == 'development':
                continue

            if expected_node.version and current_node.version != expected_node.version:
                comparison.version_mismatches.append(
                    {
                        "name": name,
                        "expected": expected_node.version,
                        "actual": current_node.version,
                    }
                )

        # Detect potential dev node renames (simple heuristic)
        if comparison.missing_nodes and comparison.extra_nodes:
            missing_dev = any(
                expected.custom_nodes[n].source == 'development'
                for n in comparison.missing_nodes
                if n in expected.custom_nodes
            )
            extra_git = any(
                (self._comfyui_path / 'custom_nodes' / n / '.git').exists()
                for n in comparison.extra_nodes
            )
            comparison.potential_dev_rename = missing_dev and extra_git

        # Package comparison is handled separately since it requires UV
        # This will be set by check_packages_sync

        return comparison

    def check_packages_sync(self) -> PackageSyncStatus:
        """Check if packages are in sync with pyproject.toml.

        Args:
            uv: UV interface for package operations

        Returns:
            Package sync status
        """
        try:
            # Use UV's dry-run to check if sync would change anything
            self._uv.sync_project(dry_run=True, all_groups=True)
            return PackageSyncStatus(
                in_sync=True, message="Packages match pyproject.toml"
            )
        except UVCommandError as e:
            return PackageSyncStatus(
                in_sync=False,
                message="Packages out of sync (run 'env sync' to update)",
                details=str(e),
            )
