"""
C04 - Memory Optimization and Deduplication Tools

Implements memory optimization, deduplication, and archiving tools to prevent
memory bloat and identify stale memories for enterprise-scale deployments.
"""

import hashlib
from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional
from collections import defaultdict
import os
from sqlalchemy import text

from ..database import get_database
from ..types import ToolResponse
from ..utils import catch_errors
from typing import cast


@catch_errors
def find_duplicates(
    table_name: str,
    content_columns: List[str],
    similarity_threshold: float = 0.95,
    sample_size: Optional[int] = None,
) -> ToolResponse:
    """
    🔍 **DUPLICATE DETECTION** - Find duplicate and near-duplicate content!

    Identifies duplicate memories using content hash comparison and semantic similarity.
    Essential for maintaining clean memory banks and reducing storage costs.

    Args:
        table_name (str): Table to analyze for duplicates
        content_columns (List[str]): Columns to compare for duplicate detection
        similarity_threshold (float): Similarity threshold for near-duplicates (0.0-1.0, default: 0.95)
        sample_size (Optional[int]): Limit analysis to sample size for performance (default: analyze all)

    Returns:
        ToolResponse: {"success": True, "duplicates": List[...], "stats": Dict}
    """
    db_path = os.environ.get("DB_PATH", "./test.db")
    db = get_database(db_path)

    try:
        # Get all rows from the table
        with db.engine.connect() as conn:
            # Build query with specified columns
            columns_str = ", ".join([f"`{col}`" for col in ["id"] + content_columns])
            query = text(f"SELECT {columns_str} FROM `{table_name}`")

            if sample_size:
                query = text(f"SELECT {columns_str} FROM `{table_name}` LIMIT {sample_size}")

            result = conn.execute(query)
            rows = [dict(row._mapping) for row in result.fetchall()]

        if not rows:
            return cast(
                ToolResponse,
                {
                    "success": True,
                    "duplicates": [],
                    "stats": {
                        "total_rows": 0,
                        "duplicate_groups": 0,
                        "potential_savings": 0,
                    },
                },
            )

        # Find exact duplicates using content hash
        content_hashes = {}
        duplicate_groups = []

        for row in rows:
            # Create content hash from specified columns
            content_values = [str(row.get(col, "")) for col in content_columns]
            content_str = "|".join(content_values)
            content_hash = hashlib.md5(content_str.encode()).hexdigest()

            if content_hash not in content_hashes:
                content_hashes[content_hash] = []
            content_hashes[content_hash].append(row)

        # Identify duplicate groups
        for content_hash, group_rows in content_hashes.items():
            if len(group_rows) > 1:
                duplicate_groups.append(
                    {
                        "content_hash": content_hash,
                        "duplicate_count": len(group_rows),
                        "rows": group_rows,
                        "suggested_action": ("keep_newest" if "timestamp" in group_rows[0] else "manual_review"),
                    }
                )

        # Calculate statistics
        total_duplicates = sum(len(group["rows"]) - 1 for group in duplicate_groups)
        potential_savings_percent = (total_duplicates / len(rows)) * 100 if rows else 0

        stats = {
            "total_rows": len(rows),
            "unique_content_hashes": len(content_hashes),
            "duplicate_groups": len(duplicate_groups),
            "total_duplicates": total_duplicates,
            "potential_savings_percent": round(potential_savings_percent, 2),
            "recommended_cleanup": total_duplicates > 0,
        }

        return cast(
            ToolResponse,
            {
                "success": True,
                "duplicates": duplicate_groups,
                "stats": stats,
                "cleanup_recommendations": _generate_cleanup_recommendations(duplicate_groups),
            },
        )

    except Exception as e:
        return cast(
            ToolResponse,
            {
                "success": False,
                "error": f"Duplicate detection failed: {str(e)}",
                "category": "DUPLICATE_DETECTION_ERROR",
                "details": {"table": table_name, "content_columns": content_columns},
            },
        )


@catch_errors
def optimize_memory_bank(table_name: str, optimization_strategy: str = "comprehensive", dry_run: bool = True) -> ToolResponse:
    """
    ⚡ **MEMORY BANK OPTIMIZATION** - Optimize storage and performance!

    Performs comprehensive memory bank optimization including deduplication,
    archiving, and index optimization for enterprise-scale performance.

    Args:
        table_name (str): Table to optimize
        optimization_strategy (str): Strategy - "conservative", "aggressive", "comprehensive"
        dry_run (bool): If True, only analyze without making changes (default: True)

    Returns:
        ToolResponse: {"success": True, "optimizations": List[...], "savings": Dict}
    """
    db_path = os.environ.get("DB_PATH", "./test.db")
    db = get_database(db_path)

    try:
        optimizations_performed = []
        potential_savings = {
            "rows_removed": 0,
            "storage_saved_mb": 0,
            "performance_improvement": 0,
        }

        with db.engine.connect() as conn:
            # Step 1: Analyze table structure and content
            schema_result = conn.execute(text(f"PRAGMA table_info(`{table_name}`)"))
            columns = [row[1] for row in schema_result.fetchall()]

            count_result = conn.execute(text(f"SELECT COUNT(*) FROM `{table_name}`"))
            count_row = count_result.fetchone()
            total_rows = count_row[0] if count_row else 0

            # Step 2: Find duplicates (using text columns)
            text_columns = [col for col in columns if col not in ["id", "timestamp", "embedding"]]
            if text_columns:
                duplicates_result = find_duplicates(table_name, text_columns)
                if duplicates_result.get("success"):
                    duplicate_count = duplicates_result.get("stats", {}).get("total_duplicates", 0)
                    if duplicate_count > 0:
                        optimizations_performed.append(
                            {
                                "type": "duplicate_removal",
                                "description": f"Found {duplicate_count} duplicate rows",
                                "action": ("Remove duplicates, keep most recent" if not dry_run else "Would remove duplicates"),
                                "rows_affected": duplicate_count,
                            }
                        )
                        potential_savings["rows_removed"] += duplicate_count

            # Step 3: Identify old/stale entries
            if "timestamp" in columns:
                cutoff_days = 365 if optimization_strategy == "conservative" else 180 if optimization_strategy == "comprehensive" else 90
                cutoff_date = (datetime.now() - timedelta(days=cutoff_days)).isoformat()

                old_count_result = conn.execute(
                    text(f"SELECT COUNT(*) FROM `{table_name}` WHERE timestamp < :cutoff_date"),
                    {"cutoff_date": cutoff_date},
                )
                old_count_row = old_count_result.fetchone()
                old_rows = old_count_row[0] if old_count_row else 0

                if old_rows > 0:
                    optimizations_performed.append(
                        {
                            "type": "archive_old_entries",
                            "description": f"Found {old_rows} entries older than {cutoff_days} days",
                            "action": ("Archive old entries" if not dry_run else f"Would archive {old_rows} old entries"),
                            "rows_affected": old_rows,
                        }
                    )
                    potential_savings["rows_removed"] += int(old_rows * 0.5)  # Archive, don't delete

            # Step 4: Analyze embedding storage efficiency
            if "embedding" in columns:
                embedding_count_result = conn.execute(text(f"SELECT COUNT(*) FROM `{table_name}` WHERE embedding IS NOT NULL"))
                embedding_count_row = embedding_count_result.fetchone()
                embedding_rows = embedding_count_row[0] if embedding_count_row else 0

                # Estimate embedding storage size
                if embedding_rows > 0:
                    sample_embedding_result = conn.execute(text(f"SELECT LENGTH(embedding) FROM `{table_name}` WHERE embedding IS NOT NULL LIMIT 1"))
                    sample_size = sample_embedding_result.fetchone()
                    if sample_size:
                        total_embedding_size_mb = (embedding_rows * sample_size[0]) / (1024 * 1024)
                        optimizations_performed.append(
                            {
                                "type": "embedding_analysis",
                                "description": f"Embedding storage: {total_embedding_size_mb:.2f} MB for {embedding_rows} rows",
                                "action": "Embeddings are efficiently stored",
                                "rows_affected": 0,
                            }
                        )

            # Step 5: Database optimization
            if not dry_run:
                # VACUUM to reclaim space
                conn.execute(text("VACUUM"))
                optimizations_performed.append(
                    {
                        "type": "database_vacuum",
                        "description": "Reclaimed deleted space and optimized database file",
                        "action": "VACUUM completed",
                        "rows_affected": 0,
                    }
                )

                # ANALYZE to update statistics
                conn.execute(text("ANALYZE"))
                optimizations_performed.append(
                    {
                        "type": "statistics_update",
                        "description": "Updated query optimization statistics",
                        "action": "ANALYZE completed",
                        "rows_affected": 0,
                    }
                )

        # Calculate estimated savings
        if potential_savings["rows_removed"] > 0:
            # Estimate storage savings (rough calculation)
            avg_row_size_kb = 2  # Conservative estimate
            potential_savings["storage_saved_mb"] = int((potential_savings["rows_removed"] * avg_row_size_kb) / 1024)
            potential_savings["performance_improvement"] = min(
                50,
                (int((potential_savings["rows_removed"] / total_rows) * 100) if total_rows > 0 else 0),
            )

        return cast(
            ToolResponse,
            {
                "success": True,
                "optimizations": optimizations_performed,
                "potential_savings": potential_savings,
                "dry_run": dry_run,
                "table_stats": {
                    "total_rows": total_rows,
                    "columns": len(columns),
                    "optimization_strategy": optimization_strategy,
                },
                "recommendations": _generate_optimization_recommendations(optimizations_performed, optimization_strategy),
            },
        )

    except Exception as e:
        return cast(
            ToolResponse,
            {
                "success": False,
                "error": f"Memory optimization failed: {str(e)}",
                "category": "OPTIMIZATION_ERROR",
                "details": {"table": table_name, "strategy": optimization_strategy},
            },
        )


@catch_errors
def archive_old_memories(
    table_name: str,
    archive_days: int = 365,
    archive_table_suffix: str = "_archive",
    delete_after_archive: bool = False,
) -> ToolResponse:
    """
    📦 **MEMORY ARCHIVING** - Archive old memories to reduce active storage!

    Moves old memories to archive tables to keep active memory bank lean while
    preserving historical data for compliance and analysis.

    Args:
        table_name (str): Source table to archive from
        archive_days (int): Archive memories older than this many days (default: 365)
        archive_table_suffix (str): Suffix for archive table name (default: "_archive")
        delete_after_archive (bool): Delete from source after archiving (default: False)

    Returns:
        ToolResponse: {"success": True, "archived": int, "archive_table": str}
    """
    db_path = os.environ.get("DB_PATH", "./test.db")
    db = get_database(db_path)

    archive_table_name = f"{table_name}{archive_table_suffix}"
    cutoff_date = (datetime.now() - timedelta(days=archive_days)).isoformat()

    try:
        with db.engine.connect() as conn:
            # Check if source table has timestamp column
            schema_result = conn.execute(text(f"PRAGMA table_info(`{table_name}`)"))
            columns = [row[1] for row in schema_result.fetchall()]

            if "timestamp" not in columns:
                return cast(
                    ToolResponse,
                    {
                        "success": False,
                        "error": "Table does not have a timestamp column for archiving",
                        "category": "ARCHIVE_ERROR",
                        "details": {"table": table_name, "available_columns": columns},
                    },
                )

            # Create archive table if it doesn't exist
            create_archive_sql = f"""
            CREATE TABLE IF NOT EXISTS `{archive_table_name}` AS
            SELECT * FROM `{table_name}` WHERE 0=1
            """
            conn.execute(text(create_archive_sql))

            # Find records to archive
            count_result = conn.execute(
                text(f"SELECT COUNT(*) FROM `{table_name}` WHERE timestamp < :cutoff_date"),
                {"cutoff_date": cutoff_date},
            )
            count_row = count_result.fetchone()
            records_to_archive = count_row[0] if count_row else 0

            if records_to_archive == 0:
                return cast(
                    ToolResponse,
                    {
                        "success": True,
                        "archived": 0,
                        "archive_table": archive_table_name,
                        "message": f"No records older than {archive_days} days found",
                    },
                )

            # Begin transaction for atomic archiving
            trans = conn.begin()
            try:
                # Copy old records to archive table
                insert_archive_sql = f"""
                INSERT INTO `{archive_table_name}`
                SELECT * FROM `{table_name}` WHERE timestamp < :cutoff_date
                """
                conn.execute(text(insert_archive_sql), {"cutoff_date": cutoff_date})

                # Optionally delete from source table
                if delete_after_archive:
                    delete_sql = f"DELETE FROM `{table_name}` WHERE timestamp < :cutoff_date"
                    conn.execute(text(delete_sql), {"cutoff_date": cutoff_date})

                trans.commit()

            except Exception as e:
                trans.rollback()
                raise e

        return cast(
            ToolResponse,
            {
                "success": True,
                "archived": records_to_archive,
                "archive_table": archive_table_name,
                "cutoff_date": cutoff_date,
                "deleted_from_source": delete_after_archive,
                "archive_days": archive_days,
                "recommendations": [
                    f"Archived {records_to_archive} records older than {archive_days} days",
                    f"Archive table '{archive_table_name}' created/updated",
                    ("Consider setting up automated archiving for ongoing maintenance" if records_to_archive > 100 else None),
                ],
            },
        )

    except Exception as e:
        return cast(
            ToolResponse,
            {
                "success": False,
                "error": f"Memory archiving failed: {str(e)}",
                "category": "ARCHIVE_ERROR",
                "details": {
                    "table": table_name,
                    "archive_table": archive_table_name,
                    "archive_days": archive_days,
                },
            },
        )


def _generate_cleanup_recommendations(
    duplicate_groups: List[Dict[str, Any]],
) -> List[str]:
    """Generate actionable cleanup recommendations based on duplicate analysis."""
    recommendations = []

    if not duplicate_groups:
        recommendations.append("✅ No duplicates found - memory bank is clean")
        return recommendations

    total_duplicates = sum(len(group["rows"]) - 1 for group in duplicate_groups)
    recommendations.append(f"🔍 Found {len(duplicate_groups)} duplicate groups with {total_duplicates} redundant records")

    # Group recommendations by suggested action
    actions = defaultdict(int)
    for group in duplicate_groups:
        actions[group["suggested_action"]] += len(group["rows"]) - 1

    for action, count in actions.items():
        if action == "keep_newest":
            recommendations.append(f"📅 {count} duplicates can be auto-removed (keeping newest versions)")
        elif action == "manual_review":
            recommendations.append(f"👁️ {count} duplicates require manual review")

    if total_duplicates > 50:
        recommendations.append("⚠️ High duplicate count suggests need for automated deduplication process")

    return recommendations


def _generate_optimization_recommendations(optimizations: List[Dict[str, Any]], strategy: str) -> List[str]:
    """Generate optimization recommendations based on analysis results."""
    recommendations = []

    if not optimizations:
        recommendations.append("✅ Memory bank is already well-optimized")
        return recommendations

    # Analyze optimization types
    has_duplicates = any(opt["type"] == "duplicate_removal" for opt in optimizations)
    has_old_data = any(opt["type"] == "archive_old_entries" for opt in optimizations)

    if has_duplicates:
        recommendations.append("🔄 Run optimize_memory_bank() with dry_run=False to remove duplicates")

    if has_old_data:
        recommendations.append("📦 Consider using archive_old_memories() to move historical data")

    if strategy == "conservative":
        recommendations.append("💡 Try 'comprehensive' strategy for more aggressive optimization")

    recommendations.append("🔧 Schedule regular optimization (monthly) for enterprise deployments")

    return recommendations


# Export the implementation functions for internal use
_find_duplicates_impl = find_duplicates
_optimize_memory_bank_impl = optimize_memory_bank
_archive_old_memories_impl = archive_old_memories
