# AUTOGENERATED! DO NOT EDIT! File to edit: InteractivePlots.ipynb (unless otherwise specified).

__all__ = ['rgb_to_plotly', 'plotly_to_html', 'plotly_rgb_lines', 'plotly_dos_lines', 'iplotfromtxt']

# Cell
def rgb_to_plotly(rgb_lines=None,mode='markers',max_width=5,showlegend=False,NBANDS=None,name='',labels=['s','p','d']):
    """
    - Returns data object of plotly's figure using `create_rgb_lines`. Returned data could be fed to a plolty's figure.
    - ** Parameters**
        - rgb_lines   : output of `create_rgb_lines`.
        - mode        : Three plotting modes are available:
            - 'markers' : Plot whole data as a single scatter object. Its too fast.
            - 'bands'   : Plot data such that each band is accessible via legend.
            - 'lines'   : A replica of `matplotlib LineCollection` object. It plots at each point separately, slower than other two modes.
        - NBANDS     : Only required when `mode='bands'` to reshape data.
        - max_width  : Line/Scatter thickness is scaled to `max_width`.
        - name       : Name to be shown on hover text or legend.
        - labels     : Optional, show red green blue colors corresponding orbitals.
        - showlegend : Optional, only suitbale if spin up/down or 'bands' mode is ON.

    """
    if mode not in ('markers','bands','lines'):
        raise TypeError("Argument `mode` expects one of ['markers','bands','lines'], got '{}'.".format(mode))
        return
    if(mode=='bands' and NBANDS==None):
        return print("For `mode = 'bands'`, NBANDS must be provided.")
    if(rgb_lines):
        import plotly.graph_objects as go
        import numpy as np
        l,a,c=rgb_lines # Object is line collections, line arrays and colors (un-normalized).
        linear_y = a[:,0,1]
        linear_x = a[:,0,0]
        if(NBANDS!=None):
            stop,step=len(linear_x),int(len(linear_x)/NBANDS)
            linear_name=[]
            for i in range(0,stop,step):
                _name=["<sub>{}</sub>".format(int(1+i/step)) for j in range(step)]
                linear_name.append(_name)
            linear_name=list(np.reshape(linear_name,(-1)))

        clrs=np.array([255*i/np.max(i) if np.max(i)!=0 else 255*i for i in c])
        _c=[[int(i) for i in j] for j in clrs]
        _txt=[[int(100*i/255) for i in j] for j in clrs]
        colors=['rgb({},{},{})'.format(*i) for i in _c]
        txts=['RGB({},{},{})'.format(*i) for i in _txt]
        lws=[0.1+max_width*i.sum() for i in c]
        h_template = "Orbs: {} <br>{} </br>Band: {}{}"
        h_text=[h_template.format([*labels],t,name,l) for t,l in zip(txts,linear_name)]
        data=[]
        if(mode=='lines'):
            for k in range(len(a)):
                arr=a[k].transpose()
                data.append(go.Scatter(x=arr[0],y=arr[1],name=name,
                mode="lines",line=dict(color=colors[k], width=lws[k]),
                showlegend=showlegend,hovertext=h_text[k]
                )
            )
        if(mode=='markers'):
            x=linear_x
            y=linear_y
            data.append(go.Scatter(x=x,y=y,mode='markers', name=name,
            marker=dict(color=colors,size=lws),line=dict(width=0.001,color='rgba(255,255,250,0)'),showlegend=showlegend,hovertext=h_text))
        if(mode=='bands'):
            nkpts=int(len(linear_x)/NBANDS)
            x1=linear_x[:nkpts]
            y1=np.reshape(linear_y,(-1,nkpts))
            c1=np.reshape(colors,(-1,nkpts))
            l1=np.reshape(lws,(-1,nkpts))
            t1=np.reshape(h_text,(-1,nkpts))
            for i,en,c,w,t in zip(range(1,len(y1)+1),y1,c1,l1,t1):
                data.append(go.Scatter(x=x1,y=en,mode='markers+lines', name="{}<sub>{}</sub>".format(name,str(i)),
                marker=dict(color=c,size=w),line=dict(width=0.001,color='rgba(255,255,250,0)'),showlegend=showlegend,
                hovertext=t)
                )

        return data

# Cell
def plotly_to_html(fig,filename='new_plot.html'):
    """
    - Writes plotly's figure as HTML file which is accessible when online. If you need to have offline working file, just use `fig.write_html('file.html')` which will be larger in size.
    - **Parameters**
        - fig      : A plotly's figure object.
        - filename : name of file to save fig.
    """

    fig_json = fig.to_json()
    # a simple HTML template
    template = """<html>
    <head>
        <script src="https://cdn.plot.ly/plotly-latest.min.js"></script>
    </head>
    <body>
        <div id='graph'></div>
        <script>
            var fig_data = {}
            Plotly.react('graph', fig_data.data, fig_data.layout);
        </script>
    </body>

    </html>"""

    # write the JSON to the HTML template
    with open(filename, 'w') as f:
        f.write(template.format(fig_json))

# Cell
def plotly_rgb_lines(path_evr    = None,
                    ions        = [],
                    orbs        = [[],[],[]],
                    orblabels   = ['','',''],
                    mode        = 'markers',
                    elim        = [],
                    E_Fermi     = None,
                    skipk       = None,
                    joinPathAt  = [],
                    max_width   = 5,
                    title        = None,
                    xt_indices  = [0,-1],
                    xt_labels   = ['Γ','M'],
                    figsize     = None,
                    interpolate = False,
                    n           = 5,
                    k           = 3

    ):
    """
    - Returns plotly's figure object, takes care of spin-polarized calculations automatically.
    - **Parameters**
        - path_ever  : Path/to/vasprun.xml or xml output of `read_asxml`.
        - ions       : List of indices of ions. If not given, picks all ions.
        - orbs       : List of size 3 of list of orbital indices, if not gievn, s,p,d plotted.
        - orblabels  : List of labels for orbs.
        - mode       : Three plotting modes are available:
            - 'markers' : Plot whole data as a single scatter object. Its too fast.
            - 'bands'   : Plot data such that each band is accessible via legend.
            - 'lines'   : A replica of `matplotlib LineCollection` object. It plots at each point separately, slower than other two modes.
        - **kwargs      : interpolate, ticks, figsize,elim,joinPathAt,max_width,title etc.
    """
    import numpy as np
    import pivotpy.vr_parser as vp
    import pivotpy.s_plots as sp
    import pivotpy.i_plots as ip
    import plotly.graph_objects as go
    if mode not in ('markers','bands','lines'):
        raise TypeError("Argument `mode` expects one of ['markers','bands','lines'], got '{}'.".format(mode))
        return
    if(len(orbs)<3):
        raise ValueError("orbs have structure [[],[],[]], do not reduce structure even if it is empty.")
        return
    #checking type of given path.
    if(path_evr==None):
        vr=vp.export_vasprun(path=path_evr,skipk=skipk,elim=elim,joinPathAt=joinPathAt)
    if(path_evr!=None):
        from os import path as pt
        if(type(path_evr)==vp.Dic2Dot):
            vr=path_evr
        elif(pt.isfile(path_evr)):
            vr=vp.export_vasprun(path=path_evr,skipk=skipk,elim=elim,joinPathAt=joinPathAt)
        else:
            return print("path_evr = `{}` does not exist".format(path_evr))
    # Apply a robust final check.
    vr_keys=['kpath','bands']
    for key in vr_keys:
        if(key not in vr.keys()):
            return print("Object: \n{} \nis like a lower tree of export_vasprun(). Expects top tree.".format(vr))
    else:
        ## Main working here.
        if(vr.pro_bands==None):
            import pivotpy.g_utils as gu
            gu.printy("Can not plot an empty eigenvalues object.")
            return gu.printg("Try with large energy range.")
        #=====================================================
        orbs=[[item] if type(item)==int else item for item in orbs] #Fix if integer given.
        elem_inds = vr.sys_info.ElemIndex
        max_ind   = elem_inds[-1]-1 # Last index is used for range in ElemIndex, not python index.
        labels=orblabels
        nfields=len(vr.pro_bands.labels)
        # If orbs not given, select all ions.
        if(ions==[]):
            ions=list(range(0,max_ind+1))
        _orb_inds = [p for orb in orbs for p in orb]
        for ind in _orb_inds:
            if(ind > nfields-1):
                return print("index {} is out of bound for {} ions.".format(ind,max_ind+1))

        if(E_Fermi==None):
            E_Fermi=vr.bands.E_Fermi
        K=vr.kpath
        xticks=[K[i] for i in xt_indices]
        xlim=[min(K),max(K)]
        if(elim):
            ylim=[min(elim),max(elim)]
        else:
            ylim=[-10,10]

        # If orbs not given, get whole projections.
        if(_orb_inds==[]):
            labels=['s','p','d']
            if(nfields==3):
                orbs=[[0],[1],[2]]
            if(nfields==9 or nfields==16):
                orbs=[[0],[1,2,3],[4,5,6,7,8]]
        #====Title Name======
        SYSTEM=vr.sys_info.SYSTEM
        E_INDS=vr.sys_info.ElemIndex
        if(ions[-1]< E_INDS[-1]):
            txt="{}[Ions: ({}), Orbs: {}]".format(SYSTEM,ions[0]+1,ions[-1]+1,','.join(labels))
        if(len(ions)==1):
            txt="{}[Ion: {}, Orbs: {}]".format(SYSTEM,ions[0]+1,','.join(labels))
        if(len(ions)== E_INDS[-1]):
            txt="{}[Ions: {}, Orbs: {}]".format(SYSTEM,E_INDS[-1],','.join(labels))
        if(title==None):
            title=txt

        # After All Fixing
        ISPIN=vr.sys_info.ISPIN
        args_dict=dict(uni_color=False,ions=ions,orbs=orbs,interpolate=interpolate,n=n,k=k) # Do not scale color there, scale here.
        data,showlegend,NBANDS,name=[],False,None,'' # Place holder
        if(mode=='bands'):
                showlegend=True
        if(ISPIN==1):
            En=vr.bands.evals-E_Fermi
            NBANDS=np.shape(En)[1]
            Pros=vr.pro_bands.pros
            new_args=dict(kpath=K, evals_set=En, pros_set=Pros,**args_dict)
            rgb_lines=sp.create_rgb_lines(**new_args)
            data=ip.rgb_to_plotly(rgb_lines=rgb_lines,mode=mode,NBANDS=NBANDS,showlegend=showlegend,labels=labels,name='B',max_width=max_width)
        if(ISPIN==2):
            if(mode=='markers'):
                showlegend=True
            En1=vr.bands.evals.SpinUp-E_Fermi
            En2=vr.bands.evals.SpinDown-E_Fermi
            NBANDS1=np.shape(En1)[1]
            NBANDS2=np.shape(En2)[1]
            Pros1=vr.pro_bands.pros.SpinUp
            Pros2=vr.pro_bands.pros.SpinDown
            new_args1=dict(kpath=K, evals_set=En1, pros_set=Pros1,**args_dict)
            rgb_lines1=sp.create_rgb_lines(**new_args1)
            data1=ip.rgb_to_plotly(rgb_lines=rgb_lines1,mode=mode,NBANDS=NBANDS1,showlegend=showlegend,labels=labels,name='B<sup>↑</sup>',max_width=max_width)
            new_args2=dict(kpath=K, evals_set=En2, pros_set=Pros2,**args_dict)
            rgb_lines2=sp.create_rgb_lines(**new_args2)
            data2=ip.rgb_to_plotly(rgb_lines=rgb_lines2,mode=mode,NBANDS=NBANDS2,showlegend=showlegend,labels=labels,name='B<sup>↓</sup>',max_width=max_width)
            data=[[d1,d2] for d1,d2 in zip(data1,data2)]
            data=[d for ds in data for d in ds]
        # Initiate figure
        fig=go.Figure(data=data)
        fig.update_layout(title=title,
            margin=go.layout.Margin(l=60,r=50,b=40,t=75,pad=0),#paper_bgcolor="whitesmoke",
            yaxis=go.layout.YAxis(title_text='Energy (eV)',range=ylim),
            xaxis=go.layout.XAxis(ticktext=xt_labels, tickvals=xticks,
            tickmode="array",range=xlim),font=dict(family="stix, serif",size=14))
        if(figsize!=None):
            fig.update_layout(width=figsize[0],height=figsize[1],autosize=False)
        #Draw lines at breakpoints
        if(joinPathAt):
            for pt in joinPathAt:
                fig.add_trace(go.Scatter(x=[K[pt],K[pt]],y=ylim,mode='lines',line=dict(color='rgb(0,0,0)',width=2),showlegend=False))
                fig.add_trace(go.Scatter(x=[K[pt],K[pt]],y=ylim,mode='lines',line=dict(color='rgb(222,222,222)',width=1.2),showlegend=False))
        fig.update_xaxes(showgrid=True, zeroline=False,showline=True, linewidth=0.1, linecolor='white', mirror=True)
        fig.update_yaxes(showgrid=False, zeroline=True,showline=True, linewidth=0.1, linecolor='white', mirror=True)
        return fig

# Cell
def plotly_dos_lines(path_evr     = None,
                    elim          = [],
                    ions          = [0,],
                    orbs          = [[0],],
                    orblabels     = ['s',],
                    colors        = None,
                    tdos_color    = (0.5,0.95,0),
                    linewidth     = 2,
                    fill_area     = True,
                    vertical      = False,
                    E_Fermi       = None,
                    figsize       = None,
                    spin          = 'both',
                    interpolate   = False,
                    n             = 5,
                    k             = 3,
                    title         = None,
                    ):
        """
        - Returns ax object (if ax!=False) and plot on which all matplotlib allowed actions could be performed, returns lists of energy,tdos and pdos and labels. If given,elements,orbs colors, and orblabels must have same length. If not given, zeroth ions is plotted with s-orbital.
        - **Parameters**)
            - path_evr   : Path/to/vasprun.xml or output of `export_vasprun`. Auto picks in CWD.
            - elim       : [min,max] of energy range.
            - E_Fermi    : If not given, automatically picked from `export_vasprun`.
            - ions       : List [] of ions indices, by defualt plot first ion's projections.
            - orbs       : List [[0],] lists of indices of orbitals, could be empty.
            - orblabels  : List [str,] of orbitals labels. len(orblabels)==len(orbs) must hold.
            - colors     : Defualt matplotlib's `gist_rainbow` colormap. List of orbitals colors {str,rgb,rgba}, len(colors)==len(orbs) must hold. Use matplotlib colors values, it will be changed to plotly's colors automatically.
            - fill_area  : Default is True and plots filled area for dos. If False, plots lines only.
            - vertical   : False, If True, plots along y-axis.
            - figsize    : Tuple in pixels (width,height).
            - interpolate: Default is False, if True, bands are interpolated.
            - n          : int, number of points, default is 5.
            - k          : int, order of interpolation 0,1,2,3. Defualt 3. `n > k` should be hold.
            - legend_kwargs: Dictionary to contain legend arguments to fix.
        - **Returns**
            - fig        : Plotly's figure object.
        """
        import pivotpy.s_plots as sp
        import numpy as np
        import pivotpy.g_utils as gu
        import matplotlib as mpl
        import plotly.graph_objects as go
        elements=[ions for orb in orbs] # Creates equal length ions set for elements.
        en,tdos,pdos,labels,vr=None,None,None,None,None # Place holders for defining
        cl_dos=sp.collect_dos(path_evr=path_evr,elim=elim, elements=elements, orbs=orbs,\
                              orblabels=orblabels, E_Fermi=E_Fermi, spin='both', interpolate=interpolate, n=n, k=k)
        try:
            en,tdos,pdos,labels,vr=cl_dos
        except TypeError:
            import pivotpy.g_utils as gu
            return gu.printg("Try with large energy range.")

        labels=[label.replace('$','').replace('^↑','<sup>↑</sup>').replace('^↓','<sup>↓</sup>') for label in labels]
        # Make additional colors for spin down. Inverted colors are better.
        if(elim):
            ylim=[min(elim),max(elim)]
        else:
            ylim=[-10,10]
        # Fix elements and colors length
        if colors == None:
            from matplotlib.pyplot import cm
            if len(tdos)==2:
                colors  = cm.gist_rainbow(np.linspace(0,1,2*len(orbs)))
            else:
                colors  = cm.gist_rainbow(np.linspace(0,1,len(orbs)))
        elif np.shape(orbs)[0]!=len(colors):
                raise ValueError("colors expect same length as orbs, valid matplotlib colors.")
                return
        else:
            clrs1=[mpl.colors.to_rgb((c)) for c in colors]
            clrs2=[gu.invert_color(color=mpl.colors.to_rgb((c))) for c in colors]
            colors=[*clrs1,*clrs2]
        # Total DOS colors
        t_color=mpl.colors.to_rgb(tdos_color)
        it_color=gu.invert_color(color=t_color)
        #========Title Name========
        SYSTEM=vr.sys_info.SYSTEM
        E_INDS=vr.sys_info.ElemIndex
        if(ions[-1]< E_INDS[-1]):
            txt="{}[Ions: ({}), Orbs: {}]".format(SYSTEM,ions[0]+1,ions[-1]+1,','.join(labels))
        if(len(ions)==1):
            txt="{}[Ion: {}, Orbs: {}]".format(SYSTEM,ions[0]+1,','.join(labels))
        if(len(ions)== E_INDS[-1]):
            txt="{}[Ions: {}, Orbs: {}]".format(SYSTEM,E_INDS[-1],','.join(labels))
        if(title==None):
            title=txt
        fig = go.Figure()
        fig.update_layout(title=title,margin=go.layout.Margin(l=60,r=50,b=40,t=75,pad=0),\
                          font=dict(family="stix, serif",size=14))
        if(figsize!=None):
            fig.update_layout(width=figsize[0],height=figsize[1],autosize=False)
        if(vertical==False):
            if(fill_area==False):
                fill=None
            if(fill_area==True):
                fill='tozeroy'
            args_dic=dict(mode='lines',line_width=linewidth,fill=fill)
            fig.update_xaxes(range=ylim,title='Energy (eV)')
            if(len(tdos)==2):   # Spin polarized.
                fig.add_scatter(x=en,y=tdos[0],line_color='rgb({},{},{})'.format(*[int(255*i) for i in t_color]),\
                                 name='TDOS<sup>↑</sup>',**args_dic)
                fig.add_scatter(x=en,y=tdos[1],line_color='rgb({},{},{})'.format(*[int(255*i) for i in it_color]),\
                                 name='TDOS<sup>↓</sup>',**args_dic)
            else:   # unpolarized.
                fig.add_trace(go.Scatter(x=en,y=tdos,line_color='rgb({},{},{})'.format(*[int(255*i) for i in t_color]),\
                              name='TDOS',**args_dic))
            for p,l,c in zip(pdos,labels,colors):
                fig.add_trace(go.Scatter(x=en,y=p,line_color='rgb({},{},{})'.format(*[int(255*i) for i in c]),\
                               name=l,**args_dic))
        if(vertical==True):
            if(fill_area==False):
                fill=None
            if(fill_area==True):
                fill='tozerox'
            args_dic=dict(mode='lines',line_width=linewidth,fill=fill)
            fig.update_yaxes(range=ylim,title='Energy (eV)')
            if(len(tdos)==2):   # Spin polarized.
                fig.add_scatter(y=en,x=tdos[0],line_color='rgb({},{},{})'.format(*[int(255*i) for i in t_color]),\
                                name='TDOS<sup>↑</sup>',**args_dic)
                fig.add_scatter(y=en,x=tdos[1],line_color='rgb({},{},{})'.format(*[int(255*i) for i in it_color]),\
                                name='TDOS<sup>↓</sup>',**args_dic)
            else:   # unpolarized.
                fig.add_trace(go.Scatter(y=en,x=tdos,line_color='rgb({},{},{})'.format(*[int(255*i) for i in t_color]),\
                                name='TDOS',**args_dic))
            for p,l,c in zip(pdos,labels,colors):
                fig.add_trace(go.Scatter(y=en,x=p,line_color='rgb({},{},{})'.format(*[int(255*i) for i in c]),\
                                name=l,**args_dic))
        fig.update_xaxes(showgrid=True, zeroline=True,showline=True, linewidth=0.1, linecolor='white', mirror=True)
        fig.update_yaxes(showgrid=True, zeroline=True,showline=True, linewidth=0.1, linecolor='white', mirror=True)
        return fig

# Cell
def iplotfromtxt(path_to_dir = '.', ions = [0], orbs = [[0],[],[]], orblabels = ['s','',''], elim = [-5,5], tick_indices= [0,-1],tick_labels=[u'\u0393','M'],force_load = False):
    """
    - Returns plotly's figure object, Not applicable for spin-polarized calculations. It is just to keep as it was starting code. Remember where you did start!
    - **Parameters**
        - path_to_dir: Path/to/directory where `Export-VaspRun`'s output from powershell is present.
        - ions       : List of indices of ions. If not given, picks all ions.
        - orbs       : List of size 3 of list of orbital indices, if not gievn, s,p,d plotted.
        - orblabels  : List of labels for orbs.
        - elim       : Energy limit.
        - force_load : If True, it will run powershell to generate text files only if files are not already present.
        - **kwargs   : tick_indices, tick_labels, elim etc.
    """
    import numpy as np
    import copy
    import plotly.graph_objects as go
    E_Limit = elim
    ProIndices = [ions,*orbs]
    ProLabels = ['',*orblabels]
    tickIndices = tick_indices
    ticklabels = tick_labels
    #====================Loading Files===================================
    import os
    loc = os.getcwd()
    os.chdir(path_to_dir)
    if force_load == True:
        check = os.path.isfile('./Bands.txt') and os.path.isfile('Projection.txt') and os.path.isfile('SysInfo.py')
        if check is False:
            import pivotpy.g_utils as gu
            gu.ps_to_std(ps_command = "Import-Module Vasp2Visual ; Export-VR")
    import importlib
    try:
        si = importlib.reload(SysInfo)
    except:
        import SysInfo as si
    os.chdir(loc)
    #=====================Loaded===========================================
    SYSTEM, NKPTS, NBANDS, NFILLED, TypeION,E_Fermi = [si.SYSTEM, si.NKPTS, si.NBANDS, si.NFILLED, si.TypeION, si.E_Fermi]
    nField_Projection = si.nField_Projection
    ElemIndex = si.ElemIndex
    ElemName = si.ElemName
    proFile = os.path.join(path_to_dir,'Projection.txt')
    bFile   = os.path.join(path_to_dir,'Bands.txt')
    data=np.loadtxt(proFile)
    KE=np.loadtxt(bFile)
    K=KE[:,3]; E=KE[:,4:]-E_Fermi; #Seperate KPOINTS and Eigenvalues in memory
    yh=max(E_Limit);yl=min(E_Limit);
    #============Calculation of ION(s)-Contribution=======
    #Get (R,G.B) values from projection and Normlalize in plot range
    maxEnergy=np.min(E,axis=0); minEnergy=np.max(E,axis=0); #Gets bands in visible energy limits.
    max_E=np.max(np.where(maxEnergy <=yh)); min_E=np.min(np.where(minEnergy >=yl))

    r_data=np.reshape(data,(-1,NKPTS,NBANDS,nField_Projection))
    s_data=np.take(r_data[:,:,min_E:max_E+1,:],ProIndices[0],axis=0).sum(axis=0)
    red=np.take(s_data,ProIndices[1],axis=2).sum(axis=2)
    green=np.take(s_data,ProIndices[2],axis=2).sum(axis=2)
    blue=np.take(s_data,ProIndices[3],axis=2).sum(axis=2)
    max_con=max(max(map(max,red[:,:])),max(map(max,green[:,:])),max(map(max,blue[:,:])))
    red=red[:,:]/max_con;green=green[:,:]/max_con;blue=blue[:,:]/max_con #Values are ready in E_Limit
    E=E[:,min_E:max_E+1]; #Updated energy in E_limit
    #===============Make Collections======================
    text_plotly=[[str(ProLabels[1:])+'<<'+'RGB'+str((int(100*red[i,j]),int(100*green[i,j]),int(100*blue[i,j]))) for i in range(NKPTS)] for j in range (np.shape(E)[1])];
    rgb_plotly=[['rgb'+str((int(255*red[i,j]),int(255*green[i,j]),int(255*blue[i,j]))) for i in range(NKPTS)] for j in range (np.shape(E)[1])];
    lw_plotly=[[np.round(1+50*(red[i,j]+green[i,j]+blue[i,j])/3,4) for i in range(NKPTS)] for j in range (np.shape(E)[1])]; # 1 as residual width
    tick_plotly=[K[tickIndices[i]] for i in range(len(tickIndices))]; tlab_plotly=ticklabels;
    #=================Plotting============================
    fig = go.Figure()
    for i in range(np.shape(E)[1]):
        fig.add_trace(go.Scatter(x=K,y=E[:,i],mode='lines', line=dict(color='rgb(20,24,222)',width=2),name='Band '+str(i+1)))
    for i in range(np.shape(E)[1]):
        fig.add_trace(go.Scatter(x=K,y=E[:,i],mode='markers+lines', visible=False,hovertext=text_plotly[:][i],
                    marker=dict(size=lw_plotly[:][i], color=rgb_plotly[:][i]) ,
                    line=dict(color='rgba(100,100,20,0)',width=0.1),name='Band '+str(i+1)))
    #====Title Name======
    if(ProIndices[0][-1]< ElemIndex[-1]):
        title=SYSTEM+'['+ProLabels[0]+': '+str(ProIndices[0][0]+1)+'-'+str(ProIndices[0][-1]+1)+']'
    if(ProIndices[0][-1]== 0):
        title=SYSTEM+'['+ProLabels[0]+': '+str(ProIndices[0][0]+1)+']'
    if(np.shape(ProIndices[0])[0]==1):
        title=SYSTEM+'['+ProLabels[0]+': '+str(ProIndices[0][0]+1)+']'
    if(ProIndices[0][-1]+1== ElemIndex[-1]):
        title=SYSTEM+'[All Sites]'
    fig.update_layout(title=title,autosize=False, width=400,height=320,
            margin=go.layout.Margin(l=60,r=50,b=40,t=90,pad=0),paper_bgcolor="whitesmoke",
            yaxis=go.layout.YAxis(title_text='E-E<sub>F</sub>',range=[yl,yh]),
            xaxis=go.layout.XAxis(ticktext=ticklabels, tickvals=tick_plotly,
            tickmode="array",range=[K[0],K[-1]]),font=dict(family="stix, serif",size=14))
    #========Update Buttons==============
    simple,projected=[],[];
    for j in range(2*np.shape(E)[1]):
        if(j>=np.shape(E)[1]):
            simple.append(False)
            projected.append(True)
        if(j<np.shape(E)[1]):
            simple.append(True)
            projected.append(False)

    fig.update_layout(updatemenus=[go.layout.Updatemenu(
                type="buttons",direction="right", active=0,x=1,y=1.2,
                buttons=list([
                dict(label="Simple", method="update", args=[{"visible": simple}]),
                dict(label="Projected", method="update", args=[{"visible": projected}])
            ]),) ])
    fig.update_xaxes(showgrid=True, zeroline=False,showline=True, linewidth=0.1, linecolor='gray', mirror=True)
    fig.update_yaxes(showgrid=False, zeroline=True,showline=True, linewidth=0.1, linecolor='gray', mirror=True)
    return fig