import hashlib
from typing import List

from pydantic import parse_obj_as

from ..api.v1.hazard_data import HazardModel, Period


class Inventory:
    """Contains an inventory of available hazard data.
    model id is given by {event_type}/{model group identifier}/{version}/{model identifier}
    """

    def __init__(self):
        osc_chronic_heat_models = [
            {
                "event_type": "ChronicHeat",
                "path": "",
                "id": "mean_degree_days/above/32c",
                "display_name": "Mean degree days above 32C",
                "description": "Description here!",
                "filename": "mean_degree_days_above_32c_{scenario}_{year}",
                "map": {"colormap": "mean_degree_days/above/32c"},
                "units": "degree days",
                "scenarios": [
                    {"id": "ssp585", "years": [2030, 2040, 2050]},
                    {"id": "historical", "years": [1980]},
                ],
            }
        ]

        wri_riverine_inundation_models = [
            {
                "event_type": "RiverineInundation",
                "path": "riverine_inundation/wri/v2",
                "id": "000000000WATCH",
                "display_name": "WRI/Baseline",
                "description": "World Resources Institute Aqueduct flood model baseline (historical)",
                "filename": "inunriver_{scenario}_{id}_{year}",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inunriver_{scenario}_{id}_{year}_rp{return_period:05d}",
                },
                "units": "metres",
                "scenarios": [{"id": "historical", "years": [1980], "periods": [{"year": 1980, "map_id": "gw4vgq"}]}],
            },
            {
                "event_type": "RiverineInundation",
                "path": "riverine_inundation/wri/v2",
                "id": "00000NorESM1-M",
                "display_name": "WRI/NorESM1-M",
                "description": "World Resources Institute Aqueduct flood model with GCM model: Bjerknes Centre for Climate Research, Norwegian Meteorological Institute",  # noqa: E501
                "filename": "inunriver_{scenario}_{id}_{year}",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inunriver_{scenario}_{id}_{year}_rp{return_period:05d}",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "RiverineInundation",
                "path": "riverine_inundation/wri/v2",
                "id": "0000GFDL-ESM2M",
                "display_name": "WRI/GFDL-ESM2M",
                "description": "World Resource Institute Aqueduct flood model with GCM model: Geophysical Fluid Dynamics Laboratory (NOAA)",  # noqa: E501
                "filename": "inunriver_{scenario}_{id}_{year}",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inunriver_{scenario}_{id}_{year}_rp{return_period:05d}",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "RiverineInundation",
                "path": "riverine_inundation/wri/v2",
                "id": "0000HadGEM2-ES",
                "display_name": "WRI/HadGEM2-ES",
                "description": "World Resource Institute Aqueduct flood model with GCM model: Met Office Hadley Centre",
                "filename": "inunriver_{scenario}_{id}_{year}",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inunriver_{scenario}_{id}_{year}_rp{return_period:05d}",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "RiverineInundation",
                "path": "riverine_inundation/wri/v2",
                "id": "00IPSL-CM5A-LR",
                "display_name": "WRI/IPSL-CM5A-LR",
                "description": "World Resource Institute Aqueduct flood model with GCM model: Institut Pierre Simon Laplace",  # noqa: E501
                "filename": "inunriver_{scenario}_{id}_{year}",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inunriver_{scenario}_{id}_{year}_rp{return_period:05d}",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "RiverineInundation",
                "path": "riverine_inundation/wri/v2",
                "id": "MIROC-ESM-CHEM",
                "display_name": "WRI/MIROC-ESM-CHEM",
                "description": """World Resource Institute Aqueduct flood model with
 GCM model: Atmosphere and Ocean Research Institute
 (The University of Tokyo), National Institute for Environmental Studies, and Japan Agency
 for Marine-Earth Science and Technology""",
                "filename": "inunriver_{scenario}_{id}_{year}",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inunriver_{scenario}_{id}_{year}_rp{return_period:05d}",
                },
                "units": "metres",
                "scenarios": [
                    {
                        "id": "rcp4p5",
                        "years": [2030, 2050, 2080],
                        "periods": [
                            {"year": 2030, "map_id": "ht2kn3"},
                            {"year": 2050, "map_id": "1k4boi"},
                            {"year": 2080, "map_id": "3rok7b"},
                        ],
                    },
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
        ]

        wri_coastal_inundation_models = [
            {
                "event_type": "CoastalInundation",
                "path": "coastal_inundation/wri/v2",
                "id": "nosub",
                "display_name": "WRI/Baseline no subsidence",
                "description": "World Resource Institute Aqueduct flood model baseline (historical); no subsidence",
                "filename": "inuncoast_historical_nosub_hist_0",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inuncoast_historical_nosub_hist_rp{return_period:04d}_0",
                },
                "units": "metres",
                "scenarios": [{"id": "historical", "years": [1980]}],
            },
            {
                "event_type": "CoastalInundation",
                "path": "coastal_inundation/wri/v2",
                "id": "nosub/95",
                "display_name": "WRI/95% no subsidence",
                "description": "World Resource Institute Aqueduct flood model with no subsidence; 95th percentile sea level rise",  # noqa: E501
                "filename": "inuncoast_{scenario}_nosub_{year}_0",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inuncoast_{scenario}_nosub_{year}_rp{return_period:04d}_0",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "CoastalInundation",
                "path": "coastal_inundation/wri/v2",
                "id": "nosub/5",
                "display_name": "WRI/5% no subsidence",
                "description": "World Resource Institute Aqueduct flood model with no subsidence; 5th percentile sea level rise",  # noqa: E501
                "filename": "inuncoast_{scenario}_nosub_{year}_0_perc_05",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inuncoast_{scenario}_nosub_{year}_rp{return_period:04d}_0_perc_05",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "CoastalInundation",
                "path": "coastal_inundation/wri/v2",
                "id": "nosub/50",
                "display_name": "WRI/50% no subsidence",
                "description": "World Resource Institute Aqueduct flood model with no subsidence; 50th percentile sea level rise",  # noqa: E501
                "filename": "inuncoast_{scenario}_nosub_{year}_0_perc_50",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inuncoast_{scenario}_nosub_{year}_rp{return_period:04d}_0_perc_50",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "CoastalInundation",
                "path": "coastal_inundation/wri/v2",
                "id": "wtsub",
                "display_name": "WRI/Baseline with subsidence",
                "description": "World Resource Institute Aqueduct flood model with baseline condition; with subsidence",  # noqa: E501
                "filename": "inuncoast_historical_wtsub_hist_0",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inuncoast_historical_wtsub_hist_rp{return_period:04d}_0",
                },
                "units": "metres",
                "scenarios": [{"id": "historical", "years": [1980]}],
            },
            {
                "event_type": "CoastalInundation",
                "path": "coastal_inundation/wri/v2",
                "id": "wtsub/95",
                "display_name": "WRI/95% with subsidence",
                "description": "World Resource Institute Aqueduct flood model with subsidence; 95th percentile sea level rise",  # noqa: E501
                "filename": "inuncoast_{scenario}_wtsub_{year}_0",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inuncoast_{scenario}_wtsub_{year}_rp{return_period:04d}_0",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "CoastalInundation",
                "path": "coastal_inundation/wri/v2",
                "id": "wtsub/5",
                "display_name": "WRI/5% with subsidence",
                "description": "World Resource Institute Aqueduct flood model with with subsidence; 5th percentile sea level rise",  # noqa: E501
                "filename": "inuncoast_{scenario}_wtsub_{year}_0_perc_05",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inuncoast_{scenario}_wtsub_{year}_rp{return_period:04d}_0_perc_05",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
            {
                "event_type": "CoastalInundation",
                "path": "coastal_inundation/wri/v2",
                "id": "wtsub/50",
                "display_name": "WRI/50% with subsidence",
                "description": "World Resource Institute Aqueduct flood model with subsidence; 50th percentile sea level rise",  # noqa: E501
                "filename": "inuncoast_{scenario}_wtsub_{year}_0_perc_50",
                "map": {
                    "colormap": "flare_intensity_0_2",
                    "filename": "inuncoast_{scenario}_wtsub_{year}_rp{return_period:04d}_0_perc_50",
                },
                "units": "metres",
                "scenarios": [
                    {"id": "rcp4p5", "years": [2030, 2050, 2080]},
                    {"id": "rcp8p5", "years": [2030, 2050, 2080]},
                ],
            },
        ]

        self.models = osc_chronic_heat_models + wri_riverine_inundation_models + wri_coastal_inundation_models

    def to_hazard_models(self) -> List[HazardModel]:
        models = parse_obj_as(List[HazardModel], self.models)

        # we populate map_id hashes programmatically
        for model in models:

            for scenario in model.scenarios:
                test_periods = scenario.periods
                scenario.periods = []
                for year in scenario.years:
                    name_format = (
                        model.filename if model.map is None or model.map.filename is None else model.map.filename
                    )
                    filename = name_format.format(scenario=scenario.id, year=year, id=model.id, return_period=1000)
                    id = alphanumeric(filename)[0:6]
                    scenario.periods.append(Period(year=year, map_id=id))
                # if a period was specifed explicitly, we check that hash is the same: a build-in check
                if test_periods is not None:
                    for (period, test_period) in zip(scenario.periods, test_periods):
                        if period.map_id != test_period.map_id:
                            raise Exception(
                                f"validation error: hash {period.map_id} different to specified hash {test_period.map_id}"  # noqa: E501
                            )

        return models

    def colormaps(self):
        """Color maps. Key can be identical to a model identifier or more descriptive (if shared by many models)."""
        return {
            "flare_intensity_0_2": {
                "colormap": {
                    "0": [255, 255, 255, 0],
                    "1": [255, 255, 255, 0],
                    "2": [236, 173, 127, 200],
                    "3": [236, 172, 126, 200],
                    "4": [236, 171, 126, 200],
                    "5": [236, 170, 125, 200],
                    "6": [236, 169, 124, 200],
                    "7": [236, 168, 124, 200],
                    "8": [236, 167, 123, 200],
                    "9": [236, 166, 123, 200],
                    "10": [236, 166, 122, 200],
                    "11": [236, 165, 121, 200],
                    "12": [235, 164, 121, 200],
                    "13": [235, 163, 120, 200],
                    "14": [235, 162, 120, 200],
                    "15": [235, 161, 119, 200],
                    "16": [235, 160, 118, 200],
                    "17": [235, 159, 118, 200],
                    "18": [235, 158, 117, 200],
                    "19": [235, 157, 117, 200],
                    "20": [235, 156, 116, 200],
                    "21": [235, 155, 116, 200],
                    "22": [235, 154, 115, 200],
                    "23": [234, 153, 114, 200],
                    "24": [234, 152, 114, 200],
                    "25": [234, 151, 113, 200],
                    "26": [234, 150, 113, 200],
                    "27": [234, 149, 112, 200],
                    "28": [234, 148, 111, 200],
                    "29": [234, 147, 111, 200],
                    "30": [234, 146, 110, 200],
                    "31": [234, 145, 110, 200],
                    "32": [233, 144, 109, 200],
                    "33": [233, 143, 109, 200],
                    "34": [233, 142, 108, 200],
                    "35": [233, 141, 107, 200],
                    "36": [233, 140, 107, 200],
                    "37": [233, 139, 106, 200],
                    "38": [233, 138, 106, 200],
                    "39": [233, 137, 105, 200],
                    "40": [232, 136, 105, 200],
                    "41": [232, 135, 104, 200],
                    "42": [232, 134, 103, 200],
                    "43": [232, 133, 103, 200],
                    "44": [232, 132, 102, 200],
                    "45": [232, 131, 102, 200],
                    "46": [232, 130, 101, 200],
                    "47": [232, 129, 101, 200],
                    "48": [231, 128, 100, 200],
                    "49": [231, 127, 100, 200],
                    "50": [231, 126, 99, 200],
                    "51": [231, 125, 99, 200],
                    "52": [231, 124, 98, 200],
                    "53": [231, 123, 98, 200],
                    "54": [230, 122, 98, 200],
                    "55": [230, 121, 97, 200],
                    "56": [230, 120, 97, 200],
                    "57": [230, 119, 96, 200],
                    "58": [230, 118, 96, 200],
                    "59": [229, 118, 96, 200],
                    "60": [229, 117, 95, 200],
                    "61": [229, 116, 95, 200],
                    "62": [229, 115, 95, 200],
                    "63": [229, 114, 94, 200],
                    "64": [228, 113, 94, 200],
                    "65": [228, 112, 94, 200],
                    "66": [228, 111, 93, 200],
                    "67": [228, 110, 93, 200],
                    "68": [227, 109, 93, 200],
                    "69": [227, 108, 93, 200],
                    "70": [227, 107, 92, 200],
                    "71": [227, 106, 92, 200],
                    "72": [226, 105, 92, 200],
                    "73": [226, 104, 92, 200],
                    "74": [226, 103, 92, 200],
                    "75": [225, 102, 92, 200],
                    "76": [225, 101, 91, 200],
                    "77": [225, 100, 91, 200],
                    "78": [224, 99, 91, 200],
                    "79": [224, 98, 91, 200],
                    "80": [224, 97, 91, 200],
                    "81": [223, 96, 91, 200],
                    "82": [223, 95, 91, 200],
                    "83": [223, 94, 91, 200],
                    "84": [222, 93, 91, 200],
                    "85": [222, 92, 91, 200],
                    "86": [221, 91, 91, 200],
                    "87": [221, 90, 91, 200],
                    "88": [221, 89, 91, 200],
                    "89": [220, 89, 92, 200],
                    "90": [220, 88, 92, 200],
                    "91": [219, 87, 92, 200],
                    "92": [219, 86, 92, 200],
                    "93": [218, 85, 92, 200],
                    "94": [218, 84, 92, 200],
                    "95": [217, 83, 93, 200],
                    "96": [217, 83, 93, 200],
                    "97": [216, 82, 93, 200],
                    "98": [215, 81, 93, 200],
                    "99": [215, 80, 94, 200],
                    "100": [214, 79, 94, 200],
                    "101": [214, 79, 94, 200],
                    "102": [213, 78, 94, 200],
                    "103": [212, 77, 95, 200],
                    "104": [212, 77, 95, 200],
                    "105": [211, 76, 95, 200],
                    "106": [211, 75, 96, 200],
                    "107": [210, 75, 96, 200],
                    "108": [209, 74, 96, 200],
                    "109": [208, 73, 97, 200],
                    "110": [208, 73, 97, 200],
                    "111": [207, 72, 97, 200],
                    "112": [206, 71, 98, 200],
                    "113": [206, 71, 98, 200],
                    "114": [205, 70, 98, 200],
                    "115": [204, 70, 99, 200],
                    "116": [203, 69, 99, 200],
                    "117": [202, 69, 99, 200],
                    "118": [202, 68, 100, 200],
                    "119": [201, 68, 100, 200],
                    "120": [200, 67, 101, 200],
                    "121": [199, 67, 101, 200],
                    "122": [198, 66, 101, 200],
                    "123": [197, 66, 102, 200],
                    "124": [197, 66, 102, 200],
                    "125": [196, 65, 102, 200],
                    "126": [195, 65, 103, 200],
                    "127": [194, 65, 103, 200],
                    "128": [193, 64, 103, 200],
                    "129": [192, 64, 104, 200],
                    "130": [191, 63, 104, 200],
                    "131": [190, 63, 104, 200],
                    "132": [189, 63, 104, 200],
                    "133": [188, 63, 105, 200],
                    "134": [187, 62, 105, 200],
                    "135": [186, 62, 105, 200],
                    "136": [186, 62, 105, 200],
                    "137": [185, 61, 106, 200],
                    "138": [184, 61, 106, 200],
                    "139": [183, 61, 106, 200],
                    "140": [182, 61, 106, 200],
                    "141": [181, 60, 107, 200],
                    "142": [180, 60, 107, 200],
                    "143": [179, 60, 107, 200],
                    "144": [178, 60, 107, 200],
                    "145": [177, 59, 108, 200],
                    "146": [176, 59, 108, 200],
                    "147": [175, 59, 108, 200],
                    "148": [174, 59, 108, 200],
                    "149": [173, 58, 108, 200],
                    "150": [172, 58, 109, 200],
                    "151": [171, 58, 109, 200],
                    "152": [170, 58, 109, 200],
                    "153": [169, 57, 109, 200],
                    "154": [168, 57, 109, 200],
                    "155": [168, 57, 110, 200],
                    "156": [167, 57, 110, 200],
                    "157": [166, 56, 110, 200],
                    "158": [165, 56, 110, 200],
                    "159": [164, 56, 110, 200],
                    "160": [163, 56, 110, 200],
                    "161": [162, 55, 110, 200],
                    "162": [161, 55, 111, 200],
                    "163": [160, 55, 111, 200],
                    "164": [159, 55, 111, 200],
                    "165": [158, 54, 111, 200],
                    "166": [157, 54, 111, 200],
                    "167": [156, 54, 111, 200],
                    "168": [155, 54, 111, 200],
                    "169": [154, 53, 111, 200],
                    "170": [154, 53, 111, 200],
                    "171": [153, 53, 112, 200],
                    "172": [152, 53, 112, 200],
                    "173": [151, 52, 112, 200],
                    "174": [150, 52, 112, 200],
                    "175": [149, 52, 112, 200],
                    "176": [148, 52, 112, 200],
                    "177": [147, 51, 112, 200],
                    "178": [146, 51, 112, 200],
                    "179": [145, 51, 112, 200],
                    "180": [144, 51, 112, 200],
                    "181": [143, 50, 112, 200],
                    "182": [142, 50, 112, 200],
                    "183": [141, 50, 112, 200],
                    "184": [141, 50, 112, 200],
                    "185": [140, 49, 112, 200],
                    "186": [139, 49, 112, 200],
                    "187": [138, 49, 112, 200],
                    "188": [137, 49, 112, 200],
                    "189": [136, 48, 112, 200],
                    "190": [135, 48, 112, 200],
                    "191": [134, 48, 112, 200],
                    "192": [133, 48, 112, 200],
                    "193": [132, 48, 112, 200],
                    "194": [131, 47, 112, 200],
                    "195": [130, 47, 112, 200],
                    "196": [129, 47, 112, 200],
                    "197": [128, 47, 112, 200],
                    "198": [127, 47, 112, 200],
                    "199": [126, 46, 112, 200],
                    "200": [126, 46, 112, 200],
                    "201": [125, 46, 111, 200],
                    "202": [124, 46, 111, 200],
                    "203": [123, 46, 111, 200],
                    "204": [122, 45, 111, 200],
                    "205": [121, 45, 111, 200],
                    "206": [120, 45, 111, 200],
                    "207": [119, 45, 111, 200],
                    "208": [118, 45, 110, 200],
                    "209": [117, 44, 110, 200],
                    "210": [116, 44, 110, 200],
                    "211": [115, 44, 110, 200],
                    "212": [114, 44, 110, 200],
                    "213": [113, 44, 109, 200],
                    "214": [112, 44, 109, 200],
                    "215": [111, 43, 109, 200],
                    "216": [110, 43, 109, 200],
                    "217": [109, 43, 109, 200],
                    "218": [108, 43, 108, 200],
                    "219": [108, 43, 108, 200],
                    "220": [107, 43, 108, 200],
                    "221": [106, 42, 108, 200],
                    "222": [105, 42, 107, 200],
                    "223": [104, 42, 107, 200],
                    "224": [103, 42, 107, 200],
                    "225": [102, 42, 107, 200],
                    "226": [101, 41, 106, 200],
                    "227": [100, 41, 106, 200],
                    "228": [99, 41, 106, 200],
                    "229": [98, 41, 105, 200],
                    "230": [97, 41, 105, 200],
                    "231": [96, 40, 105, 200],
                    "232": [95, 40, 104, 200],
                    "233": [94, 40, 104, 200],
                    "234": [93, 40, 104, 200],
                    "235": [93, 40, 104, 200],
                    "236": [92, 39, 103, 200],
                    "237": [91, 39, 103, 200],
                    "238": [90, 39, 103, 200],
                    "239": [89, 39, 102, 200],
                    "240": [88, 38, 102, 200],
                    "241": [87, 38, 102, 200],
                    "242": [86, 38, 101, 200],
                    "243": [85, 38, 101, 200],
                    "244": [84, 38, 101, 200],
                    "245": [83, 37, 100, 200],
                    "246": [82, 37, 100, 200],
                    "247": [82, 37, 100, 200],
                    "248": [81, 36, 100, 200],
                    "249": [80, 36, 99, 200],
                    "250": [79, 36, 99, 200],
                    "251": [78, 36, 99, 200],
                    "252": [77, 35, 98, 200],
                    "253": [76, 35, 98, 200],
                    "254": [75, 35, 98, 200],
                    "255": [74, 34, 98, 200],
                },
                "nodata": {"color_index": 0},
                "min": {"data": 0.0, "color_index": 1},
                "max": {"data": 2.0, "color_index": 255},
                "units": "m",
            },
            "mean_degree_days/above/32c": {
                "colormap": {
                    "0": [255, 255, 255, 0],
                    "1": [255, 255, 255, 0],
                    "2": [222, 219, 218, 200],
                    "3": [222, 219, 218, 200],
                    "4": [223, 219, 217, 200],
                    "5": [223, 219, 217, 200],
                    "6": [224, 218, 215, 200],
                    "7": [224, 218, 215, 200],
                    "8": [225, 218, 214, 200],
                    "9": [225, 218, 214, 200],
                    "10": [226, 217, 212, 200],
                    "11": [226, 217, 212, 200],
                    "12": [227, 217, 211, 200],
                    "13": [227, 217, 211, 200],
                    "14": [228, 216, 209, 200],
                    "15": [228, 216, 209, 200],
                    "16": [229, 216, 208, 200],
                    "17": [229, 216, 208, 200],
                    "18": [230, 215, 207, 200],
                    "19": [230, 215, 207, 200],
                    "20": [231, 214, 205, 200],
                    "21": [231, 214, 205, 200],
                    "22": [231, 214, 204, 200],
                    "23": [231, 214, 204, 200],
                    "24": [232, 213, 202, 200],
                    "25": [232, 213, 202, 200],
                    "26": [233, 212, 201, 200],
                    "27": [233, 212, 201, 200],
                    "28": [234, 211, 199, 200],
                    "29": [234, 211, 199, 200],
                    "30": [235, 211, 198, 200],
                    "31": [235, 211, 198, 200],
                    "32": [236, 210, 196, 200],
                    "33": [236, 210, 196, 200],
                    "34": [236, 209, 195, 200],
                    "35": [236, 209, 195, 200],
                    "36": [237, 208, 193, 200],
                    "37": [237, 208, 193, 200],
                    "38": [237, 207, 192, 200],
                    "39": [237, 207, 192, 200],
                    "40": [238, 207, 190, 200],
                    "41": [238, 207, 190, 200],
                    "42": [239, 206, 188, 200],
                    "43": [239, 206, 188, 200],
                    "44": [239, 205, 187, 200],
                    "45": [239, 205, 187, 200],
                    "46": [240, 204, 185, 200],
                    "47": [240, 204, 185, 200],
                    "48": [241, 203, 184, 200],
                    "49": [241, 203, 184, 200],
                    "50": [241, 202, 182, 200],
                    "51": [241, 202, 182, 200],
                    "52": [242, 201, 181, 200],
                    "53": [242, 201, 181, 200],
                    "54": [242, 200, 179, 200],
                    "55": [242, 200, 179, 200],
                    "56": [242, 199, 178, 200],
                    "57": [242, 199, 178, 200],
                    "58": [243, 198, 176, 200],
                    "59": [243, 198, 176, 200],
                    "60": [243, 197, 175, 200],
                    "61": [243, 197, 175, 200],
                    "62": [244, 196, 173, 200],
                    "63": [244, 196, 173, 200],
                    "64": [244, 195, 171, 200],
                    "65": [244, 195, 171, 200],
                    "66": [244, 194, 170, 200],
                    "67": [244, 194, 170, 200],
                    "68": [245, 193, 168, 200],
                    "69": [245, 193, 168, 200],
                    "70": [245, 192, 167, 200],
                    "71": [245, 192, 167, 200],
                    "72": [245, 191, 165, 200],
                    "73": [245, 191, 165, 200],
                    "74": [246, 189, 164, 200],
                    "75": [246, 189, 164, 200],
                    "76": [246, 188, 162, 200],
                    "77": [246, 188, 162, 200],
                    "78": [246, 187, 160, 200],
                    "79": [246, 187, 160, 200],
                    "80": [246, 186, 159, 200],
                    "81": [246, 186, 159, 200],
                    "82": [246, 185, 157, 200],
                    "83": [246, 185, 157, 200],
                    "84": [246, 183, 156, 200],
                    "85": [246, 183, 156, 200],
                    "86": [246, 182, 154, 200],
                    "87": [246, 182, 154, 200],
                    "88": [247, 181, 152, 200],
                    "89": [247, 181, 152, 200],
                    "90": [247, 179, 151, 200],
                    "91": [247, 179, 151, 200],
                    "92": [247, 178, 149, 200],
                    "93": [247, 178, 149, 200],
                    "94": [247, 177, 148, 200],
                    "95": [247, 177, 148, 200],
                    "96": [247, 176, 146, 200],
                    "97": [247, 176, 146, 200],
                    "98": [247, 174, 145, 200],
                    "99": [247, 174, 145, 200],
                    "100": [247, 173, 143, 200],
                    "101": [247, 173, 143, 200],
                    "102": [246, 171, 141, 200],
                    "103": [246, 171, 141, 200],
                    "104": [246, 170, 140, 200],
                    "105": [246, 170, 140, 200],
                    "106": [246, 169, 138, 200],
                    "107": [246, 169, 138, 200],
                    "108": [246, 167, 137, 200],
                    "109": [246, 167, 137, 200],
                    "110": [246, 166, 135, 200],
                    "111": [246, 166, 135, 200],
                    "112": [246, 164, 134, 200],
                    "113": [246, 164, 134, 200],
                    "114": [246, 163, 132, 200],
                    "115": [246, 163, 132, 200],
                    "116": [245, 161, 130, 200],
                    "117": [245, 161, 130, 200],
                    "118": [245, 160, 129, 200],
                    "119": [245, 160, 129, 200],
                    "120": [245, 158, 127, 200],
                    "121": [245, 158, 127, 200],
                    "122": [244, 157, 126, 200],
                    "123": [244, 157, 126, 200],
                    "124": [244, 155, 124, 200],
                    "125": [244, 155, 124, 200],
                    "126": [244, 154, 123, 200],
                    "127": [244, 154, 123, 200],
                    "128": [243, 152, 121, 200],
                    "129": [243, 152, 121, 200],
                    "130": [243, 150, 120, 200],
                    "131": [243, 150, 120, 200],
                    "132": [243, 149, 118, 200],
                    "133": [243, 149, 118, 200],
                    "134": [242, 147, 117, 200],
                    "135": [242, 147, 117, 200],
                    "136": [242, 145, 115, 200],
                    "137": [242, 145, 115, 200],
                    "138": [241, 144, 114, 200],
                    "139": [241, 144, 114, 200],
                    "140": [241, 142, 112, 200],
                    "141": [241, 142, 112, 200],
                    "142": [240, 141, 111, 200],
                    "143": [240, 141, 111, 200],
                    "144": [240, 139, 109, 200],
                    "145": [240, 139, 109, 200],
                    "146": [239, 137, 108, 200],
                    "147": [239, 137, 108, 200],
                    "148": [238, 135, 106, 200],
                    "149": [238, 135, 106, 200],
                    "150": [238, 134, 105, 200],
                    "151": [238, 134, 105, 200],
                    "152": [237, 132, 103, 200],
                    "153": [237, 132, 103, 200],
                    "154": [236, 130, 102, 200],
                    "155": [236, 130, 102, 200],
                    "156": [236, 128, 100, 200],
                    "157": [236, 128, 100, 200],
                    "158": [235, 127, 99, 200],
                    "159": [235, 127, 99, 200],
                    "160": [234, 125, 97, 200],
                    "161": [234, 125, 97, 200],
                    "162": [234, 123, 96, 200],
                    "163": [234, 123, 96, 200],
                    "164": [233, 121, 94, 200],
                    "165": [233, 121, 94, 200],
                    "166": [232, 119, 93, 200],
                    "167": [232, 119, 93, 200],
                    "168": [231, 117, 92, 200],
                    "169": [231, 117, 92, 200],
                    "170": [230, 116, 90, 200],
                    "171": [230, 116, 90, 200],
                    "172": [230, 114, 89, 200],
                    "173": [230, 114, 89, 200],
                    "174": [229, 112, 87, 200],
                    "175": [229, 112, 87, 200],
                    "176": [228, 110, 86, 200],
                    "177": [228, 110, 86, 200],
                    "178": [227, 108, 84, 200],
                    "179": [227, 108, 84, 200],
                    "180": [226, 106, 83, 200],
                    "181": [226, 106, 83, 200],
                    "182": [225, 104, 82, 200],
                    "183": [225, 104, 82, 200],
                    "184": [224, 102, 80, 200],
                    "185": [224, 102, 80, 200],
                    "186": [223, 100, 79, 200],
                    "187": [223, 100, 79, 200],
                    "188": [222, 98, 78, 200],
                    "189": [222, 98, 78, 200],
                    "190": [221, 96, 76, 200],
                    "191": [221, 96, 76, 200],
                    "192": [220, 94, 75, 200],
                    "193": [220, 94, 75, 200],
                    "194": [219, 92, 74, 200],
                    "195": [219, 92, 74, 200],
                    "196": [218, 90, 72, 200],
                    "197": [218, 90, 72, 200],
                    "198": [217, 88, 71, 200],
                    "199": [217, 88, 71, 200],
                    "200": [216, 86, 70, 200],
                    "201": [216, 86, 70, 200],
                    "202": [215, 84, 68, 200],
                    "203": [215, 84, 68, 200],
                    "204": [214, 82, 67, 200],
                    "205": [214, 82, 67, 200],
                    "206": [212, 79, 66, 200],
                    "207": [212, 79, 66, 200],
                    "208": [211, 77, 64, 200],
                    "209": [211, 77, 64, 200],
                    "210": [210, 75, 63, 200],
                    "211": [210, 75, 63, 200],
                    "212": [209, 73, 62, 200],
                    "213": [209, 73, 62, 200],
                    "214": [207, 70, 61, 200],
                    "215": [207, 70, 61, 200],
                    "216": [206, 68, 60, 200],
                    "217": [206, 68, 60, 200],
                    "218": [205, 66, 58, 200],
                    "219": [205, 66, 58, 200],
                    "220": [204, 63, 57, 200],
                    "221": [204, 63, 57, 200],
                    "222": [202, 61, 56, 200],
                    "223": [202, 61, 56, 200],
                    "224": [201, 59, 55, 200],
                    "225": [201, 59, 55, 200],
                    "226": [200, 56, 53, 200],
                    "227": [200, 56, 53, 200],
                    "228": [198, 53, 52, 200],
                    "229": [198, 53, 52, 200],
                    "230": [197, 50, 51, 200],
                    "231": [197, 50, 51, 200],
                    "232": [196, 48, 50, 200],
                    "233": [196, 48, 50, 200],
                    "234": [194, 45, 49, 200],
                    "235": [194, 45, 49, 200],
                    "236": [193, 42, 48, 200],
                    "237": [193, 42, 48, 200],
                    "238": [191, 40, 46, 200],
                    "239": [191, 40, 46, 200],
                    "240": [190, 35, 45, 200],
                    "241": [190, 35, 45, 200],
                    "242": [188, 31, 44, 200],
                    "243": [188, 31, 44, 200],
                    "244": [187, 26, 43, 200],
                    "245": [187, 26, 43, 200],
                    "246": [185, 22, 42, 200],
                    "247": [185, 22, 42, 200],
                    "248": [184, 17, 41, 200],
                    "249": [184, 17, 41, 200],
                    "250": [182, 13, 40, 200],
                    "251": [182, 13, 40, 200],
                    "252": [181, 8, 39, 200],
                    "253": [181, 8, 39, 200],
                    "254": [179, 3, 38, 200],
                    "255": [179, 3, 38, 200],
                },
                "nodata": {"color_index": 0},
                "min": {"data": 0.0, "color_index": 1},
                "max": {"data": 3158.1914, "color_index": 255},
                "units": "deg days",
            },
        }


def alphanumeric(text):
    """Return alphanumeric hash from supplied string."""
    hash_int = int.from_bytes(hashlib.sha1(text.encode("utf-8")).digest(), "big")
    return base36encode(hash_int)


def base36encode(number, alphabet="0123456789abcdefghijklmnopqrstuvwxyz"):
    """Converts an integer to a base36 string."""
    if not isinstance(number, int):
        raise TypeError("number must be an integer")

    base36 = ""

    if number < 0:
        raise TypeError("number must be positive")

    if 0 <= number < len(alphabet):
        return alphabet[number]

    while number != 0:
        number, i = divmod(number, len(alphabet))
        base36 = alphabet[i] + base36

    return base36
