# Copyright Iris contributors
#
# This file is part of Iris and is released under the LGPL license.
# See COPYING and COPYING.LESSER in the root of the repository for full
# licensing details.
"""
Provides UM/CF phenomenon translations.

"""

from collections import namedtuple

CFName = namedtuple('CFName', 'standard_name long_name units')


LBFC_TO_CF = {
    5: CFName('atmosphere_boundary_layer_thickness', None, 'm'),
    16: CFName('air_temperature', None, 'K'),
    23: CFName('soil_temperature', None, 'K'),
    27: CFName('air_density', None, 'kg m-3'),
    36: CFName('land_area_fraction', None, '1'),
    37: CFName('sea_ice_area_fraction', None, '1'),
    50: CFName('wind_speed', None, 'm s-1'),
    56: CFName('x_wind', None, 'm s-1'),
    57: CFName('y_wind', None, 'm s-1'),
    73: CFName('atmosphere_relative_vorticity', None, 's-1'),
    74: CFName('divergence_of_wind', None, 's-1'),
    83: CFName('potential_vorticity_of_atmosphere_layer', None, 'Pa-1 s-1'),
    94: CFName('convective_rainfall_amount', None, 'kg m-2'),
    97: CFName('rainfall_flux', None, 'kg m-2 s-1'),
    102: CFName('stratiform_rainfall_amount', None, 'kg m-2'),
    108: CFName('snowfall_flux', None, 'kg m-2 s-1'),
    111: CFName('surface_runoff_amount', None, 'kg m-2'),
    116: CFName('stratiform_snowfall_amount', None, 'kg m-2'),
    117: CFName('convective_snowfall_amount', None, 'kg m-2'),
    122: CFName('moisture_content_of_soil_layer', None, 'kg m-2'),
    183: CFName('wind_speed', None, 'm s-1'),
    200: CFName('toa_incoming_shortwave_flux', None, 'W m-2'),
    203: CFName('surface_downwelling_shortwave_flux_in_air', None, 'W m-2'),
    206: CFName('toa_outgoing_longwave_flux', None, 'W m-2'),
    208: CFName('surface_downwelling_shortwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    209: CFName('sea_ice_temperature', None, 'K'),
    253: CFName('tendency_of_air_temperature_due_to_longwave_heating', None, 'K s-1'),
    261: CFName('downward_heat_flux_in_sea_ice', None, 'W m-2'),
    321: CFName('root_depth', None, 'm'),
    326: CFName('vegetation_area_fraction', None, '1'),
    328: CFName('surface_albedo_assuming_deep_snow', None, '1'),
    329: CFName('volume_fraction_of_condensed_water_in_soil_at_wilting_point', None, '1'),
    330: CFName('volume_fraction_of_condensed_water_in_soil_at_critical_point', None, '1'),
    332: CFName('soil_porosity', None, '1'),
    333: CFName('soil_hydraulic_conductivity_at_saturation', None, 'm s-1'),
    335: CFName('soil_thermal_capacity', None, 'J kg-1 K-1'),
    336: CFName('soil_thermal_conductivity', None, 'W m-1 K-1'),
    342: CFName('soil_suction_at_saturation', None, 'Pa'),
    687: CFName('sea_ice_thickness', None, 'm'),
    701: CFName('surface_eastward_sea_water_velocity', None, 'm s-1'),
    702: CFName('surface_northward_sea_water_velocity', None, 'm s-1'),
    1025: CFName('surface_downward_eastward_stress', None, 'Pa'),
    1026: CFName('surface_downward_northward_stress', None, 'Pa'),
    1373: CFName('mass_fraction_of_dimethyl_sulfide_in_air', None, '1'),
    1374: CFName('mass_fraction_of_sulfur_dioxide_in_air', None, '1'),
    1382: CFName('leaf_area_index', None, '1'),
    1383: CFName('canopy_height', None, 'm'),
    1385: CFName('mass_fraction_of_unfrozen_water_in_soil_moisture', None, '1'),
    1386: CFName('mass_fraction_of_frozen_water_in_soil_moisture', None, '1'),
    1392: CFName('leaf_area_index', None, '1'),
    1393: CFName('canopy_height', None, 'm'),
    1395: CFName('soil_albedo', None, '1'),
    1507: CFName('snow_grain_size', None, '1e-6 m'),
    1559: CFName('soil_moisture_content_at_field_capacity', None, 'kg m-2'),
    1720: CFName('cloud_area_fraction_in_atmosphere_layer', None, '1'),
    }

STASH_TO_CF = {
    'm01s00i001': CFName('surface_air_pressure', None, 'Pa'),
    'm01s00i002': CFName('x_wind', None, 'm s-1'),
    'm01s00i003': CFName('y_wind', None, 'm s-1'),
    'm01s00i004': CFName('air_potential_temperature', None, 'K'),
    'm01s00i009': CFName('moisture_content_of_soil_layer', None, 'kg m-2'),
    'm01s00i010': CFName('specific_humidity', None, 'kg kg-1'),
    'm01s00i012': CFName('mass_fraction_of_cloud_ice_in_air', None, 'kg kg-1'),
    'm01s00i013': CFName('convective_cloud_area_fraction', None, '1'),
    'm01s00i020': CFName('soil_temperature', None, 'K'),
    'm01s00i023': CFName('snowfall_amount', None, 'kg m-2'),
    'm01s00i024': CFName('surface_temperature', None, 'K'),
    'm01s00i025': CFName('atmosphere_boundary_layer_thickness', None, 'm'),
    'm01s00i026': CFName('surface_roughness_length', None, 'm'),
    'm01s00i028': CFName('surface_eastward_sea_water_velocity', None, 'm s-1'),
    'm01s00i029': CFName('surface_northward_sea_water_velocity', None, 'm s-1'),
    'm01s00i030': CFName('land_binary_mask', None, '1'),
    'm01s00i031': CFName('sea_ice_area_fraction', None, '1'),
    'm01s00i032': CFName('sea_ice_thickness', None, 'm'),
    'm01s00i033': CFName('surface_altitude', None, 'm'),
    'm01s00i040': CFName('volume_fraction_of_condensed_water_in_soil_at_wilting_point', None, '1'),
    'm01s00i041': CFName('volume_fraction_of_condensed_water_in_soil_at_critical_point', None, '1'),
    'm01s00i043': CFName('soil_porosity', None, '1'),
    'm01s00i044': CFName('soil_hydraulic_conductivity_at_saturation', None, 'm s-1'),
    'm01s00i046': CFName('soil_thermal_capacity', None, 'J kg-1 K-1'),
    'm01s00i047': CFName('soil_thermal_conductivity', None, 'W m-1 K-1'),
    'm01s00i048': CFName('soil_suction_at_saturation', None, 'Pa'),
    'm01s00i049': CFName('sea_ice_temperature', None, 'K'),
    'm01s00i050': CFName('vegetation_area_fraction', None, '1'),
    'm01s00i051': CFName('root_depth', None, 'm'),
    'm01s00i052': CFName('surface_albedo_assuming_no_snow', None, '1'),
    'm01s00i053': CFName('surface_albedo_assuming_deep_snow', None, '1'),
    'm01s00i058': CFName(None, 'tendency_of_atmosphere_mass_content_of_sulfur_dioxide_expressed_as_sulfur_due_to_low_level_emission', 'kg/m2/s'),
    'm01s00i059': CFName(None, 'tendency_of_atmosphere_mass_content_of_dimethyl_sulfide_expressed_as_sulfur_due_to_emission', 'kg/m2/s'),
    'm01s00i060': CFName('mass_fraction_of_ozone_in_air', None, '1'),
    'm01s00i075': CFName(None, 'number_of_cloud_droplets_per_kg_of_air', 'kg-1'),
    'm01s00i076': CFName(None, 'number_of_rain_drops_per_kg_of_air', 'kg-1'),
    'm01s00i077': CFName(None, 'rain_third_moment', '1'),
    'm01s00i078': CFName(None, 'number_of_ice_particles_per_kg_of_air', 'kg-1'),
    'm01s00i079': CFName(None, 'number_of_snow_aggregates_per_kg_of_air', 'kg-1'),
    'm01s00i080': CFName(None, 'snow_third_moment', '1'),
    'm01s00i081': CFName(None, 'number_of_graupel_particles_per_kg_of_air', 'kg-1'),
    'm01s00i082': CFName(None, 'graupel_third_moment', '1'),
    'm01s00i090': CFName(None, 'visibility_murk_aerosol', '1e-9 kg kg-1'),
    'm01s00i091': CFName(None, 'lightning_flash_potential', '1'),
    'm01s00i095': CFName(None, 'Snow amount on sea ice', 'kg/m^2'),
    'm01s00i099': CFName(None, 'variance_of_vertical_velocity_from_boundary_layer_for_turbulent_mixed_phase_scheme', 'm s-1'),
    'm01s00i101': CFName(None, 'mass_fraction_of_sulfur_dioxide_expressed_as_sulfur_in_air', 'kg/kg'),
    'm01s00i102': CFName(None, 'mass_fraction_of_dimethyl_sulfide_expressed_as_sulfur_in_air', 'kg/kg'),
    'm01s00i103': CFName(None, 'mass_fraction_of_aitken_mode_sulfate_dry_aerosol_expressed_as_sulfur_in_air', 'kg/kg'),
    'm01s00i104': CFName(None, 'mass_fraction_of_accumulation_mode_sulfate_dry_aerosol_expressed_as_sulfur_in_air', 'kg/kg'),
    'm01s00i105': CFName(None, 'mass_fraction_of_dissolved_sulfate_dry_aerosol_expressed_as_sulfur_in_air', 'kg/kg'),
    'm01s00i106': CFName('mass_fraction_of_hydrogen_peroxide_in_air', None, 'kg kg-1'),
    'm01s00i107': CFName(None, 'mass_fraction_of_ammonia_expressed_as_nitrogen_in_air', 'kg/kg'),
    'm01s00i108': CFName(None, 'mass_fraction_of_fresh_black_carbon_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i109': CFName(None, 'mass_fraction_of_aged_black_carbon_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i110': CFName(None, 'mass_fraction_of_cloud_black_carbon_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i111': CFName(None, 'mass_fraction_of_fresh_biomass_burning_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i112': CFName(None, 'mass_fraction_of_aged_biomass_burning_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i113': CFName(None, 'mass_fraction_of_cloud_biomass_burning_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i114': CFName(None, 'mass_fraction_of_fresh_organic_carbon_from_fossil_fuel_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i115': CFName(None, 'mass_fraction_of_aged_organic_carbon_from_fossil_fuel_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i116': CFName(None, 'mass_fraction_of_cloud_organic_carbon_from_fossil_fuel_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i117': CFName(None, 'mass_fraction_of_accumulation_mode_nitrate_dry_aerosol_expressed_as_nitrogen_in_air', 'kg/kg'),
    'm01s00i118': CFName(None, 'mass_fraction_of_dissolved_nitrate_dry_aerosol_expressed_as_nitrogen_in_air', 'kg/kg'),
    'm01s00i121': CFName(None, '3D NATURAL SO2 EMISSIONS', 'kg m-2 s-1'),
    'm01s00i122': CFName(None, 'molecular_concentration_of_hydroxyl_radical_in_air', 'cm-3'),
    'm01s00i123': CFName(None, 'molecular_concentration_of_hydroperoxyl_radical_in_air', 'cm-3'),
    'm01s00i124': CFName('mass_fraction_of_hydrogen_peroxide_in_air', None, 'kg kg-1'),
    'm01s00i125': CFName('mass_fraction_of_ozone_in_air', None, 'kg kg-1'),
    'm01s00i126': CFName(None, 'tendency_of_atmosphere_mass_content_of_sulfur_dioxide_expressed_as_sulfur_due_to_high_level_emission', 'kg/m2/s'),
    'm01s00i127': CFName(None, 'tendency_of_atmosphere_mass_content_of_ammonia_expressed_as_nitrogen_due_to_emission', 'kg/m2/s'),
    'm01s00i128': CFName(None, 'tendency_of_atmosphere_mass_content_of_black_carbon_dry_aerosol_due_to_low_level_emission', 'kg/m2/s'),
    'm01s00i129': CFName(None, 'tendency_of_atmosphere_mass_content_of_black_carbon_dry_aerosol_due_to_high_level_emission', 'kg/m2/s'),
    'm01s00i130': CFName(None, 'tendency_of_atmosphere_mass_content_of_biomass_burning_dry_aerosol_due_to_low_level_emission', 'kg/m2/s'),
    'm01s00i131': CFName(None, 'tendency_of_atmosphere_mass_content_of_biomass_burning_dry_aerosol_due_to_high_level_emission', 'kg/m2/s'),
    'm01s00i132': CFName('mole_concentration_of_dimethyl_sulfide_in_sea_water', None, 'nanomole/l'),
    'm01s00i134': CFName(None, 'tendency_of_atmosphere_mass_content_of_organic_carbon_from_fossil_fuel_dry_aerosol_due_to_low_level_emission', 'kg/m2/s'),
    'm01s00i135': CFName(None, 'tendency_of_atmosphere_mass_content_of_organic_carbon_from_fossil_fuel_dry_aerosol_due_to_high_level_emission', 'kg/m2/s'),
    'm01s00i150': CFName('upward_air_velocity', None, 'm s-1'),
    'm01s00i205': CFName('land_area_fraction', None, '1'),
    'm01s00i208': CFName('leaf_area_index', None, '1'),
    'm01s00i209': CFName('canopy_height', None, 'm'),
    'm01s00i211': CFName(None, 'Convective cloud amount with anvil', '1'),
    'm01s00i214': CFName('mass_fraction_of_unfrozen_water_in_soil_moisture', None, '1'),
    'm01s00i215': CFName('mass_fraction_of_frozen_water_in_soil_moisture', None, '1'),
    'm01s00i217': CFName('leaf_area_index', None, '1'),
    'm01s00i218': CFName('canopy_height', None, 'm'),
    'm01s00i220': CFName('soil_albedo', None, '1'),
    'm01s00i223': CFName('soil_carbon_content', None, 'kg m-2'),
    'm01s00i231': CFName('snow_grain_size', None, '1e-6 m'),
    'm01s00i243': CFName(None, 'surface_diffuse_albedo_assuming_no_snow', '1'),
    'm01s00i244': CFName(None, 'surface_diffuse_albedo_of_photosynthetically_active_radiation_assuming_no_snow', '1'),
    'm01s00i245': CFName(None, 'surface_diffuse_albedo_of_near_infra_red_radiation_assuming_no_snow', '1'),
    'm01s00i252': CFName('mass_fraction_of_carbon_dioxide_in_air', None, '1'),
    'm01s00i254': CFName('mass_fraction_of_cloud_liquid_water_in_air', None, 'kg kg-1'),
    'm01s00i255': CFName('dimensionless_exner_function', None, '1'),
    'm01s00i265': CFName('cloud_area_fraction_in_atmosphere_layer', None, '1'),
    'm01s00i266': CFName(None, 'cloud_volume_fraction_in_atmosphere_layer', '1'),
    'm01s00i267': CFName(None, 'liquid_cloud_volume_fraction_in_atmosphere_layer', '1'),
    'm01s00i268': CFName(None, 'ice_cloud_volume_fraction_in_atmosphere_layer', '1'),
    'm01s00i269': CFName('surface_eastward_sea_water_velocity', None, 'm s-1'),
    'm01s00i270': CFName('surface_northward_sea_water_velocity', None, 'm s-1'),
    'm01s00i271': CFName(None, 'mass_fraction_of_cloud_ice_crystals_in_air', 'kg kg-1'),
    'm01s00i272': CFName('mass_fraction_of_rain_in_air', None, 'kg kg-1'),
    'm01s00i273': CFName('mass_fraction_of_graupel_in_air', None, 'kg kg-1'),
    'm01s00i351': CFName(None, 'mass_concentration_of_biogenic_nmvoc_in_air', 'kg/kg'),
    'm01s00i352': CFName(None, 'mass_fraction_of_fresh_biomass_burning_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i353': CFName(None, 'mass_fraction_of_aged_biomass_burning_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i354': CFName(None, 'mass_fraction_of_cloud_biomass_burning_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i355': CFName(None, 'mass_fraction_of_fresh_black_carbon_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i356': CFName(None, 'mass_fraction_of_aged_black_carbon_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i357': CFName(None, 'atmosphere_number_concentration_of_film_mode_sea_salt_particles', 'kg/kg'),
    'm01s00i358': CFName(None, 'atmosphere_number_concentration_of_jet_mode_sea_salt_particles', 'kg/kg'),
    'm01s00i359': CFName(None, 'mass_fraction_of_aitken_mode_sulfate_dry_aerosol_in_air_expressed_as_sulfur', 'kg/kg'),
    'm01s00i360': CFName(None, 'mass_fraction_of_accumulation_mode_sulfate_dry_aerosol_in_air_expressed_as_sulfur', 'kg/kg'),
    'm01s00i361': CFName(None, 'mass_fraction_of_dissolved_sulfate_dry_aerosol_in_air_expressed_as_sulfur', 'kg/kg'),
    'm01s00i362': CFName(None, 'mass_fraction_of_dust_ukmo_division_1_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i363': CFName(None, 'mass_fraction_of_dust_ukmo_division_2_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i364': CFName(None, 'mass_fraction_of_dust_ukmo_division_3_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i365': CFName(None, 'mass_fraction_of_dust_ukmo_division_4_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i366': CFName(None, 'mass_fraction_of_dust_ukmo_division_5_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i367': CFName(None, 'mass_fraction_of_dust_ukmo_division_6_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i368': CFName(None, 'mass_fraction_of_fresh_organic_carbon_from_fossil_fuel_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i369': CFName(None, 'mass_fraction_of_aged_organic_carbon_from_fossil_fuel_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i370': CFName(None, 'mass_fraction_of_cloud_organic_carbon_from_fossil_fuel_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i371': CFName(None, 'mass_concentration_of_unspecified_aerosol_in_air', 'kg/kg'),
    'm01s00i388': CFName(None, 'virtual_potential_temperature', 'K'),
    'm01s00i389': CFName('air_density', None, 'kg m-3'),
    'm01s00i391': CFName('humidity_mixing_ratio', None, 'kg kg-1'),
    'm01s00i392': CFName('cloud_liquid_water_mixing_ratio', None, 'kg kg-1'),
    'm01s00i393': CFName('cloud_ice_mixing_ratio', None, 'kg kg-1'),
    'm01s00i394': CFName(None, 'rain_mixing_ratio', 'kg kg-1'),
    'm01s00i395': CFName(None, 'graupel_mixing_ratio', 'kg kg-1'),
    'm01s00i406': CFName('dimensionless_exner_function', None, '1'),
    'm01s00i407': CFName('air_pressure', None, 'Pa'),
    'm01s00i408': CFName('air_pressure', None, 'Pa'),
    'm01s00i409': CFName('surface_air_pressure', None, 'Pa'),
    'm01s00i413': CFName(None, 'Sea ice concentration by categories', '1'),
    'm01s00i414': CFName(None, 'Sea ice thickness GBM by categories', 'm'),
    'm01s00i415': CFName('sea_ice_surface_temperature', None, 'K'),
    'm01s00i416': CFName(None, 'Snow thickness on sea ice', 'm'),
    'm01s00i418': CFName('volume_fraction_of_clay_in_soil', None, 'm3 m-3'),
    'm01s00i419': CFName('volume_fraction_of_silt_in_soil', None, 'm3 m-3'),
    'm01s00i420': CFName('volume_fraction_of_sand_in_soil', None, 'm3 m-3'),
    'm01s00i421': CFName(None, 'mass_fraction_of_soil_particles_in_ukmo_division1', 'kg/kg'),
    'm01s00i422': CFName(None, 'mass_fraction_of_soil_particles_in_ukmo_division2', 'kg/kg'),
    'm01s00i423': CFName(None, 'mass_fraction_of_soil_particles_in_ukmo_division3', 'kg/kg'),
    'm01s00i424': CFName(None, 'mass_fraction_of_soil_particles_in_ukmo_division4', 'kg/kg'),
    'm01s00i425': CFName(None, 'mass_fraction_of_soil_particles_in_ukmo_division5', 'kg/kg'),
    'm01s00i426': CFName(None, 'mass_fraction_of_soil_particles_in_ukmo_division6', 'kg/kg'),
    'm01s00i431': CFName(None, 'mass_fraction_of_dust_ukmo_division_1_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i432': CFName(None, 'mass_fraction_of_dust_ukmo_division_2_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i433': CFName(None, 'mass_fraction_of_dust_ukmo_division_3_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i434': CFName(None, 'mass_fraction_of_dust_ukmo_division_4_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i435': CFName(None, 'mass_fraction_of_dust_ukmo_division_5_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i436': CFName(None, 'mass_fraction_of_dust_ukmo_division_6_dry_aerosol_in_air', 'kg/kg'),
    'm01s00i505': CFName('land_area_fraction', None, '1'),
    'm01s00i506': CFName('surface_temperature', None, 'K'),
    'm01s00i507': CFName('surface_temperature', None, 'K'),
    'm01s00i508': CFName('surface_temperature', None, 'K'),
    'm01s00i509': CFName(None, 'product_of_sea_ice_albedo_and_sunlit_binary_mask', '1'),
    'm01s00i510': CFName(None, 'product_of_land_albedo_and_sunlit_binary_mask', '1'),
    'm01s01i004': CFName('air_temperature', None, 'K'),
    'm01s01i101': CFName(None, 'northward_horizon_angle_from_zenith', 'rad'),
    'm01s01i102': CFName(None, 'northeastward_horizon_angle_from_zenith', 'rad'),
    'm01s01i103': CFName(None, 'eastward_horizon_angle_from_zenith', 'rad'),
    'm01s01i104': CFName(None, 'southeastward_horizon_angle_from_zenith', 'rad'),
    'm01s01i105': CFName(None, 'southward_horizon_angle_from_zenith', 'rad'),
    'm01s01i106': CFName(None, 'southwestward_horizon_angle_from_zenith', 'rad'),
    'm01s01i107': CFName(None, 'westward_horizon_angle_from_zenith', 'rad'),
    'm01s01i108': CFName(None, 'northwestward_horizon_angle_from_zenith', 'rad'),
    'm01s01i109': CFName(None, 'northnortheastward_horizon_angle_from_zenith', 'rad'),
    'm01s01i110': CFName(None, 'eastnortheastward_horizon_angle_from_zenith', 'rad'),
    'm01s01i111': CFName(None, 'eastsoutheastward_horizon_angle_from_zenith', 'rad'),
    'm01s01i112': CFName(None, 'southsoutheastward_horizon_angle_from_zenith', 'rad'),
    'm01s01i113': CFName(None, 'southsouthwestward_horizon_angle_from_zenith', 'rad'),
    'm01s01i114': CFName(None, 'westsouthwestward_horizon_angle_from_zenith', 'rad'),
    'm01s01i115': CFName(None, 'westnorthwestward_horizon_angle_from_zenith', 'rad'),
    'm01s01i116': CFName(None, 'northnorthwestward_horizon_angle_from_zenith', 'rad'),
    'm01s01i181': CFName(None, 'change_over_time_in_air_temperature_due_to_shortwave_heating', 'K'),
    'm01s01i182': CFName(None, 'change_over_time_in_specific_humidity_due_to_shortwave_heating', 'kg kg-1'),
    'm01s01i183': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_shortwave_heating', 'kg kg-1'),
    'm01s01i192': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer_due_to_shortwave_heating', '1'),
    'm01s01i193': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer_due_to_shortwave_heating', '1'),
    'm01s01i201': CFName('surface_net_downward_shortwave_flux', None, 'W m-2'),
    'm01s01i203': CFName('surface_net_downward_shortwave_flux', None, 'W m-2'),
    'm01s01i205': CFName('toa_outgoing_shortwave_flux', None, 'W m-2'),
    'm01s01i207': CFName('toa_incoming_shortwave_flux', None, 'W m-2'),
    'm01s01i208': CFName('toa_outgoing_shortwave_flux', None, 'W m-2'),
    'm01s01i209': CFName('toa_outgoing_shortwave_flux_assuming_clear_sky', None, 'W m-2'),
    'm01s01i210': CFName('surface_downwelling_shortwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    'm01s01i211': CFName('surface_upwelling_shortwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    'm01s01i217': CFName('upwelling_shortwave_flux_in_air', None, 'W m-2'),
    'm01s01i218': CFName('downwelling_shortwave_flux_in_air', None, 'W m-2'),
    'm01s01i219': CFName('upwelling_shortwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    'm01s01i220': CFName('downwelling_shortwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    'm01s01i221': CFName(None, 'product_of_effective_radius_of_stratiform_cloud_liquid_water_particle_and_stratiform_cloud_liquid_water_area_fraction_and_sunlit_binary_mask', 'um'),
    'm01s01i223': CFName(None, 'product_of_stratiform_cloud_liquid_water_area_fraction_and_sunlit_binary_mask', '1'),
    'm01s01i224': CFName(None, 'product_of_stratiform_cloud_liquid_water_path_and_stratiform_cloud_liquid_water_area_fraction_and_sunlit_binary_mask', 'kg m-2'),
    'm01s01i225': CFName(None, 'product_of_effective_radius_of_convective_cloud_liquid_water_particle_and_convective_cloud_liquid_water_area_fraction_and_sunlit_binary_mask', 'um'),
    'm01s01i226': CFName(None, 'product_of_convective_cloud_liquid_water_area_fraction_and_sunlit_binary_mask', '1'),
    'm01s01i232': CFName('tendency_of_air_temperature_due_to_shortwave_heating', None, 'K s-1'),
    'm01s01i233': CFName('tendency_of_air_temperature_due_to_shortwave_heating_assuming_clear_sky', None, 'K s-1'),
    'm01s01i235': CFName('surface_downwelling_shortwave_flux_in_air', None, 'W m-2'),
    'm01s01i237': CFName('net_downward_shortwave_flux_in_air', None, 'W m-2'),
    'm01s01i238': CFName('tropopause_upwelling_shortwave_flux', None, 'W m-2'),
    'm01s01i241': CFName(None, 'product_of_number_concentration_of_stratiform_cloud_liquid_water_particles_and_stratiform_cloud_liquid_water_area_fraction_and_sunlit_binary_mask', 'cm-3'),
    'm01s01i242': CFName(None, 'product_of_stratiform_cloud_liquid_water_content_and_stratiform_cloud_liquid_water_area_fraction_and_sunlit_binary_mask', 'g cm-3'),
    'm01s01i243': CFName(None, 'product_of_mass_concentration_of_sulfate_ion_and_sunlit_binary_mask', 'ug m-3'),
    'm01s01i244': CFName(None, 'sunlit_binary_mask_in_atmosphere_layer_below_cloud_top', '1'),
    'm01s01i245': CFName(None, 'product_of_effective_radius_of_cloud_liquid_water_particle_and_cloud_liquid_water_area_fraction_exposed_to_space_and_sunlit_binary_mask', 'um'),
    'm01s01i246': CFName(None, 'product_of_cloud_liquid_water_area_fraction_exposed_to_space_and_sunlit_binary_mask', '1'),
    'm01s01i247': CFName(None, 'atmosphere_number_concentration_of_film_mode_sea_salt_particles', 'm-3'),
    'm01s01i248': CFName(None, 'atmosphere_number_concentration_of_jet_mode_sea_salt_particles', 'm-3'),
    'm01s01i254': CFName(None, 'product_of_effective_radius_of_warm_cloud_liquid_water_particle_and_warm_cloud_liquid_water_area_fraction_exposed_to_space_and_sunlit_binary_mask', 'um'),
    'm01s01i255': CFName(None, 'product_of_warm_cloud_liquid_water_area_fraction_exposed_to_space_and_sunlit_binary_mask', '1'),
    'm01s01i268': CFName(None, 'surface_direct_beam_albedo_assuming_no_snow', '1'),
    'm01s01i269': CFName(None, 'surface_diffuse_albedo_assuming_no_snow', '1'),
    'm01s01i270': CFName(None, 'scaling_factor_for_surface_diffuse_albedo_of_photosynthetically_active_radiation_assuming_no_snow', '1'),
    'm01s01i271': CFName(None, 'scaling_factor_for_surface_diffuse_albedo_of_near_infra_red_radiation_assuming_no_snow', '1'),
    'm01s01i280': CFName(None, 'product_of_atmosphere_number_content_of_cloud_droplets_and_warm_cloud_area_fraction_and_sunlit_binary_mask', 'm-2'),
    'm01s01i281': CFName(None, 'product_of_warm_cloud_area_fraction_and_sunlit_binary_mask', '1'),
    'm01s01i294': CFName(None, 'surface_mean_slope_angle', 'rad'),
    'm01s01i295': CFName(None, 'orographic_correction_factor_to_surface_direct_downwelling_shortwave_flux', '1'),
    'm01s01i410': CFName('surface_downwelling_shortwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    'm01s01i435': CFName('surface_downwelling_shortwave_flux_in_air', None, 'W m-2'),
    'm01s02i004': CFName('air_temperature', None, 'K'),
    'm01s02i101': CFName(None, 'ratio_of_skyview_factor_to_cosine_of_surface_mean_slope_angle', '1'),
    'm01s02i181': CFName(None, 'change_over_time_in_air_temperature_due_to_longwave_heating', 'K'),
    'm01s02i182': CFName(None, 'change_over_time_in_specific_humidity_due_to_longwave_heating', 'kg kg-1'),
    'm01s02i183': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_longwave_heating', 'kg kg-1'),
    'm01s02i192': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer_due_to_longwave_heating', '1'),
    'm01s02i193': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer_due_to_longwave_heating', '1'),
    'm01s02i201': CFName('surface_net_downward_longwave_flux', None, 'W m-2'),
    'm01s02i203': CFName('surface_net_downward_longwave_flux', None, 'W m-2'),
    'm01s02i204': CFName('cloud_area_fraction', None, '1'),
    'm01s02i205': CFName('toa_outgoing_longwave_flux', None, 'W m-2'),
    'm01s02i206': CFName('toa_outgoing_longwave_flux_assuming_clear_sky', None, 'W m-2'),
    'm01s02i207': CFName('surface_downwelling_longwave_flux_in_air', None, 'W m-2'),
    'm01s02i208': CFName('surface_downwelling_longwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    'm01s02i217': CFName('upwelling_longwave_flux_in_air', None, 'W m-2'),
    'm01s02i218': CFName('downwelling_longwave_flux_in_air', None, 'W m-2'),
    'm01s02i219': CFName('upwelling_longwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    'm01s02i220': CFName('downwelling_longwave_flux_in_air_assuming_clear_sky', None, 'W m-2'),
    'm01s02i232': CFName('tendency_of_air_temperature_due_to_longwave_heating', None, 'K s-1'),
    'm01s02i233': CFName('tendency_of_air_temperature_due_to_longwave_heating_assuming_clear_sky', None, 'K s-1'),
    'm01s02i237': CFName('tropopause_net_downward_longwave_flux', None, 'W m-2'),
    'm01s02i238': CFName('tropopause_downwelling_longwave_flux', None, 'W m-2'),
    'm01s02i260': CFName('mass_fraction_of_ozone_in_air', None, '1'),
    'm01s02i261': CFName('cloud_area_fraction_in_atmosphere_layer', None, '1'),
    'm01s02i262': CFName(None, 'product_of_mass_absorption_coefficient_due_to_cloud_and_upwelling_longwave_flux_assuming_clear_sky_and_cloud_area_fraction_in_atmosphere_layer', 'W kg-1'),
    'm01s02i263': CFName(None, 'product_of_upwelling_longwave_flux_assuming_clear_sky_and_cloud_area_fraction_in_atmosphere_layer', 'W m-2'),
    'm01s02i264': CFName(None, 'product_of_mass_absorption_coefficient_due_to_stratiform_cloud_and_upwelling_longwave_flux_assuming_clear_sky_and_stratiform_cloud_area_fraction_in_atmosphere_layer', 'W kg-1'),
    'm01s02i265': CFName(None, 'product_of_upwelling_longwave_flux_assuming_clear_sky_and_stratiform_cloud_area_fraction_in_atmosphere_layer', 'W m-2'),
    'm01s02i266': CFName(None, 'product_of_mass_absorption_coefficient_due_to_convective_cloud_and_upwelling_longwave_flux_assuming_clear_sky_and_convective_cloud_area_fraction_in_atmosphere_layer', 'W kg-1'),
    'm01s02i267': CFName(None, 'product_of_upwelling_longwave_flux_assuming_clear_sky_and_convective_cloud_area_fraction_in_atmosphere_layer', 'W m-2'),
    'm01s02i280': CFName(None, 'model_level_number_at_ozone_tropopause', '1'),
    'm01s02i281': CFName(None, 'ozone_tropopause_altitude', 'm'),
    'm01s02i282': CFName(None, 'model_level_number_at_thermal_tropopause', '1'),
    'm01s02i283': CFName(None, 'thermal_tropopause_altitude', 'm'),
    'm01s02i284': CFName(None, 'atmosphere_optical_thickness_due_to_sulphate_ambient_aerosol', '1'),
    'm01s02i285': CFName('atmosphere_optical_thickness_due_to_dust_ambient_aerosol', None, '1'),
    'm01s02i286': CFName('atmosphere_optical_thickness_due_to_seasalt_ambient_aerosol', None, '1'),
    'm01s02i287': CFName('atmosphere_optical_thickness_due_to_black_carbon_ambient_aerosol', None, '1'),
    'm01s02i288': CFName(None, 'atmosphere_optical_thickness_due_to_biomass_burning_ambient_aerosol', '1'),
    'm01s02i289': CFName(None, 'atmosphere_optical_thickness_due_to_biogenic_aerosol', '1'),
    'm01s02i295': CFName(None, 'atmosphere_optical_thickness_due_to_fossil_fuel_organic_carbon_ambient_aerosol', '1'),
    'm01s02i296': CFName(None, 'atmosphere_optical_thickness_due_to_unspecified_aerosol', '1'),
    'm01s02i297': CFName(None, 'atmosphere_optical_thickness_due_to_ammonium_nitrate_ambient_aerosol', '1'),
    'm01s02i298': CFName(None, 'atmosphere_optical_thickness_due_all_ambient_aerosol', '1'),
    'm01s02i299': CFName('angstrom_exponent_of_ambient_aerosol_in_air', None, '1'),
    'm01s02i300': CFName(None, 'atmosphere_optical_thickness_due_to_soluble_aitken_mode_sulphate_aerosol', '1'),
    'm01s02i301': CFName(None, 'atmosphere_optical_thickness_due_to_soluble_accumulation_mode_sulphate_aerosol', '1'),
    'm01s02i302': CFName(None, 'atmosphere_optical_thickness_due_to_soluble_coarse_mode_sulphate_aerosol', '1'),
    'm01s02i303': CFName(None, 'atmosphere_optical_thickness_due_to_insoluble_aitken_mode_sulphate_aerosol', '1'),
    'm01s02i304': CFName(None, 'atmosphere_optical_thickness_due_to_unsoluble_accumulation_mode_sulphate_aerosol', '1'),
    'm01s02i305': CFName(None, 'atmosphere_optical_thickness_due_to_unsoluble_coarse_mode_sulphate_aerosol', '1'),
    'm01s02i308': CFName('mass_fraction_of_stratiform_cloud_liquid_water_in_air', None, '1'),
    'm01s02i309': CFName('mass_fraction_of_stratiform_cloud_ice_in_air', None, '1'),
    'm01s02i310': CFName('mass_fraction_of_convective_cloud_liquid_water_in_air', None, '1'),
    'm01s02i311': CFName('mass_fraction_of_convective_cloud_ice_in_air', None, '1'),
    'm01s02i312': CFName(None, 'stratiform_cloud_liquid_water_area_fraction_in_atmosphere_layer', '1'),
    'm01s02i313': CFName(None, 'stratiform_cloud_ice_area_fraction_in_atmosphere_layer', '1'),
    'm01s02i314': CFName(None, 'convective_cloud_liquid_water_area_fraction_in_atmosphere_layer', '1'),
    'm01s02i315': CFName(None, 'convective_cloud_ice_area_fraction_in_atmosphere_layer', '1'),
    'm01s02i348': CFName('toa_bidirectional_reflectance', None, '1'),
    'm01s02i351': CFName('equivalent_reflectivity_factor', None, 'dBZ'),
    'm01s02i370': CFName('histogram_of_backscattering_ratio_over_height_above_reference_ellipsoid', None, '1'),
    'm01s02i372': CFName('histogram_of_equivalent_reflectivity_factor_over_height_above_reference_ellipsoid', None, '1'),
    'm01s02i375': CFName('atmosphere_optical_thickness_due_to_stratiform_cloud', None, '1'),
    'm01s02i376': CFName('stratiform_cloud_longwave_emissivity', None, '1'),
    'm01s02i377': CFName('atmosphere_optical_thickness_due_to_convective_cloud', None, '1'),
    'm01s02i378': CFName('convective_cloud_longwave_emissivity', None, '1'),
    'm01s02i380': CFName('effective_radius_of_stratiform_cloud_liquid_water_particle', None, 'm'),
    'm01s02i381': CFName('effective_radius_of_stratiform_cloud_ice_particle', None, 'm'),
    'm01s02i382': CFName('effective_radius_of_stratiform_cloud_rain_particle', None, 'm'),
    'm01s02i383': CFName('effective_radius_of_stratiform_cloud_snow_particle', None, 'm'),
    'm01s02i384': CFName('effective_radius_of_convective_cloud_liquid_water_particle', None, 'm'),
    'm01s02i385': CFName('effective_radius_of_convective_cloud_ice_particle', None, 'm'),
    'm01s02i386': CFName('effective_radius_of_convective_cloud_rain_particle', None, 'm'),
    'm01s02i387': CFName('effective_radius_of_convective_cloud_snow_particle', None, 'm'),
    'm01s02i388': CFName('effective_radius_of_stratiform_cloud_graupel_particle', None, 'm'),
    'm01s02i421': CFName(None, 'atmosphere_optical_thickness_due_to_sulphate_ambient_aerosol', '1'),
    'm01s02i422': CFName('atmosphere_optical_thickness_due_to_dust_ambient_aerosol', None, '1'),
    'm01s02i423': CFName('atmosphere_optical_thickness_due_to_seasalt_ambient_aerosol', None, '1'),
    'm01s02i424': CFName('atmosphere_optical_thickness_due_to_black_carbon_ambient_aerosol', None, '1'),
    'm01s02i425': CFName(None, 'atmosphere_optical_thickness_due_to_biomass_burning_ambient_aerosol', '1'),
    'm01s02i426': CFName(None, 'atmosphere_optical_thickness_due_to_fossil_fuel_organic_carbon_ambient_aerosol', '1'),
    'm01s02i427': CFName(None, 'atmosphere_optical_thickness_due_to_ammonium_nitrate_ambient_aerosol', '1'),
    'm01s03i004': CFName('air_temperature', None, 'K'),
    'm01s03i010': CFName('specific_humidity', None, '1'),
    'm01s03i025': CFName('atmosphere_boundary_layer_thickness', None, 'm'),
    'm01s03i181': CFName(None, 'change_over_time_in_air_temperature_due_to_boundary_layer_mixing', 'K'),
    'm01s03i182': CFName(None, 'change_over_time_in_specific_humidity_due_to_boundary_layer_mixing', 'kg kg-1'),
    'm01s03i183': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_boundary_layer_mixing', 'kg kg-1'),
    'm01s03i184': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_ice_in_air_due_to_boundary_layer_mixing', 'kg kg-1'),
    'm01s03i185': CFName(None, 'change_over_time_in_x_wind_due_to_boundary_layer_mixing', 'm s-1'),
    'm01s03i186': CFName(None, 'change_over_time_in_y_wind_due_to_boundary_layer_mixing', 'm s-1'),
    'm01s03i187': CFName(None, 'change_over_time_in_upward_air_velocity_due_to_boundary_layer_mixing', 'm s-1'),
    'm01s03i192': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer_due_to_boundary_layer_mixing', '1'),
    'm01s03i193': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer_due_to_boundary_layer_mixing', '1'),
    'm01s03i194': CFName(None, 'change_over_time_in_ice_cloud_volume_fraction_in_atmosphere_layer_due_to_boundary_layer_mixing', '1'),
    'm01s03i201': CFName('downward_heat_flux_in_sea_ice', None, 'W m-2'),
    'm01s03i202': CFName('downward_heat_flux_in_soil', None, 'W m-2'),
    'm01s03i209': CFName('x_wind', None, 'm s-1'),
    'm01s03i210': CFName('y_wind', None, 'm s-1'),
    'm01s03i216': CFName('upward_heat_flux_in_air', None, 'W m-2'),
    'm01s03i217': CFName('surface_upward_sensible_heat_flux', None, 'W m-2'),
    'm01s03i219': CFName(None, 'atmosphere_downward_eastward_stress', 'Pa'),
    'm01s03i220': CFName(None, 'atmosphere_downward_northward_stress', 'Pa'),
    'm01s03i222': CFName('upward_water_vapor_flux_in_air', None, 'kg m-2 s-1'),
    'm01s03i223': CFName('surface_upward_water_flux', None, 'kg m-2 s-1'),
    'm01s03i224': CFName('wind_mixing_energy_flux_into_sea_water', None, 'W m-2'),
    'm01s03i225': CFName('x_wind', None, 'm s-1'),
    'm01s03i226': CFName('y_wind', None, 'm s-1'),
    'm01s03i227': CFName('wind_speed', None, 'm s-1'),
    'm01s03i228': CFName('surface_upward_sensible_heat_flux', None, 'W m-2'),
    'm01s03i230': CFName('wind_speed', None, 'm s-1'),
    'm01s03i231': CFName(None, 'water_sublimation_flux_in_timestep', 'kg m-2'),
    'm01s03i232': CFName(None, 'Evaporation flux from open sea', 'kg/m^2/s'),
    'm01s03i234': CFName('surface_upward_latent_heat_flux', None, 'W m-2'),
    'm01s03i235': CFName(None, 'Latent heat flux from sea ice top melt', 'W/m^2'),
    'm01s03i236': CFName('air_temperature', None, 'K'),
    'm01s03i237': CFName('specific_humidity', None, '1'),
    'm01s03i238': CFName('soil_temperature', None, 'K'),
    'm01s03i245': CFName('relative_humidity', None, '%'),
    'm01s03i247': CFName('visibility_in_air', None, 'm'),
    'm01s03i248': CFName('fog_area_fraction', None, '1'),
    'm01s03i249': CFName('wind_speed', None, 'm s-1'),
    'm01s03i250': CFName('dew_point_temperature', None, 'K'),
    'm01s03i256': CFName(None, 'Heat flux through sea ice', 'W/m^2'),
    'm01s03i257': CFName(None, 'Heat flux in sea ice surface melt', 'W/m^2'),
    'm01s03i258': CFName('surface_snow_melt_heat_flux', None, 'W m-2'),
    'm01s03i261': CFName('gross_primary_productivity_of_carbon', None, 'kg m-2 s-1'),
    'm01s03i262': CFName('net_primary_productivity_of_carbon', None, 'kg m-2 s-1'),
    'm01s03i263': CFName('plant_respiration_carbon_flux', None, 'kg m-2 s-1'),
    'm01s03i270': CFName('tendency_of_atmosphere_mass_content_of_sulfur_dioxide_due_to_dry_deposition', None, 'kg m-2 s-1'),
    'm01s03i281': CFName('visibility_in_air', None, 'm'),
    'm01s03i293': CFName('soil_respiration_carbon_flux', None, 'kg m-2 s-1'),
    'm01s03i295': CFName(None, 'surface_snow_area_fraction_where_land', '%'),
    'm01s03i296': CFName(None, 'Evaporation from soil surface', 'kg/m^2/s'),
    'm01s03i297': CFName(None, 'Evaporation from canopy', 'kg/m^2/s'),
    'm01s03i298': CFName('water_sublimation_flux', None, 'kg m-2 s-1'),
    'm01s03i300': CFName('tendency_of_atmosphere_mass_content_of_ammonia_due_to_dry_deposition', None, 'kg m-2 s-1'),
    'm01s03i304': CFName(None, 'Turbulent mixing height after boundary layer', 'm'),
    'm01s03i305': CFName(None, 'Stable boundary layer indicator', '1'),
    'm01s03i306': CFName(None, 'Stratocumulus over stable boundary layer indicator', '1'),
    'm01s03i307': CFName(None, 'Well-mixed boundary layer indicator', '1'),
    'm01s03i308': CFName(None, 'Decoupled stratocumulus not over cumulus indicator', '1'),
    'm01s03i309': CFName(None, 'Decoupled stratocumulus over cumulus indicator', '1'),
    'm01s03i310': CFName(None, 'Cumulus capped boundary layer indicator', '1'),
    'm01s03i313': CFName('soil_moisture_content_at_field_capacity', None, 'kg m-2'),
    'm01s03i321': CFName(None, 'Canopy water on tiles', 'kg/m^2'),
    'm01s03i331': CFName(None, 'Sublimation moisture flux on tiles', 'kg/m^2/s'),
    'm01s03i332': CFName('toa_outgoing_longwave_flux', None, 'W m-2'),
    'm01s03i334': CFName('water_potential_evaporation_flux', None, 'kg m-2 s-1'),
    'm01s03i337': CFName('downward_heat_flux_in_soil', None, 'W m-2'),
    'm01s03i339': CFName(None, 'bulk_richardson_number', '1'),
    'm01s03i340': CFName(None, 'Shear driven boundary layer indicator', '1'),
    'm01s03i353': CFName(None, 'Sublimation of sea ice meaned over sea portion of grid box', 'kg/m^2/s'),
    'm01s03i365': CFName('x_wind', None, 'm s-1'),
    'm01s03i366': CFName('y_wind', None, 'm s-1'),
    'm01s03i380': CFName('surface_net_downward_radiative_flux', None, 'W m-2'),
    'm01s03i390': CFName('wind_speed_shear', None, 'm s-1'),
    'm01s03i391': CFName('surface_downward_eastward_stress', None, 'Pa'),
    'm01s03i392': CFName('surface_downward_eastward_stress', None, 'Pa'),
    'm01s03i393': CFName('surface_downward_northward_stress', None, 'Pa'),
    'm01s03i394': CFName('surface_downward_northward_stress', None, 'Pa'),
    'm01s03i395': CFName('land_area_fraction', None, '1'),
    'm01s03i401': CFName(None, 'Dust emissions division 1', 'kg/m^2/s'),
    'm01s03i402': CFName(None, 'Dust emissions division 2', 'kg/m^2/s'),
    'm01s03i403': CFName(None, 'Dust emissions division 3', 'kg/m^2/s'),
    'm01s03i404': CFName(None, 'Dust emissions division 4', 'kg/m^2/s'),
    'm01s03i405': CFName(None, 'Dust emissions division 5', 'kg/m^2/s'),
    'm01s03i406': CFName(None, 'Dust emissions division 6', 'kg/m^2/s'),
    'm01s03i430': CFName(None, 'Dust friction velocity', 'm/s'),
    'm01s03i441': CFName(None, 'Dust dry deposition flux division 1 from level 1', 'kg/m^2/s'),
    'm01s03i442': CFName(None, 'Dust dry deposition flux division 2 from level 1', 'kg/m^2/s'),
    'm01s03i443': CFName(None, 'Dust dry deposition flux division 3 from level 1', 'kg/m^2/s'),
    'm01s03i444': CFName(None, 'Dust dry deposition flux division 4 from level 1', 'kg/m^2/s'),
    'm01s03i445': CFName(None, 'Dust dry deposition flux division 5 from level 1', 'kg/m^2/s'),
    'm01s03i446': CFName(None, 'Dust dry deposition flux division 6 from level 1', 'kg/m^2/s'),
    'm01s03i451': CFName(None, 'Dust dry deposition flux division 1 from level 2', 'kg/m^2/s'),
    'm01s03i452': CFName(None, 'Dust dry deposition flux division 2 from level 2', 'kg/m^2/s'),
    'm01s03i453': CFName(None, 'Dust dry deposition flux division 3 from level 2', 'kg/m^2/s'),
    'm01s03i454': CFName(None, 'Dust dry deposition flux division 4 from level 2', 'kg/m^2/s'),
    'm01s03i455': CFName(None, 'Dust dry deposition flux division 5 from level 2', 'kg/m^2/s'),
    'm01s03i456': CFName(None, 'Dust dry deposition flux division 6 from level 2', 'kg/m^2/s'),
    'm01s03i460': CFName('surface_downward_eastward_stress', None, 'Pa'),
    'm01s03i461': CFName('surface_downward_northward_stress', None, 'Pa'),
    'm01s03i463': CFName('wind_speed_of_gust', None, 'm s-1'),
    'm01s03i471': CFName('atmosphere_momentum_diffusivity', None, 'm2 s-1'),
    'm01s03i472': CFName('atmosphere_heat_diffusivity', None, 'm2 s-1'),
    'm01s03i491': CFName('surface_carbon_dioxide_mole_flux', None, 'mol m-2 s-1'),
    'm01s03i538': CFName('surface_drag_coefficient_for_momentum_in_air', None, '1'),
    'm01s03i541': CFName('surface_drag_coefficient_for_heat_in_air', None, '1'),
    'm01s04i004': CFName('air_temperature', None, 'K'),
    'm01s04i010': CFName('specific_humidity', None, '1'),
    'm01s04i100': CFName(None, 'ice_aggregate_fraction', '1'),
    'm01s04i101': CFName(None, 'flag_to_indicate_microphysics_code_has_been_run', '1'),
    'm01s04i102': CFName(None, 'fall_speed_of_ice_crystals', 'm s-1'),
    'm01s04i103': CFName(None, 'fall_speed_of_ice_aggregates', 'm s-1'),
    'm01s04i104': CFName(None, 'flag_for_ice_fall_speed_in_use', '1'),
    'm01s04i105': CFName(None, 'assumed_fall_speed_of_ice', 'm s-1'),
    'm01s04i110': CFName(None, 'radar_reflectivity_due_to_all_hydrometeors_at_the_surface', 'dBZ'),
    'm01s04i111': CFName(None, 'maximum_radar_reflectivity_in_the_grid_column_due_to_all_hydrometeors', 'dBZ'),
    'm01s04i112': CFName(None, 'radar_reflectivity_due_to_all_hydrometeors_at_1km_altitude', 'dBZ'),
    'm01s04i113': CFName(None, 'radar_reflectivity_due_to_graupel_alone', 'dBZ'),
    'm01s04i114': CFName(None, 'radar_reflectivity_due_to_ice_aggregates_alone', 'dBZ'),
    'm01s04i115': CFName(None, 'radar_reflectivity_due_to_ice_crystals_alone', 'dBZ'),
    'm01s04i116': CFName(None, 'radar_reflectivity_due_to_rain_alone', 'dBZ'),
    'm01s04i117': CFName(None, 'radar_reflectivity_due_to_cloud_alone', 'dBZ'),
    'm01s04i118': CFName(None, 'radar_reflectivity_due_to_all_hydrometeor_species', 'dBZ'),
    'm01s04i119': CFName(None, 'cloud_top_altitude_derived_using_radar_echo_top_altitude', 'm'),
    'm01s04i141': CFName(None, 'change_over_time_in_air_temperature_due_to_pc2_checks', 'K'),
    'm01s04i142': CFName(None, 'change_over_time_in_specific_humidity_due_to_pc2_checks', 'kg kg-1'),
    'm01s04i143': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_pc2_checks', 'kg kg-1'),
    'm01s04i144': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_ice_in_air_due_to_pc2_checks', 'kg kg-1'),
    'm01s04i152': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_checks', '1'),
    'm01s04i153': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_checks', '1'),
    'm01s04i154': CFName(None, 'change_over_time_in_ice_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_checks', '1'),
    'm01s04i181': CFName(None, 'change_over_time_in_air_temperature_due_to_stratiform_precipitation', 'K'),
    'm01s04i182': CFName(None, 'change_over_time_in_specific_humidity_due_to_stratiform_precipitation', 'kg kg-1'),
    'm01s04i183': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_stratiform_precipitation', 'kg kg-1'),
    'm01s04i184': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_ice_in_air_due_to_stratiform_precipitation', 'kg kg-1'),
    'm01s04i189': CFName(None, 'change_over_time_in_mass_fraction_of_rain_in_air_due_to_stratiform_precipitation', 'kg kg-1'),
    'm01s04i190': CFName(None, 'graupel_mixing_ratio_increment_due_to_grid_scale_precipitation', '1'),
    'm01s04i191': CFName(None, 'change_over_time_in_mass_fraction_of_graupel_in_air_due_to_stratiform_precipitation', 'kg kg-1'),
    'm01s04i192': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer_due_to_stratiform_precipitation', '1'),
    'm01s04i193': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer_due_to_stratiform_precipitation', '1'),
    'm01s04i194': CFName(None, 'change_over_time_in_ice_cloud_volume_fraction_in_atmosphere_layer_due_to_stratiform_precipitation', '1'),
    'm01s04i201': CFName('stratiform_rainfall_amount', None, 'kg m-2'),
    'm01s04i202': CFName('stratiform_snowfall_amount', None, 'kg m-2'),
    'm01s04i203': CFName('stratiform_rainfall_flux', None, 'kg m-2 s-1'),
    'm01s04i204': CFName('stratiform_snowfall_flux', None, 'kg m-2 s-1'),
    'm01s04i205': CFName('mass_fraction_of_cloud_liquid_water_in_air', None, '1'),
    'm01s04i206': CFName('mass_fraction_of_cloud_ice_in_air', None, '1'),
    'm01s04i207': CFName(None, 'relative_humidity_with_respect_to_liquid_water', '%'),
    'm01s04i208': CFName(None, 'relative_humidity_with_respect_to_water_and_ice', '%'),
    'm01s04i209': CFName(None, 'graupel_fall_amount_at_surface', 'kg m-2'),
    'm01s04i210': CFName(None, 'cloud_drop_number_concentration_where_cloud_is_present', 'm-3'),
    'm01s04i211': CFName(None, 'cloud_drop_number_concentration_ignoring_prescence_of_cloud', 'm-3'),
    'm01s04i212': CFName(None, 'graupel_fall_flux_at_surface', 'kg m-2 s-1'),
    'm01s04i222': CFName('large_scale_rainfall_flux', None, 'kg m-2 s-1'),
    'm01s04i223': CFName('large_scale_snowfall_flux', None, 'kg m-2 s-1'),
    'm01s04i224': CFName(None, 'supercooled_liquid_water_content', '1'),
    'm01s04i225': CFName(None, 'supercooled_rainfall_flux', 'kg m-2 s-1'),
    'm01s04i226': CFName(None, 'graupel_fall_flux_on_model_levels', 'kg m-2 s-1'),
    'm01s04i227': CFName(None, 'fraction_of_grid_box_assumed_to_be_rain', '1'),
    'm01s04i231': CFName(None, 'Dust wet deposition flux due to large scale precipitation division 1', 'kg/m^2/s'),
    'm01s04i232': CFName(None, 'Dust wet deposition flux due to large scale precipitation division 2', 'kg/m^2/s'),
    'm01s04i233': CFName(None, 'Dust wet deposition flux due to large scale precipitation division 3', 'kg/m^2/s'),
    'm01s04i234': CFName(None, 'Dust wet deposition flux due to large scale precipitation division 4', 'kg/m^2/s'),
    'm01s04i235': CFName(None, 'Dust wet deposition flux due to large scale precipitation division 5', 'kg/m^2/s'),
    'm01s04i236': CFName(None, 'Dust wet deposition flux due to large scale precipitation division 6', 'kg/m^2/s'),
    'm01s04i240': CFName('tendency_of_mass_fraction_of_stratiform_cloud_ice_in_air_due_to_homogeneous_nucleation', None, 's-1'),
    'm01s04i241': CFName('tendency_of_mass_fraction_of_stratiform_cloud_ice_in_air_due_to_heterogeneous_nucleation_from_water_vapor', None, 's-1'),
    'm01s04i242': CFName('tendency_of_mass_fraction_of_stratiform_cloud_ice_in_air_due_to_heterogeneous_nucleation_from_cloud_liquid', None, 's-1'),
    'm01s04i243': CFName(None, 'rate_of_increase_of_ice_mass_due_to_vapour_deposition', 's-1'),
    'm01s04i245': CFName(None, 'rate_of_increase_of_snow_mass_due_to_vapour_deposition', 's-1'),
    'm01s04i247': CFName('tendency_of_mass_fraction_of_stratiform_cloud_ice_in_air_due_to_riming_from_cloud_liquid', None, 's-1'),
    'm01s04i248': CFName(None, 'rate_of_increase_of_snow_mass_due_to_riming_of_liquid_cloud', 's-1'),
    'm01s04i249': CFName(None, 'rate_of_increase_of_ice_mass_due_to_capture_of_raindrops', 's-1'),
    'm01s04i250': CFName(None, 'rate_of_increase_of_snow_mass_due_to_capture_of_raindrops', 's-1'),
    'm01s04i251': CFName(None, 'rate_of_loss_of_ice_mass_due_to_sublimation', 's-1'),
    'm01s04i252': CFName(None, 'rate_of_loss_of_snow_mass_due_to_sublimation', 's-1'),
    'm01s04i253': CFName(None, 'rate_of_increase_of_rain_mass_due_to_melting_of_ice_crystals', 's-1'),
    'm01s04i254': CFName(None, 'rate_of_increase_of_rain_mass_due_to_melting_of_snow', 's-1'),
    'm01s04i255': CFName(None, 'rate_of_increase_of_snow_mass_due_to_autoconversion_from_ice_crystals', 's-1'),
    'm01s04i256': CFName(None, 'rate_of_increase_of_snow_mass_due_to_capture_of_ice_crystals', 's-1'),
    'm01s04i257': CFName(None, 'rate_of_increase_of_rain_mass_due_to_autoconversion_from_liquid_cloud', 's-1'),
    'm01s04i258': CFName(None, 'rate_of_increase_of_rain_mass_due_to_accretion_of_liquid_cloud', 's-1'),
    'm01s04i259': CFName(None, 'rate_of_loss_of_rain_mass_due_to_evaporation', 's-1'),
    'm01s04i260': CFName(None, 'rate_of_increase_of_graupel_mass_due_to_autoconversion_from_snow', 's-1'),
    'm01s04i261': CFName(None, 'rate_of_change_of_graupel_mass_due_to_riming_liquid_water', 's-1'),
    'm01s04i262': CFName(None, 'rate_of_change_of_graupel_mass_due_to_capturing_snow', 's-1'),
    'm01s04i263': CFName(None, 'melting_rate_of_graupel_mass', 's-1'),
    'm01s04i264': CFName(None, 'loss_of_graupel_mass_due_to_sublimation', 's-1'),
    'm01s04i265': CFName(None, 'sedimentation_rate_of_ice_crystal_mass', 's-1'),
    'm01s04i266': CFName(None, 'sedimentation_rate_of_ice_aggregate_mass', 's-1'),
    'm01s04i267': CFName(None, 'sedimentation_rate_of_rain_mass', 's-1'),
    'm01s04i268': CFName(None, 'sedimentation_rate_of_graupel_mass', 's-1'),
    'm01s04i269': CFName(None, 'rate_of_sedimentation_of_settling_cloud_droplets', 's-1'),
    'm01s04i270': CFName(None, 'rate_of_evaporation_of_settling_cloud_droplets', 's-1'),
    'm01s04i271': CFName(None, 'rate_of_change_of_ice_mass_due_to_homogeneous_freezing_of_rain', 's-1'),
    'm01s04i272': CFName(None, 'rate_of_change_of_ice_mass_due_to_heterogeneous_freezing_of_rain', 's-1'),
    'm01s04i275': CFName(None, 'maximum_predicted_hailstone_size_at_surface', 'mm'),
    'm01s04i276': CFName(None, 'maximum_predicted_hailstone_size_in_model_vertical_column', 'mm'),
    'm01s04i277': CFName(None, 'maximum_predicted_hailstone_size', 'mm'),
    'm01s04i294': CFName(None, 'cloud_liquid_content_diagnosed_by_turbulent_mixed_phase_scheme', '1'),
    'm01s04i295': CFName(None, 'liquid_cloud_fraction_diagnosed_by_turbulent_mixed_phase_scheme', '1'),
    'm01s04i296': CFName(None, 'turbulent_decorrelation_timescale_diagnosed_by_turbulent_mixed_phase_scheme', 's'),
    'm01s04i297': CFName(None, 'time_for_in_cloud_air_to_adjust_to_ice_saturation_via_deposition', 's-1'),
    'm01s04i298': CFName(None, 'turbulent_dissipation_rate_diagnosed_by_turbulent_mixed_phase_scheme', 'm2 s-3'),
    'm01s04i299': CFName(None, 'timescale_for_tubulence_to_mix_cloud_and_environment', 's-1'),
    'm01s04i300': CFName(None, 'mean_of_subgrid_pdf_of_supersaturation_with_respect_to_ice', '1'),
    'm01s04i301': CFName(None, 'variance_of_subgrid_pdf_of_supersaturation_with_respect_to_ice', '1'),
    'm01s04i302': CFName(None, 'surface_snow_amount_ignoring_graupel', 'kg m-2'),
    'm01s04i303': CFName(None, 'cloud_liquid_content_increment_by_turbulent_mixed_phase_scheme_and_PC2_scheme', 's-1'),
    'm01s04i304': CFName(None, 'surface_snowfall_rate_ignoring_graupel', 'kg m-2 s-1'),
    'm01s04i323': CFName(None, 'snowfall_flux_on_model_levels_ignoring_graupel', 'kg m-2 s-1'),
    'm01s04i325': CFName(None, 'rate_of_change_of_liquid_cloud_mass_due_to_vapour_condensation_or_evaporation', 's-1'),
    'm01s04i336': CFName(None, 'sedimentation_rate_of_ice_cloud_mass', 's-1'),
    'm01s04i350': CFName(None, 'rate_of_change_of_ice_number_due_to_homogeneous_freezing_of_cloud', 'kg s-1'),
    'm01s04i351': CFName(None, 'rate_of_change_of_ice_number_due_to_homogeneous_freezing_of_rain', 'kg s-1'),
    'm01s04i352': CFName(None, 'rate_of_change_of_ice_number_due_to_hallett_mossop_process', 'kg-1 s-1'),
    'm01s04i353': CFName(None, 'rate_of_change_of_ice_number_due_to_ice_nucleation', 'kg-1 s-1'),
    'm01s04i354': CFName(None, 'rate_of_change_of_ice_number_due_to_snow_sedimentation', 'kg-1 s-1'),
    'm01s04i355': CFName(None, 'rate_of_change_of_snow_number_due_to_snow_sedimentation', 'kg-1 s-1'),
    'm01s04i356': CFName(None, 'rate_of_change_of_graupel_number_due_to_graupel_sedimentation', 'kg-1 s-1'),
    'm01s04i400': CFName(None, 'subgrid_orographic_cloud_mixing_ratio', '1'),
    'm01s04i401': CFName(None, 'subgrid_orographic_rain_accretion_rate', 's-1'),
    'm01s04i402': CFName(None, 'subgrid_orographic_snow_riming_rate', 's-1'),
    'm01s04i982': CFName(None, 'change_over_time_in_specific_humidity_due_to_methane_oxidation', 'kg kg-1'),
    'm01s05i010': CFName('specific_humidity', None, '1'),
    'm01s05i181': CFName(None, 'change_over_time_in_air_temperature_due_to_convection', 'K'),
    'm01s05i182': CFName(None, 'change_over_time_in_specific_humidity_due_to_convection', 'kg kg-1'),
    'm01s05i185': CFName(None, 'change_over_time_in_x_wind_due_to_convection', 'm s-1'),
    'm01s05i186': CFName(None, 'change_over_time_in_y_wind_due_to_convection', 'm s-1'),
    'm01s05i201': CFName('convective_rainfall_amount', None, 'kg m-2'),
    'm01s05i202': CFName('convective_snowfall_amount', None, 'kg m-2'),
    'm01s05i205': CFName('convective_rainfall_flux', None, 'kg m-2 s-1'),
    'm01s05i206': CFName('convective_snowfall_flux', None, 'kg m-2 s-1'),
    'm01s05i207': CFName('air_pressure_at_convective_cloud_base', None, 'Pa'),
    'm01s05i208': CFName('air_pressure_at_convective_cloud_top', None, 'Pa'),
    'm01s05i209': CFName('air_temperature', None, 'K'),
    'm01s05i212': CFName('convective_cloud_area_fraction_in_atmosphere_layer', None, '1'),
    'm01s05i213': CFName('mass_fraction_of_convective_cloud_liquid_water_in_air', None, '1'),
    'm01s05i214': CFName('rainfall_flux', None, 'kg m-2 s-1'),
    'm01s05i215': CFName('snowfall_flux', None, 'kg m-2 s-1'),
    'm01s05i216': CFName('precipitation_flux', None, 'kg m-2 s-1'),
    'm01s05i222': CFName('air_pressure_at_convective_cloud_base', None, 'Pa'),
    'm01s05i226': CFName('precipitation_amount', None, 'kg m-2'),
    'm01s05i227': CFName('convective_rainfall_flux', None, 'kg m-2 s-1'),
    'm01s05i228': CFName('convective_snowfall_flux', None, 'kg m-2 s-1'),
    'm01s05i231': CFName(None, 'Cape time scale (deep)', 's'),
    'm01s05i232': CFName(None, 'reduced cape time scale indicator', '1'),
    'm01s05i233': CFName(None, 'undilute_cape', 'J kg-1'),
    'm01s05i269': CFName(None, 'deep convection indicator', '1'),
    'm01s05i270': CFName(None, 'shallow convection indicator', '1'),
    'm01s05i272': CFName(None, 'mid level convection indicator', '1'),
    'm01s05i277': CFName(None, 'deep convective precipitation rate', 'kg/m^2/s'),
    'm01s05i278': CFName(None, 'shallow convective precipitation rate', 'kg/m^2/s'),
    'm01s05i279': CFName(None, 'mid level convective precipitation rate', 'kg/m^2/s'),
    'm01s05i281': CFName(None, 'Dust wet deposition flux due to convective precipitation division 1', 'kg/m^2/s'),
    'm01s05i282': CFName(None, 'Dust wet deposition flux due to convective precipitation division 2', 'kg/m^2/s'),
    'm01s05i283': CFName(None, 'Dust wet deposition flux due to convective precipitation division 3', 'kg/m^2/s'),
    'm01s05i284': CFName(None, 'Dust wet deposition flux due to convective precipitation division 4', 'kg/m^2/s'),
    'm01s05i285': CFName(None, 'Dust wet deposition flux due to convective precipitation division 5', 'kg/m^2/s'),
    'm01s05i286': CFName(None, 'Dust wet deposition flux due to convective precipitation division 6', 'kg/m^2/s'),
    'm01s06i111': CFName('upward_eastward_momentum_flux_in_air_due_to_nonorographic_eastward_gravity_waves', None, 'Pa'),
    'm01s06i113': CFName('upward_eastward_momentum_flux_in_air_due_to_nonorographic_westward_gravity_waves', None, 'Pa'),
    'm01s06i115': CFName('tendency_of_eastward_wind_due_to_nonorographic_gravity_wave_drag', None, 'm s-2'),
    'm01s06i181': CFName(None, 'change_over_time_in_air_temperature_due_to_gravity_wave_drag', 'K'),
    'm01s06i185': CFName(None, 'change_over_time_in_x_wind_due_to_gravity_wave_drag', 'm s-1'),
    'm01s06i186': CFName(None, 'change_over_time_in_y_wind_due_to_gravity_wave_drag', 'm s-1'),
    'm01s06i201': CFName('atmosphere_eastward_stress_due_to_gravity_wave_drag', None, 'Pa'),
    'm01s06i202': CFName('atmosphere_northward_stress_due_to_gravity_wave_drag', None, 'Pa'),
    'm01s06i241': CFName('upward_eastward_momentum_flux_in_air_due_to_orographic_gravity_waves', None, 'Pa'),
    'm01s06i247': CFName('tendency_of_eastward_wind_due_to_orographic_gravity_wave_drag', None, 'm s-2'),
    'm01s08i023': CFName('surface_snow_amount', None, 'kg m-2'),
    'm01s08i202': CFName(None, 'surface_snow_melt_flux_where_land', 'W m-2'),
    'm01s08i204': CFName('surface_runoff_amount', None, 'kg m-2'),
    'm01s08i205': CFName('subsurface_runoff_amount', None, 'kg m-2'),
    'm01s08i208': CFName('soil_moisture_content', None, 'kg m-2'),
    'm01s08i209': CFName('canopy_water_amount', None, 'kg m-2'),
    'm01s08i223': CFName('moisture_content_of_soil_layer', None, 'kg m-2'),
    'm01s08i225': CFName('soil_temperature', None, 'K'),
    'm01s08i229': CFName('mass_fraction_of_unfrozen_water_in_soil_moisture', None, 'kg kg-1'),
    'm01s08i230': CFName('mass_fraction_of_frozen_water_in_soil_moisture', None, 'kg kg-1'),
    'm01s08i231': CFName(None, 'surface_snow_melt_flux_where_land', 'kg m-2 s-1'),
    'm01s08i233': CFName('canopy_throughfall_flux', None, 'kg m-2 s-1'),
    'm01s08i234': CFName('surface_runoff_flux', None, 'kg m-2 s-1'),
    'm01s08i235': CFName('subsurface_runoff_flux', None, 'kg m-2 s-1'),
    'm01s08i245': CFName(None, 'Inland basin flow on atmospheric grid', 'kg/m^2/s'),
    'm01s08i258': CFName('surface_runoff_flux', None, 'kg m-2 s-1'),
    'm01s09i004': CFName('air_temperature', None, 'K'),
    'm01s09i010': CFName('specific_humidity', None, '1'),
    'm01s09i201': CFName('stratiform_cloud_area_fraction_in_atmosphere_layer', None, '1'),
    'm01s09i203': CFName('low_type_cloud_area_fraction', None, '1'),
    'm01s09i204': CFName('medium_type_cloud_area_fraction', None, '1'),
    'm01s09i205': CFName('high_type_cloud_area_fraction', None, '1'),
    'm01s09i208': CFName(None, 'cloud_base_altitude_assuming_only_consider_cloud_area_fraction_greater_than_0p1_oktas', 'kft'),
    'm01s09i209': CFName(None, 'cloud_base_altitude_assuming_only_consider_cloud_area_fraction_greater_than_1p5_oktas', 'kft'),
    'm01s09i210': CFName(None, 'cloud_base_altitude_assuming_only_consider_cloud_area_fraction_greater_than_2p5_oktas', 'kft'),
    'm01s09i211': CFName(None, 'cloud_base_altitude_assuming_only_consider_cloud_area_fraction_greater_than_3p5_oktas', 'kft'),
    'm01s09i212': CFName(None, 'cloud_base_altitude_assuming_only_consider_cloud_area_fraction_greater_than_4p5_oktas', 'kft'),
    'm01s09i213': CFName(None, 'cloud_base_altitude_assuming_only_consider_cloud_area_fraction_greater_than_5p5_oktas', 'kft'),
    'm01s09i214': CFName(None, 'cloud_base_altitude_assuming_only_consider_cloud_area_fraction_greater_than_6p5_oktas', 'kft'),
    'm01s09i215': CFName(None, 'cloud_base_altitude_assuming_only_consider_cloud_area_fraction_greater_than_7p9_oktas', 'kft'),
    'm01s09i216': CFName(None, 'cloud_area_fraction_assuming_random_overlap', '1'),
    'm01s09i217': CFName(None, 'cloud_area_fraction_assuming_maximum_random_overlap', '1'),
    'm01s09i218': CFName(None, 'cloud_area_fraction_assuming_only_consider_surface_to_1000_feet_asl', '1'),
    'm01s09i219': CFName('cloud_base_altitude', None, 'ft'),
    'm01s09i221': CFName(None, 'wet_bulb_freezing_level_altitude', 'm'),
    'm01s09i222': CFName('wet_bulb_temperature', None, 'K'),
    'm01s09i226': CFName(None, 'binary_mask_where_cloud_area_fraction_in_atmosphere_layer_gt_0', '1'),
    'm01s09i228': CFName(None, 'relative_humidity_at_which_cloud_assumed_to_form', '%'),
    'm01s09i229': CFName('relative_humidity', None, '%'),
    'm01s09i230': CFName(None, 'visibility_in_atmosphere_layer', 'm'),
    'm01s10i181': CFName(None, 'change_over_time_in_air_temperature_due_to_pressure_solver', 'K'),
    'm01s10i185': CFName(None, 'change_over_time_in_x_wind_due_to_pressure_solver', 'm s-1'),
    'm01s10i186': CFName(None, 'change_over_time_in_y_wind_due_to_pressure_solver', 'm s-1'),
    'm01s10i187': CFName(None, 'change_over_time_in_upward_air_velocity_due_to_pressure_solver', 'm s-1'),
    'm01s12i004': CFName('air_temperature', None, 'K'),
    'm01s12i010': CFName('specific_humidity', None, '1'),
    'm01s12i012': CFName('mass_fraction_of_cloud_ice_in_air', None, '1'),
    'm01s12i181': CFName(None, 'change_over_time_in_air_temperature_due_to_advection', 'K'),
    'm01s12i182': CFName(None, 'change_over_time_in_specific_humidity_due_to_advection', 'kg kg-1'),
    'm01s12i183': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_advection', 'kg kg-1'),
    'm01s12i184': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_ice_in_air_due_to_advection', 'kg kg-1'),
    'm01s12i185': CFName(None, 'change_over_time_in_x_wind_due_to_advection', 'm s-1'),
    'm01s12i186': CFName(None, 'change_over_time_in_y_wind_due_to_advection', 'm s-1'),
    'm01s12i187': CFName(None, 'change_over_time_in_upward_air_velocity_due_to_advection', 'm s-1'),
    'm01s12i189': CFName(None, 'change_over_time_in_mass_fraction_of_rain_in_air_due_to_advection', 'kg kg-1'),
    'm01s12i190': CFName(None, 'change_over_time_in_mass_fraction_of_graupel_in_air_due_to_advection', 'kg kg-1'),
    'm01s12i192': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer_due_to_advection', '1'),
    'm01s12i193': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer_due_to_advection', '1'),
    'm01s12i194': CFName(None, 'change_over_time_in_ice_cloud_volume_fraction_in_atmosphere_layer_due_to_advection', '1'),
    'm01s12i195': CFName(None, 'change_over_time_in_humidity_mixing_ratio_due_to_advection', 'kg kg-1'),
    'm01s12i196': CFName(None, 'change_over_time_in_cloud_liquid_water_mixing_ratio_due_to_advection', 'kg kg-1'),
    'm01s12i197': CFName(None, 'change_over_time_in_cloud_ice_mixing_ratio_due_to_advection', 'kg kg-1'),
    'm01s12i198': CFName(None, 'change_over_time_in_rain_mixing_ratio_due_to_advection', 'kg kg-1'),
    'm01s12i199': CFName(None, 'change_over_time_in_graupel_mixing_ratio_due_to_advection', 'kg kg-1'),
    'm01s12i201': CFName('lagrangian_tendency_of_air_pressure', None, 'Pa s-1'),
    'm01s12i202': CFName('lagrangian_tendency_of_air_pressure', None, 'Pa s-1'),
    'm01s12i381': CFName(None, 'change_over_time_in_air_temperature_due_to_advection_corrections', 'K'),
    'm01s12i382': CFName(None, 'change_over_time_in_specific_humidity_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i383': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i384': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_ice_in_air_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i389': CFName(None, 'change_over_time_in_mass_fraction_of_rain_in_air_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i391': CFName(None, 'change_over_time_in_mass_fraction_of_graupel_in_air_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i395': CFName(None, 'change_over_time_in_humidity_mixing_ratio_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i396': CFName(None, 'change_over_time_in_cloud_liquid_water_mixing_ratio_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i397': CFName(None, 'change_over_time_in_cloud_ice_mixing_ratio_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i398': CFName(None, 'change_over_time_in_rain_mixing_ratio_due_to_advection_corrections', 'kg kg-1'),
    'm01s12i399': CFName(None, 'change_over_time_in_graupel_mixing_ratio_due_to_advection_corrections', 'kg kg-1'),
    'm01s13i002': CFName('eastward_wind', None, 'm s-1'),
    'm01s13i003': CFName('northward_wind', None, 'm s-1'),
    'm01s13i004': CFName('air_temperature', None, 'K'),
    'm01s13i181': CFName(None, 'change_over_time_in_air_temperature_due_to_diffusion', 'K'),
    'm01s13i182': CFName(None, 'change_over_time_in_specific_humidity_due_to_diffusion', 'kg kg-1'),
    'm01s13i185': CFName(None, 'change_over_time_in_x_wind_due_to_diffusion', 'm s-1'),
    'm01s13i186': CFName(None, 'change_over_time_in_y_wind_due_to_diffusion', 'm s-1'),
    'm01s13i187': CFName(None, 'change_over_time_in_upward_air_velocity_due_to_diffusion', 'm s-1'),
    'm01s14i181': CFName(None, 'change_over_time_in_air_temperature_due_to_energy_correction', 'K'),
    'm01s15i101': CFName('height_above_reference_ellipsoid', None, 'm'),
    'm01s15i102': CFName('height_above_reference_ellipsoid', None, 'm'),
    'm01s15i108': CFName('air_pressure', None, 'Pa'),
    'm01s15i119': CFName('air_potential_temperature', None, 'K'),
    'm01s15i127': CFName('air_density', None, 'kg m-3'),
    'm01s15i142': CFName('upward_air_velocity', None, 'm s-1'),
    'm01s15i143': CFName('x_wind', None, 'm s-1'),
    'm01s15i144': CFName('y_wind', None, 'm s-1'),
    'm01s15i201': CFName('x_wind', None, 'm s-1'),
    'm01s15i202': CFName('y_wind', None, 'm s-1'),
    'm01s15i212': CFName('x_wind', None, 'm s-1'),
    'm01s15i213': CFName('y_wind', None, 'm s-1'),
    'm01s15i214': CFName('ertel_potential_vorticity', None, 'K m2 kg-1 s-1'),
    'm01s15i215': CFName('air_potential_temperature', None, 'K'),
    'm01s15i216': CFName('air_potential_temperature', None, 'K'),
    'm01s15i217': CFName('potential_vorticity_of_atmosphere_layer', None, 'Pa-1 s-1'),
    'm01s15i218': CFName('potential_vorticity_of_atmosphere_layer', None, 'Pa-1 s-1'),
    'm01s15i219': CFName('square_of_air_temperature', None, 'K2'),
    'm01s15i220': CFName(None, 'square_of_x_wind', 'm2 s-2'),
    'm01s15i221': CFName(None, 'square_of_y_wind', 'm2 s-2'),
    'm01s15i222': CFName('lagrangian_tendency_of_air_pressure', None, 'Pa s-1'),
    'm01s15i223': CFName('product_of_omega_and_air_temperature', None, 'K Pa s-1'),
    'm01s15i224': CFName(None, 'product_of_x_wind_and_omega', 'Pa m s-2'),
    'm01s15i225': CFName(None, 'product_of_y_wind_and_omega', 'Pa m s-2'),
    'm01s15i226': CFName('specific_humidity', None, 'kg kg-1'),
    'm01s15i227': CFName(None, 'product_of_x_wind_and_specific_humidity', 'm s-1'),
    'm01s15i228': CFName(None, 'product_of_y_wind_and_specific_humidity', 'm s-1'),
    'm01s15i235': CFName('product_of_omega_and_specific_humidity', None, 'Pa s-1'),
    'm01s15i238': CFName('geopotential_height', None, 'm'),
    'm01s15i239': CFName(None, 'product_of_x_wind_and_geopotential_height', 'm2 s-1'),
    'm01s15i240': CFName(None, 'product_of_y_wind_and_geopotential_height', 'm2 s-1'),
    'm01s15i242': CFName('upward_air_velocity', None, 'm s-1'),
    'm01s15i243': CFName('x_wind', None, 'm s-1'),
    'm01s15i244': CFName('y_wind', None, 'm s-1'),
    'm01s16i004': CFName('air_temperature', None, 'K'),
    'm01s16i161': CFName(None, 'change_over_time_in_air_temperature_due_to_pc2_initialisation', 'K'),
    'm01s16i162': CFName(None, 'change_over_time_in_specific_humidity_due_to_pc2_initialisation', 'kg kg-1'),
    'm01s16i163': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_pc2_initialisation', 'kg kg-1'),
    'm01s16i164': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_ice_in_air_due_to_pc2_initialisation', 'kg kg-1'),
    'm01s16i172': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_initialisation', '1'),
    'm01s16i173': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_initialisation', '1'),
    'm01s16i174': CFName(None, 'change_over_time_in_ice_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_initialisation', '1'),
    'm01s16i181': CFName(None, 'change_over_time_in_air_temperature_due_to_pc2_pressure_change', 'K'),
    'm01s16i182': CFName(None, 'change_over_time_in_specific_humidity_due_to_pc2_pressure_change', 'kg kg-1'),
    'm01s16i183': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air_due_to_pc2_pressure_change', 'kg kg-1'),
    'm01s16i184': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_ice_in_air_due_to_pc2_pressure_change', 'kg kg-1'),
    'm01s16i192': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_pressure_change', '1'),
    'm01s16i193': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_pressure_change', '1'),
    'm01s16i194': CFName(None, 'change_over_time_in_ice_cloud_volume_fraction_in_atmosphere_layer_due_to_pc2_pressure_change', '1'),
    'm01s16i201': CFName('geopotential_height', None, 'm'),
    'm01s16i202': CFName('geopotential_height', None, 'm'),
    'm01s16i203': CFName('air_temperature', None, 'K'),
    'm01s16i204': CFName('relative_humidity', None, '%'),
    'm01s16i205': CFName('wet_bulb_potential_temperature', None, 'K'),
    'm01s16i222': CFName('air_pressure_at_sea_level', None, 'Pa'),
    'm01s16i224': CFName(None, 'square_of_height', 'm2'),
    'm01s16i255': CFName('geopotential_height', None, 'm'),
    'm01s16i256': CFName('relative_humidity', None, '%'),
    'm01s17i220': CFName(None, 'mass_concentration_of_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i221': CFName(None, 'mass_concentration_of_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i222': CFName(None, 'mass_concentration_of_ammonium_sulfate_in_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i223': CFName(None, 'mass_concentration_of_ammonium_sulfate_in_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i224': CFName(None, 'mass_concentration_of_black_carbon_in_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i225': CFName(None, 'mass_concentration_of_black_carbon_in_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i226': CFName(None, 'mass_concentration_of_biomass_burning_aerosol_in_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i227': CFName(None, 'mass_concentration_of_biomass_burning_aerosol_in_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i228': CFName(None, 'mass_concentration_of_organic_carbon_from_fossil_fuel_combustion_in_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i229': CFName(None, 'mass_concentration_of_organic_carbon_from_fossil_fuel_combustion_in_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i230': CFName(None, 'mass_concentration_of_secondary_particulate_organic_matter_in_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i231': CFName(None, 'mass_concentration_of_secondary_particulate_organic_matter_in_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i232': CFName(None, 'mass_concentration_of_seasalt_in_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i233': CFName(None, 'mass_concentration_of_seasalt_in_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i234': CFName(None, 'mass_concentration_of_dust_in_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i235': CFName(None, 'mass_concentration_of_dust_in_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i236': CFName(None, 'mass_concentration_of_ammonium_nitrate_in_pm10_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i237': CFName(None, 'mass_concentration_of_ammonium_nitrate_in_pm2p5_dry_aerosol_in_air', 'ug m-3'),
    'm01s17i257': CFName('mass_concentration_of_dust_dry_aerosol_in_air', None, 'ug m-3'),
    'm01s19i002': CFName('vegetation_carbon_content', None, 'kg m-2'),
    'm01s19i016': CFName('soil_carbon_content', None, 'kg m-2'),
    'm01s20i003': CFName('wind_speed', None, 'm s-1'),
    'm01s20i004': CFName('wind_speed', None, 'm s-1'),
    'm01s20i005': CFName('divergence_of_wind', None, 's-1'),
    'm01s20i006': CFName('atmosphere_relative_vorticity', None, 's-1'),
    'm01s20i024': CFName('tropopause_air_pressure', None, 'Pa'),
    'm01s20i025': CFName('tropopause_air_temperature', None, 'K'),
    'm01s20i026': CFName('tropopause_altitude', None, 'm'),
    'm01s20i034': CFName('air_pressure_at_freezing_level', None, 'Pa'),
    'm01s20i064': CFName('tropopause_air_pressure', None, 'Pa'),
    'm01s20i065': CFName('tropopause_air_temperature', None, 'K'),
    'm01s20i066': CFName('tropopause_altitude', None, 'm'),
    'm01s21i100': CFName(None, 'lightning_flash_rate', 's-1'),
    'm01s21i101': CFName(None, 'flag_for_location_of_storms', '1'),
    'm01s21i102': CFName(None, 'graupel_water_path', 'kg m-2'),
    'm01s21i103': CFName(None, 'total_ice_water_path', 'kg m-2'),
    'm01s21i104': CFName(None, 'Number_of_lightning_flashes', '1'),
    'm01s21i105': CFName(None, 'lightning_flash_rate_due_to_graupel_flux', 's-1'),
    'm01s21i106': CFName(None, 'lightning_flash_rate_due_to_total_ice_water_path', 's-1'),
    'm01s26i001': CFName(None, 'river water storage', 'kg'),
    'm01s26i002': CFName(None, 'gridbox outflow', 'kg/s'),
    'm01s26i003': CFName(None, 'gridbox inflow', 'kg/s'),
    'm01s26i004': CFName('water_flux_into_sea_water_from_rivers', None, 'kg m-2 s-1'),
    'm01s26i006': CFName(None, 'Inland basin flow on trip grid', 'kg/s'),
    'm01s30i003': CFName('upward_air_velocity', None, 'm s-1'),
    'm01s30i004': CFName('air_temperature', None, 'K'),
    'm01s30i005': CFName('specific_humidity', None, '1'),
    'm01s30i007': CFName('specific_kinetic_energy_of_air', None, 'm2 s-2'),
    'm01s30i008': CFName('lagrangian_tendency_of_air_pressure', None, 'Pa s-1'),
    'm01s30i111': CFName('air_temperature', None, 'K'),
    'm01s30i113': CFName('relative_humidity', None, '%'),
    'm01s30i181': CFName(None, 'change_over_time_in_air_temperature', 'K'),
    'm01s30i182': CFName(None, 'change_over_time_in_specific_humidity', 'kg kg-1'),
    'm01s30i183': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_liquid_water_in_air', 'kg kg-1'),
    'm01s30i184': CFName(None, 'change_over_time_in_mass_fraction_of_cloud_ice_in_air', 'kg kg-1'),
    'm01s30i185': CFName(None, 'change_over_time_in_x_wind', 'm s-1'),
    'm01s30i186': CFName(None, 'change_over_time_in_y_wind', 'm s-1'),
    'm01s30i187': CFName(None, 'change_over_time_in_upward_air_velocity', 'm s-1'),
    'm01s30i188': CFName('tendency_of_air_density', None, 'kg m-3 s-1'),
    'm01s30i189': CFName(None, 'change_over_time_in_mass_fraction_of_rain_in_air', 'kg kg-1'),
    'm01s30i191': CFName(None, 'change_over_time_in_mass_fraction_of_graupel_in_air', 'kg kg-1'),
    'm01s30i192': CFName(None, 'change_over_time_in_cloud_volume_fraction_in_atmosphere_layer', '1'),
    'm01s30i193': CFName(None, 'change_over_time_in_liquid_water_cloud_volume_fraction_in_atmosphere_layer', '1'),
    'm01s30i194': CFName(None, 'change_over_time_in_ice_cloud_volume_fraction_in_atmosphere_layer', '1'),
    'm01s30i195': CFName(None, 'change_over_time_in_humidity_mixing_ratio', 'kg kg-1'),
    'm01s30i196': CFName(None, 'change_over_time_in_cloud_liquid_water_mixing_ratio', 'kg kg-1'),
    'm01s30i197': CFName(None, 'change_over_time_in_cloud_ice_mixing_ratio', 'kg kg-1'),
    'm01s30i198': CFName(None, 'change_over_time_in_rain_mixing_ratio', 'kg kg-1'),
    'm01s30i199': CFName(None, 'change_over_time_in_graupel_mixing_ratio', 'kg kg-1'),
    'm01s30i201': CFName('x_wind', None, 'm s-1'),
    'm01s30i202': CFName('y_wind', None, 'm s-1'),
    'm01s30i203': CFName('upward_air_velocity', None, 'm s-1'),
    'm01s30i204': CFName('air_temperature', None, 'K'),
    'm01s30i205': CFName('specific_humidity', None, '1'),
    'm01s30i206': CFName('relative_humidity', None, '%'),
    'm01s30i207': CFName('geopotential_height', None, 'm'),
    'm01s30i208': CFName('lagrangian_tendency_of_air_pressure', None, 'Pa s-1'),
    'm01s30i211': CFName('square_of_eastward_wind', None, 'm2 s-2'),
    'm01s30i212': CFName('product_of_eastward_wind_and_northward_wind', None, 'm2 s-2'),
    'm01s30i213': CFName('product_of_eastward_wind_and_upward_air_velocity', None, 'm2 s-2'),
    'm01s30i214': CFName('product_of_eastward_wind_and_air_temperature', None, 'K m s-1'),
    'm01s30i215': CFName('product_of_eastward_wind_and_specific_humidity', None, 'm s-1'),
    'm01s30i217': CFName('product_of_eastward_wind_and_geopotential_height', None, 'm2 s-1'),
    'm01s30i218': CFName(None, 'product_of_x_wind_and_omega', 'Pa m s-2'),
    'm01s30i222': CFName('square_of_northward_wind', None, 'm2 s-2'),
    'm01s30i223': CFName('product_of_northward_wind_and_upward_air_velocity', None, 'm2 s-2'),
    'm01s30i224': CFName('product_of_northward_wind_and_air_temperature', None, 'K m s-1'),
    'm01s30i225': CFName('product_of_northward_wind_and_specific_humidity', None, 'm s-1'),
    'm01s30i227': CFName('product_of_northward_wind_and_geopotential_height', None, 'm2 s-1'),
    'm01s30i228': CFName(None, 'product_of_y_wind_and_omega', 'Pa m s-2'),
    'm01s30i233': CFName('square_of_upward_air_velocity', None, 'm2 s-2'),
    'm01s30i234': CFName('product_of_upward_air_velocity_and_air_temperature', None, 'K m s-1'),
    'm01s30i235': CFName('product_of_upward_air_velocity_and_specific_humidity', None, 'm s-1'),
    'm01s30i244': CFName('square_of_air_temperature', None, 'K2'),
    'm01s30i245': CFName('product_of_air_temperature_and_specific_humidity', None, 'K'),
    'm01s30i248': CFName('product_of_air_temperature_and_omega', None, 'K Pa s-1'),
    'm01s30i258': CFName('product_of_specific_humidity_and_omega', None, 'Pa s-1'),
    'm01s30i277': CFName('square_of_geopotential_height', None, 'm2'),
    'm01s30i278': CFName('product_of_geopotential_height_and_omega', None, 'Pa m s-1'),
    'm01s30i288': CFName('square_of_lagrangian_tendency_of_air_pressure', None, 'Pa2 s-2'),
    'm01s30i301': CFName(None, 'Heavyside function on pressure levels', '1'),
    'm01s30i302': CFName('virtual_temperature', None, 'K'),
    'm01s30i310': CFName('northward_transformed_eulerian_mean_air_velocity', None, 'm s-1'),
    'm01s30i311': CFName('northward_transformed_eulerian_mean_air_velocity', None, 'm s-1'),
    'm01s30i312': CFName('northward_eliassen_palm_flux_in_air', None, 'kg s-2'),
    'm01s30i313': CFName('upward_eliassen_palm_flux_in_air', None, 'kg s-2'),
    'm01s30i314': CFName('tendency_of_eastward_wind_due_to_eliassen_palm_flux_divergence', None, 'm s-2'),
    'm01s30i401': CFName('atmosphere_kinetic_energy_content', None, 'J m-2'),
    'm01s30i404': CFName('atmosphere_mass_per_unit_area', None, 'kg m-2'),
    'm01s30i405': CFName('atmosphere_cloud_liquid_water_content', None, 'kg m-2'),
    'm01s30i406': CFName('atmosphere_cloud_ice_content', None, 'kg m-2'),
    'm01s30i417': CFName('surface_air_pressure', None, 'Pa'),
    'm01s30i418': CFName('surface_air_pressure', None, 'Pa'),
    'm01s30i451': CFName('tropopause_air_pressure', None, 'Pa'),
    'm01s30i452': CFName('tropopause_air_temperature', None, 'K'),
    'm01s30i453': CFName('tropopause_altitude', None, 'm'),
    'm01s30i901': CFName(None, 'change_over_time_in_air_potential_temperature', 'K'),
    'm01s30i902': CFName(None, 'change_over_time_in_virtual_potential_temperature', 'K'),
    'm01s30i903': CFName(None, 'change_over_time_in_air_density', 'kg m-3'),
    'm01s33i001': CFName('mole_fraction_of_ozone_in_air', None, 'mole mole-1'),
    'm01s33i004': CFName(None, 'mole_fraction_of_nitrogen_trioxide_in_air', 'mole mole-1'),
    'm01s33i005': CFName('mole_fraction_of_dinitrogen_pentoxide_in_air', None, 'mole mole-1'),
    'm01s33i006': CFName('mole_fraction_of_peroxynitric_acid_in_air', None, 'mole mole-1'),
    'm01s33i007': CFName('mole_fraction_of_chlorine_nitrate_in_air', None, 'mole mole-1'),
    'm01s33i009': CFName('mole_fraction_of_methane_in_air', None, 'mole mole-1'),
    'm01s33i041': CFName('mole_fraction_of_atomic_chlorine_in_air', None, '1'),
    'm01s33i042': CFName('mole_fraction_of_chlorine_monoxide_in_air', None, '1'),
    'm01s33i043': CFName('mole_fraction_of_dichlorine_peroxide_in_air', None, '1'),
    'm01s33i044': CFName('mole_fraction_of_chlorine_dioxide_in_air', None, '1'),
    'm01s33i047': CFName('mole_fraction_of_bromine_chloride_in_air', None, '1'),
    'm01s33i048': CFName('mole_fraction_of_bromine_nitrate_in_air', None, '1'),
    'm01s33i049': CFName('mole_fraction_of_nitrous_oxide_in_air', None, '1'),
    'm01s33i051': CFName('mole_fraction_of_hypochlorous_acid_in_air', None, '1'),
    'm01s33i054': CFName('mole_fraction_of_chlorine_nitrate_in_air', None, '1'),
    'm01s33i055': CFName('mole_fraction_of_cfc11_in_air', None, '1'),
    'm01s33i056': CFName('mole_fraction_of_cfc12_in_air', None, '1'),
    'm01s33i058': CFName('mole_fraction_of_atomic_nitrogen_in_air', None, '1'),
    'm01s33i150': CFName('age_of_stratospheric_air', None, 's'),
    'm01s34i001': CFName('mass_fraction_of_ozone_in_air', None, 'kg kg-1'),
    'm01s34i002': CFName('mass_fraction_of_nitrogen_monoxide_in_air', None, 'kg kg-1'),
    'm01s34i003': CFName('mass_fraction_of_nitrate_radical_in_air', None, 'kg kg-1'),
    'm01s34i004': CFName('mass_fraction_of_nitrogen_dioxide_in_air', None, 'kg kg-1'),
    'm01s34i005': CFName('mass_fraction_of_dinitrogen_pentoxide_in_air', None, 'kg kg-1'),
    'm01s34i006': CFName('mass_fraction_of_peroxynitric_acid_in_air', None, 'kg kg-1'),
    'm01s34i007': CFName('mass_fraction_of_nitric_acid_in_air', None, 'kg kg-1'),
    'm01s34i008': CFName('mass_fraction_of_hydrogen_peroxide_in_air', None, 'kg kg-1'),
    'm01s34i009': CFName('mass_fraction_of_methane_in_air', None, 'kg kg-1'),
    'm01s34i010': CFName('mass_fraction_of_carbon_monoxide_in_air', None, 'kg kg-1'),
    'm01s34i011': CFName('mass_fraction_of_formaldehyde_in_air', None, 'kg kg-1'),
    'm01s34i012': CFName('mass_fraction_of_methyl_hydroperoxide_in_air', None, 'kg kg-1'),
    'm01s34i013': CFName('mass_fraction_of_nitrous_acid_in_air', None, 'kg kg-1'),
    'm01s34i014': CFName('mass_fraction_of_ethane_in_air', None, 'kg kg-1'),
    'm01s34i015': CFName(None, 'mass_fraction_of_ethyl_hydroperoxide_in_air', 'kg kg-1'),
    'm01s34i016': CFName(None, 'mass_fraction_of_acetaldehyde_in_air', 'kg kg-1'),
    'm01s34i017': CFName('mass_fraction_of_peroxyacetyl_nitrate_in_air', None, 'kg kg-1'),
    'm01s34i018': CFName('mass_fraction_of_propane_in_air', None, 'kg kg-1'),
    'm01s34i019': CFName(None, 'mass_fraction_of_n-propyl_hydroperoxide_in_air', 'kg kg-1'),
    'm01s34i020': CFName(None, 'mass_fraction_of_i-propyl_hydroperoxide_in_air', 'kg kg-1'),
    'm01s34i021': CFName(None, 'mass_fraction_of_propanal_in_air', 'kg kg-1'),
    'm01s34i022': CFName(None, 'mass_fraction_of_acetone_in_air', 'kg kg-1'),
    'm01s34i023': CFName(None, 'mass_fraction_of_acetonylhydroperoxide_in_air', 'kg kg-1'),
    'm01s34i024': CFName(None, 'mass_fraction_of_peroxypropionyl_nitrate_in_air', 'kg kg-1'),
    'm01s34i025': CFName(None, 'mass_fraction_of_methyl_nitrate_in_air', 'kg kg-1'),
    'm01s34i026': CFName(None, 'mass_fraction_of_stratospheric_ozone_in_air', 'kg kg-1'),
    'm01s34i027': CFName('mass_fraction_of_isoprene_in_air', None, 'kg kg-1'),
    'm01s34i028': CFName(None, 'mass_fraction_of_isoprene_hydroperoxide_in_air', 'kg kg-1'),
    'm01s34i030': CFName(None, 'mass_fraction_of_methacrolein_in_air', 'kg kg-1'),
    'm01s34i031': CFName(None, 'mass_fraction_of_methacroyl_hydroperoxide_in_air', 'kg kg-1'),
    'm01s34i032': CFName(None, 'mass_fraction_of_methacryloylperoxy_nitrate_in_air', 'kg kg-1'),
    'm01s34i033': CFName(None, 'mass_fraction_of_hydroxyacetone_in_air', 'kg kg-1'),
    'm01s34i034': CFName(None, 'mass_fraction_of_methlyglyoxal_in_air', 'kg kg-1'),
    'm01s34i035': CFName(None, 'mass_fraction_of_second_generation_isoprene_nitrate_in_air', 'kg kg-1'),
    'm01s34i036': CFName('mass_fraction_of_formic_acid_in_air', None, 'kg kg-1'),
    'm01s34i037': CFName(None, 'mass_fraction_of_peracetic_acid_in_air', 'kg kg-1'),
    'm01s34i038': CFName('mass_fraction_of_acetic_acid_in_air', None, 'kg kg-1'),
    'm01s34i041': CFName('mass_fraction_of_atomic_chlorine_in_air', None, 'kg kg-1'),
    'm01s34i042': CFName('mass_fraction_of_chlorine_monoxide_in_air', None, 'kg kg-1'),
    'm01s34i043': CFName('mass_fraction_of_dichlorine_peroxide_in_air', None, 'kg kg-1'),
    'm01s34i044': CFName('mass_fraction_of_chlorine_dioxide_in_air', None, 'kg kg-1'),
    'm01s34i045': CFName('mass_fraction_of_atomic_bromine_in_air', None, 'kg kg-1'),
    'm01s34i047': CFName('mass_fraction_of_bromine_chloride_in_air', None, 'kg kg-1'),
    'm01s34i048': CFName('mass_fraction_of_bromine_nitrate_in_air', None, 'kg kg-1'),
    'm01s34i049': CFName('mass_fraction_of_nitrous_oxide_in_air', None, 'kg kg-1'),
    'm01s34i051': CFName('mass_fraction_of_hypochlorous_acid_in_air', None, 'kg kg-1'),
    'm01s34i052': CFName('mass_fraction_of_hydrogen_bromide_in_air', None, 'kg kg-1'),
    'm01s34i053': CFName('mole_fraction_of_hypobromous_acid_in_air', None, 'kg kg-1'),
    'm01s34i054': CFName('mass_fraction_of_chlorine_nitrate_in_air', None, 'kg kg-1'),
    'm01s34i055': CFName('mass_fraction_of_cfc11_in_air', None, 'kg kg-1'),
    'm01s34i056': CFName('mass_fraction_of_cfc12_in_air', None, 'kg kg-1'),
    'm01s34i057': CFName('mass_fraction_of_methyl_bromide_in_air', None, 'kg kg-1'),
    'm01s34i058': CFName('mass_fraction_of_atomic_nitrogen_in_air', None, 'kg kg-1'),
    'm01s34i059': CFName(None, 'mass_fraction_of_ground_state_atomic_oxygen_in_air', 'kg kg-1'),
    'm01s34i070': CFName('mass_fraction_of_molecular_hydrogen_in_air', None, 'kg kg-1'),
    'm01s34i071': CFName('mass_fraction_of_dimethyl_sulfide_in_air', None, 'kg kg-1'),
    'm01s34i072': CFName('mass_fraction_of_sulfur_dioxide_in_air', None, 'kg kg-1'),
    'm01s34i073': CFName('mass_fraction_of_sulfuric_acid_in_air', None, 'kg kg-1'),
    'm01s34i074': CFName(None, 'mass_fraction_of_methanesulfonic_acid_in_air', 'kg kg-1'),
    'm01s34i075': CFName(None, 'mass_fraction_of_dimethyl_sulfoxide', 'kg kg-1'),
    'm01s34i076': CFName('mass_fraction_of_ammonia_in_air', None, 'kg kg-1'),
    'm01s34i077': CFName(None, 'mass_fraction_of_carbon_disulfide_in_air', 'kg kg-1'),
    'm01s34i078': CFName(None, 'mass_fraction_of_carbonyl_sulfide_in_air', 'kg kg-1'),
    'm01s34i079': CFName(None, 'mass_fraction_of_hydrogen_sulfide_in_air', 'kg kg-1'),
    'm01s34i080': CFName(None, 'mass_fraction_of_atomic_hydrogen_in_air', 'kg kg-1'),
    'm01s34i081': CFName('mass_fraction_of_hydroxyl_radical_in_air', None, 'kg kg-1'),
    'm01s34i082': CFName('mass_fraction_of_hydroperoxyl_radical_in_air', None, 'kg kg-1'),
    'm01s34i083': CFName('mass_fraction_of_methyl_peroxy_radical_in_air', None, 'kg kg-1'),
    'm01s34i084': CFName(None, 'mass_fraction_of_ethyl_peroxy_radical_in_air', 'kg kg-1'),
    'm01s34i085': CFName(None, 'mass_fraction_of_peroxyacetyl_radical_in_air', 'kg kg-1'),
    'm01s34i086': CFName(None, 'mass_fraction_of_n-propylperoxy_radical_in_air', 'kg kg-1'),
    'm01s34i087': CFName(None, 'mass_fraction_of_isopropylperoxy_radical_in_air', 'kg kg-1'),
    'm01s34i088': CFName(None, 'mass_fraction_of_peroxypropanoyl_radical_in_air', 'kg kg-1'),
    'm01s34i089': CFName(None, 'mass_fraction_of_acetonyl_peroxy_radical_in_air', 'kg kg-1'),
    'm01s34i093': CFName('mass_fraction_of_propene_in_air', None, 'kg kg-1'),
    'm01s34i096': CFName(None, 'mass_fraction_of_methyl_ethyl_ketone_in_air', 'kg kg-1'),
    'm01s34i097': CFName('mass_fraction_of_toluene_in_air', None, 'kg kg-1'),
    'm01s34i100': CFName(None, 'mass_fraction_of_lumped_chlorine_expressed_as_hydrogen_chloride', 'kg kg-1'),
    'm01s34i101': CFName(None, 'number_of_particles_per_air_molecule_of_soluble_nucleation_mode_aerosol_in_air', '1'),
    'm01s34i102': CFName(None, 'mass_fraction_of_sulfuric_acid_in_soluble_nucleation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i103': CFName(None, 'number_of_particles_per_air_molecule_of_soluble_aitken_mode_aerosol_in_air', '1'),
    'm01s34i104': CFName(None, 'mass_fraction_of_sulfuric_acid_in_soluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i105': CFName(None, 'mass_fraction_of_black_carbon_in_soluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i106': CFName(None, 'mass_fraction_of_particulate_organic_matter_in_soluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i107': CFName(None, 'number_of_particles_per_air_molecule_of_soluble_accumulation_mode_aerosol_in_air', '1'),
    'm01s34i108': CFName(None, 'mass_fraction_of_sulfuric_acid_in_soluble_accumulation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i109': CFName(None, 'mass_fraction_of_black_carbon_in_soluble_accumulation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i110': CFName(None, 'mass_fraction_of_particulate_organic_matter_in_soluble_accumulation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i111': CFName(None, 'mass_fraction_of_seasalt_in_soluble_accumulation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i112': CFName(None, 'mass_fraction_of_dust_in_soluble_accumulation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i113': CFName(None, 'number_of_particles_per_air_molecule_of_soluble_coarse_mode_aerosol_in_air', '1'),
    'm01s34i114': CFName(None, 'mass_fraction_of_sulfuric_acid_in_soluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i115': CFName(None, 'mass_fraction_of_black_carbon_in_soluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i116': CFName(None, 'mass_fraction_of_particulate_organic_matter_in_soluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i117': CFName(None, 'mass_fraction_of_seasalt_in_soluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i118': CFName(None, 'mass_fraction_of_dust_in_soluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i119': CFName(None, 'number_of_particles_per_air_molecule_of_insoluble_aitken_mode_aerosol_in_air', '1'),
    'm01s34i120': CFName(None, 'mass_fraction_of_black_carbon_in_insoluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i121': CFName(None, 'mass_fraction_of_particulate_organic_matter_in_insoluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i122': CFName(None, 'number_of_particles_per_air_molecule_of_insoluble_accumulation_mode_aerosol_in_air', '1'),
    'm01s34i123': CFName(None, 'mass_fraction_of_dust_in_insoluble_accumulation_mode_aerosol_in_air', 'kg kg-1'),
    'm01s34i124': CFName(None, 'number_of_particles_per_air_molecule_of_insoluble_coarse_mode_aerosol_in_air', '1'),
    'm01s34i125': CFName(None, 'mass_fraction_of_dust_in_insoluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i126': CFName(None, 'mass_fraction_of_particulate_organic_matter_in_soluble_nucleation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i127': CFName(None, 'mass_fraction_of_seasalt_in_soluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i128': CFName(None, 'mass_fraction_of_secondary_particulate_organic_matter_in_soluble_nucleation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i129': CFName(None, 'mass_fraction_of_secondary_particulate_organic_matter_in_soluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i130': CFName(None, 'mass_fraction_of_secondary_particulate_organic_matter_in_soluble_accumulation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i131': CFName(None, 'mass_fraction_of_secondary_particulate_organic_matter_in_soluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i132': CFName(None, 'mass_fraction_of_ammonium_in_soluble_nucleation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i133': CFName(None, 'mass_fraction_of_ammonium_in_soluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i134': CFName(None, 'mass_fraction_of_ammonium_in_soluble_accumulation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i135': CFName(None, 'mass_fraction_of_ammonium_in_soluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i136': CFName(None, 'mass_fraction_of_nitrate_in_soluble_nucleation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i137': CFName(None, 'mass_fraction_of_nitrate_in_soluble_aitken_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i138': CFName(None, 'mass_fraction_of_nitrate_in_soluble_accumulation_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i139': CFName(None, 'mass_fraction_of_nitrate_in_soluble_coarse_mode_dry_aerosol_in_air', 'kg kg-1'),
    'm01s34i150': CFName('age_of_stratospheric_air', None, 's'),
    'm01s34i159': CFName('equivalent_thickness_at_stp_of_atmosphere_ozone_content', None, 'DU'),
    'm01s35i003': CFName(None, 'change_over_time_in_x_wind_due_to_stochastic_kinetic_energy_backscatter', 'm s-1'),
    'm01s35i004': CFName(None, 'change_over_time_in_y_wind_due_to_stochastic_kinetic_energy_backscatter', 'm s-1'),
    'm01s35i024': CFName(None, 'change_over_time_in_air_potential_temperature_due_to_stochastic_perturbation_of_tendencies', 'K'),
    'm01s35i025': CFName(None, 'change_over_time_in_specific_humidity_due_to_stochastic_perturbation_of_tendencies', 'kg kg-1'),
    'm01s35i026': CFName(None, 'change_over_time_in_x_wind_due_to_stochastic_perturbation_of_tendencies', 'm s-1'),
    'm01s35i027': CFName(None, 'change_over_time_in_y_wind_due_to_stochastic_perturbation_of_tendencies', 'm s-1'),
    'm01s35i029': CFName(None, 'change_over_time_in_air_temperature_due_to_stochastic_perturbation_of_tendencies', 'K'),
    'm01s50i228': CFName('photolysis_rate_of_ozone_to_1D_oxygen_atom', None, 's-1'),
    'm01s50i229': CFName('photolysis_rate_of_nitrogen_dioxide', None, 's-1'),
    'm01s50i230': CFName('mass_concentration_of_nmvoc_expressed_as_carbon_in_air', None, 'ug m-3'),
    'm02s00i101': CFName('sea_water_potential_temperature', None, 'degC'),
    'm02s00i102': CFName('sea_water_salinity', None, '1e3 @0.035'),
    'm02s00i121': CFName('baroclinic_eastward_sea_water_velocity', None, 'cm s-1'),
    'm02s00i122': CFName('baroclinic_northward_sea_water_velocity', None, 'cm s-1'),
    'm02s00i130': CFName('ocean_barotropic_streamfunction', None, 'cm3 s-1'),
    'm02s00i131': CFName('ocean_barotropic_streamfunction', None, 'cm3 s-1'),
    'm02s00i132': CFName('tendency_of_ocean_barotropic_streamfunction', None, 'cm3 s-2'),
    'm02s00i133': CFName('tendency_of_ocean_barotropic_streamfunction', None, 'cm3 s-2'),
    'm02s00i134': CFName('surface_air_pressure', None, 'g cm-1 s-2'),
    'm02s00i135': CFName('barotropic_eastward_sea_water_velocity', None, 'cm s-1'),
    'm02s00i136': CFName('barotropic_northward_sea_water_velocity', None, 'cm s-1'),
    'm02s00i137': CFName('ocean_mixed_layer_thickness', None, 'm'),
    'm02s00i139': CFName('downward_eastward_stress_at_sea_ice_base', None, 'Pa'),
    'm02s00i140': CFName('downward_northward_stress_at_sea_ice_base', None, 'Pa'),
    'm02s00i141': CFName('surface_snow_thickness', None, 'm'),
    'm02s00i143': CFName('upward_sea_ice_basal_heat_flux', None, 'W m-2'),
    'm02s00i146': CFName('sea_ice_area_fraction', None, '1'),
    'm02s00i147': CFName('sea_ice_thickness', None, 'm'),
    'm02s00i148': CFName('eastward_sea_ice_velocity', None, 'm s-1'),
    'm02s00i149': CFName('northward_sea_ice_velocity', None, 'm s-1'),
    'm02s00i150': CFName('surface_downward_eastward_stress', None, 'Pa'),
    'm02s00i151': CFName('surface_downward_northward_stress', None, 'Pa'),
    'm02s00i152': CFName('wind_mixing_energy_flux_into_sea_water', None, 'W m-2'),
    'm02s00i166': CFName('water_flux_into_sea_water_from_rivers', None, 'kg m-2 s-1'),
    'm02s00i171': CFName('snowfall_flux', None, 'kg m-2 s-1'),
    'm02s00i172': CFName('surface_snow_and_ice_sublimation_flux', None, 'kg m-2 s-1'),
    'm02s00i180': CFName('sea_surface_temperature', None, 'K'),
    'm02s00i181': CFName('sea_surface_salinity', None, '1e3 @0.035'),
    'm02s00i182': CFName('air_temperature', None, 'K'),
    'm02s00i183': CFName('sea_ice_thickness', None, 'm'),
    'm02s00i185': CFName('heat_flux_correction', None, 'W m-2'),
    'm02s00i186': CFName('water_flux_correction', None, 'kg m-2 s-1'),
    'm02s00i190': CFName('surface_snow_and_ice_melt_heat_flux', None, 'W m-2'),
    'm02s00i191': CFName('downward_heat_flux_in_sea_ice', None, 'W m-2'),
    'm02s00i192': CFName('water_flux_into_sea_water_due_to_sea_ice_thermodynamics', None, 'kg m-2 s-1'),
    'm02s30i201': CFName('upward_sea_water_velocity', None, 'cm s-1'),
    'm02s30i202': CFName('ocean_mixed_layer_thickness', None, 'm'),
    'm02s30i211': CFName('northward_ocean_heat_transport', None, 'PW'),
    'm02s30i212': CFName('northward_ocean_salt_transport', None, '1e7kg s-1'),
    'm02s30i320': CFName('eastward_sea_water_velocity', None, 'cm s-1'),
    'm02s30i321': CFName('northward_sea_water_velocity', None, 'cm s-1'),
    'm02s30i324': CFName('ocean_mixed_layer_thickness', None, 'm'),
    'm02s30i406': CFName(None, 'mole_concentration_of_dimethyl_sulphide_in_seawater', 'mol m-3'),
    'm02s32i201': CFName('tendency_of_sea_ice_area_fraction_due_to_dynamics', None, 's-1'),
    'm02s32i202': CFName('tendency_of_sea_ice_thickness_due_to_dynamics', None, 'm s-1'),
    'm02s32i209': CFName('eastward_sea_ice_velocity', None, 'm s-1'),
    'm02s32i210': CFName('northward_sea_ice_velocity', None, 'm s-1'),
    'm02s32i211': CFName('tendency_of_sea_ice_area_fraction_due_to_thermodynamics', None, 's-1'),
    'm02s32i212': CFName('tendency_of_sea_ice_thickness_due_to_thermodynamics', None, 'm s-1'),
    'm02s32i215': CFName('snowfall_flux', None, 'kg m-2 s-1'),
    'm02s32i219': CFName('downward_eastward_stress_at_sea_ice_base', None, 'Pa'),
    'm02s32i220': CFName('downward_northward_stress_at_sea_ice_base', None, 'Pa'),
    'm03s00i177': CFName(None, 'prescribed_heat_flux_into_slab_ocean', 'W m-2'),
    'm04s06i001': CFName('sea_surface_wind_wave_significant_height', None, 'm'),
    }

STASHCODE_IMPLIED_HEIGHTS = {
    'm01s03i209': (10.0,),
    'm01s03i210': (10.0,),
    'm01s03i225': (10.0,),
    'm01s03i226': (10.0,),
    'm01s03i227': (10.0,),
    'm01s03i230': (10.0,),
    'm01s03i236': (1.5,),
    'm01s03i237': (1.5,),
    'm01s03i245': (1.5,),
    'm01s03i247': (1.5,),
    'm01s03i250': (1.5,),
    'm01s03i281': (1.5,),
    'm01s03i365': (10.0,),
    'm01s03i366': (10.0,),
    'm01s03i463': (10.0,),
    'm01s15i212': (50.0,),
    'm01s15i213': (50.0,),
    }

CF_TO_LBFC = {
    CFName(None, 'stratiform_snowfall_rate', 'kg m-2 s-1'): 118,
    CFName('age_of_stratospheric_air', None, '1'): 501,
    CFName('air_density', None, 'kg m-3'): 27,
    CFName('air_potential_temperature', None, 'K'): 19,
    CFName('air_pressure', None, 'Pa'): 8,
    CFName('air_pressure_at_freezing_level', None, 'Pa'): 8,
    CFName('air_pressure_at_sea_level', None, 'Pa'): 8,
    CFName('air_temperature', None, 'K'): 16,
    CFName('atmosphere_boundary_layer_thickness', None, 'm'): 5,
    CFName('atmosphere_eastward_stress_due_to_gravity_wave_drag', None, 'Pa'): 61,
    CFName('atmosphere_kinetic_energy_content', None, 'J m-2'): 63,
    CFName('atmosphere_northward_stress_due_to_gravity_wave_drag', None, 'Pa'): 62,
    CFName('atmosphere_relative_vorticity', None, 's-1'): 73,
    CFName('cloud_area_fraction', None, '1'): 30,
    CFName('cloud_area_fraction_in_atmosphere_layer', None, '1'): 1720,
    CFName('convective_cloud_area_fraction', None, '1'): 34,
    CFName('convective_rainfall_amount', None, 'kg m-2'): 94,
    CFName('convective_snowfall_amount', None, 'kg m-2'): 117,
    CFName('dimensionless_exner_function', None, '1'): 7,
    CFName('divergence_of_wind', None, 's-1'): 74,
    CFName('downward_heat_flux_in_sea_ice', None, 'W m-2'): 261,
    CFName('downward_heat_flux_in_soil', None, 'W m-2'): 1564,
    CFName('eastward_wind', None, 'm s-1'): 56,
    CFName('ertel_potential_vorticity', None, 'K m2 kg-1 s-1'): 82,
    CFName('geopotential_height', None, 'm'): 1,
    CFName('lagrangian_tendency_of_air_pressure', None, 'Pa s-1'): 40,
    CFName('land_binary_mask', None, '1'): 395,
    CFName('large_scale_rainfall_rate', None, 'm s-1'): 99,
    CFName('mass_fraction_of_carbon_dioxide_in_air', None, '1'): 1564,
    CFName('mass_fraction_of_cloud_liquid_water_in_air', None, '1'): 79,
    CFName('mass_fraction_of_dimethyl_sulfide_in_air', None, '1'): 1373,
    CFName('mass_fraction_of_frozen_water_in_soil_moisture', None, '1'): 1386,
    CFName('mass_fraction_of_ozone_in_air', None, '1'): 453,
    CFName('mass_fraction_of_sulfur_dioxide_in_air', None, '1'): 1374,
    CFName('mass_fraction_of_unfrozen_water_in_soil_moisture', None, '1'): 1385,
    CFName('moisture_content_of_soil_layer', None, 'kg m-2'): 122,
    CFName('mole_fraction_of_atomic_chlorine_in_air', None, '1'): 501,
    CFName('mole_fraction_of_atomic_nitrogen_in_air', None, '1'): 501,
    CFName('mole_fraction_of_bromine_chloride_in_air', None, '1'): 501,
    CFName('mole_fraction_of_bromine_nitrate_in_air', None, '1'): 501,
    CFName('mole_fraction_of_cfc11_in_air', None, '1'): 501,
    CFName('mole_fraction_of_cfc12_in_air', None, '1'): 501,
    CFName('mole_fraction_of_chlorine_dioxide_in_air', None, '1'): 501,
    CFName('mole_fraction_of_chlorine_monoxide_in_air', None, '1'): 501,
    CFName('mole_fraction_of_chlorine_nitrate_in_air', None, '1'): 501,
    CFName('mole_fraction_of_dichlorine_peroxide_in_air', None, '1'): 501,
    CFName('mole_fraction_of_hypochlorous_acid_in_air', None, '1'): 501,
    CFName('mole_fraction_of_nitrous_oxide_in_air', None, '1'): 501,
    CFName('northward_wind', None, 'm s-1'): 57,
    CFName('rainfall_flux', None, 'kg m-2 s-1'): 97,
    CFName('relative_humidity', None, '%'): 88,
    CFName('root_depth', None, 'm'): 321,
    CFName('sea_ice_albedo', None, '1'): 322,
    CFName('sea_ice_area_fraction', None, '1'): 37,
    CFName('sea_ice_temperature', None, 'K'): 209,
    CFName('sea_ice_thickness', None, 'm'): 687,
    CFName('sea_surface_elevation', None, 'm'): 608,
    CFName('snow_grain_size', None, '1e-6 m'): 1507,
    CFName('snowfall_amount', None, 'kg m-2'): 93,
    CFName('snowfall_flux', None, 'kg m-2 s-1'): 108,
    CFName('soil_albedo', None, '1'): 1395,
    CFName('soil_carbon_content', None, 'kg m-2'): 1397,
    CFName('soil_hydraulic_conductivity_at_saturation', None, 'm s-1'): 333,
    CFName('soil_moisture_content_at_field_capacity', None, 'kg m-2'): 1559,
    CFName('soil_porosity', None, '1'): 332,
    CFName('soil_suction_at_saturation', None, 'Pa'): 342,
    CFName('soil_temperature', None, 'K'): 23,
    CFName('soil_thermal_capacity', None, 'J kg-1 K-1'): 335,
    CFName('soil_thermal_conductivity', None, 'W m-1 K-1'): 336,
    CFName('specific_kinetic_energy_of_air', None, 'm2 s-2'): 60,
    CFName('stratiform_cloud_area_fraction_in_atmosphere_layer', None, '1'): 220,
    CFName('stratiform_rainfall_amount', None, 'kg m-2'): 102,
    CFName('stratiform_rainfall_rate', None, 'kg m-2 s-1'): 99,
    CFName('stratiform_snowfall_amount', None, 'kg m-2'): 116,
    CFName('subsurface_runoff_amount', None, 'kg m-2'): 112,
    CFName('subsurface_runoff_flux', None, 'kg m-2 s-1'): 1533,
    CFName('surface_albedo_assuming_deep_snow', None, '1'): 328,
    CFName('surface_albedo_assuming_no_snow', None, '1'): 322,
    CFName('surface_altitude', None, 'm'): 1,
    CFName('surface_downwelling_shortwave_flux_in_air', None, 'W m-2'): 203,
    CFName('surface_downwelling_shortwave_flux_in_air_assuming_clear_sky', None, 'W m-2'): 208,
    CFName('surface_eastward_sea_water_velocity', None, 'm s-1'): 701,
    CFName('surface_net_downward_longwave_flux', None, 'W m-2'): 187,
    CFName('surface_net_downward_shortwave_flux', None, 'W m-2'): 186,
    CFName('surface_northward_sea_water_velocity', None, 'm s-1'): 702,
    CFName('surface_roughness_length', None, 'm'): 324,
    CFName('surface_runoff_amount', None, 'kg m-2'): 111,
    CFName('surface_runoff_flux', None, 'kg m-2 s-1'): 1532,
    CFName('surface_snow_amount', None, 'kg m-2'): 93,
    CFName('surface_temperature', None, 'K'): 16,
    CFName('surface_upward_sensible_heat_flux', None, 'W m-2'): 178,
    CFName('surface_upward_water_flux', None, 'kg m-2 s-1'): 184,
    CFName('surface_upwelling_shortwave_flux_in_air_assuming_clear_sky', None, 'W m-2'): 207,
    CFName('tendency_of_air_density', None, 'kg m-3 s-1'): 7,
    CFName('tendency_of_air_temperature', None, 'K s-1'): 16,
    CFName('tendency_of_air_temperature_due_to_diffusion', None, 'K s-1'): 16,
    CFName('tendency_of_air_temperature_due_to_longwave_heating', None, 'K s-1'): 253,
    CFName('tendency_of_eastward_wind', None, 'm s-1'): 56,
    CFName('tendency_of_eastward_wind_due_to_diffusion', None, 'm s-1'): 56,
    CFName('tendency_of_mass_fraction_of_cloud_ice_in_air', None, 's-1'): 78,
    CFName('tendency_of_mass_fraction_of_cloud_liquid_water_in_air', None, 's-1'): 79,
    CFName('tendency_of_northward_wind', None, 'm s-1'): 57,
    CFName('tendency_of_northward_wind_due_to_diffusion', None, 'm s-1'): 57,
    CFName('tendency_of_specific_humidity', None, 's-1'): 95,
    CFName('tendency_of_specific_humidity_due_to_diffusion', None, 's-1'): 95,
    CFName('tendency_of_upward_air_velocity', None, 'm s-1'): 42,
    CFName('toa_incoming_shortwave_flux', None, 'W m-2'): 200,
    CFName('toa_outgoing_longwave_flux', None, 'W m-2'): 206,
    CFName('toa_outgoing_longwave_flux_assuming_clear_sky', None, 'W m-2'): 210,
    CFName('toa_outgoing_shortwave_flux', None, 'W m-2'): 201,
    CFName('toa_outgoing_shortwave_flux_assuming_clear_sky', None, 'W m-2'): 207,
    CFName('tropopause_air_pressure', None, 'Pa'): 8,
    CFName('tropopause_air_temperature', None, 'K'): 16,
    CFName('tropopause_altitude', None, 'm'): 1,
    CFName('upward_air_velocity', None, 'm s-1'): 42,
    CFName('vegetation_area_fraction', None, '1'): 326,
    CFName('virtual_temperature', None, 'K'): 16,
    CFName('volume_fraction_of_condensed_water_in_soil_at_critical_point', None, '1'): 330,
    CFName('volume_fraction_of_condensed_water_in_soil_at_wilting_point', None, '1'): 329,
    CFName('water_potential_evaporation_flux', None, 'kg m-2 s-1'): 115,
    CFName('wind_mixing_energy_flux_into_sea_water', None, 'W m-2'): 182,
    CFName('wind_speed', None, 'm s-1'): 50,
    CFName('x_wind', None, 'm s-1'): 56,
    CFName('y_wind', None, 'm s-1'): 57,
    }
