"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const client_vpn_endpoint_1 = require("./client-vpn-endpoint");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const subnet_1 = require("./subnet");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * The type of Subnet.
 *
 * @stability stable
 */
var SubnetType;
(function (SubnetType) {
    SubnetType["ISOLATED"] = "Isolated";
    SubnetType["PRIVATE"] = "Private";
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * (deprecated) Dependencies for NAT connectivity.
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets.
         *
         * @stability stable
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets.
     *
     * @stability stable
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            get availabilityZones() { return subnets.map(s => s.availabilityZone); },
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
        };
    }
    /**
     * Adds a VPN Gateway to this VPC.
     *
     * @stability stable
     */
    enableVpnGateway(options) {
        var _e;
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = (_e = options.vpnRoutePropagation) !== null && _e !== void 0 ? _e : [{}];
        const routeTableIds = util_1.allRouteTableIds(util_1.flatten(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            core_1.Annotations.of(this).addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * Adds a new VPN connection to this VPC.
     *
     * @stability stable
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new client VPN endpoint to this VPC.
     *
     * @stability stable
     */
    addClientVpnEndpoint(id, options) {
        return new client_vpn_endpoint_1.ClientVpnEndpoint(this, id, {
            ...options,
            vpc: this,
        });
    }
    /**
     * Adds a new interface endpoint to this VPC.
     *
     * @stability stable
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new gateway endpoint to this VPC.
     *
     * @stability stable
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new flow log to this VPC.
     *
     * @stability stable
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * Returns the id of the VPN Gateway (if enabled).
     *
     * @stability stable
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * Return the subnets appropriate for the placement strategy.
     *
     * @stability stable
     */
    selectSubnetObjects(selection = {}) {
        var _e;
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE;
            subnets = this.selectSubnetObjectsByType(type);
        }
        // Apply all the filters
        subnets = this.applySubnetFilters(subnets, (_e = selection.subnetFilters) !== null && _e !== void 0 ? _e : []);
        return subnets;
    }
    applySubnetFilters(subnets, filters) {
        let filtered = subnets;
        // Apply each filter in sequence
        for (const filter of filters) {
            filtered = filter.selectSubnets(filtered);
        }
        return filtered;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        var _e;
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            else {
                core_1.Annotations.of(this).addWarning('Usage of \'subnetName\' in SubnetSelection is deprecated, use \'subnetGroupName\' instead');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            let subnetType = this.privateSubnets.length ? SubnetType.PRIVATE : this.isolatedSubnets.length ? SubnetType.ISOLATED : SubnetType.PUBLIC;
            placement = { ...placement, subnetType: subnetType };
        }
        // Establish which subnet filters are going to be used
        let subnetFilters = (_e = placement.subnetFilters) !== null && _e !== void 0 ? _e : [];
        // Backwards compatibility with existing `availabilityZones` and `onePerAz` functionality
        if (placement.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnetFilters.push(subnet_1.SubnetFilter.availabilityZones(placement.availabilityZones));
        }
        if (!!placement.onePerAz) { // Ensure one per AZ if specified
            subnetFilters.push(subnet_1.SubnetFilter.onePerAz());
        }
        // Overwrite the provided placement filters and remove the availabilityZones and onePerAz properties
        placement = { ...placement, subnetFilters: subnetFilters, availabilityZones: undefined, onePerAz: undefined };
        const { availabilityZones, onePerAz, ...rest } = placement;
        return rest;
    }
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 *
 * @stability stable
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    DefaultInstanceTenancy["DEFAULT"] = "default";
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud.
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *    cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *    subnetType: ec2.SubnetType.PRIVATE
 * });
 *
 * for (const subnet of selection.subnets) {
 *    // ...
 * }
 * ```
 *
 * @stability stable
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Vpc creates a VPC that spans a whole region.
     *
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     *
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _e;
        super(scope, id);
        /**
         * List of public subnets in this VPC.
         *
         * @stability stable
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC.
         *
         * @stability stable
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC.
         *
         * @stability stable
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        this.dnsHostnamesEnabled = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        this.dnsSupportEnabled = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames: this.dnsHostnamesEnabled,
            enableDnsSupport: this.dnsSupportEnabled,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = (_e = props.maxAzs) !== null && _e !== void 0 ? _e : 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * Import a VPC by supplying all attributes directly.
     *
     * NOTE: using `fromVpcAttributes()` with deploy-time parameters (like a `Fn.importValue()` or
     * `CfnParameter` to represent a list of subnet IDs) sometimes accidentally works. It happens
     * to work for constructs that need a list of subnets (like `AutoScalingGroup` and `eks.Cluster`)
     * but it does not work for constructs that need individual subnets (like
     * `Instance`). See https://github.com/aws/aws-cdk/issues/4118 for more
     * information.
     *
     * Prefer to use `Vpc.fromLookup()` instead.
     *
     * @stability stable
     */
    static fromVpcAttributes(scope, id, attrs) {
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * @stability stable
     */
    static fromLookup(scope, id, options) {
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * (deprecated) Adds a new S3 gateway endpoint to this VPC.
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * (deprecated) Adds a new DynamoDB gateway endpoint to this VPC.
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaining = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaining);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: (subnetConfig.subnetType === SubnetType.PUBLIC),
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            core_1.Tags.of(subnet).add(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes });
            core_1.Tags.of(subnet).add(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes });
        });
    }
}
exports.Vpc = Vpc;
_a = JSII_RTTI_SYMBOL_1;
Vpc[_a] = { fqn: "@aws-cdk/aws-ec2.Vpc", version: "1.101.0" };
/**
 * The default CIDR range used when creating VPCs.
 *
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 *
 * @stability stable
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration.
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 *
 * @stability stable
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE),
    },
];
/**
 * The default subnet configuration if natGateways specified to be 0.
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 *
 * @stability stable
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.ISOLATED,
        name: util_1.defaultSubnetName(SubnetType.ISOLATED),
    },
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE: return 'Private';
        case SubnetType.ISOLATED: return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource.
 *
 * @stability stable
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet.
         *
         * @stability stable
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZone = props.availabilityZone;
        this.ipv4CidrBlock = props.cidrBlock;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        this.subnetOutpostArn = subnet.attrOutpostArn;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.string({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    /**
     * @stability stable
     */
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    /**
     * @stability stable
     */
    static fromSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    // eslint-disable-next-line @typescript-eslint/no-shadow
    /**
     * Import existing subnet from id.
     *
     * @stability stable
     */
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC.
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency.
     * @stability stable
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet.
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explicit ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     *
     * @stability stable
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatewayId.
     *
     * @param natGatewayId The ID of the NAT gateway.
     * @stability stable
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table.
     *
     * @stability stable
     */
    addRoute(id, options) {
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    /**
     * Associate a Network ACL with this subnet.
     *
     * @stability stable
     */
    associateNetworkAcl(id, networkAcl) {
        this._networkAcl = networkAcl;
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
_b = JSII_RTTI_SYMBOL_1;
Subnet[_b] = { fqn: "@aws-cdk/aws-ec2.Subnet", version: "1.101.0" };
/**
 * Type of router used in route.
 *
 * @stability stable
 */
var RouterType;
(function (RouterType) {
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    RouterType["GATEWAY"] = "Gateway";
    RouterType["INSTANCE"] = "Instance";
    RouterType["NAT_GATEWAY"] = "NatGateway";
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource.
 *
 * @stability stable
 */
class PublicSubnet extends Subnet {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * @stability stable
     */
    static fromPublicSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     *
     * Also adds the EIP for the managed NAT.
     *
     * @returns A ref to the the NAT Gateway ID
     * @stability stable
     */
    addNatGateway() {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
_c = JSII_RTTI_SYMBOL_1;
PublicSubnet[_c] = { fqn: "@aws-cdk/aws-ec2.PublicSubnet", version: "1.101.0" };
/**
 * Represents a private VPC subnet resource.
 *
 * @stability stable
 */
class PrivateSubnet extends Subnet {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * @stability stable
     */
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
}
exports.PrivateSubnet = PrivateSubnet;
_d = JSII_RTTI_SYMBOL_1;
PrivateSubnet[_d] = { fqn: "@aws-cdk/aws-ec2.PrivateSubnet", version: "1.101.0" };
function ifUndefined(value, defaultValue) {
    return value !== null && value !== void 0 ? value : defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        // None of the values may be unresolved list tokens
        for (const k of Object.keys(props)) {
            if (Array.isArray(props[k]) && core_1.Token.isUnresolved(props[k])) {
                core_1.Annotations.of(this).addWarning(`fromVpcAttributes: '${k}' is a list token: the imported VPC will not work with constructs that require a list of subnets at synthesis time. Use 'Vpc.fromLookup()' or 'Fn.importListValue' instead.`);
            }
        }
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
                ipv4CidrBlock: vpcSubnet.cidr,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            // The following looks a little weird, but comes down to:
            //
            // * Is the subnetId itself unresolved ({ Ref: Subnet }); or
            // * Was it the accidentally extracted first element of a list-encoded
            //   token? ({ Fn::ImportValue: Subnets } => ['#{Token[1234]}'] =>
            //   '#{Token[1234]}'
            //
            // There's no other API to test for the second case than to the string back into
            // a list and see if the combination is Unresolved.
            //
            // In both cases we can't output the subnetId literally into the metadata (because it'll
            // be useless). In the 2nd case even, if we output it to metadata, the `resolve()` call
            // that gets done on the metadata will even `throw`, because the '#{Token}' value will
            // occur in an illegal position (not in a list context).
            const ref = core_1.Token.isUnresolved(attrs.subnetId) || core_1.Token.isUnresolved([attrs.subnetId])
                ? `at '${constructs_1.Node.of(scope).path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            core_1.Annotations.of(this).addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._ipv4CidrBlock = attrs.ipv4CidrBlock;
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error('You cannot reference a Subnet\'s availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()');
        }
        return this._availabilityZone;
    }
    get ipv4CidrBlock() {
        if (!this._ipv4CidrBlock) {
            // tslint:disable-next-line: max-line-length
            throw new Error('You cannot reference an imported Subnet\'s IPv4 CIDR if it was not supplied. Add the ipv4CidrBlock when importing using Subnet.fromSubnetAttributes()');
        }
        return this._ipv4CidrBlock;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + core_1.Names.nodeUniqueId(other.node), {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,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