"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Port = exports.Protocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
/**
 * Protocol for use in Connection Rules.
 *
 * https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml
 *
 * @stability stable
 */
var Protocol;
(function (Protocol) {
    Protocol["ALL"] = "-1";
    Protocol["HOPOPT"] = "0";
    Protocol["ICMP"] = "icmp";
    Protocol["IGMP"] = "2";
    Protocol["GGP"] = "3";
    Protocol["IPV4"] = "4";
    Protocol["ST"] = "5";
    Protocol["TCP"] = "tcp";
    Protocol["CBT"] = "7";
    Protocol["EGP"] = "8";
    Protocol["IGP"] = "9";
    Protocol["BBN_RCC_MON"] = "10";
    Protocol["NVP_II"] = "11";
    Protocol["PUP"] = "12";
    Protocol["EMCON"] = "14";
    Protocol["XNET"] = "15";
    Protocol["CHAOS"] = "16";
    Protocol["UDP"] = "udp";
    Protocol["MUX"] = "18";
    Protocol["DCN_MEAS"] = "19";
    Protocol["HMP"] = "20";
    Protocol["PRM"] = "21";
    Protocol["XNS_IDP"] = "22";
    Protocol["TRUNK_1"] = "23";
    Protocol["TRUNK_2"] = "24";
    Protocol["LEAF_1"] = "25";
    Protocol["LEAF_2"] = "26";
    Protocol["RDP"] = "27";
    Protocol["IRTP"] = "28";
    Protocol["ISO_TP4"] = "29";
    Protocol["NETBLT"] = "30";
    Protocol["MFE_NSP"] = "31";
    Protocol["MERIT_INP"] = "32";
    Protocol["DCCP"] = "33";
    Protocol["THREEPC"] = "34";
    Protocol["IDPR"] = "35";
    Protocol["XTP"] = "36";
    Protocol["DDP"] = "37";
    Protocol["IDPR_CMTP"] = "38";
    Protocol["TPPLUSPLUS"] = "39";
    Protocol["IL"] = "40";
    Protocol["IPV6"] = "41";
    Protocol["SDRP"] = "42";
    Protocol["IPV6_ROUTE"] = "43";
    Protocol["IPV6_FRAG"] = "44";
    Protocol["IDRP"] = "45";
    Protocol["RSVP"] = "46";
    Protocol["GRE"] = "47";
    Protocol["DSR"] = "48";
    Protocol["BNA"] = "49";
    Protocol["ESP"] = "50";
    Protocol["AH"] = "51";
    Protocol["I_NLSP"] = "52";
    Protocol["SWIPE"] = "53";
    Protocol["NARP"] = "54";
    Protocol["MOBILE"] = "55";
    Protocol["TLSP"] = "56";
    Protocol["SKIP"] = "57";
    Protocol["ICMPV6"] = "icmpv6";
    Protocol["IPV6_NONXT"] = "59";
    Protocol["IPV6_OPTS"] = "60";
    Protocol["CFTP"] = "62";
    Protocol["ANY_LOCAL"] = "63";
    Protocol["SAT_EXPAK"] = "64";
    Protocol["KRYPTOLAN"] = "65";
    Protocol["RVD"] = "66";
    Protocol["IPPC"] = "67";
    Protocol["ANY_DFS"] = "68";
    Protocol["SAT_MON"] = "69";
    Protocol["VISA"] = "70";
    Protocol["IPCV"] = "71";
    Protocol["CPNX"] = "72";
    Protocol["CPHB"] = "73";
    Protocol["WSN"] = "74";
    Protocol["PVP"] = "75";
    Protocol["BR_SAT_MON"] = "76";
    Protocol["SUN_ND"] = "77";
    Protocol["WB_MON"] = "78";
    Protocol["WB_EXPAK"] = "79";
    Protocol["ISO_IP"] = "80";
    Protocol["VMTP"] = "81";
    Protocol["SECURE_VMTP"] = "82";
    Protocol["VINES"] = "83";
    Protocol["TTP"] = "84";
    Protocol["IPTM"] = "84";
    Protocol["NSFNET_IGP"] = "85";
    Protocol["DGP"] = "86";
    Protocol["TCF"] = "87";
    Protocol["EIGRP"] = "88";
    Protocol["OSPFIGP"] = "89";
    Protocol["SPRITE_RPC"] = "90";
    Protocol["LARP"] = "91";
    Protocol["MTP"] = "92";
    Protocol["AX_25"] = "93";
    Protocol["IPIP"] = "94";
    Protocol["MICP"] = "95";
    Protocol["SCC_SP"] = "96";
    Protocol["ETHERIP"] = "97";
    Protocol["ENCAP"] = "98";
    Protocol["ANY_ENC"] = "99";
    Protocol["GMTP"] = "100";
    Protocol["IFMP"] = "101";
    Protocol["PNNI"] = "102";
    Protocol["PIM"] = "103";
    Protocol["ARIS"] = "104";
    Protocol["SCPS"] = "105";
    Protocol["QNX"] = "106";
    Protocol["A_N"] = "107";
    Protocol["IPCOMP"] = "108";
    Protocol["SNP"] = "109";
    Protocol["COMPAQ_PEER"] = "110";
    Protocol["IPX_IN_IP"] = "111";
    Protocol["VRRP"] = "112";
    Protocol["PGM"] = "113";
    Protocol["ANY_0_HOP"] = "114";
    Protocol["L2_T_P"] = "115";
    Protocol["DDX"] = "116";
    Protocol["IATP"] = "117";
    Protocol["STP"] = "118";
    Protocol["SRP"] = "119";
    Protocol["UTI"] = "120";
    Protocol["SMP"] = "121";
    Protocol["SM"] = "122";
    Protocol["PTP"] = "123";
    Protocol["ISIS_IPV4"] = "124";
    Protocol["FIRE"] = "125";
    Protocol["CRTP"] = "126";
    Protocol["CRUDP"] = "127";
    Protocol["SSCOPMCE"] = "128";
    Protocol["IPLT"] = "129";
    Protocol["SPS"] = "130";
    Protocol["PIPE"] = "131";
    Protocol["SCTP"] = "132";
    Protocol["FC"] = "133";
    Protocol["RSVP_E2E_IGNORE"] = "134";
    Protocol["MOBILITY_HEADER"] = "135";
    Protocol["UDPLITE"] = "136";
    Protocol["MPLS_IN_IP"] = "137";
    Protocol["MANET"] = "138";
    Protocol["HIP"] = "139";
    Protocol["SHIM6"] = "140";
    Protocol["WESP"] = "141";
    Protocol["ROHC"] = "142";
    Protocol["ETHERNET"] = "143";
    Protocol["EXPERIMENT_1"] = "253";
    Protocol["EXPERIMENT_2"] = "254";
    Protocol["RESERVED"] = "255";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Interface for classes that provide the connection-specification parts of a security group rule.
 *
 * @stability stable
 */
class Port {
    /**
     * @stability stable
     */
    constructor(props) {
        this.props = props;
        this.canInlineRule = !core_1.Token.isUnresolved(props.fromPort) && !core_1.Token.isUnresolved(props.toPort);
    }
    /**
     * A single TCP port.
     *
     * @stability stable
     */
    static tcp(port) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: port,
            toPort: port,
            stringRepresentation: renderPort(port),
        });
    }
    /**
     * A TCP port range.
     *
     * @stability stable
     */
    static tcpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any TCP traffic.
     *
     * @stability stable
     */
    static allTcp() {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'ALL PORTS',
        });
    }
    /**
     * A single UDP port.
     *
     * @stability stable
     */
    static udp(port) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: port,
            toPort: port,
            stringRepresentation: `UDP ${renderPort(port)}`,
        });
    }
    /**
     * A UDP port range.
     *
     * @stability stable
     */
    static udpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `UDP ${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any UDP traffic.
     *
     * @stability stable
     */
    static allUdp() {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'UDP ALL PORTS',
        });
    }
    /**
     * A specific combination of ICMP type and code.
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     * @stability stable
     */
    static icmpTypeAndCode(type, code) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: code,
            stringRepresentation: `ICMP Type ${type} Code ${code}`,
        });
    }
    /**
     * All codes for a single ICMP type.
     *
     * @stability stable
     */
    static icmpType(type) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: -1,
            stringRepresentation: `ICMP Type ${type}`,
        });
    }
    /**
     * ICMP ping (echo) traffic.
     *
     * @stability stable
     */
    static icmpPing() {
        return Port.icmpType(8);
    }
    /**
     * All ICMP traffic.
     *
     * @stability stable
     */
    static allIcmp() {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: -1,
            toPort: -1,
            stringRepresentation: 'ALL ICMP',
        });
    }
    /**
     * All traffic.
     *
     * @stability stable
     */
    static allTraffic() {
        return new Port({
            protocol: Protocol.ALL,
            stringRepresentation: 'ALL TRAFFIC',
        });
    }
    /**
     * A single ESP port.
     *
     * @stability stable
     */
    static esp() {
        return new Port({
            protocol: Protocol.ESP,
            fromPort: 50,
            toPort: 50,
            stringRepresentation: 'ESP 50',
        });
    }
    /**
     * A single AH port.
     *
     * @stability stable
     */
    static ah() {
        return new Port({
            protocol: Protocol.AH,
            fromPort: 51,
            toPort: 51,
            stringRepresentation: 'AH 51',
        });
    }
    /**
     * Produce the ingress/egress rule JSON for the given connection.
     *
     * @stability stable
     */
    toRuleJson() {
        return {
            ipProtocol: this.props.protocol,
            fromPort: this.props.fromPort,
            toPort: this.props.toPort,
        };
    }
    /**
     * @stability stable
     */
    toString() {
        return this.props.stringRepresentation;
    }
}
exports.Port = Port;
_a = JSII_RTTI_SYMBOL_1;
Port[_a] = { fqn: "@aws-cdk/aws-ec2.Port", version: "1.101.0" };
function renderPort(port) {
    return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
}
//# sourceMappingURL=data:application/json;base64,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