"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
const cdk = require("aws-cdk-lib");
/**
 * @summary The ApiGatewayToSqs class.
 */
class ApiGatewayToSqs extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSqsProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.defaultCreateRequestTemplate = 'Action=SendMessage&MessageBody=$util.urlEncode("$input.body")';
        this.defaultReadRequestTemplate = 'Action=ReceiveMessage';
        this.defaultDeleteRequestTemplate = "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))";
        defaults.CheckSqsProps(props);
        defaults.CheckApiProps(props);
        this.CheckAllRequestProps(props);
        // Setup the queue
        const buildQueueResponse = defaults.buildQueue(this, 'queue', {
            existingQueueObj: props.existingQueueObj,
            queueProps: props.queueProps,
            deployDeadLetterQueue: props.deployDeadLetterQueue,
            deadLetterQueueProps: props.deadLetterQueueProps,
            maxReceiveCount: props.maxReceiveCount,
            enableEncryptionWithCustomerManagedKey: props.enableEncryptionWithCustomerManagedKey,
            encryptionKey: props.encryptionKey,
            encryptionKeyProps: props.encryptionKeyProps
        });
        this.sqsQueue = buildQueueResponse.queue;
        this.deadLetterQueue = buildQueueResponse.dlq;
        // Setup the API Gateway
        const globalRestApiResponse = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps, props.createUsagePlan);
        this.apiGateway = globalRestApiResponse.api;
        this.apiGatewayCloudWatchRole = globalRestApiResponse.role;
        this.apiGatewayLogGroup = globalRestApiResponse.logGroup;
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Create
        const createRequestTemplate = props.createRequestTemplate ?? this.defaultCreateRequestTemplate;
        if (defaults.CheckBooleanWithDefault(props.allowCreateOperation, false)) {
            let createMethodOptions = {};
            // If the client supplied model definitions, set requestModels
            if (props.messageSchema) {
                const requestModels = {};
                Object.keys(props.messageSchema).forEach(key => {
                    const contentType = key;
                    const schema = props.messageSchema[key];
                    const newModel = new api.Model(this, `${id}-model-${defaults.removeNonAlphanumeric(contentType)}`, {
                        restApi: this.apiGateway,
                        contentType,
                        schema
                    });
                    requestModels[contentType] = newModel;
                    createMethodOptions = defaults.overrideProps(createMethodOptions, {
                        requestModels,
                        requestValidatorOptions: {
                            validateRequestBody: true
                        }
                    });
                });
            }
            if (props.createMethodResponses) {
                createMethodOptions = defaults.overrideProps(createMethodOptions, { methodResponses: props.createMethodResponses });
            }
            this.addActionToPolicy("sqs:SendMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate,
                additionalRequestTemplates: props.additionalCreateRequestTemplates,
                contentType: "'application/x-www-form-urlencoded'",
                integrationResponses: props.createIntegrationResponses,
                methodOptions: createMethodOptions
            });
        }
        // Read
        const readRequestTemplate = props.readRequestTemplate ?? this.defaultReadRequestTemplate;
        const readMethodOptions = props.readMethodResponses ? { methodResponses: props.readMethodResponses } : {};
        if (defaults.CheckBooleanWithDefault(props.allowReadOperation, true)) {
            this.addActionToPolicy("sqs:ReceiveMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: this.apiGateway.root,
                requestTemplate: readRequestTemplate,
                additionalRequestTemplates: props.additionalReadRequestTemplates,
                contentType: "'application/x-www-form-urlencoded'",
                integrationResponses: props.readIntegrationResponses,
                methodOptions: readMethodOptions
            });
        }
        // Delete
        const deleteRequestTemplate = props.deleteRequestTemplate ?? this.defaultDeleteRequestTemplate;
        const deleteMethodOptions = props.deleteMethodResponses ? { methodResponses: props.deleteMethodResponses } : {};
        if (defaults.CheckBooleanWithDefault(props.allowDeleteOperation, false)) {
            const apiGatewayResource = this.apiGateway.root.addResource('message');
            this.addActionToPolicy("sqs:DeleteMessage");
            defaults.addProxyMethodToApiResource({
                service: "sqs",
                path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate,
                additionalRequestTemplates: props.additionalDeleteRequestTemplates,
                contentType: "'application/x-www-form-urlencoded'",
                integrationResponses: props.deleteIntegrationResponses,
                methodOptions: deleteMethodOptions
            });
        }
    }
    CheckAllRequestProps(props) {
        if (this.CheckCreateRequestProps(props)) {
            throw new Error(`The 'allowCreateOperation' property must be set to true when setting any of the following: ` +
                `'createRequestTemplate', 'additionalCreateRequestTemplates', 'createIntegrationResponses', 'messageSchema'`);
        }
        if (this.CheckReadRequestProps(props)) {
            throw new Error(`The 'allowReadOperation' property must be set to true or undefined when setting any of the following: ` +
                `'readRequestTemplate', 'additionalReadRequestTemplates', 'readIntegrationResponses'`);
        }
        if (this.CheckDeleteRequestProps(props)) {
            throw new Error(`The 'allowDeleteOperation' property must be set to true when setting any of the following: ` +
                `'deleteRequestTemplate', 'additionalDeleteRequestTemplates', 'deleteIntegrationResponses'`);
        }
    }
    CheckReadRequestProps(props) {
        if ((props.readRequestTemplate || props.additionalReadRequestTemplates || props.readIntegrationResponses)
            && props.allowReadOperation === false) {
            return true;
        }
        return false;
    }
    CheckDeleteRequestProps(props) {
        if ((props.deleteRequestTemplate || props.additionalDeleteRequestTemplates || props.deleteIntegrationResponses)
            && props.allowDeleteOperation !== true) {
            return true;
        }
        return false;
    }
    CheckCreateRequestProps(props) {
        if ((props.createRequestTemplate || props.additionalCreateRequestTemplates || props.createIntegrationResponses || props.messageSchema)
            && props.allowCreateOperation !== true) {
            return true;
        }
        return false;
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.sqsQueue.queueArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToSqs = ApiGatewayToSqs;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToSqs[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-sqs.ApiGatewayToSqs", version: "2.92.0" };
//# sourceMappingURL=data:application/json;base64,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