import glob
import json
import os
import sys
import tempfile
from importlib.util import module_from_spec, spec_from_file_location
from pathlib import Path
from pydantic import BaseModel
import pytest

from fhircraft.fhir.resources.factory import construct_resource_model, factory
from fhircraft.fhir.resources.generator import CodeGenerator

VERSIONS = ["R4B", "R5"]
CORE_EXAMPLES_DIRECTORY = f"test/static/fhir-core-examples"
CORE_DEFINITIONS_DIRECTORY = f"test/static/fhir-core-definitions"
PROFILES_EXAMPLES_DIRECTORY = f"test/static/fhir-profiles-examples"
PROFILES_DEFINTIONS_DIRECTORY = f"test/static/fhir-profiles-definitions"


def _get_core_example_filenames(prefix, version):
    pattern = os.path.join(f"{CORE_EXAMPLES_DIRECTORY}/{version}", prefix)
    example_files = [os.path.basename(f) for f in glob.glob(pattern)]
    return example_files


fhir_resources_test_cases = {
    version: [
        case
        for cases in [
            [
                (resource_label, filename)
                for filename in _get_core_example_filenames(
                    f"{resource_label.lower()}-*", version
                )
            ]
            for resource_label in [
                "Observation",
                "Condition",
                "Patient",
                "Practitioner",
                "Procedure",
                "MedicationAdministration",
                "Organization",
                "StructureMap",
                "ConceptMap",
                "DiagnosticReport",
                ""
            ]
        ]
        for case in cases
    ]
    for version in VERSIONS
}


def _assert_construct_core_resource(version, resource_label, filename):
    # Disable internet access to ensure we use local definitions
    factory.disable_internet_access()
    # Load the FHIR resource definition from local files
    factory.load_definitions_from_files(
        Path(CORE_DEFINITIONS_DIRECTORY)
        / Path(version)
        / Path(f"{resource_label.lower()}.profile.json")
    )

    fhir_version = {
        "R4B": "4.3.0",
        "R5": "5.0.0",
    }.get(version, version)    

    # Generate source code for Pydantic FHIR model
    resource = factory.construct_resource_model(
        canonical_url=f"http://hl7.org/fhir/StructureDefinition/{resource_label}|{fhir_version}"
    )
    # Load example FHIR resource data
    with open(
        os.path.join(
            os.path.abspath(f"{CORE_EXAMPLES_DIRECTORY}/{version}"), filename
        ),
        encoding="utf8",
    ) as file:
        fhir_resource_data = json.load(file)

    # Use the factory-constructed model to validate the FHIR resource data
    assert (instance := resource.model_validate(fhir_resource_data)), 'Factory model failed to validate the example resource data'
    assert fhir_resource_data == json.loads(instance.model_dump_json()), 'Factory model failed to recreate the original resource data'

    # Create temp directory for storing generated code
    with tempfile.TemporaryDirectory() as d:

        source_code = CodeGenerator().generate_resource_model_code(resource)
        # Store source code in a file
        temp_file_name = os.path.join(d, "temp_test.py")
        with open(temp_file_name, "w") as test_file:
            test_file.write(source_code)
        # Load the Pydantic FHIR model
        spec = spec_from_file_location("module", temp_file_name)
        if not spec or not spec.loader:
            raise ImportError(f"Could not load module from {temp_file_name}")
        module = module_from_spec(spec)
        if not module:
            raise ImportError(f"Could not create module from spec {spec}")
        sys.modules["module.name"] = module
        spec.loader.exec_module(module)
        # Use the auto-generated model to validate a FHIR resource
        coded_model: BaseModel = getattr(
            module, resource_label
        )

        # Use the code-loaded model to validate the FHIR resource data
        assert (instance := coded_model.model_validate(fhir_resource_data)), 'Autogenerated code model failed to validate the example resource data'
        assert fhir_resource_data == json.loads(instance.model_dump_json()), 'Autogenerated code model failed to recreate the original resource data'


@pytest.mark.parametrize("resource_label, filename", fhir_resources_test_cases["R4B"])
def test_construct_R4B_core_resource(resource_label, filename):
    _assert_construct_core_resource("R4B", resource_label, filename)


@pytest.mark.parametrize("resource_label, filename", fhir_resources_test_cases["R5"])
def test_construct_R5_core_resource(resource_label, filename):
    _assert_construct_core_resource("R5", resource_label, filename)


def _get_profiles_example_filenames(prefix):
    pattern = os.path.join(f"{PROFILES_EXAMPLES_DIRECTORY}", prefix)
    example_files = [os.path.basename(f) for f in glob.glob(pattern)]
    return example_files


fhir_profiles_test_cases = [
    case
    for cases in [
        [
            filename
            for filename in _get_profiles_example_filenames(f"{resource_label}-*")
        ]
        for resource_label in [
            "Observation",
            "Condition",
            "Patient",
        ]
    ]
    for case in cases
]


def mock_resolve_profile_canonical_url(canonical_url: str):
    MAP = {
        "http://hl7.org/fhir/us/mcode/StructureDefinition/mcode-primary-cancer-condition": "mcode-primary-cancer-condition.json",
        "http://hl7.org/fhir/us/mcode/StructureDefinition/mcode-cancer-related-medication-administration": "mcode-cancer-related-medication-administration.json",
        "http://hl7.org/fhir/us/mcode/StructureDefinition/mcode-tnm-distant-metastases-category": "mcode-tnm-distant-metastases-category.json",
        "http://hl7.org/fhir/us/mcode/StructureDefinition/mcode-cancer-patient": "mcode-cancer-patient.json",
        "http://hl7.org/fhir/us/core/StructureDefinition/us-core-practitioner": "us-core-practitioner.json",
        "http://hl7.org/fhir/us/mcode/StructureDefinition/mcode-radiotherapy-course-summary": "mcode-radiotherapy-course-summary.json",
        "http://hl7.org/fhir/us/core/StructureDefinition/us-core-procedure": "us-core-procedure.json",
        "http://hl7.org/fhir/us/mcode/StructureDefinition/mcode-human-specimen": "mcode-human-specimen.json",
    }
    # Use the auto-generated model to validate a FHIR resource
    with open(
        os.path.join(
            os.path.abspath(f"{PROFILES_DEFINTIONS_DIRECTORY}"), MAP[canonical_url]
        ),
        encoding="utf8",
    ) as file:
        return json.load(file)


@pytest.mark.parametrize("filename", fhir_profiles_test_cases)
def test_construct_profiled_resource(filename):
    # Use the auto-generated model to validate a FHIR resource
    with open(
        os.path.join(os.path.abspath(f"{PROFILES_EXAMPLES_DIRECTORY}"), filename),
        encoding="utf8",
    ) as file:
        fhir_resource = json.load(file)

    # Create temp directory for storing generated code
    with tempfile.TemporaryDirectory() as d:
        # Disable internet access to ensure we use local definitions
        factory.disable_internet_access()
        # Load the FHIR resource definition from local files
        factory.load_definitions_from_directory(Path(PROFILES_DEFINTIONS_DIRECTORY))

        # Generate source code for Pydantic FHIR model
        resource = construct_resource_model(
            canonical_url=fhir_resource["meta"]["profile"][0]
        )
        source_code = CodeGenerator().generate_resource_model_code(resource)
        # Store source code in a file
        temp_file_name = os.path.join(d, f"temp_test_{resource.__name__}.py")
        with open(temp_file_name, "w") as test_file:
            test_file.write(source_code)
        # Load the Pydantic FHIR model
        spec = spec_from_file_location("module", temp_file_name)
        if not spec or not spec.loader:
            raise ImportError(f"Could not load module from {temp_file_name}")
        module = module_from_spec(spec)
        if not module:
            raise ImportError(f"Could not create module from spec {spec}")
        sys.modules["module.name"] = module
        spec.loader.exec_module(module)

    fhir_resource_instance = getattr(module, resource.__name__).model_validate(
        fhir_resource
    )
    assert json.loads(fhir_resource_instance.model_dump_json()) == fhir_resource