# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    Decimal,
    DateTime,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    BackboneElement,
    DomainResource,
)


class TestReportParticipant(BackboneElement):
    """
    A participant in the test execution, either the execution engine, a client, or a server.
    """

    type: Optional[Code] = Field(
        description="test-engine | client | server",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    uri: Optional[Uri] = Field(
        description="The uri of the participant. An absolute URL is preferred",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )
    display: Optional[String] = Field(
        description="The display name of the participant",
        default=None,
    )
    display_ext: Optional[Element] = Field(
        description="Placeholder element for display extensions",
        default=None,
        alias="_display",
    )

    @field_validator(
        *(
            "display",
            "uri",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReportSetupActionOperation(BackboneElement):
    """
    The operation performed.
    """

    result: Optional[Code] = Field(
        description="pass | skip | fail | warning | error",
        default=None,
    )
    result_ext: Optional[Element] = Field(
        description="Placeholder element for result extensions",
        default=None,
        alias="_result",
    )
    message: Optional[Markdown] = Field(
        description="A message associated with the result",
        default=None,
    )
    message_ext: Optional[Element] = Field(
        description="Placeholder element for message extensions",
        default=None,
        alias="_message",
    )
    detail: Optional[Uri] = Field(
        description="A link to further details on the result",
        default=None,
    )
    detail_ext: Optional[Element] = Field(
        description="Placeholder element for detail extensions",
        default=None,
        alias="_detail",
    )

    @field_validator(
        *(
            "detail",
            "message",
            "result",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReportSetupActionAssertRequirement(BackboneElement):
    """
    Links or references providing traceability to the testing requirements for this assert.
    """

    linkUri: Optional[Uri] = Field(
        description="Link or reference to the testing requirement",
        default=None,
    )
    linkUri_ext: Optional[Element] = Field(
        description="Placeholder element for linkUri extensions",
        default=None,
        alias="_linkUri",
    )
    linkCanonical: Optional[Canonical] = Field(
        description="Link or reference to the testing requirement",
        default=None,
    )
    linkCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for linkCanonical extensions",
        default=None,
        alias="_linkCanonical",
    )

    @property
    def link(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="link",
        )

    @model_validator(mode="after")
    def link_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Uri, Canonical],
            field_name_base="link",
            required=False,
        )


class TestReportSetupActionAssert(BackboneElement):
    """
    The results of the assertion performed on the previous operations.
    """

    result: Optional[Code] = Field(
        description="pass | skip | fail | warning | error",
        default=None,
    )
    result_ext: Optional[Element] = Field(
        description="Placeholder element for result extensions",
        default=None,
        alias="_result",
    )
    message: Optional[Markdown] = Field(
        description="A message associated with the result",
        default=None,
    )
    message_ext: Optional[Element] = Field(
        description="Placeholder element for message extensions",
        default=None,
        alias="_message",
    )
    detail: Optional[String] = Field(
        description="A link to further details on the result",
        default=None,
    )
    detail_ext: Optional[Element] = Field(
        description="Placeholder element for detail extensions",
        default=None,
        alias="_detail",
    )
    requirement: Optional[List[TestReportSetupActionAssertRequirement]] = Field(
        description="Links or references to the testing requirements",
        default=None,
    )

    @field_validator(
        *(
            "requirement",
            "detail",
            "message",
            "result",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReportSetupAction(BackboneElement):
    """
    Action would contain either an operation or an assertion.
    """

    operation: Optional[TestReportSetupActionOperation] = Field(
        description="The operation to perform",
        default=None,
    )
    assert_: Optional[TestReportSetupActionAssert] = Field(
        description="The assertion to perform",
        default=None,
    )

    @field_validator(
        *(
            "assert_",
            "operation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReportSetup(BackboneElement):
    """
    The results of the series of required setup operations before the tests were executed.
    """

    action: Optional[List[TestReportSetupAction]] = Field(
        description="A setup operation or assert that was executed",
        default=None,
    )

    @field_validator(
        *("action", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("action",), mode="after", check_fields=None)
    @classmethod
    def FHIR_inv_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="operation.exists() xor assert.exists()",
            human="Setup action SHALL contain either an operation or assert but not both.",
            key="inv-1",
            severity="error",
        )


class TestReportTestActionOperation(BackboneElement):
    """
    An operation would involve a REST request to a server.
    """

    result: Optional[Code] = Field(
        description="pass | skip | fail | warning | error",
        default=None,
    )
    result_ext: Optional[Element] = Field(
        description="Placeholder element for result extensions",
        default=None,
        alias="_result",
    )
    message: Optional[Markdown] = Field(
        description="A message associated with the result",
        default=None,
    )
    message_ext: Optional[Element] = Field(
        description="Placeholder element for message extensions",
        default=None,
        alias="_message",
    )
    detail: Optional[Uri] = Field(
        description="A link to further details on the result",
        default=None,
    )
    detail_ext: Optional[Element] = Field(
        description="Placeholder element for detail extensions",
        default=None,
        alias="_detail",
    )

    @field_validator(
        *(
            "detail",
            "message",
            "result",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReportSetupActionAssertRequirement(BackboneElement):
    """
    Links or references providing traceability to the testing requirements for this assert.
    """

    linkUri: Optional[Uri] = Field(
        description="Link or reference to the testing requirement",
        default=None,
    )
    linkUri_ext: Optional[Element] = Field(
        description="Placeholder element for linkUri extensions",
        default=None,
        alias="_linkUri",
    )
    linkCanonical: Optional[Canonical] = Field(
        description="Link or reference to the testing requirement",
        default=None,
    )
    linkCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for linkCanonical extensions",
        default=None,
        alias="_linkCanonical",
    )

    @property
    def link(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="link",
        )

    @model_validator(mode="after")
    def link_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Uri, Canonical],
            field_name_base="link",
            required=False,
        )


class TestReportTestActionAssert(BackboneElement):
    """
    The results of the assertion performed on the previous operations.
    """

    result: Optional[Code] = Field(
        description="pass | skip | fail | warning | error",
        default=None,
    )
    result_ext: Optional[Element] = Field(
        description="Placeholder element for result extensions",
        default=None,
        alias="_result",
    )
    message: Optional[Markdown] = Field(
        description="A message associated with the result",
        default=None,
    )
    message_ext: Optional[Element] = Field(
        description="Placeholder element for message extensions",
        default=None,
        alias="_message",
    )
    detail: Optional[String] = Field(
        description="A link to further details on the result",
        default=None,
    )
    detail_ext: Optional[Element] = Field(
        description="Placeholder element for detail extensions",
        default=None,
        alias="_detail",
    )
    requirement: Optional[List[TestReportSetupActionAssertRequirement]] = Field(
        description="Links or references to the testing requirements",
        default=None,
    )

    @field_validator(
        *(
            "requirement",
            "detail",
            "message",
            "result",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReportTestAction(BackboneElement):
    """
    Action would contain either an operation or an assertion.
    """

    operation: Optional[TestReportTestActionOperation] = Field(
        description="The operation performed",
        default=None,
    )
    assert_: Optional[TestReportTestActionAssert] = Field(
        description="The assertion performed",
        default=None,
    )

    @field_validator(
        *(
            "assert_",
            "operation",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReportTest(BackboneElement):
    """
    A test executed from the test script.
    """

    name: Optional[String] = Field(
        description="Tracking/logging name of this test",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    description: Optional[String] = Field(
        description="Tracking/reporting short description of the test",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    action: Optional[List[TestReportTestAction]] = Field(
        description="A test operation or assert that was performed",
        default=None,
    )

    @field_validator(
        *(
            "action",
            "description",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("action",), mode="after", check_fields=None)
    @classmethod
    def FHIR_inv_2_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="operation.exists() xor assert.exists()",
            human="Test action SHALL contain either an operation or assert but not both.",
            key="inv-2",
            severity="error",
        )


class TestReportTeardownAction(BackboneElement):
    """
    The teardown action will only contain an operation.
    """

    operation: Optional[TestReportSetupActionOperation] = Field(
        description="The teardown operation performed",
        default=None,
    )

    @field_validator(
        *("operation", "modifierExtension", "extension"),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReportTeardown(BackboneElement):
    """
    The results of the series of operations required to clean up after all the tests were executed (successfully or otherwise).
    """

    action: Optional[List[TestReportTeardownAction]] = Field(
        description="One or more teardown operations performed",
        default=None,
    )

    @field_validator(
        *("action", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestReport(DomainResource):
    """
    A summary of information based on the results of executing a TestScript.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/TestReport"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[Identifier] = Field(
        description="External identifier",
        default=None,
    )
    name: Optional[String] = Field(
        description="Informal name of the executed TestReport",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    status: Optional[Code] = Field(
        description="completed | in-progress | waiting | stopped | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    testScript: Optional[Canonical] = Field(
        description="Canonical URL to the  version-specific TestScript that was executed to produce this TestReport",
        default=None,
    )
    testScript_ext: Optional[Element] = Field(
        description="Placeholder element for testScript extensions",
        default=None,
        alias="_testScript",
    )
    result: Optional[Code] = Field(
        description="pass | fail | pending",
        default=None,
    )
    result_ext: Optional[Element] = Field(
        description="Placeholder element for result extensions",
        default=None,
        alias="_result",
    )
    score: Optional[Decimal] = Field(
        description="The final score (percentage of tests passed) resulting from the execution of the TestScript",
        default=None,
    )
    score_ext: Optional[Element] = Field(
        description="Placeholder element for score extensions",
        default=None,
        alias="_score",
    )
    tester: Optional[String] = Field(
        description="Name of the tester producing this report (Organization or individual)",
        default=None,
    )
    tester_ext: Optional[Element] = Field(
        description="Placeholder element for tester extensions",
        default=None,
        alias="_tester",
    )
    issued: Optional[DateTime] = Field(
        description="When the TestScript was executed and this TestReport was generated",
        default=None,
    )
    issued_ext: Optional[Element] = Field(
        description="Placeholder element for issued extensions",
        default=None,
        alias="_issued",
    )
    participant: Optional[List[TestReportParticipant]] = Field(
        description="A participant in the test execution, either the execution engine, a client, or a server",
        default=None,
    )
    setup: Optional[TestReportSetup] = Field(
        description="The results of the series of required setup operations before the tests were executed",
        default=None,
    )
    test: Optional[List[TestReportTest]] = Field(
        description="A test executed from the test script",
        default=None,
    )
    teardown: Optional[TestReportTeardown] = Field(
        description="The results of running the series of required clean up steps",
        default=None,
    )
    resourceType: Literal["TestReport"] = Field(
        description=None,
        default="TestReport",
    )

    @field_validator(
        *(
            "teardown",
            "test",
            "setup",
            "participant",
            "issued",
            "tester",
            "score",
            "result",
            "testScript",
            "status",
            "name",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
