# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Integer,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    Reference,
    BackboneElement,
    CodeableReference,
    DomainResource,
)


class TestPlanDependency(BackboneElement):
    """
    The required criteria to execute the test plan - e.g. preconditions, previous tests...
    """

    description: Optional[Markdown] = Field(
        description="Description of the dependency criterium",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    predecessor: Optional[Reference] = Field(
        description="Link to predecessor test plans",
        default=None,
    )

    @field_validator(
        *(
            "predecessor",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestPlanTestCaseDependency(BackboneElement):
    """
    The required criteria to execute the test case - e.g. preconditions, previous tests.
    """

    description: Optional[Markdown] = Field(
        description="Description of the criteria",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    predecessor: Optional[Reference] = Field(
        description="Link to predecessor test plans",
        default=None,
    )

    @field_validator(
        *(
            "predecessor",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestPlanTestCaseTestRunScript(BackboneElement):
    """
    The test cases in a structured language e.g. gherkin, Postman, or FHIR TestScript.
    """

    language: Optional[CodeableConcept] = Field(
        description="The language for the test cases e.g. \u0027gherkin\u0027, \u0027testscript\u0027",
        default=None,
    )
    sourceString: Optional[String] = Field(
        description="The actual content of the cases - references to TestScripts or externally defined content",
        default=None,
    )
    sourceString_ext: Optional[Element] = Field(
        description="Placeholder element for sourceString extensions",
        default=None,
        alias="_sourceString",
    )
    sourceReference: Optional[Reference] = Field(
        description="The actual content of the cases - references to TestScripts or externally defined content",
        default=None,
    )

    @property
    def source(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="source",
        )

    @field_validator(
        *("language", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def source_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Reference],
            field_name_base="source",
            required=False,
        )


class TestPlanTestCaseTestRun(BackboneElement):
    """
    The actual test to be executed.
    """

    narrative: Optional[Markdown] = Field(
        description="The narrative description of the tests",
        default=None,
    )
    narrative_ext: Optional[Element] = Field(
        description="Placeholder element for narrative extensions",
        default=None,
        alias="_narrative",
    )
    script: Optional[TestPlanTestCaseTestRunScript] = Field(
        description="The test cases in a structured language e.g. gherkin, Postman, or FHIR TestScript",
        default=None,
    )

    @field_validator(
        *(
            "script",
            "narrative",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestPlanTestCaseTestData(BackboneElement):
    """
    The test data used in the test case.
    """

    type: Optional[Coding] = Field(
        description="The type of test data description, e.g. \u0027synthea\u0027",
        default=None,
    )
    content: Optional[Reference] = Field(
        description="The actual test resources when they exist",
        default=None,
    )
    sourceString: Optional[String] = Field(
        description="Pointer to a definition of test resources - narrative or structured e.g. synthetic data generation, etc",
        default=None,
    )
    sourceString_ext: Optional[Element] = Field(
        description="Placeholder element for sourceString extensions",
        default=None,
        alias="_sourceString",
    )
    sourceReference: Optional[Reference] = Field(
        description="Pointer to a definition of test resources - narrative or structured e.g. synthetic data generation, etc",
        default=None,
    )

    @property
    def source(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="source",
        )

    @field_validator(
        *(
            "content",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def source_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Reference],
            field_name_base="source",
            required=False,
        )


class TestPlanTestCaseAssertion(BackboneElement):
    """
    The test assertions - the expectations of test results from the execution of the test case.
    """

    type: Optional[List[CodeableConcept]] = Field(
        description="Assertion type - for example \u0027informative\u0027 or \u0027required\u0027 ",
        default=None,
    )
    object: Optional[List[CodeableReference]] = Field(
        description="The focus or object of the assertion",
        default=None,
    )
    result: Optional[List[CodeableReference]] = Field(
        description="The actual result assertion",
        default=None,
    )

    @field_validator(
        *(
            "result",
            "object",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestPlanTestCase(BackboneElement):
    """
    The individual test cases that are part of this plan, when they they are made explicit.
    """

    sequence: Optional[Integer] = Field(
        description="Sequence of test case in the test plan",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    scope: Optional[List[Reference]] = Field(
        description="The scope or artifact covered by the case",
        default=None,
    )
    dependency: Optional[List[TestPlanTestCaseDependency]] = Field(
        description="Required criteria to execute the test case",
        default=None,
    )
    testRun: Optional[List[TestPlanTestCaseTestRun]] = Field(
        description="The actual test to be executed",
        default=None,
    )
    testData: Optional[List[TestPlanTestCaseTestData]] = Field(
        description="The test data used in the test case",
        default=None,
    )
    assertion: Optional[List[TestPlanTestCaseAssertion]] = Field(
        description="Test assertions or expectations",
        default=None,
    )

    @field_validator(
        *(
            "assertion",
            "testData",
            "testRun",
            "dependency",
            "scope",
            "sequence",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TestPlan(DomainResource):
    """
    A plan for executing testing on an artifact or specifications
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/TestPlan"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this test plan, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier identifier for the test plan",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the test plan",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this test plan (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this test plan (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the test plan",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction where the test plan applies (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this test plan is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    category: Optional[List[CodeableConcept]] = Field(
        description="The category of the Test Plan - can be acceptance, unit, performance",
        default=None,
    )
    scope: Optional[List[Reference]] = Field(
        description="What is being tested with this Test Plan - a conformance resource, or narrative criteria, or an external reference",
        default=None,
    )
    testTools: Optional[Markdown] = Field(
        description="A description of test tools to be used in the test plan - narrative for now",
        default=None,
    )
    testTools_ext: Optional[Element] = Field(
        description="Placeholder element for testTools extensions",
        default=None,
        alias="_testTools",
    )
    dependency: Optional[List[TestPlanDependency]] = Field(
        description="The required criteria to execute the test plan - e.g. preconditions, previous tests",
        default=None,
    )
    exitCriteria: Optional[Markdown] = Field(
        description="The threshold or criteria for the test plan to be considered successfully executed - narrative",
        default=None,
    )
    exitCriteria_ext: Optional[Element] = Field(
        description="Placeholder element for exitCriteria extensions",
        default=None,
        alias="_exitCriteria",
    )
    testCase: Optional[List[TestPlanTestCase]] = Field(
        description="The test cases that constitute this plan",
        default=None,
    )
    resourceType: Literal["TestPlan"] = Field(
        description=None,
        default="TestPlan",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "testCase",
            "exitCriteria",
            "dependency",
            "testTools",
            "scope",
            "category",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )
