# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Boolean,
    DateTime,
    Markdown,
    Url,
    Canonical,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    DomainResource,
)


class TerminologyCapabilitiesSoftware(BackboneElement):
    """
    Software that is covered by this terminology capability statement.  It is used when the statement describes the capabilities of a particular software version, independent of an installation.
    """

    name: Optional[String] = Field(
        description="A name the software is known by",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    version: Optional[String] = Field(
        description="Version covered by this statement",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )

    @field_validator(
        *(
            "version",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesImplementation(BackboneElement):
    """
    Identifies a specific implementation instance that is described by the terminology capability statement - i.e. a particular installation, rather than the capabilities of a software program.
    """

    description: Optional[String] = Field(
        description="Describes this specific instance",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    url: Optional[Url] = Field(
        description="Base URL for the implementation",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )

    @field_validator(
        *(
            "url",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesCodeSystemVersionFilter(BackboneElement):
    """
    Filter Properties supported.
    """

    code: Optional[Code] = Field(
        description="Code of the property supported",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    op: Optional[List[Code]] = Field(
        description="Operations supported for the property",
        default=None,
    )
    op_ext: Optional[Element] = Field(
        description="Placeholder element for op extensions",
        default=None,
        alias="_op",
    )

    @field_validator(
        *(
            "op",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesCodeSystemVersion(BackboneElement):
    """
    For the code system, a list of versions that are supported by the server.
    """

    code: Optional[String] = Field(
        description="Version identifier for this version",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    isDefault: Optional[Boolean] = Field(
        description="If this is the default version for this code system",
        default=None,
    )
    isDefault_ext: Optional[Element] = Field(
        description="Placeholder element for isDefault extensions",
        default=None,
        alias="_isDefault",
    )
    compositional: Optional[Boolean] = Field(
        description="If compositional grammar is supported",
        default=None,
    )
    compositional_ext: Optional[Element] = Field(
        description="Placeholder element for compositional extensions",
        default=None,
        alias="_compositional",
    )
    language: Optional[List[Code]] = Field(
        description="Language Displays supported",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    filter: Optional[List[TerminologyCapabilitiesCodeSystemVersionFilter]] = Field(
        description="Filter Properties supported",
        default=None,
    )
    property_: Optional[List[Code]] = Field(
        description="Properties supported for $lookup",
        default=None,
    )
    property_ext: Optional[Element] = Field(
        description="Placeholder element for property extensions",
        default=None,
        alias="_property",
    )

    @field_validator(
        *(
            "property_",
            "filter",
            "language",
            "compositional",
            "isDefault",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesCodeSystem(BackboneElement):
    """
    Identifies a code system that is supported by the server. If there is a no code system URL, then this declares the general assumptions a client can make about support for any CodeSystem resource.
    """

    uri: Optional[Canonical] = Field(
        description="Canonical identifier for the code system, represented as a URI",
        default=None,
    )
    uri_ext: Optional[Element] = Field(
        description="Placeholder element for uri extensions",
        default=None,
        alias="_uri",
    )
    version: Optional[List[TerminologyCapabilitiesCodeSystemVersion]] = Field(
        description="Version of Code System supported",
        default=None,
    )
    content: Optional[Code] = Field(
        description="not-present | example | fragment | complete | supplement",
        default=None,
    )
    content_ext: Optional[Element] = Field(
        description="Placeholder element for content extensions",
        default=None,
        alias="_content",
    )
    subsumption: Optional[Boolean] = Field(
        description="Whether subsumption is supported",
        default=None,
    )
    subsumption_ext: Optional[Element] = Field(
        description="Placeholder element for subsumption extensions",
        default=None,
        alias="_subsumption",
    )

    @field_validator(
        *(
            "subsumption",
            "content",
            "version",
            "uri",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesExpansionParameter(BackboneElement):
    """
    Supported expansion parameter.
    """

    name: Optional[Code] = Field(
        description="Name of the supported expansion parameter",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    documentation: Optional[String] = Field(
        description="Description of support for parameter",
        default=None,
    )
    documentation_ext: Optional[Element] = Field(
        description="Placeholder element for documentation extensions",
        default=None,
        alias="_documentation",
    )

    @field_validator(
        *(
            "documentation",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesExpansion(BackboneElement):
    """
    Information about the [ValueSet/$expand](valueset-operation-expand.html) operation.
    """

    hierarchical: Optional[Boolean] = Field(
        description="Whether the server can return nested value sets",
        default=None,
    )
    hierarchical_ext: Optional[Element] = Field(
        description="Placeholder element for hierarchical extensions",
        default=None,
        alias="_hierarchical",
    )
    paging: Optional[Boolean] = Field(
        description="Whether the server supports paging on expansion",
        default=None,
    )
    paging_ext: Optional[Element] = Field(
        description="Placeholder element for paging extensions",
        default=None,
        alias="_paging",
    )
    incomplete: Optional[Boolean] = Field(
        description="Allow request for incomplete expansions?",
        default=None,
    )
    incomplete_ext: Optional[Element] = Field(
        description="Placeholder element for incomplete extensions",
        default=None,
        alias="_incomplete",
    )
    parameter: Optional[List[TerminologyCapabilitiesExpansionParameter]] = Field(
        description="Supported expansion parameter",
        default=None,
    )
    textFilter: Optional[Markdown] = Field(
        description="Documentation about text searching works",
        default=None,
    )
    textFilter_ext: Optional[Element] = Field(
        description="Placeholder element for textFilter extensions",
        default=None,
        alias="_textFilter",
    )

    @field_validator(
        *(
            "textFilter",
            "parameter",
            "incomplete",
            "paging",
            "hierarchical",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesValidateCode(BackboneElement):
    """
    Information about the [ValueSet/$validate-code](valueset-operation-validate-code.html) operation.
    """

    translations: Optional[Boolean] = Field(
        description="Whether translations are validated",
        default=None,
    )
    translations_ext: Optional[Element] = Field(
        description="Placeholder element for translations extensions",
        default=None,
        alias="_translations",
    )

    @field_validator(
        *("translations", "modifierExtension", "extension"),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesTranslation(BackboneElement):
    """
    Information about the [ConceptMap/$translate](conceptmap-operation-translate.html) operation.
    """

    needsMap: Optional[Boolean] = Field(
        description="Whether the client must identify the map",
        default=None,
    )
    needsMap_ext: Optional[Element] = Field(
        description="Placeholder element for needsMap extensions",
        default=None,
        alias="_needsMap",
    )

    @field_validator(
        *("needsMap", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilitiesClosure(BackboneElement):
    """
    Whether the $closure operation is supported.
    """

    translation: Optional[Boolean] = Field(
        description="If cross-system closure is supported",
        default=None,
    )
    translation_ext: Optional[Element] = Field(
        description="Placeholder element for translation extensions",
        default=None,
        alias="_translation",
    )

    @field_validator(
        *("translation", "modifierExtension", "extension"),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class TerminologyCapabilities(DomainResource):
    """
    A TerminologyCapabilities resource documents a set of capabilities (behaviors) of a FHIR Terminology Server that may be used as a statement of actual server functionality or a statement of required or desired server implementation.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/TerminologyCapabilities"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this terminology capabilities, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the terminology capabilities",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the terminology capabilities",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this terminology capabilities (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this terminology capabilities (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the terminology capabilities",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for terminology capabilities (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this terminology capabilities is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    kind: Optional[Code] = Field(
        description="instance | capability | requirements",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    software: Optional[TerminologyCapabilitiesSoftware] = Field(
        description="Software that is covered by this terminology capability statement",
        default=None,
    )
    implementation: Optional[TerminologyCapabilitiesImplementation] = Field(
        description="If this describes a specific instance",
        default=None,
    )
    lockedDate: Optional[Boolean] = Field(
        description="Whether lockedDate is supported",
        default=None,
    )
    lockedDate_ext: Optional[Element] = Field(
        description="Placeholder element for lockedDate extensions",
        default=None,
        alias="_lockedDate",
    )
    codeSystem: Optional[List[TerminologyCapabilitiesCodeSystem]] = Field(
        description="A code system supported by the server",
        default=None,
    )
    expansion: Optional[TerminologyCapabilitiesExpansion] = Field(
        description="Information about the [ValueSet/$expand](valueset-operation-expand.html) operation",
        default=None,
    )
    codeSearch: Optional[Code] = Field(
        description="in-compose | in-expansion | in-compose-or-expansion",
        default=None,
    )
    codeSearch_ext: Optional[Element] = Field(
        description="Placeholder element for codeSearch extensions",
        default=None,
        alias="_codeSearch",
    )
    validateCode: Optional[TerminologyCapabilitiesValidateCode] = Field(
        description="Information about the [ValueSet/$validate-code](valueset-operation-validate-code.html) operation",
        default=None,
    )
    translation: Optional[TerminologyCapabilitiesTranslation] = Field(
        description="Information about the [ConceptMap/$translate](conceptmap-operation-translate.html) operation",
        default=None,
    )
    closure: Optional[TerminologyCapabilitiesClosure] = Field(
        description="Information about the [ConceptMap/$closure](conceptmap-operation-closure.html) operation",
        default=None,
    )
    resourceType: Literal["TerminologyCapabilities"] = Field(
        description=None,
        default="TerminologyCapabilities",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "closure",
            "translation",
            "validateCode",
            "codeSearch",
            "expansion",
            "codeSystem",
            "lockedDate",
            "implementation",
            "software",
            "kind",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @field_validator(*("codeSystem",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tcp_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="version.count() > 1 implies version.all(code.exists())",
            human="If there is more than one version, a version code must be defined",
            key="tcp-1",
            severity="error",
        )

    @field_validator(*("codeSystem",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tcp_7_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="version.code.isDistinct()",
            human="Each version.code element must be distinct for a particular code system.",
            key="tcp-7",
            severity="error",
        )

    @field_validator(*("codeSystem",), mode="after", check_fields=None)
    @classmethod
    def FHIR_tcp_8_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="version.where(isDefault = true).count() <= 1",
            human="A codeSystem element instance may have at most one version.isDefault element with a value of 'true'.",
            key="tcp-8",
            severity="error",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_tcp_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(description.count() + software.count() + implementation.count()) > 0",
            human="A Terminology Capability statement SHALL have at least one of description, software, or implementation element",
            key="tcp-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_tcp_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(kind != 'instance') or implementation.exists()",
            human="If kind = instance, implementation must be present and software may be present",
            key="tcp-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_tcp_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(kind != 'capability') or (implementation.exists().not() and software.exists())",
            human="If kind = capability, implementation must be absent, software must be present",
            key="tcp-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_tcp_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(kind!='requirements') or (implementation.exists().not() and software.exists().not())",
            human="If kind = requirements, implementation and software must be absent",
            key="tcp-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_tcp_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="codeSystem.uri.isDistinct()",
            human="Each instance of the codeSystem element must represent a distinct code system.",
            key="tcp-6",
            severity="error",
        )
