# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, Integer

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    CodeableConcept,
    BackboneElement,
    Attachment,
    Identifier,
    DomainResource,
)


class SubstanceNucleicAcidSubunitLinkage(BackboneElement):
    """
    The linkages between sugar residues will also be captured.
    """

    connectivity: Optional[String] = Field(
        description="The entity that links the sugar residues together should also be captured for nearly all naturally occurring nucleic acid the linkage is a phosphate group. For many synthetic oligonucleotides phosphorothioate linkages are often seen. Linkage connectivity is assumed to be 3\u2019-5\u2019. If the linkage is either 3\u2019-3\u2019 or 5\u2019-5\u2019 this should be specified",
        default=None,
    )
    connectivity_ext: Optional[Element] = Field(
        description="Placeholder element for connectivity extensions",
        default=None,
        alias="_connectivity",
    )
    identifier: Optional[Identifier] = Field(
        description="Each linkage will be registered as a fragment and have an ID",
        default=None,
    )
    name: Optional[String] = Field(
        description="Each linkage will be registered as a fragment and have at least one name. A single name shall be assigned to each linkage",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    residueSite: Optional[String] = Field(
        description="Residues shall be captured as described in 5.3.6.8.3",
        default=None,
    )
    residueSite_ext: Optional[Element] = Field(
        description="Placeholder element for residueSite extensions",
        default=None,
        alias="_residueSite",
    )

    @field_validator(
        *(
            "residueSite",
            "name",
            "identifier",
            "connectivity",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceNucleicAcidSubunitSugar(BackboneElement):
    """
    5.3.6.8.1 Sugar ID (Mandatory).
    """

    identifier: Optional[Identifier] = Field(
        description="The Substance ID of the sugar or sugar-like component that make up the nucleotide",
        default=None,
    )
    name: Optional[String] = Field(
        description="The name of the sugar or sugar-like component that make up the nucleotide",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    residueSite: Optional[String] = Field(
        description="The residues that contain a given sugar will be captured. The order of given residues will be captured in the 5\u2018-3\u2018direction consistent with the base sequences listed above",
        default=None,
    )
    residueSite_ext: Optional[Element] = Field(
        description="Placeholder element for residueSite extensions",
        default=None,
        alias="_residueSite",
    )

    @field_validator(
        *(
            "residueSite",
            "name",
            "identifier",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceNucleicAcidSubunit(BackboneElement):
    """
    Subunits are listed in order of decreasing length; sequences of the same length will be ordered by molecular weight; subunits that have identical sequences will be repeated multiple times.
    """

    subunit: Optional[Integer] = Field(
        description="Index of linear sequences of nucleic acids in order of decreasing length. Sequences of the same length will be ordered by molecular weight. Subunits that have identical sequences will be repeated and have sequential subscripts",
        default=None,
    )
    subunit_ext: Optional[Element] = Field(
        description="Placeholder element for subunit extensions",
        default=None,
        alias="_subunit",
    )
    sequence: Optional[String] = Field(
        description="Actual nucleotide sequence notation from 5\u0027 to 3\u0027 end using standard single letter codes. In addition to the base sequence, sugar and type of phosphate or non-phosphate linkage should also be captured",
        default=None,
    )
    sequence_ext: Optional[Element] = Field(
        description="Placeholder element for sequence extensions",
        default=None,
        alias="_sequence",
    )
    length: Optional[Integer] = Field(
        description="The length of the sequence shall be captured",
        default=None,
    )
    length_ext: Optional[Element] = Field(
        description="Placeholder element for length extensions",
        default=None,
        alias="_length",
    )
    sequenceAttachment: Optional[Attachment] = Field(
        description="(TBC)",
        default=None,
    )
    fivePrime: Optional[CodeableConcept] = Field(
        description="The nucleotide present at the 5\u2019 terminal shall be specified based on a controlled vocabulary. Since the sequence is represented from the 5\u0027 to the 3\u0027 end, the 5\u2019 prime nucleotide is the letter at the first position in the sequence. A separate representation would be redundant",
        default=None,
    )
    threePrime: Optional[CodeableConcept] = Field(
        description="The nucleotide present at the 3\u2019 terminal shall be specified based on a controlled vocabulary. Since the sequence is represented from the 5\u0027 to the 3\u0027 end, the 5\u2019 prime nucleotide is the letter at the last position in the sequence. A separate representation would be redundant",
        default=None,
    )
    linkage: Optional[List[SubstanceNucleicAcidSubunitLinkage]] = Field(
        description="The linkages between sugar residues will also be captured",
        default=None,
    )
    sugar: Optional[List[SubstanceNucleicAcidSubunitSugar]] = Field(
        description="5.3.6.8.1 Sugar ID (Mandatory)",
        default=None,
    )

    @field_validator(
        *(
            "sugar",
            "linkage",
            "threePrime",
            "fivePrime",
            "sequenceAttachment",
            "length",
            "sequence",
            "subunit",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceNucleicAcid(DomainResource):
    """
    Nucleic acids are defined by three distinct elements: the base, sugar and linkage. Individual substance/moiety IDs will be created for each of these elements. The nucleotide sequence will be always entered in the 5’-3’ direction.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/SubstanceNucleicAcid"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    sequenceType: Optional[CodeableConcept] = Field(
        description="The type of the sequence shall be specified based on a controlled vocabulary",
        default=None,
    )
    numberOfSubunits: Optional[Integer] = Field(
        description="The number of linear sequences of nucleotides linked through phosphodiester bonds shall be described. Subunits would be strands of nucleic acids that are tightly associated typically through Watson-Crick base pairing. NOTE: If not specified in the reference source, the assumption is that there is 1 subunit",
        default=None,
    )
    numberOfSubunits_ext: Optional[Element] = Field(
        description="Placeholder element for numberOfSubunits extensions",
        default=None,
        alias="_numberOfSubunits",
    )
    areaOfHybridisation: Optional[String] = Field(
        description="The area of hybridisation shall be described if applicable for double stranded RNA or DNA. The number associated with the subunit followed by the number associated to the residue shall be specified in increasing order. The underscore \u201c\u201d shall be used as separator as follows: \u201cSubunitnumber Residue\u201d",
        default=None,
    )
    areaOfHybridisation_ext: Optional[Element] = Field(
        description="Placeholder element for areaOfHybridisation extensions",
        default=None,
        alias="_areaOfHybridisation",
    )
    oligoNucleotideType: Optional[CodeableConcept] = Field(
        description="(TBC)",
        default=None,
    )
    subunit: Optional[List[SubstanceNucleicAcidSubunit]] = Field(
        description="Subunits are listed in order of decreasing length; sequences of the same length will be ordered by molecular weight; subunits that have identical sequences will be repeated multiple times",
        default=None,
    )
    resourceType: Literal["SubstanceNucleicAcid"] = Field(
        description=None,
        default="SubstanceNucleicAcid",
    )

    @field_validator(
        *(
            "subunit",
            "oligoNucleotideType",
            "areaOfHybridisation",
            "numberOfSubunits",
            "sequenceType",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
