# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Markdown,
    Date,
    Boolean,
    DateTime,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    Annotation,
    BackboneElement,
    Quantity,
    Attachment,
    Ratio,
    DomainResource,
)


class SubstanceDefinitionMoiety(BackboneElement):
    """
    Moiety, for structural modifications.
    """

    role: Optional[CodeableConcept] = Field(
        description="Role that the moiety is playing",
        default=None,
    )
    identifier: Optional[Identifier] = Field(
        description="Identifier by which this moiety substance is known",
        default=None,
    )
    name: Optional[String] = Field(
        description="Textual name for this moiety substance",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    stereochemistry: Optional[CodeableConcept] = Field(
        description="Stereochemistry type",
        default=None,
    )
    opticalActivity: Optional[CodeableConcept] = Field(
        description="Optical activity type",
        default=None,
    )
    molecularFormula: Optional[String] = Field(
        description="Molecular formula for this moiety (e.g. with the Hill system)",
        default=None,
    )
    molecularFormula_ext: Optional[Element] = Field(
        description="Placeholder element for molecularFormula extensions",
        default=None,
        alias="_molecularFormula",
    )
    amountQuantity: Optional[Quantity] = Field(
        description="Quantitative value for this moiety",
        default=None,
    )
    amountString: Optional[String] = Field(
        description="Quantitative value for this moiety",
        default=None,
    )
    amountString_ext: Optional[Element] = Field(
        description="Placeholder element for amountString extensions",
        default=None,
        alias="_amountString",
    )
    measurementType: Optional[CodeableConcept] = Field(
        description="The measurement type of the quantitative value",
        default=None,
    )

    @property
    def amount(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="amount",
        )

    @field_validator(
        *(
            "measurementType",
            "molecularFormula",
            "opticalActivity",
            "stereochemistry",
            "name",
            "identifier",
            "role",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def amount_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Quantity, String],
            field_name_base="amount",
            required=False,
        )


class SubstanceDefinitionCharacterization(BackboneElement):
    """
    General specifications for this substance.
    """

    technique: Optional[CodeableConcept] = Field(
        description="The method used to find the characterization e.g. HPLC",
        default=None,
    )
    form: Optional[CodeableConcept] = Field(
        description="Describes the nature of the chemical entity and explains, for instance, whether this is a base or a salt form",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="The description or justification in support of the interpretation of the data file",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    file: Optional[List[Attachment]] = Field(
        description="The data produced by the analytical instrument or a pictorial representation of that data. Examples: a JCAMP, JDX, or ADX file, or a chromatogram or spectrum analysis",
        default=None,
    )

    @field_validator(
        *(
            "file",
            "description",
            "form",
            "technique",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinitionProperty(BackboneElement):
    """
    General specifications for this substance.
    """

    type: Optional[CodeableConcept] = Field(
        description="A code expressing the type of property",
        default=None,
    )
    valueCodeableConcept: Optional[CodeableConcept] = Field(
        description="A value for the property",
        default=None,
    )
    valueQuantity: Optional[Quantity] = Field(
        description="A value for the property",
        default=None,
    )
    valueDate: Optional[Date] = Field(
        description="A value for the property",
        default=None,
    )
    valueDate_ext: Optional[Element] = Field(
        description="Placeholder element for valueDate extensions",
        default=None,
        alias="_valueDate",
    )
    valueBoolean: Optional[Boolean] = Field(
        description="A value for the property",
        default=None,
    )
    valueBoolean_ext: Optional[Element] = Field(
        description="Placeholder element for valueBoolean extensions",
        default=None,
        alias="_valueBoolean",
    )
    valueAttachment: Optional[Attachment] = Field(
        description="A value for the property",
        default=None,
    )

    @property
    def value(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="value",
        )

    @field_validator(
        *("type", "modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def value_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Quantity, Date, Boolean, Attachment],
            field_name_base="value",
            required=False,
        )


class SubstanceDefinitionMolecularWeight(BackboneElement):
    """
    The average mass of a molecule of a compound compared to 1/12 the mass of carbon 12 and calculated as the sum of the atomic weights of the constituent atoms.
    """

    method: Optional[CodeableConcept] = Field(
        description="The method by which the weight was determined",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Type of molecular weight e.g. exact, average, weight average",
        default=None,
    )
    amount: Optional[Quantity] = Field(
        description="Used to capture quantitative values for a variety of elements",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "type",
            "method",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinitionStructureMolecularWeight(BackboneElement):
    """
    The molecular weight or weight range (for proteins, polymers or nucleic acids).
    """

    method: Optional[CodeableConcept] = Field(
        description="The method by which the weight was determined",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="Type of molecular weight e.g. exact, average, weight average",
        default=None,
    )
    amount: Optional[Quantity] = Field(
        description="Used to capture quantitative values for a variety of elements",
        default=None,
    )

    @field_validator(
        *(
            "amount",
            "type",
            "method",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinitionStructureRepresentation(BackboneElement):
    """
    A depiction of the structure of the substance.
    """

    type: Optional[CodeableConcept] = Field(
        description="The kind of structural representation (e.g. full, partial)",
        default=None,
    )
    representation: Optional[String] = Field(
        description="The structural representation as a text string in a standard format",
        default=None,
    )
    representation_ext: Optional[Element] = Field(
        description="Placeholder element for representation extensions",
        default=None,
        alias="_representation",
    )
    format: Optional[CodeableConcept] = Field(
        description="The format of the representation e.g. InChI, SMILES, MOLFILE (note: not the physical file format)",
        default=None,
    )
    document: Optional[Reference] = Field(
        description="An attachment with the structural representation e.g. a structure graphic or AnIML file",
        default=None,
    )

    @field_validator(
        *(
            "document",
            "format",
            "representation",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinitionStructure(BackboneElement):
    """
    Structural information.
    """

    stereochemistry: Optional[CodeableConcept] = Field(
        description="Stereochemistry type",
        default=None,
    )
    opticalActivity: Optional[CodeableConcept] = Field(
        description="Optical activity type",
        default=None,
    )
    molecularFormula: Optional[String] = Field(
        description="An expression which states the number and type of atoms present in a molecule of a substance",
        default=None,
    )
    molecularFormula_ext: Optional[Element] = Field(
        description="Placeholder element for molecularFormula extensions",
        default=None,
        alias="_molecularFormula",
    )
    molecularFormulaByMoiety: Optional[String] = Field(
        description="Specified per moiety according to the Hill system",
        default=None,
    )
    molecularFormulaByMoiety_ext: Optional[Element] = Field(
        description="Placeholder element for molecularFormulaByMoiety extensions",
        default=None,
        alias="_molecularFormulaByMoiety",
    )
    molecularWeight: Optional[SubstanceDefinitionStructureMolecularWeight] = Field(
        description="The molecular weight or weight range",
        default=None,
    )
    technique: Optional[List[CodeableConcept]] = Field(
        description="The method used to find the structure e.g. X-ray, NMR",
        default=None,
    )
    sourceDocument: Optional[List[Reference]] = Field(
        description="Source of information for the structure",
        default=None,
    )
    representation: Optional[List[SubstanceDefinitionStructureRepresentation]] = Field(
        description="A depiction of the structure of the substance",
        default=None,
    )

    @field_validator(
        *(
            "representation",
            "sourceDocument",
            "technique",
            "molecularWeight",
            "molecularFormulaByMoiety",
            "molecularFormula",
            "opticalActivity",
            "stereochemistry",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinitionCode(BackboneElement):
    """
    Codes associated with the substance.
    """

    code: Optional[CodeableConcept] = Field(
        description="The specific code",
        default=None,
    )
    status: Optional[CodeableConcept] = Field(
        description="Status of the code assignment, for example \u0027provisional\u0027, \u0027approved\u0027",
        default=None,
    )
    statusDate: Optional[DateTime] = Field(
        description="The date at which the code status was changed",
        default=None,
    )
    statusDate_ext: Optional[Element] = Field(
        description="Placeholder element for statusDate extensions",
        default=None,
        alias="_statusDate",
    )
    note: Optional[List[Annotation]] = Field(
        description="Any comment can be provided in this field",
        default=None,
    )
    source: Optional[List[Reference]] = Field(
        description="Supporting literature",
        default=None,
    )

    @field_validator(
        *(
            "source",
            "note",
            "statusDate",
            "status",
            "code",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinitionNameOfficial(BackboneElement):
    """
    Details of the official nature of this name.
    """

    authority: Optional[CodeableConcept] = Field(
        description="Which authority uses this official name",
        default=None,
    )
    status: Optional[CodeableConcept] = Field(
        description="The status of the official name, for example \u0027draft\u0027, \u0027active\u0027",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="Date of official name change",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )

    @field_validator(
        *(
            "date",
            "status",
            "authority",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinitionName(BackboneElement):
    """
    Names applicable to this substance.
    """

    name: Optional[String] = Field(
        description="The actual name",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[CodeableConcept] = Field(
        description="Name type e.g. \u0027systematic\u0027,  \u0027scientific, \u0027brand\u0027",
        default=None,
    )
    status: Optional[CodeableConcept] = Field(
        description="The status of the name e.g. \u0027current\u0027, \u0027proposed\u0027",
        default=None,
    )
    preferred: Optional[Boolean] = Field(
        description="If this is the preferred name for this substance",
        default=None,
    )
    preferred_ext: Optional[Element] = Field(
        description="Placeholder element for preferred extensions",
        default=None,
        alias="_preferred",
    )
    language: Optional[List[CodeableConcept]] = Field(
        description="Human language that the name is written in",
        default=None,
    )
    domain: Optional[List[CodeableConcept]] = Field(
        description="The use context of this name e.g. as an active ingredient or as a food colour additive",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="The jurisdiction where this name applies",
        default=None,
    )
    synonym: Optional[List["SubstanceDefinitionName"]] = Field(
        description="A synonym of this particular name, by which the substance is also known",
        default=None,
    )
    translation: Optional[List["SubstanceDefinitionName"]] = Field(
        description="A translation for this name into another human language",
        default=None,
    )
    official: Optional[List[SubstanceDefinitionNameOfficial]] = Field(
        description="Details of the official nature of this name",
        default=None,
    )
    source: Optional[List[Reference]] = Field(
        description="Supporting literature",
        default=None,
    )

    @field_validator(
        *(
            "source",
            "official",
            "translation",
            "synonym",
            "jurisdiction",
            "domain",
            "language",
            "preferred",
            "status",
            "type",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinitionRelationship(BackboneElement):
    """
    A link between this substance and another, with details of the relationship.
    """

    substanceDefinitionReference: Optional[Reference] = Field(
        description="A pointer to another substance, as a resource or a representational code",
        default=None,
    )
    substanceDefinitionCodeableConcept: Optional[CodeableConcept] = Field(
        description="A pointer to another substance, as a resource or a representational code",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description='For example "salt to parent", "active moiety"',
        default=None,
    )
    isDefining: Optional[Boolean] = Field(
        description="For example where an enzyme strongly bonds with a particular substance, this is a defining relationship for that enzyme, out of several possible relationships",
        default=None,
    )
    isDefining_ext: Optional[Element] = Field(
        description="Placeholder element for isDefining extensions",
        default=None,
        alias="_isDefining",
    )
    amountQuantity: Optional[Quantity] = Field(
        description="A numeric factor for the relationship, e.g. that a substance salt has some percentage of active substance in relation to some other",
        default=None,
    )
    amountRatio: Optional[Ratio] = Field(
        description="A numeric factor for the relationship, e.g. that a substance salt has some percentage of active substance in relation to some other",
        default=None,
    )
    amountString: Optional[String] = Field(
        description="A numeric factor for the relationship, e.g. that a substance salt has some percentage of active substance in relation to some other",
        default=None,
    )
    amountString_ext: Optional[Element] = Field(
        description="Placeholder element for amountString extensions",
        default=None,
        alias="_amountString",
    )
    ratioHighLimitAmount: Optional[Ratio] = Field(
        description="For use when the numeric has an uncertain range",
        default=None,
    )
    comparator: Optional[CodeableConcept] = Field(
        description='An operator for the amount, for example "average", "approximately", "less than"',
        default=None,
    )
    source: Optional[List[Reference]] = Field(
        description="Supporting literature",
        default=None,
    )

    @property
    def substanceDefinition(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="substanceDefinition",
        )

    @property
    def amount(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="amount",
        )

    @field_validator(
        *(
            "source",
            "comparator",
            "ratioHighLimitAmount",
            "isDefining",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def substanceDefinition_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Reference, CodeableConcept],
            field_name_base="substanceDefinition",
            required=False,
        )

    @model_validator(mode="after")
    def amount_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Quantity, Ratio, String],
            field_name_base="amount",
            required=False,
        )


class SubstanceDefinitionSourceMaterial(BackboneElement):
    """
    Material or taxonomic/anatomical source for the substance.
    """

    type: Optional[CodeableConcept] = Field(
        description="Classification of the origin of the raw material. e.g. cat hair is an Animal source type",
        default=None,
    )
    genus: Optional[CodeableConcept] = Field(
        description="The genus of an organism e.g. the Latin epithet of the plant/animal scientific name",
        default=None,
    )
    species: Optional[CodeableConcept] = Field(
        description="The species of an organism e.g. the Latin epithet of the species of the plant/animal",
        default=None,
    )
    part: Optional[CodeableConcept] = Field(
        description="An anatomical origin of the source material within an organism",
        default=None,
    )
    countryOfOrigin: Optional[List[CodeableConcept]] = Field(
        description="The country or countries where the material is harvested",
        default=None,
    )

    @field_validator(
        *(
            "countryOfOrigin",
            "part",
            "species",
            "genus",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubstanceDefinition(DomainResource):
    """
    The detailed description of a substance, typically at a level beyond what is used for prescribing.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/SubstanceDefinition"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Identifier by which this substance is known",
        default=None,
    )
    version: Optional[String] = Field(
        description="A business level version identifier of the substance",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    status: Optional[CodeableConcept] = Field(
        description="Status of substance within the catalogue e.g. active, retired",
        default=None,
    )
    classification: Optional[List[CodeableConcept]] = Field(
        description="A categorization, high level e.g. polymer or nucleic acid, or food, chemical, biological, or lower e.g. polymer linear or branch chain, or type of impurity",
        default=None,
    )
    domain: Optional[CodeableConcept] = Field(
        description="If the substance applies to human or veterinary use",
        default=None,
    )
    grade: Optional[List[CodeableConcept]] = Field(
        description="The quality standard, established benchmark, to which substance complies (e.g. USP/NF, BP)",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Textual description of the substance",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    informationSource: Optional[List[Reference]] = Field(
        description="Supporting literature",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Textual comment about the substance\u0027s catalogue or registry record",
        default=None,
    )
    manufacturer: Optional[List[Reference]] = Field(
        description="The entity that creates, makes, produces or fabricates the substance",
        default=None,
    )
    supplier: Optional[List[Reference]] = Field(
        description="An entity that is the source for the substance. It may be different from the manufacturer",
        default=None,
    )
    moiety: Optional[List[SubstanceDefinitionMoiety]] = Field(
        description="Moiety, for structural modifications",
        default=None,
    )
    characterization: Optional[List[SubstanceDefinitionCharacterization]] = Field(
        description="General specifications for this substance",
        default=None,
    )
    property_: Optional[List[SubstanceDefinitionProperty]] = Field(
        description="General specifications for this substance",
        default=None,
    )
    referenceInformation: Optional[Reference] = Field(
        description="General information detailing this substance",
        default=None,
    )
    molecularWeight: Optional[List[SubstanceDefinitionMolecularWeight]] = Field(
        description="The average mass of a molecule of a compound",
        default=None,
    )
    structure: Optional[SubstanceDefinitionStructure] = Field(
        description="Structural information",
        default=None,
    )
    code: Optional[List[SubstanceDefinitionCode]] = Field(
        description="Codes associated with the substance",
        default=None,
    )
    name: Optional[List[SubstanceDefinitionName]] = Field(
        description="Names applicable to this substance",
        default=None,
    )
    relationship: Optional[List[SubstanceDefinitionRelationship]] = Field(
        description="A link between this substance and another",
        default=None,
    )
    nucleicAcid: Optional[Reference] = Field(
        description="Data items specific to nucleic acids",
        default=None,
    )
    polymer: Optional[Reference] = Field(
        description="Data items specific to polymers",
        default=None,
    )
    protein: Optional[Reference] = Field(
        description="Data items specific to proteins",
        default=None,
    )
    sourceMaterial: Optional[SubstanceDefinitionSourceMaterial] = Field(
        description="Material or taxonomic/anatomical source",
        default=None,
    )
    resourceType: Literal["SubstanceDefinition"] = Field(
        description=None,
        default="SubstanceDefinition",
    )

    @field_validator(
        *(
            "sourceMaterial",
            "protein",
            "polymer",
            "nucleicAcid",
            "relationship",
            "name",
            "code",
            "structure",
            "molecularWeight",
            "referenceInformation",
            "property_",
            "characterization",
            "moiety",
            "supplier",
            "manufacturer",
            "note",
            "informationSource",
            "description",
            "grade",
            "domain",
            "classification",
            "status",
            "version",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
