# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    Boolean,
    DateTime,
    Markdown,
    Date,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    Period,
    BackboneElement,
    DomainResource,
)


class SubscriptionTopicResourceTriggerQueryCriteria(BackboneElement):
    """
    The FHIR query based rules that the server should use to determine when to trigger a notification for this subscription topic.
    """

    previous: Optional[String] = Field(
        description="Rule applied to previous resource state",
        default=None,
    )
    previous_ext: Optional[Element] = Field(
        description="Placeholder element for previous extensions",
        default=None,
        alias="_previous",
    )
    resultForCreate: Optional[Code] = Field(
        description="test-passes | test-fails",
        default=None,
    )
    resultForCreate_ext: Optional[Element] = Field(
        description="Placeholder element for resultForCreate extensions",
        default=None,
        alias="_resultForCreate",
    )
    current: Optional[String] = Field(
        description="Rule applied to current resource state",
        default=None,
    )
    current_ext: Optional[Element] = Field(
        description="Placeholder element for current extensions",
        default=None,
        alias="_current",
    )
    resultForDelete: Optional[Code] = Field(
        description="test-passes | test-fails",
        default=None,
    )
    resultForDelete_ext: Optional[Element] = Field(
        description="Placeholder element for resultForDelete extensions",
        default=None,
        alias="_resultForDelete",
    )
    requireBoth: Optional[Boolean] = Field(
        description="Both must be true flag",
        default=None,
    )
    requireBoth_ext: Optional[Element] = Field(
        description="Placeholder element for requireBoth extensions",
        default=None,
        alias="_requireBoth",
    )

    @field_validator(
        *(
            "requireBoth",
            "resultForDelete",
            "current",
            "resultForCreate",
            "previous",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubscriptionTopicResourceTrigger(BackboneElement):
    """
    A definition of a resource-based event that triggers a notification based on the SubscriptionTopic. The criteria may be just a human readable description and/or a full FHIR search string or FHIRPath expression. Multiple triggers are considered OR joined (e.g., a resource update matching ANY of the definitions will trigger a notification).
    """

    description: Optional[Markdown] = Field(
        description="Text representation of the resource trigger",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    resource: Optional[Uri] = Field(
        description="Data Type or Resource (reference to definition) for this trigger definition",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    supportedInteraction: Optional[List[Code]] = Field(
        description="create | update | delete",
        default=None,
    )
    supportedInteraction_ext: Optional[Element] = Field(
        description="Placeholder element for supportedInteraction extensions",
        default=None,
        alias="_supportedInteraction",
    )
    queryCriteria: Optional[SubscriptionTopicResourceTriggerQueryCriteria] = Field(
        description="Query based trigger rule",
        default=None,
    )
    fhirPathCriteria: Optional[String] = Field(
        description="FHIRPath based trigger rule",
        default=None,
    )
    fhirPathCriteria_ext: Optional[Element] = Field(
        description="Placeholder element for fhirPathCriteria extensions",
        default=None,
        alias="_fhirPathCriteria",
    )

    @field_validator(
        *(
            "fhirPathCriteria",
            "queryCriteria",
            "supportedInteraction",
            "resource",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubscriptionTopicEventTrigger(BackboneElement):
    """
    Event definition which can be used to trigger the SubscriptionTopic.
    """

    description: Optional[Markdown] = Field(
        description="Text representation of the event trigger",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    event: Optional[CodeableConcept] = Field(
        description="Event which can trigger a notification from the SubscriptionTopic",
        default=None,
    )
    resource: Optional[Uri] = Field(
        description="Data Type or Resource (reference to definition) for this trigger definition",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )

    @field_validator(
        *(
            "resource",
            "event",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubscriptionTopicCanFilterBy(BackboneElement):
    """
    List of properties by which Subscriptions on the SubscriptionTopic can be filtered. May be defined Search Parameters (e.g., Encounter.patient) or parameters defined within this SubscriptionTopic context (e.g., hub.event).
    """

    description: Optional[Markdown] = Field(
        description="Description of this filter parameter",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    resource: Optional[Uri] = Field(
        description="URL of the triggering Resource that this filter applies to",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    filterParameter: Optional[String] = Field(
        description="Human-readable and computation-friendly name for a filter parameter usable by subscriptions on this topic, via Subscription.filterBy.filterParameter",
        default=None,
    )
    filterParameter_ext: Optional[Element] = Field(
        description="Placeholder element for filterParameter extensions",
        default=None,
        alias="_filterParameter",
    )
    filterDefinition: Optional[Uri] = Field(
        description="Canonical URL for a filterParameter definition",
        default=None,
    )
    filterDefinition_ext: Optional[Element] = Field(
        description="Placeholder element for filterDefinition extensions",
        default=None,
        alias="_filterDefinition",
    )
    comparator: Optional[List[Code]] = Field(
        description="eq | ne | gt | lt | ge | le | sa | eb | ap",
        default=None,
    )
    comparator_ext: Optional[Element] = Field(
        description="Placeholder element for comparator extensions",
        default=None,
        alias="_comparator",
    )
    modifier: Optional[List[Code]] = Field(
        description="missing | exact | contains | not | text | in | not-in | below | above | type | identifier | of-type | code-text | text-advanced | iterate",
        default=None,
    )
    modifier_ext: Optional[Element] = Field(
        description="Placeholder element for modifier extensions",
        default=None,
        alias="_modifier",
    )

    @field_validator(
        *(
            "modifier",
            "comparator",
            "filterDefinition",
            "filterParameter",
            "resource",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubscriptionTopicNotificationShape(BackboneElement):
    """
    List of properties to describe the shape (e.g., resources) included in notifications from this Subscription Topic.
    """

    resource: Optional[Uri] = Field(
        description="URL of the Resource that is the focus (main) resource in a notification shape",
        default=None,
    )
    resource_ext: Optional[Element] = Field(
        description="Placeholder element for resource extensions",
        default=None,
        alias="_resource",
    )
    include: Optional[List[String]] = Field(
        description="Include directives, rooted in the resource for this shape",
        default=None,
    )
    include_ext: Optional[Element] = Field(
        description="Placeholder element for include extensions",
        default=None,
        alias="_include",
    )
    revInclude: Optional[List[String]] = Field(
        description="Reverse include directives, rooted in the resource for this shape",
        default=None,
    )
    revInclude_ext: Optional[Element] = Field(
        description="Placeholder element for revInclude extensions",
        default=None,
        alias="_revInclude",
    )

    @field_validator(
        *(
            "revInclude",
            "include",
            "resource",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubscriptionTopic(DomainResource):
    """
    Describes a stream of resource state changes identified by trigger criteria and annotated with labels useful to filter projections from this topic.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/SubscriptionTopic"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this subscription topic, represented as an absolute URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier for subscription topic",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the subscription topic",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this subscription topic (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this subscription topic (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    derivedFrom: Optional[List[Canonical]] = Field(
        description="Based on FHIR protocol or definition",
        default=None,
    )
    derivedFrom_ext: Optional[Element] = Field(
        description="Placeholder element for derivedFrom extensions",
        default=None,
        alias="_derivedFrom",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="If for testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date status first applied",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="The name of the individual or organization that published the SubscriptionTopic",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the SubscriptionTopic",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="Content intends to support these contexts",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction of the SubscriptionTopic (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this SubscriptionTopic is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    approvalDate: Optional[Date] = Field(
        description="When SubscriptionTopic is/was approved by publisher",
        default=None,
    )
    approvalDate_ext: Optional[Element] = Field(
        description="Placeholder element for approvalDate extensions",
        default=None,
        alias="_approvalDate",
    )
    lastReviewDate: Optional[Date] = Field(
        description="Date the Subscription Topic was last reviewed by the publisher",
        default=None,
    )
    lastReviewDate_ext: Optional[Element] = Field(
        description="Placeholder element for lastReviewDate extensions",
        default=None,
        alias="_lastReviewDate",
    )
    effectivePeriod: Optional[Period] = Field(
        description="The effective date range for the SubscriptionTopic",
        default=None,
    )
    resourceTrigger: Optional[List[SubscriptionTopicResourceTrigger]] = Field(
        description="Definition of a resource-based trigger for the subscription topic",
        default=None,
    )
    eventTrigger: Optional[List[SubscriptionTopicEventTrigger]] = Field(
        description="Event definitions the SubscriptionTopic",
        default=None,
    )
    canFilterBy: Optional[List[SubscriptionTopicCanFilterBy]] = Field(
        description="Properties by which a Subscription can filter notifications from the SubscriptionTopic",
        default=None,
    )
    notificationShape: Optional[List[SubscriptionTopicNotificationShape]] = Field(
        description="Properties for describing the shape of notifications generated by this topic",
        default=None,
    )
    resourceType: Literal["SubscriptionTopic"] = Field(
        description=None,
        default="SubscriptionTopic",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "notificationShape",
            "canFilterBy",
            "eventTrigger",
            "resourceTrigger",
            "effectivePeriod",
            "lastReviewDate",
            "approvalDate",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "derivedFrom",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )
