# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Integer64,
    Instant,
    Canonical,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    BackboneElement,
    Reference,
    CodeableConcept,
    DomainResource,
)


class SubscriptionStatusNotificationEvent(BackboneElement):
    """
    Detailed information about events relevant to this subscription notification.
    """

    eventNumber: Optional[Integer64] = Field(
        description="Sequencing index of this event",
        default=None,
    )
    eventNumber_ext: Optional[Element] = Field(
        description="Placeholder element for eventNumber extensions",
        default=None,
        alias="_eventNumber",
    )
    timestamp: Optional[Instant] = Field(
        description="The instant this event occurred",
        default=None,
    )
    timestamp_ext: Optional[Element] = Field(
        description="Placeholder element for timestamp extensions",
        default=None,
        alias="_timestamp",
    )
    focus: Optional[Reference] = Field(
        description="Reference to the primary resource or information of this event",
        default=None,
    )
    additionalContext: Optional[List[Reference]] = Field(
        description="References related to the focus resource and/or context of this event",
        default=None,
    )

    @field_validator(
        *(
            "additionalContext",
            "focus",
            "timestamp",
            "eventNumber",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubscriptionStatus(DomainResource):
    """
    The SubscriptionStatus resource describes the state of a Subscription during notifications. It is not persisted.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/SubscriptionStatus"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    status: Optional[Code] = Field(
        description="requested | active | error | off | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[Code] = Field(
        description="handshake | heartbeat | event-notification | query-status | query-event",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    eventsSinceSubscriptionStart: Optional[Integer64] = Field(
        description="Events since the Subscription was created",
        default=None,
    )
    eventsSinceSubscriptionStart_ext: Optional[Element] = Field(
        description="Placeholder element for eventsSinceSubscriptionStart extensions",
        default=None,
        alias="_eventsSinceSubscriptionStart",
    )
    notificationEvent: Optional[List[SubscriptionStatusNotificationEvent]] = Field(
        description="Detailed information about any events relevant to this notification",
        default=None,
    )
    subscription: Optional[Reference] = Field(
        description="Reference to the Subscription responsible for this notification",
        default=None,
    )
    topic: Optional[Canonical] = Field(
        description="Reference to the SubscriptionTopic this notification relates to",
        default=None,
    )
    topic_ext: Optional[Element] = Field(
        description="Placeholder element for topic extensions",
        default=None,
        alias="_topic",
    )
    error: Optional[List[CodeableConcept]] = Field(
        description="List of errors on the subscription",
        default=None,
    )
    resourceType: Literal["SubscriptionStatus"] = Field(
        description=None,
        default="SubscriptionStatus",
    )

    @field_validator(
        *(
            "error",
            "topic",
            "subscription",
            "notificationEvent",
            "eventsSinceSubscriptionStart",
            "type",
            "status",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_sst_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="(type = 'event-notification' or type = 'query-event') implies notificationEvent.exists()",
            human="Event notifications must contain events",
            key="sst-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_sst_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="type = 'query-status' implies status.exists()",
            human="Status messages must contain status",
            key="sst-2",
            severity="error",
        )
