# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    Instant,
    Url,
    UnsignedInt,
    PositiveInt,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    ContactPoint,
    Reference,
    BackboneElement,
    Coding,
    DomainResource,
)


class SubscriptionFilterBy(BackboneElement):
    """
    The filter properties to be applied to narrow the subscription topic stream.  When multiple filters are applied, evaluates to true if all the conditions applicable to that resource are met; otherwise it returns false (i.e., logical AND).
    """

    resourceType: Optional[Uri] = Field(
        description="Allowed Resource (reference to definition) for this Subscription filter",
        default=None,
    )
    resourceType_ext: Optional[Element] = Field(
        description="Placeholder element for resourceType extensions",
        default=None,
        alias="_resourceType",
    )
    filterParameter: Optional[String] = Field(
        description="Filter label defined in SubscriptionTopic",
        default=None,
    )
    filterParameter_ext: Optional[Element] = Field(
        description="Placeholder element for filterParameter extensions",
        default=None,
        alias="_filterParameter",
    )
    comparator: Optional[Code] = Field(
        description="eq | ne | gt | lt | ge | le | sa | eb | ap",
        default=None,
    )
    comparator_ext: Optional[Element] = Field(
        description="Placeholder element for comparator extensions",
        default=None,
        alias="_comparator",
    )
    modifier: Optional[Code] = Field(
        description="missing | exact | contains | not | text | in | not-in | below | above | type | identifier | of-type | code-text | text-advanced | iterate",
        default=None,
    )
    modifier_ext: Optional[Element] = Field(
        description="Placeholder element for modifier extensions",
        default=None,
        alias="_modifier",
    )
    value: Optional[String] = Field(
        description="Literal value or resource path",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "modifier",
            "comparator",
            "filterParameter",
            "resourceType",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SubscriptionParameter(BackboneElement):
    """
    Channel-dependent information to send as part of the notification (e.g., HTTP Headers).
    """

    name: Optional[String] = Field(
        description="Name (key) of the parameter",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    value: Optional[String] = Field(
        description="Value of the parameter to use or pass through",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Subscription(DomainResource):
    """
    The subscription resource describes a particular client's request to be notified about a SubscriptionTopic.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Subscription"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifiers (business identifier)",
        default=None,
    )
    name: Optional[String] = Field(
        description="Human readable name for this subscription",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    status: Optional[Code] = Field(
        description="requested | active | error | off | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    topic: Optional[Canonical] = Field(
        description="Reference to the subscription topic being subscribed to",
        default=None,
    )
    topic_ext: Optional[Element] = Field(
        description="Placeholder element for topic extensions",
        default=None,
        alias="_topic",
    )
    contact: Optional[List[ContactPoint]] = Field(
        description="Contact details for source (e.g. troubleshooting)",
        default=None,
    )
    end: Optional[Instant] = Field(
        description="When to automatically delete the subscription",
        default=None,
    )
    end_ext: Optional[Element] = Field(
        description="Placeholder element for end extensions",
        default=None,
        alias="_end",
    )
    managingEntity: Optional[Reference] = Field(
        description="Entity responsible for Subscription changes",
        default=None,
    )
    reason: Optional[String] = Field(
        description="Description of why this subscription was created",
        default=None,
    )
    reason_ext: Optional[Element] = Field(
        description="Placeholder element for reason extensions",
        default=None,
        alias="_reason",
    )
    filterBy: Optional[List[SubscriptionFilterBy]] = Field(
        description="Criteria for narrowing the subscription topic stream",
        default=None,
    )
    channelType: Optional[Coding] = Field(
        description="Channel type for notifications",
        default=None,
    )
    endpoint: Optional[Url] = Field(
        description="Where the channel points to",
        default=None,
    )
    endpoint_ext: Optional[Element] = Field(
        description="Placeholder element for endpoint extensions",
        default=None,
        alias="_endpoint",
    )
    parameter: Optional[List[SubscriptionParameter]] = Field(
        description="Channel type",
        default=None,
    )
    heartbeatPeriod: Optional[UnsignedInt] = Field(
        description="Interval in seconds to send \u0027heartbeat\u0027 notification",
        default=None,
    )
    heartbeatPeriod_ext: Optional[Element] = Field(
        description="Placeholder element for heartbeatPeriod extensions",
        default=None,
        alias="_heartbeatPeriod",
    )
    timeout: Optional[UnsignedInt] = Field(
        description="Timeout in seconds to attempt notification delivery",
        default=None,
    )
    timeout_ext: Optional[Element] = Field(
        description="Placeholder element for timeout extensions",
        default=None,
        alias="_timeout",
    )
    contentType: Optional[Code] = Field(
        description="MIME type to send, or omit for no payload",
        default=None,
    )
    contentType_ext: Optional[Element] = Field(
        description="Placeholder element for contentType extensions",
        default=None,
        alias="_contentType",
    )
    content: Optional[Code] = Field(
        description="empty | id-only | full-resource",
        default=None,
    )
    content_ext: Optional[Element] = Field(
        description="Placeholder element for content extensions",
        default=None,
        alias="_content",
    )
    maxCount: Optional[PositiveInt] = Field(
        description="Maximum number of events that can be combined in a single notification",
        default=None,
    )
    maxCount_ext: Optional[Element] = Field(
        description="Placeholder element for maxCount extensions",
        default=None,
        alias="_maxCount",
    )
    resourceType: Literal["Subscription"] = Field(
        description=None,
        default="Subscription",
    )

    @field_validator(
        *(
            "maxCount",
            "content",
            "contentType",
            "timeout",
            "heartbeatPeriod",
            "parameter",
            "endpoint",
            "channelType",
            "filterBy",
            "reason",
            "managingEntity",
            "end",
            "contact",
            "topic",
            "status",
            "name",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("filterBy",), mode="after", check_fields=None)
    @classmethod
    def FHIR_scr_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="(comparator.exists() and modifier.exists()).not()",
            human="Subscription filters may only contain a modifier or a comparator",
            key="scr-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
