# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import String, Uri, Code, DateTime

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    CodeableConcept,
    Reference,
    BackboneElement,
    Period,
    Duration,
    Quantity,
    CodeableReference,
    Annotation,
    DomainResource,
)


class SpecimenFeature(BackboneElement):
    """
    A physical feature or landmark on a specimen, highlighted for context by the collector of the specimen (e.g. surgeon), that identifies the type of feature as well as its meaning (e.g. the red ink indicating the resection margin of the right lobe of the excised prostate tissue or wire loop at radiologically suspected tumor location).
    """

    type: Optional[CodeableConcept] = Field(
        description="Highlighted feature",
        default=None,
    )
    description: Optional[String] = Field(
        description="Information about the feature",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SpecimenCollection(BackboneElement):
    """
    Details concerning the specimen collection.
    """

    collector: Optional[Reference] = Field(
        description="Who collected the specimen",
        default=None,
    )
    collectedDateTime: Optional[DateTime] = Field(
        description="Collection time",
        default=None,
    )
    collectedDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for collectedDateTime extensions",
        default=None,
        alias="_collectedDateTime",
    )
    collectedPeriod: Optional[Period] = Field(
        description="Collection time",
        default=None,
    )
    duration: Optional[Duration] = Field(
        description="How long it took to collect specimen",
        default=None,
    )
    quantity: Optional[Quantity] = Field(
        description="The quantity of specimen collected",
        default=None,
    )
    method: Optional[CodeableConcept] = Field(
        description="Technique used to perform collection",
        default=None,
    )
    device: Optional[CodeableReference] = Field(
        description="Device used to perform collection",
        default=None,
    )
    procedure: Optional[Reference] = Field(
        description="The procedure that collects the specimen",
        default=None,
    )
    bodySite: Optional[CodeableReference] = Field(
        description="Anatomical collection site",
        default=None,
    )
    fastingStatusCodeableConcept: Optional[CodeableConcept] = Field(
        description="Whether or how long patient abstained from food and/or drink",
        default=None,
    )
    fastingStatusDuration: Optional[Duration] = Field(
        description="Whether or how long patient abstained from food and/or drink",
        default=None,
    )

    @property
    def collected(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="collected",
        )

    @property
    def fastingStatus(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="fastingStatus",
        )

    @field_validator(
        *(
            "bodySite",
            "procedure",
            "device",
            "method",
            "quantity",
            "duration",
            "collector",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def collected_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period],
            field_name_base="collected",
            required=False,
        )

    @model_validator(mode="after")
    def fastingStatus_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[CodeableConcept, Duration],
            field_name_base="fastingStatus",
            required=False,
        )


class SpecimenProcessing(BackboneElement):
    """
    Details concerning processing and processing steps for the specimen.
    """

    description: Optional[String] = Field(
        description="Textual description of procedure",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    method: Optional[CodeableConcept] = Field(
        description="Indicates the treatment step  applied to the specimen",
        default=None,
    )
    additive: Optional[List[Reference]] = Field(
        description="Material used in the processing step",
        default=None,
    )
    timeDateTime: Optional[DateTime] = Field(
        description="Date and time of specimen processing",
        default=None,
    )
    timeDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for timeDateTime extensions",
        default=None,
        alias="_timeDateTime",
    )
    timePeriod: Optional[Period] = Field(
        description="Date and time of specimen processing",
        default=None,
    )

    @property
    def time(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="time",
        )

    @field_validator(
        *(
            "additive",
            "method",
            "description",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def time_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Period],
            field_name_base="time",
            required=False,
        )


class SpecimenContainer(BackboneElement):
    """
    The container holding the specimen.  The recursive nature of containers; i.e. blood in tube in tray in rack is not addressed here.
    """

    device: Optional[Reference] = Field(
        description="Device resource for the container",
        default=None,
    )
    location: Optional[Reference] = Field(
        description="Where the container is",
        default=None,
    )
    specimenQuantity: Optional[Quantity] = Field(
        description="Quantity of specimen within container",
        default=None,
    )

    @field_validator(
        *(
            "specimenQuantity",
            "location",
            "device",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class Specimen(DomainResource):
    """
    A sample to be used for analysis.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/Specimen"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="External Identifier",
        default=None,
    )
    accessionIdentifier: Optional[Identifier] = Field(
        description="Identifier assigned by the lab",
        default=None,
    )
    status: Optional[Code] = Field(
        description="available | unavailable | unsatisfactory | entered-in-error",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    type: Optional[CodeableConcept] = Field(
        description="Kind of material that forms the specimen",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Where the specimen came from. This may be from patient(s), from a location (e.g., the source of an environmental sample), or a sampling of a substance, a biologically-derived product, or a device",
        default=None,
    )
    receivedTime: Optional[DateTime] = Field(
        description="The time when specimen is received by the testing laboratory",
        default=None,
    )
    receivedTime_ext: Optional[Element] = Field(
        description="Placeholder element for receivedTime extensions",
        default=None,
        alias="_receivedTime",
    )
    parent: Optional[List[Reference]] = Field(
        description="Specimen from which this specimen originated",
        default=None,
    )
    request: Optional[List[Reference]] = Field(
        description="Why the specimen was collected",
        default=None,
    )
    combined: Optional[Code] = Field(
        description="grouped | pooled",
        default=None,
    )
    combined_ext: Optional[Element] = Field(
        description="Placeholder element for combined extensions",
        default=None,
        alias="_combined",
    )
    role: Optional[List[CodeableConcept]] = Field(
        description="The role the specimen serves",
        default=None,
    )
    feature: Optional[List[SpecimenFeature]] = Field(
        description="The physical feature of a specimen",
        default=None,
    )
    collection: Optional[SpecimenCollection] = Field(
        description="Collection details",
        default=None,
    )
    processing: Optional[List[SpecimenProcessing]] = Field(
        description="Processing and processing step details",
        default=None,
    )
    container: Optional[List[SpecimenContainer]] = Field(
        description="Direct container of specimen (tube/slide, etc.)",
        default=None,
    )
    condition: Optional[List[CodeableConcept]] = Field(
        description="State of the specimen",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments",
        default=None,
    )
    resourceType: Literal["Specimen"] = Field(
        description=None,
        default="Specimen",
    )

    @field_validator(
        *(
            "note",
            "condition",
            "container",
            "processing",
            "collection",
            "feature",
            "role",
            "combined",
            "request",
            "parent",
            "receivedTime",
            "subject",
            "type",
            "status",
            "accessionIdentifier",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
