# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    Boolean,
    DateTime,
    Markdown,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Coding,
    ContactDetail,
    UsageContext,
    CodeableConcept,
    BackboneElement,
    DomainResource,
)


class SearchParameterComponent(BackboneElement):
    """
    Used to define the parts of a composite search parameter.
    """

    definition: Optional[Canonical] = Field(
        description="Defines how the part works",
        default=None,
    )
    definition_ext: Optional[Element] = Field(
        description="Placeholder element for definition extensions",
        default=None,
        alias="_definition",
    )
    expression: Optional[String] = Field(
        description="Subexpression relative to main expression",
        default=None,
    )
    expression_ext: Optional[Element] = Field(
        description="Placeholder element for expression extensions",
        default=None,
        alias="_expression",
    )

    @field_validator(
        *(
            "expression",
            "definition",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class SearchParameter(DomainResource):
    """
    A search parameter that defines a named search item that can be used to search/filter on a resource.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/SearchParameter"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this search parameter, represented as a URI (globally unique)",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Additional identifier for the search parameter (business identifier)",
        default=None,
    )
    version: Optional[String] = Field(
        description="Business version of the search parameter",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    versionAlgorithmString: Optional[String] = Field(
        description="How to compare versions",
        default=None,
    )
    versionAlgorithmString_ext: Optional[Element] = Field(
        description="Placeholder element for versionAlgorithmString extensions",
        default=None,
        alias="_versionAlgorithmString",
    )
    versionAlgorithmCoding: Optional[Coding] = Field(
        description="How to compare versions",
        default=None,
    )
    name: Optional[String] = Field(
        description="Name for this search parameter (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Name for this search parameter (human friendly)",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    derivedFrom: Optional[Canonical] = Field(
        description="Original definition for the search parameter",
        default=None,
    )
    derivedFrom_ext: Optional[Element] = Field(
        description="Placeholder element for derivedFrom extensions",
        default=None,
        alias="_derivedFrom",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    experimental: Optional[Boolean] = Field(
        description="For testing purposes, not real usage",
        default=None,
    )
    experimental_ext: Optional[Element] = Field(
        description="Placeholder element for experimental extensions",
        default=None,
        alias="_experimental",
    )
    date: Optional[DateTime] = Field(
        description="Date last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    publisher: Optional[String] = Field(
        description="Name of the publisher/steward (organization or individual)",
        default=None,
    )
    publisher_ext: Optional[Element] = Field(
        description="Placeholder element for publisher extensions",
        default=None,
        alias="_publisher",
    )
    contact: Optional[List[ContactDetail]] = Field(
        description="Contact details for the publisher",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Natural language description of the search parameter",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    useContext: Optional[List[UsageContext]] = Field(
        description="The context that the content is intended to support",
        default=None,
    )
    jurisdiction: Optional[List[CodeableConcept]] = Field(
        description="Intended jurisdiction for search parameter (if applicable)",
        default=None,
    )
    purpose: Optional[Markdown] = Field(
        description="Why this search parameter is defined",
        default=None,
    )
    purpose_ext: Optional[Element] = Field(
        description="Placeholder element for purpose extensions",
        default=None,
        alias="_purpose",
    )
    copyright: Optional[Markdown] = Field(
        description="Use and/or publishing restrictions",
        default=None,
    )
    copyright_ext: Optional[Element] = Field(
        description="Placeholder element for copyright extensions",
        default=None,
        alias="_copyright",
    )
    copyrightLabel: Optional[String] = Field(
        description="Copyright holder and year(s)",
        default=None,
    )
    copyrightLabel_ext: Optional[Element] = Field(
        description="Placeholder element for copyrightLabel extensions",
        default=None,
        alias="_copyrightLabel",
    )
    code: Optional[Code] = Field(
        description="Recommended name for parameter in search url",
        default=None,
    )
    code_ext: Optional[Element] = Field(
        description="Placeholder element for code extensions",
        default=None,
        alias="_code",
    )
    base: Optional[List[Code]] = Field(
        description="The resource type(s) this search parameter applies to",
        default=None,
    )
    base_ext: Optional[Element] = Field(
        description="Placeholder element for base extensions",
        default=None,
        alias="_base",
    )
    type: Optional[Code] = Field(
        description="number | date | string | token | reference | composite | quantity | uri | special",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    expression: Optional[String] = Field(
        description="FHIRPath expression that extracts the values",
        default=None,
    )
    expression_ext: Optional[Element] = Field(
        description="Placeholder element for expression extensions",
        default=None,
        alias="_expression",
    )
    processingMode: Optional[Code] = Field(
        description="normal | phonetic | other",
        default=None,
    )
    processingMode_ext: Optional[Element] = Field(
        description="Placeholder element for processingMode extensions",
        default=None,
        alias="_processingMode",
    )
    constraint: Optional[String] = Field(
        description="FHIRPath expression that constraints the usage of this SearchParamete",
        default=None,
    )
    constraint_ext: Optional[Element] = Field(
        description="Placeholder element for constraint extensions",
        default=None,
        alias="_constraint",
    )
    target: Optional[List[Code]] = Field(
        description="Types of resource (if a resource reference)",
        default=None,
    )
    target_ext: Optional[Element] = Field(
        description="Placeholder element for target extensions",
        default=None,
        alias="_target",
    )
    multipleOr: Optional[Boolean] = Field(
        description="Allow multiple values per parameter (or)",
        default=None,
    )
    multipleOr_ext: Optional[Element] = Field(
        description="Placeholder element for multipleOr extensions",
        default=None,
        alias="_multipleOr",
    )
    multipleAnd: Optional[Boolean] = Field(
        description="Allow multiple parameters (and)",
        default=None,
    )
    multipleAnd_ext: Optional[Element] = Field(
        description="Placeholder element for multipleAnd extensions",
        default=None,
        alias="_multipleAnd",
    )
    comparator: Optional[List[Code]] = Field(
        description="eq | ne | gt | lt | ge | le | sa | eb | ap",
        default=None,
    )
    comparator_ext: Optional[Element] = Field(
        description="Placeholder element for comparator extensions",
        default=None,
        alias="_comparator",
    )
    modifier: Optional[List[Code]] = Field(
        description="missing | exact | contains | not | text | in | not-in | below | above | type | identifier | of-type | code-text | text-advanced | iterate",
        default=None,
    )
    modifier_ext: Optional[Element] = Field(
        description="Placeholder element for modifier extensions",
        default=None,
        alias="_modifier",
    )
    chain: Optional[List[String]] = Field(
        description="Chained names supported",
        default=None,
    )
    chain_ext: Optional[Element] = Field(
        description="Placeholder element for chain extensions",
        default=None,
        alias="_chain",
    )
    component: Optional[List[SearchParameterComponent]] = Field(
        description="For Composite resources to define the parts",
        default=None,
    )
    resourceType: Literal["SearchParameter"] = Field(
        description=None,
        default="SearchParameter",
    )

    @property
    def versionAlgorithm(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="versionAlgorithm",
        )

    @field_validator(
        *(
            "component",
            "chain",
            "modifier",
            "comparator",
            "multipleAnd",
            "multipleOr",
            "target",
            "constraint",
            "processingMode",
            "expression",
            "type",
            "base",
            "code",
            "copyrightLabel",
            "copyright",
            "purpose",
            "jurisdiction",
            "useContext",
            "description",
            "contact",
            "publisher",
            "date",
            "experimental",
            "status",
            "derivedFrom",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("url",), mode="after", check_fields=None)
    @classmethod
    def FHIR_cnl_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="exists() implies matches('^[^|# ]+$')",
            human="URL should not contain | or # - these characters make processing canonical references problematic",
            key="cnl-1",
            severity="warning",
        )

    @model_validator(mode="after")
    def versionAlgorithm_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[String, Coding],
            field_name_base="versionAlgorithm",
            required=False,
        )

    @model_validator(mode="after")
    def FHIR_cnl_0_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="name.exists() implies name.matches('^[A-Z]([A-Za-z0-9_]){1,254}$')",
            human="Name should be usable as an identifier for the module by machine processing applications such as code generation",
            key="cnl-0",
            severity="warning",
        )

    @model_validator(mode="after")
    def FHIR_spd_1_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="expression.empty() or processingMode.exists()",
            human="If an expression is present, there SHALL be a processingMode",
            key="spd-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_spd_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="chain.empty() or type = 'reference'",
            human="Search parameters can only have chain names when the search parameter type is 'reference'",
            key="spd-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_spd_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="comparator.empty() or (type in ('number' | 'date' | 'quantity' | 'special'))",
            human="Search parameters comparator can only be used on type 'number', 'date', 'quantity' or 'special'.",
            key="spd-3",
            severity="error",
        )
