# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    DateTime,
    Markdown,
    Boolean,
    UnsignedInt,
    Id,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    BackboneElement,
    CodeableConcept,
    Reference,
    RelatedArtifact,
    CodeableReference,
    Period,
    Annotation,
    DomainResource,
)


class ResearchStudyLabel(BackboneElement):
    """
    Additional names for the study.
    """

    type: Optional[CodeableConcept] = Field(
        description="primary | official | scientific | plain-language | subtitle | short-title | acronym | earlier-title | language | auto-translated | human-use | machine-use | duplicate-uid",
        default=None,
    )
    value: Optional[String] = Field(
        description="The name",
        default=None,
    )
    value_ext: Optional[Element] = Field(
        description="Placeholder element for value extensions",
        default=None,
        alias="_value",
    )

    @field_validator(
        *(
            "value",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ResearchStudyAssociatedParty(BackboneElement):
    """
    Sponsors, collaborators, and other parties.
    """

    name: Optional[String] = Field(
        description="Name of associated party",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    role: Optional[CodeableConcept] = Field(
        description="sponsor | lead-sponsor | sponsor-investigator | primary-investigator | collaborator | funding-source | general-contact | recruitment-contact | sub-investigator | study-director | study-chair",
        default=None,
    )
    period: Optional[List[Period]] = Field(
        description="When active in the role",
        default=None,
    )
    classifier: Optional[List[CodeableConcept]] = Field(
        description="nih | fda | government | nonprofit | academic | industry",
        default=None,
    )
    party: Optional[Reference] = Field(
        description="Individual or organization associated with study (use practitionerRole to specify their organisation)",
        default=None,
    )

    @field_validator(
        *(
            "party",
            "classifier",
            "period",
            "role",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ResearchStudyProgressStatus(BackboneElement):
    """
    Status of study with time for that status.
    """

    state: Optional[CodeableConcept] = Field(
        description="Label for status or state (e.g. recruitment status)",
        default=None,
    )
    actual: Optional[Boolean] = Field(
        description="Actual if true else anticipated",
        default=None,
    )
    actual_ext: Optional[Element] = Field(
        description="Placeholder element for actual extensions",
        default=None,
        alias="_actual",
    )
    period: Optional[Period] = Field(
        description="Date range",
        default=None,
    )

    @field_validator(
        *(
            "period",
            "actual",
            "state",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ResearchStudyRecruitment(BackboneElement):
    """
    Target or actual group of participants enrolled in study.
    """

    targetNumber: Optional[UnsignedInt] = Field(
        description="Estimated total number of participants to be enrolled",
        default=None,
    )
    targetNumber_ext: Optional[Element] = Field(
        description="Placeholder element for targetNumber extensions",
        default=None,
        alias="_targetNumber",
    )
    actualNumber: Optional[UnsignedInt] = Field(
        description="Actual total number of participants enrolled in study",
        default=None,
    )
    actualNumber_ext: Optional[Element] = Field(
        description="Placeholder element for actualNumber extensions",
        default=None,
        alias="_actualNumber",
    )
    eligibility: Optional[Reference] = Field(
        description="Inclusion and exclusion criteria",
        default=None,
    )
    actualGroup: Optional[Reference] = Field(
        description="Group of participants who were enrolled in study",
        default=None,
    )

    @field_validator(
        *(
            "actualGroup",
            "eligibility",
            "actualNumber",
            "targetNumber",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ResearchStudyComparisonGroup(BackboneElement):
    """
    Describes an expected event or sequence of events for one of the subjects of a study. E.g. for a living subject: exposure to drug A, wash-out, exposure to drug B, wash-out, follow-up. E.g. for a stability study: {store sample from lot A at 25 degrees for 1 month}, {store sample from lot A at 40 degrees for 1 month}.
    """

    linkId: Optional[Id] = Field(
        description="Allows the comparisonGroup for the study and the comparisonGroup for the subject to be linked easily",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    name: Optional[String] = Field(
        description="Label for study comparisonGroup",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[CodeableConcept] = Field(
        description="Categorization of study comparisonGroup",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Short explanation of study path",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    intendedExposure: Optional[List[Reference]] = Field(
        description="Interventions or exposures in this comparisonGroup or cohort",
        default=None,
    )
    observedGroup: Optional[Reference] = Field(
        description="Group of participants who were enrolled in study comparisonGroup",
        default=None,
    )

    @field_validator(
        *(
            "observedGroup",
            "intendedExposure",
            "description",
            "type",
            "name",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ResearchStudyObjective(BackboneElement):
    """
    A goal that the study is aiming to achieve in terms of a scientific question to be answered by the analysis of data collected during the study.
    """

    name: Optional[String] = Field(
        description="Label for the objective",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[CodeableConcept] = Field(
        description="primary | secondary | exploratory",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Description of the objective",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )

    @field_validator(
        *(
            "description",
            "type",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ResearchStudyOutcomeMeasure(BackboneElement):
    """
    An "outcome measure", "endpoint", "effect measure" or "measure of effect" is a specific measurement or observation used to quantify the effect of experimental variables on the participants in a study, or for observational studies, to describe patterns of diseases or traits or associations with exposures, risk factors or treatment.
    """

    name: Optional[String] = Field(
        description="Label for the outcome",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    type: Optional[List[CodeableConcept]] = Field(
        description="primary | secondary | exploratory",
        default=None,
    )
    description: Optional[Markdown] = Field(
        description="Description of the outcome",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    reference: Optional[Reference] = Field(
        description="Structured outcome definition",
        default=None,
    )

    @field_validator(
        *(
            "reference",
            "description",
            "type",
            "name",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class ResearchStudy(DomainResource):
    """
    A scientific study of nature that sometimes includes processes involved in health and disease. For example, clinical trials are research studies that involve people. These studies may be related to new ways to screen, prevent, diagnose, and treat disease. They may also study certain outcomes and certain groups of people by looking at data collected in the past or future.
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/ResearchStudy"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    url: Optional[Uri] = Field(
        description="Canonical identifier for this study resource",
        default=None,
    )
    url_ext: Optional[Element] = Field(
        description="Placeholder element for url extensions",
        default=None,
        alias="_url",
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business Identifier for study",
        default=None,
    )
    version: Optional[String] = Field(
        description="The business version for the study record",
        default=None,
    )
    version_ext: Optional[Element] = Field(
        description="Placeholder element for version extensions",
        default=None,
        alias="_version",
    )
    name: Optional[String] = Field(
        description="Name for this study (computer friendly)",
        default=None,
    )
    name_ext: Optional[Element] = Field(
        description="Placeholder element for name extensions",
        default=None,
        alias="_name",
    )
    title: Optional[String] = Field(
        description="Human readable name of the study",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    label: Optional[List[ResearchStudyLabel]] = Field(
        description="Additional names for the study",
        default=None,
    )
    protocol: Optional[List[Reference]] = Field(
        description="Steps followed in executing study",
        default=None,
    )
    partOf: Optional[List[Reference]] = Field(
        description="Part of larger study",
        default=None,
    )
    relatedArtifact: Optional[List[RelatedArtifact]] = Field(
        description="References, URLs, and attachments",
        default=None,
    )
    date: Optional[DateTime] = Field(
        description="Date the resource last changed",
        default=None,
    )
    date_ext: Optional[Element] = Field(
        description="Placeholder element for date extensions",
        default=None,
        alias="_date",
    )
    status: Optional[Code] = Field(
        description="draft | active | retired | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    primaryPurposeType: Optional[CodeableConcept] = Field(
        description="treatment | prevention | diagnostic | supportive-care | screening | health-services-research | basic-science | device-feasibility",
        default=None,
    )
    phase: Optional[CodeableConcept] = Field(
        description="n-a | early-phase-1 | phase-1 | phase-1-phase-2 | phase-2 | phase-2-phase-3 | phase-3 | phase-4",
        default=None,
    )
    studyDesign: Optional[List[CodeableConcept]] = Field(
        description="Classifications of the study design characteristics",
        default=None,
    )
    focus: Optional[List[CodeableReference]] = Field(
        description="Drugs, devices, etc. under study",
        default=None,
    )
    condition: Optional[List[CodeableConcept]] = Field(
        description="Condition being studied",
        default=None,
    )
    keyword: Optional[List[CodeableConcept]] = Field(
        description="Used to search for the study",
        default=None,
    )
    region: Optional[List[CodeableConcept]] = Field(
        description="Geographic area for the study",
        default=None,
    )
    descriptionSummary: Optional[Markdown] = Field(
        description="Brief text explaining the study",
        default=None,
    )
    descriptionSummary_ext: Optional[Element] = Field(
        description="Placeholder element for descriptionSummary extensions",
        default=None,
        alias="_descriptionSummary",
    )
    description: Optional[Markdown] = Field(
        description="Detailed narrative of the study",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    period: Optional[Period] = Field(
        description="When the study began and ended",
        default=None,
    )
    site: Optional[List[Reference]] = Field(
        description="Facility where study activities are conducted",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Comments made about the study",
        default=None,
    )
    classifier: Optional[List[CodeableConcept]] = Field(
        description="Classification for the study",
        default=None,
    )
    associatedParty: Optional[List[ResearchStudyAssociatedParty]] = Field(
        description="Sponsors, collaborators, and other parties",
        default=None,
    )
    progressStatus: Optional[List[ResearchStudyProgressStatus]] = Field(
        description="Status of study with time for that status",
        default=None,
    )
    whyStopped: Optional[CodeableConcept] = Field(
        description="accrual-goal-met | closed-due-to-toxicity | closed-due-to-lack-of-study-progress | temporarily-closed-per-study-design",
        default=None,
    )
    recruitment: Optional[ResearchStudyRecruitment] = Field(
        description="Target or actual group of participants enrolled in study",
        default=None,
    )
    comparisonGroup: Optional[List[ResearchStudyComparisonGroup]] = Field(
        description="Defined path through the study for a subject",
        default=None,
    )
    objective: Optional[List[ResearchStudyObjective]] = Field(
        description="A goal for the study",
        default=None,
    )
    outcomeMeasure: Optional[List[ResearchStudyOutcomeMeasure]] = Field(
        description="A variable measured during the study",
        default=None,
    )
    result: Optional[List[Reference]] = Field(
        description="Link to results generated during the study",
        default=None,
    )
    resourceType: Literal["ResearchStudy"] = Field(
        description=None,
        default="ResearchStudy",
    )

    @field_validator(
        *(
            "result",
            "outcomeMeasure",
            "objective",
            "comparisonGroup",
            "recruitment",
            "whyStopped",
            "progressStatus",
            "associatedParty",
            "classifier",
            "note",
            "site",
            "period",
            "description",
            "descriptionSummary",
            "region",
            "keyword",
            "condition",
            "focus",
            "studyDesign",
            "phase",
            "primaryPurposeType",
            "status",
            "date",
            "relatedArtifact",
            "partOf",
            "protocol",
            "label",
            "title",
            "name",
            "version",
            "identifier",
            "url",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
