# Fhircraft modules
import fhircraft
from fhircraft.utils import model_rebuild_all
from fhircraft.fhir.resources.datatypes.primitives import *
from fhircraft.fhir.resources.base import FHIRBaseModel
import fhircraft.fhir.resources.validators as fhir_validators

# Pydantic modules
from pydantic import Field, field_validator, model_validator, BaseModel
from pydantic.fields import FieldInfo

# Standard modules
from typing import Optional, Literal, Union
from enum import Enum

NoneType = type(None)

# Dynamic modules

from fhircraft.fhir.resources.base import FHIRBaseModel

from typing import Optional, List, Literal

from fhircraft.fhir.resources.datatypes.primitives import (
    String,
    Uri,
    Code,
    Canonical,
    DateTime,
    Markdown,
    Id,
)

from fhircraft.fhir.resources.datatypes.R5.complex import (
    Element,
    Meta,
    Narrative,
    Resource,
    Extension,
    Identifier,
    Reference,
    CodeableConcept,
    CodeableReference,
    Annotation,
    BackboneElement,
    RelatedArtifact,
    Expression,
    DataRequirement,
    Duration,
    Range,
    Age,
    Period,
    Timing,
    DomainResource,
)


class RequestOrchestrationActionCondition(BackboneElement):
    """
    An expression that describes applicability criteria, or start/stop conditions for the action.
    """

    kind: Optional[Code] = Field(
        description="applicability | start | stop",
        default=None,
    )
    kind_ext: Optional[Element] = Field(
        description="Placeholder element for kind extensions",
        default=None,
        alias="_kind",
    )
    expression: Optional[Expression] = Field(
        description="Boolean-valued expression",
        default=None,
    )

    @field_validator(
        *(
            "expression",
            "kind",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class RequestOrchestrationActionInput(BackboneElement):
    """
    Defines input data requirements for the action.
    """

    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    requirement: Optional[DataRequirement] = Field(
        description="What data is provided",
        default=None,
    )
    relatedData: Optional[Id] = Field(
        description="What data is provided",
        default=None,
    )
    relatedData_ext: Optional[Element] = Field(
        description="Placeholder element for relatedData extensions",
        default=None,
        alias="_relatedData",
    )

    @field_validator(
        *(
            "relatedData",
            "requirement",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class RequestOrchestrationActionOutput(BackboneElement):
    """
    Defines the outputs of the action, if any.
    """

    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    requirement: Optional[DataRequirement] = Field(
        description="What data is provided",
        default=None,
    )
    relatedData: Optional[String] = Field(
        description="What data is provided",
        default=None,
    )
    relatedData_ext: Optional[Element] = Field(
        description="Placeholder element for relatedData extensions",
        default=None,
        alias="_relatedData",
    )

    @field_validator(
        *(
            "relatedData",
            "requirement",
            "title",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class RequestOrchestrationActionRelatedAction(BackboneElement):
    """
    A relationship to another action such as "before" or "30-60 minutes after start of".
    """

    targetId: Optional[Id] = Field(
        description="What action this is related to",
        default=None,
    )
    targetId_ext: Optional[Element] = Field(
        description="Placeholder element for targetId extensions",
        default=None,
        alias="_targetId",
    )
    relationship: Optional[Code] = Field(
        description="before | before-start | before-end | concurrent | concurrent-with-start | concurrent-with-end | after | after-start | after-end",
        default=None,
    )
    relationship_ext: Optional[Element] = Field(
        description="Placeholder element for relationship extensions",
        default=None,
        alias="_relationship",
    )
    endRelationship: Optional[Code] = Field(
        description="before | before-start | before-end | concurrent | concurrent-with-start | concurrent-with-end | after | after-start | after-end",
        default=None,
    )
    endRelationship_ext: Optional[Element] = Field(
        description="Placeholder element for endRelationship extensions",
        default=None,
        alias="_endRelationship",
    )
    offsetDuration: Optional[Duration] = Field(
        description="Time offset for the relationship",
        default=None,
    )
    offsetRange: Optional[Range] = Field(
        description="Time offset for the relationship",
        default=None,
    )

    @property
    def offset(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="offset",
        )

    @field_validator(
        *(
            "endRelationship",
            "relationship",
            "targetId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def offset_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Duration, Range],
            field_name_base="offset",
            required=False,
        )


class RequestOrchestrationActionParticipant(BackboneElement):
    """
    The participant that should perform or be responsible for this action.
    """

    type: Optional[Code] = Field(
        description="careteam | device | group | healthcareservice | location | organization | patient | practitioner | practitionerrole | relatedperson",
        default=None,
    )
    type_ext: Optional[Element] = Field(
        description="Placeholder element for type extensions",
        default=None,
        alias="_type",
    )
    typeCanonical: Optional[Canonical] = Field(
        description="Who or what can participate",
        default=None,
    )
    typeCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for typeCanonical extensions",
        default=None,
        alias="_typeCanonical",
    )
    typeReference: Optional[Reference] = Field(
        description="Who or what can participate",
        default=None,
    )
    role: Optional[CodeableConcept] = Field(
        description="E.g. Nurse, Surgeon, Parent, etc",
        default=None,
    )
    function: Optional[CodeableConcept] = Field(
        description="E.g. Author, Reviewer, Witness, etc",
        default=None,
    )
    actorCanonical: Optional[Canonical] = Field(
        description="Who/what is participating?",
        default=None,
    )
    actorCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for actorCanonical extensions",
        default=None,
        alias="_actorCanonical",
    )
    actorReference: Optional[Reference] = Field(
        description="Who/what is participating?",
        default=None,
    )

    @property
    def actor(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="actor",
        )

    @field_validator(
        *(
            "function",
            "role",
            "typeReference",
            "typeCanonical",
            "type",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @model_validator(mode="after")
    def actor_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Canonical, Reference],
            field_name_base="actor",
            required=False,
        )


class RequestOrchestrationActionDynamicValue(BackboneElement):
    """
    Customizations that should be applied to the statically defined resource. For example, if the dosage of a medication must be computed based on the patient's weight, a customization would be used to specify an expression that calculated the weight, and the path on the resource that would contain the result.
    """

    path: Optional[String] = Field(
        description="The path to the element to be set dynamically",
        default=None,
    )
    path_ext: Optional[Element] = Field(
        description="Placeholder element for path extensions",
        default=None,
        alias="_path",
    )
    expression: Optional[Expression] = Field(
        description="An expression that provides the dynamic value for the customization",
        default=None,
    )

    @field_validator(
        *(
            "expression",
            "path",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )


class RequestOrchestrationAction(BackboneElement):
    """
    The actions, if any, produced by the evaluation of the artifact.
    """

    linkId: Optional[String] = Field(
        description="Pointer to specific item from the PlanDefinition",
        default=None,
    )
    linkId_ext: Optional[Element] = Field(
        description="Placeholder element for linkId extensions",
        default=None,
        alias="_linkId",
    )
    prefix: Optional[String] = Field(
        description="User-visible prefix for the action (e.g. 1. or A.)",
        default=None,
    )
    prefix_ext: Optional[Element] = Field(
        description="Placeholder element for prefix extensions",
        default=None,
        alias="_prefix",
    )
    title: Optional[String] = Field(
        description="User-visible title",
        default=None,
    )
    title_ext: Optional[Element] = Field(
        description="Placeholder element for title extensions",
        default=None,
        alias="_title",
    )
    description: Optional[Markdown] = Field(
        description="Short description of the action",
        default=None,
    )
    description_ext: Optional[Element] = Field(
        description="Placeholder element for description extensions",
        default=None,
        alias="_description",
    )
    textEquivalent: Optional[Markdown] = Field(
        description="Static text equivalent of the action, used if the dynamic aspects cannot be interpreted by the receiving system",
        default=None,
    )
    textEquivalent_ext: Optional[Element] = Field(
        description="Placeholder element for textEquivalent extensions",
        default=None,
        alias="_textEquivalent",
    )
    priority: Optional[Code] = Field(
        description="routine | urgent | asap | stat",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    code: Optional[List[CodeableConcept]] = Field(
        description="Code representing the meaning of the action or sub-actions",
        default=None,
    )
    documentation: Optional[List[RelatedArtifact]] = Field(
        description="Supporting documentation for the intended performer of the action",
        default=None,
    )
    goal: Optional[List[Reference]] = Field(
        description="What goals",
        default=None,
    )
    condition: Optional[List[RequestOrchestrationActionCondition]] = Field(
        description="Whether or not the action is applicable",
        default=None,
    )
    input: Optional[List[RequestOrchestrationActionInput]] = Field(
        description="Input data requirements",
        default=None,
    )
    output: Optional[List[RequestOrchestrationActionOutput]] = Field(
        description="Output data definition",
        default=None,
    )
    relatedAction: Optional[List[RequestOrchestrationActionRelatedAction]] = Field(
        description="Relationship to another action",
        default=None,
    )
    timingDateTime: Optional[DateTime] = Field(
        description="When the action should take place",
        default=None,
    )
    timingDateTime_ext: Optional[Element] = Field(
        description="Placeholder element for timingDateTime extensions",
        default=None,
        alias="_timingDateTime",
    )
    timingAge: Optional[Age] = Field(
        description="When the action should take place",
        default=None,
    )
    timingPeriod: Optional[Period] = Field(
        description="When the action should take place",
        default=None,
    )
    timingDuration: Optional[Duration] = Field(
        description="When the action should take place",
        default=None,
    )
    timingRange: Optional[Range] = Field(
        description="When the action should take place",
        default=None,
    )
    timingTiming: Optional[Timing] = Field(
        description="When the action should take place",
        default=None,
    )
    location: Optional[CodeableReference] = Field(
        description="Where it should happen",
        default=None,
    )
    participant: Optional[List[RequestOrchestrationActionParticipant]] = Field(
        description="Who should perform the action",
        default=None,
    )
    type: Optional[CodeableConcept] = Field(
        description="create | update | remove | fire-event",
        default=None,
    )
    groupingBehavior: Optional[Code] = Field(
        description="visual-group | logical-group | sentence-group",
        default=None,
    )
    groupingBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for groupingBehavior extensions",
        default=None,
        alias="_groupingBehavior",
    )
    selectionBehavior: Optional[Code] = Field(
        description="any | all | all-or-none | exactly-one | at-most-one | one-or-more",
        default=None,
    )
    selectionBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for selectionBehavior extensions",
        default=None,
        alias="_selectionBehavior",
    )
    requiredBehavior: Optional[Code] = Field(
        description="must | could | must-unless-documented",
        default=None,
    )
    requiredBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for requiredBehavior extensions",
        default=None,
        alias="_requiredBehavior",
    )
    precheckBehavior: Optional[Code] = Field(
        description="yes | no",
        default=None,
    )
    precheckBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for precheckBehavior extensions",
        default=None,
        alias="_precheckBehavior",
    )
    cardinalityBehavior: Optional[Code] = Field(
        description="single | multiple",
        default=None,
    )
    cardinalityBehavior_ext: Optional[Element] = Field(
        description="Placeholder element for cardinalityBehavior extensions",
        default=None,
        alias="_cardinalityBehavior",
    )
    resource: Optional[Reference] = Field(
        description="The target of the action",
        default=None,
    )
    definitionCanonical: Optional[Canonical] = Field(
        description="Description of the activity to be performed",
        default=None,
    )
    definitionCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for definitionCanonical extensions",
        default=None,
        alias="_definitionCanonical",
    )
    definitionUri: Optional[Uri] = Field(
        description="Description of the activity to be performed",
        default=None,
    )
    definitionUri_ext: Optional[Element] = Field(
        description="Placeholder element for definitionUri extensions",
        default=None,
        alias="_definitionUri",
    )
    transform: Optional[Canonical] = Field(
        description="Transform to apply the template",
        default=None,
    )
    transform_ext: Optional[Element] = Field(
        description="Placeholder element for transform extensions",
        default=None,
        alias="_transform",
    )
    dynamicValue: Optional[List[RequestOrchestrationActionDynamicValue]] = Field(
        description="Dynamic aspects of the definition",
        default=None,
    )
    action: Optional[List["RequestOrchestrationAction"]] = Field(
        description="Sub action",
        default=None,
    )

    @property
    def timing(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="timing",
        )

    @property
    def definition(self):
        return fhir_validators.get_type_choice_value_by_base(
            self,
            base="definition",
        )

    @field_validator(
        *(
            "action",
            "dynamicValue",
            "transform",
            "resource",
            "cardinalityBehavior",
            "precheckBehavior",
            "requiredBehavior",
            "selectionBehavior",
            "groupingBehavior",
            "type",
            "participant",
            "location",
            "relatedAction",
            "output",
            "input",
            "condition",
            "goal",
            "documentation",
            "code",
            "priority",
            "textEquivalent",
            "description",
            "title",
            "prefix",
            "linkId",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
            "modifierExtension",
            "extension",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(*("input",), mode="after", check_fields=None)
    @classmethod
    def FHIR_pld_0_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="requirement.exists() xor relatedData.exists()",
            human="Input data elements must have a requirement or a relatedData, but not both",
            key="pld-0",
            severity="error",
        )

    @field_validator(*("output",), mode="after", check_fields=None)
    @classmethod
    def FHIR_pld_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="requirement.exists() xor relatedData.exists()",
            human="Output data element must have a requirement or a relatedData, but not both",
            key="pld-1",
            severity="error",
        )

    @model_validator(mode="after")
    def timing_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[DateTime, Age, Period, Duration, Range, Timing],
            field_name_base="timing",
            required=False,
        )

    @model_validator(mode="after")
    def definition_type_choice_validator(self):
        return fhir_validators.validate_type_choice_element(
            self,
            field_types=[Canonical, Uri],
            field_name_base="definition",
            required=False,
        )


class RequestOrchestration(DomainResource):
    """
    A set of related requests that can be used to capture intended activities that have inter-dependencies such as "give this medication after that one".
    """

    id: Optional[String] = Field(
        description="Logical id of this artifact",
        default=None,
    )
    id_ext: Optional[Element] = Field(
        description="Placeholder element for id extensions",
        default=None,
        alias="_id",
    )
    meta: Optional[Meta] = Field(
        description="Metadata about the resource.",
        default_factory=lambda: Meta(
            profile=["http://hl7.org/fhir/StructureDefinition/RequestOrchestration"]
        ),
    )
    implicitRules: Optional[Uri] = Field(
        description="A set of rules under which this content was created",
        default=None,
    )
    implicitRules_ext: Optional[Element] = Field(
        description="Placeholder element for implicitRules extensions",
        default=None,
        alias="_implicitRules",
    )
    language: Optional[Code] = Field(
        description="Language of the resource content",
        default=None,
    )
    language_ext: Optional[Element] = Field(
        description="Placeholder element for language extensions",
        default=None,
        alias="_language",
    )
    text: Optional[Narrative] = Field(
        description="Text summary of the resource, for human interpretation",
        default=None,
    )
    contained: Optional[List[Resource]] = Field(
        description="Contained, inline Resources",
        default=None,
    )
    extension: Optional[List[Extension]] = Field(
        description="Additional content defined by implementations",
        default=None,
    )
    modifierExtension: Optional[List[Extension]] = Field(
        description="Extensions that cannot be ignored",
        default=None,
    )
    identifier: Optional[List[Identifier]] = Field(
        description="Business identifier",
        default=None,
    )
    instantiatesCanonical: Optional[List[Canonical]] = Field(
        description="Instantiates FHIR protocol or definition",
        default=None,
    )
    instantiatesCanonical_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesCanonical extensions",
        default=None,
        alias="_instantiatesCanonical",
    )
    instantiatesUri: Optional[List[Uri]] = Field(
        description="Instantiates external protocol or definition",
        default=None,
    )
    instantiatesUri_ext: Optional[Element] = Field(
        description="Placeholder element for instantiatesUri extensions",
        default=None,
        alias="_instantiatesUri",
    )
    basedOn: Optional[List[Reference]] = Field(
        description="Fulfills plan, proposal, or order",
        default=None,
    )
    replaces: Optional[List[Reference]] = Field(
        description="Request(s) replaced by this request",
        default=None,
    )
    groupIdentifier: Optional[Identifier] = Field(
        description="Composite request this is part of",
        default=None,
    )
    status: Optional[Code] = Field(
        description="draft | active | on-hold | revoked | completed | entered-in-error | unknown",
        default=None,
    )
    status_ext: Optional[Element] = Field(
        description="Placeholder element for status extensions",
        default=None,
        alias="_status",
    )
    intent: Optional[Code] = Field(
        description="proposal | plan | directive | order | original-order | reflex-order | filler-order | instance-order | option",
        default=None,
    )
    intent_ext: Optional[Element] = Field(
        description="Placeholder element for intent extensions",
        default=None,
        alias="_intent",
    )
    priority: Optional[Code] = Field(
        description="routine | urgent | asap | stat",
        default=None,
    )
    priority_ext: Optional[Element] = Field(
        description="Placeholder element for priority extensions",
        default=None,
        alias="_priority",
    )
    code: Optional[CodeableConcept] = Field(
        description="What\u0027s being requested/ordered",
        default=None,
    )
    subject: Optional[Reference] = Field(
        description="Who the request orchestration is about",
        default=None,
    )
    encounter: Optional[Reference] = Field(
        description="Created as part of",
        default=None,
    )
    authoredOn: Optional[DateTime] = Field(
        description="When the request orchestration was authored",
        default=None,
    )
    authoredOn_ext: Optional[Element] = Field(
        description="Placeholder element for authoredOn extensions",
        default=None,
        alias="_authoredOn",
    )
    author: Optional[Reference] = Field(
        description="Device or practitioner that authored the request orchestration",
        default=None,
    )
    reason: Optional[List[CodeableReference]] = Field(
        description="Why the request orchestration is needed",
        default=None,
    )
    goal: Optional[List[Reference]] = Field(
        description="What goals",
        default=None,
    )
    note: Optional[List[Annotation]] = Field(
        description="Additional notes about the response",
        default=None,
    )
    action: Optional[List[RequestOrchestrationAction]] = Field(
        description="Proposed actions, if any",
        default=None,
    )
    resourceType: Literal["RequestOrchestration"] = Field(
        description=None,
        default="RequestOrchestration",
    )

    @field_validator(
        *(
            "action",
            "note",
            "goal",
            "reason",
            "author",
            "authoredOn",
            "encounter",
            "subject",
            "code",
            "priority",
            "intent",
            "status",
            "groupIdentifier",
            "replaces",
            "basedOn",
            "instantiatesUri",
            "instantiatesCanonical",
            "identifier",
            "modifierExtension",
            "extension",
            "text",
            "language",
            "implicitRules",
            "meta",
        ),
        mode="after",
        check_fields=None,
    )
    @classmethod
    def FHIR_ele_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="hasValue() or (children().count() > id.count())",
            human="All FHIR elements must have a @value or children",
            key="ele-1",
            severity="error",
        )

    @field_validator(
        *("modifierExtension", "extension"), mode="after", check_fields=None
    )
    @classmethod
    def FHIR_ext_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="extension.exists() != value.exists()",
            human="Must have either extensions or value[x], not both",
            key="ext-1",
            severity="error",
        )

    @field_validator(*("action",), mode="after", check_fields=None)
    @classmethod
    def FHIR_rqg_1_constraint_validator(cls, value):
        return fhir_validators.validate_element_constraint(
            cls,
            value,
            expression="resource.exists() != action.exists()",
            human="Must have resource or action but not both",
            key="rqg-1",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_2_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.contained.empty()",
            human="If the resource is contained in another resource, it SHALL NOT contain nested Resources",
            key="dom-2",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_3_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.where((('#'+id in (%resource.descendants().reference | %resource.descendants().ofType(canonical) | %resource.descendants().ofType(uri) | %resource.descendants().ofType(url))) or descendants().where(reference = '#').exists() or descendants().where(ofType(canonical) = '#').exists() or descendants().where(ofType(canonical) = '#').exists()).not()).trace('unmatched', id).empty()",
            human="If the resource is contained in another resource, it SHALL be referred to from elsewhere in the resource or SHALL refer to the containing resource",
            key="dom-3",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_4_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.versionId.empty() and contained.meta.lastUpdated.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a meta.versionId or a meta.lastUpdated",
            key="dom-4",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_5_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="contained.meta.security.empty()",
            human="If a resource is contained in another resource, it SHALL NOT have a security label",
            key="dom-5",
            severity="error",
        )

    @model_validator(mode="after")
    def FHIR_dom_6_constraint_model_validator(self):
        return fhir_validators.validate_model_constraint(
            self,
            expression="text.`div`.exists()",
            human="A resource should have narrative for robust management",
            key="dom-6",
            severity="warning",
        )
